<?php

/**
 * Template parts.
 *
 * @package     Blogun
 * @author      Peregrine Themes
 * @since       1.0.0
 */

/**
 * Do not allow direct script access.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds the meta tag to the site header.
 *
 * @since 1.0.0
 */
function blogun_meta_viewport() {
	echo '<meta name="viewport" content="width=device-width, initial-scale=1">';
}
add_action( 'wp_head', 'blogun_meta_viewport', 1 );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 *
 * @since 1.0.0
 */
function blogun_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">' . "\n", esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'blogun_pingback_header' );

/**
 * Adds the meta tag for website accent color.
 *
 * @since 1.0.0
 */
function blogun_meta_theme_color() {

	$color = blogun_option( 'accent_color' );

	if ( $color ) {
		printf( '<meta name="theme-color" content="%s">', esc_attr( $color ) );
	}
}
add_action( 'wp_head', 'blogun_meta_theme_color' );

/**
 * Outputs the theme top bar area.
 *
 * @since 1.0.0
 */
function blogun_topbar_output() {

	if ( ! blogun_is_top_bar_displayed() ) {
		return;
	}

	get_template_part( 'template-parts/topbar/topbar' );
}
add_action( 'blogun_header', 'blogun_topbar_output', 10 );

/**
 * Outputs the top bar widgets.
 *
 * @since 1.0.0
 * @param string $location Widget location in top bar.
 */
function blogun_topbar_widgets_output( $location ) {

	do_action( 'blogun_top_bar_widgets_before_' . $location );

	$blogun_top_bar_widgets = blogun_option( 'top_bar_widgets' );

	if ( is_array( $blogun_top_bar_widgets ) && ! empty( $blogun_top_bar_widgets ) ) {
		foreach ( $blogun_top_bar_widgets as $widget ) {

			if ( ! isset( $widget['values'] ) ) {
				continue;
			}

			if ( $location !== $widget['values']['location'] ) {
				continue;
			}

			if ( function_exists( 'blogun_top_bar_widget_' . $widget['type'] ) ) {

				$classes   = array();
				$classes[] = 'blogun-topbar-widget__' . esc_attr( $widget['type'] );
				$classes[] = 'blogun-topbar-widget';

				if ( isset( $widget['values']['visibility'] ) && $widget['values']['visibility'] ) {
					$classes[] = 'blogun-' . esc_attr( $widget['values']['visibility'] );
				}

				$classes = apply_filters( 'blogun_topbar_widget_classes', $classes, $widget );
				$classes = trim( implode( ' ', $classes ) );

				printf( '<div class="%s">', esc_attr( $classes ) );
				call_user_func( 'blogun_top_bar_widget_' . $widget['type'], $widget['values'] );
				printf( '</div><!-- END .blogun-topbar-widget -->' );
			}
		}
	}

	do_action( 'blogun_top_bar_widgets_after_' . $location );
}
add_action( 'blogun_topbar_widgets', 'blogun_topbar_widgets_output' );

/**
 * Outputs the theme header area.
 *
 * @since 1.0.0
 */
function blogun_header_output() {

	if ( ! blogun_is_header_displayed() ) {
		return;
	}

	get_template_part( 'template-parts/header/base' );
}
add_action( 'blogun_header', 'blogun_header_output', 20 );

/**
 * Outputs the header widgets in Header Widget Locations.
 *
 * @since 1.0.0
 * @param string $locations Widget location.
 */
function blogun_header_widgets( $locations ) {

	$locations   = (array) $locations;
	$all_widgets = (array) blogun_option( 'header_widgets' );

	$header_widgets = $all_widgets;
	$header_class   = '';

	if ( ! empty( $locations ) ) {

		$header_widgets = array();

		foreach ( $locations as $location ) {

			$header_class = ' blogun-widget-location-' . $location;

			$header_widgets[ $location ] = array();

			if ( ! empty( $all_widgets ) ) {
				foreach ( $all_widgets as $i => $widget ) {
					if ( $location === $widget['values']['location'] ) {
						$header_widgets[ $location ][] = $widget;
					}
				}
			}
		}
	}

	echo '<div class="blogun-header-widgets blogun-header-element' . esc_attr( $header_class ) . '">';

	if ( ! empty( $header_widgets ) ) {
		foreach ( $header_widgets as $location => $widgets ) {

			do_action( 'blogun_header_widgets_before_' . $location );

			if ( ! empty( $widgets ) ) {
				foreach ( $widgets as $widget ) {
					if ( function_exists( 'blogun_header_widget_' . $widget['type'] ) ) {

						$classes   = array();
						$classes[] = 'blogun-header-widget__' . esc_attr( $widget['type'] );
						$classes[] = 'blogun-header-widget';

						if ( isset( $widget['values']['visibility'] ) && $widget['values']['visibility'] ) {
							$classes[] = 'blogun-' . esc_attr( $widget['values']['visibility'] );
						}

						$classes = apply_filters( 'blogun_header_widget_classes', $classes, $widget );
						$classes = trim( implode( ' ', $classes ) );

						printf( '<div class="%s"><div class="blogun-widget-wrapper">', esc_attr( $classes ) );
						call_user_func( 'blogun_header_widget_' . $widget['type'], $widget['values'] );
						printf( '</div></div><!-- END .blogun-header-widget -->' );
					}
				}
			}

			do_action( 'blogun_header_widgets_after_' . $location );
		}
	}

	echo '</div><!-- END .blogun-header-widgets -->';
}
add_action( 'blogun_header_widget_location', 'blogun_header_widgets', 1 );

/**
 * Outputs the content of theme header.
 *
 * @since 1.0.0
 */
function blogun_header_content_output() {

	// Get the selected header layout from Customizer.
	$header_layout = blogun_option( 'header_layout' );

	?>
	<div id="blogun-header-inner">
		<?php

		// Load header layout template.
		get_template_part( 'template-parts/header/header', $header_layout );

		?>
	</div><!-- END #blogun-header-inner -->
	<?php
}
add_action( 'blogun_header_content', 'blogun_header_content_output' );

/**
 * Outputs the main footer area.
 *
 * @since 1.0.0
 */
function blogun_footer_output() {

	if ( ! blogun_is_footer_displayed() ) {
		return;
	}

	get_template_part( 'template-parts/footer/base' );
}
add_action( 'blogun_footer', 'blogun_footer_output', 20 );

/**
 * Outputs the copyright area.
 *
 * @since 1.0.0
 */
function blogun_copyright_bar_output() {

	if ( ! blogun_is_copyright_bar_displayed() ) {
		return;
	}

	get_template_part( 'template-parts/footer/copyright/copyright' );
}
add_action( 'blogun_footer', 'blogun_copyright_bar_output', 30 );

/**
 * Outputs the copyright widgets.
 *
 * @since 1.0.0
 * @param string $location Widget location in copyright.
 */
function blogun_copyright_widgets_output( $location ) {

	do_action( 'blogun_copyright_widgets_before_' . $location );

	$blogun_widgets = blogun_option( 'copyright_widgets' );

	if ( is_array( $blogun_widgets ) && ! empty( $blogun_widgets ) ) {
		foreach ( $blogun_widgets as $widget ) {

			if ( ! isset( $widget['values'] ) ) {
				continue;
			}

			if ( isset( $widget['values'], $widget['values']['location'] ) && $location !== $widget['values']['location'] ) {
				continue;
			}

			if ( function_exists( 'blogun_copyright_widget_' . $widget['type'] ) ) {

				$classes   = array();
				$classes[] = 'blogun-copyright-widget__' . esc_attr( $widget['type'] );
				$classes[] = 'blogun-copyright-widget';

				if ( isset( $widget['values']['visibility'] ) && $widget['values']['visibility'] ) {
					$classes[] = 'blogun-' . esc_attr( $widget['values']['visibility'] );
				}

				$classes = apply_filters( 'blogun_copyright_widget_classes', $classes, $widget );
				$classes = trim( implode( ' ', $classes ) );

				printf( '<div class="%s">', esc_attr( $classes ) );
				call_user_func( 'blogun_copyright_widget_' . $widget['type'], $widget['values'] );
				printf( '</div><!-- END .blogun-copyright-widget -->' );
			}
		}
	}

	do_action( 'blogun_copyright_widgets_after_' . $location );
}
add_action( 'blogun_copyright_widgets', 'blogun_copyright_widgets_output' );

/**
 * Outputs the theme sidebar area.
 *
 * @since 1.0.0
 */
function blogun_sidebar_output() {

	if ( blogun_is_sidebar_displayed() ) {
		get_sidebar();
	}
}
add_action( 'blogun_sidebar', 'blogun_sidebar_output' );

/**
 * Outputs the back to top button.
 *
 * @since 1.0.0
 */
function blogun_back_to_top_output() {

	if ( ! blogun_option( 'enable_scroll_top' ) ) {
		return;
	}

	get_template_part( 'template-parts/misc/back-to-top' );
}
add_action( 'blogun_after_page_wrapper', 'blogun_back_to_top_output' );

/**
 * Outputs the cursor dot.
 *
 * @since 1.0.0
 */
function blogun_cursor_dot_output() {

	if ( ! blogun_option( 'enable_cursor_dot' ) ) {
		return;
	}

	get_template_part( 'template-parts/misc/cursor-dot' );
}
add_action( 'blogun_after_page_wrapper', 'blogun_cursor_dot_output' );

/**
 * Outputs the theme page content.
 *
 * @since 1.0.0
 */
function blogun_page_header_template() {

	do_action( 'blogun_before_page_header' );

	if ( blogun_is_page_header_displayed() ) {
		if ( is_singular( 'post' ) ) {
			get_template_part( 'template-parts/header-page-title-single' );
		} else {
			get_template_part( 'template-parts/header-page-title' );
		}
	}

	do_action( 'blogun_after_page_header' );
}
add_action( 'blogun_page_header', 'blogun_page_header_template' );

/**
 * Outputs the theme hero content.
 *
 * @since 1.0.0
 */
function blogun_blog_hero() {

	if ( ! blogun_is_hero_displayed() ) {
		return;
	}

	// Hero type.
	$hero_type = blogun_option( 'hero_type' );

	do_action( 'blogun_before_hero' );

	// Enqueue Blogun Slider script.
	wp_enqueue_script( 'blogun-slider' );

	?>
	<div id="hero" <?php blogun_hero_classes(); ?>>
		<?php get_template_part( 'template-parts/hero/hero', $hero_type ); ?>
	</div><!-- END #hero -->
	<?php

	do_action( 'blogun_after_hero' );
}
add_action( 'blogun_after_masthead', 'blogun_blog_hero', 30 );


/**
 * Outputs the theme PYML content.
 *
 * @since 1.0.0
 */
function blogun_blog_pyml() {

	if ( ! blogun_is_pyml_displayed() ) {
		return;
	}

	do_action( 'blogun_before_pyml' );

	?>
	<div id="pyml" <?php blogun_pyml_classes(); ?>>
		<?php get_template_part( 'template-parts/pyml/pyml' ); ?>
	</div><!-- END #pyml -->
	<?php

	do_action( 'blogun_after_pyml' );
}
add_action( 'blogun_after_container', 'blogun_blog_pyml', 30 );


/**
 * Outputs the theme Ticker News content.
 *
 * @since 1.0.0
 */
function blogun_blog_ticker() {

	if ( ! blogun_is_ticker_displayed() ) {
		return;
	}

	do_action( 'blogun_before_ticker' );

	// Enqueue Blogun Marquee script.
	wp_enqueue_script( 'blogun-marquee' );

	?>
	<div id="ticker" <?php blogun_ticker_classes(); ?>>
		<?php get_template_part( 'template-parts/ticker/ticker' ); ?>
	</div><!-- END #ticker -->
	<?php

	do_action( 'blogun_after_ticker' );
}
add_action( 'blogun_after_masthead', 'blogun_blog_ticker', 29 );


/**
 * Outputs the queried articles.
 *
 * @since 1.0.0
 */
function blogun_content() {

	$blogun_blog_layout        = blogun()->options->get( 'blogun_blog_layout' ) == 'blog-masonry' ? 'masonries' : '';
	$blogun_blog_layout_column = 12;

	if ( blogun()->options->get( 'blogun_blog_layout' ) != 'blog-horizontal' ) :
		$blogun_blog_layout_column = blogun()->options->get( 'blogun_blog_layout_column' );
	endif;

	if ( have_posts() ) :

		echo '<div class="blogun-flex-row g-4 ' . $blogun_blog_layout . '">';
		while ( have_posts() ) :
			the_post();

			echo '<div class="col-md-' . $blogun_blog_layout_column . ' col-sm-' . $blogun_blog_layout_column . ' col-xs-12">';
			get_template_part( 'template-parts/content/content', blogun_get_article_feed_layout() );
			echo '</div>';
		endwhile;
		echo '</div>';
		blogun_pagination();

	else :
		get_template_part( 'template-parts/content/content', 'none' );
	endif;
}
add_action( 'blogun_content', 'blogun_content' );
add_action( 'blogun_content_archive', 'blogun_content' );
add_action( 'blogun_content_search', 'blogun_content' );

/**
 * Outputs the theme single content.
 *
 * @since 1.0.0
 */
function blogun_content_singular() {

	if ( have_posts() ) :
		while ( have_posts() ) :
			the_post();

			if ( is_singular( 'post' ) ) {
				do_action( 'blogun_content_single' );
			} else {
				do_action( 'blogun_content_page' );
			}

		endwhile;
	else :
		get_template_part( 'template-parts/content/content', 'none' );
	endif;
}
add_action( 'blogun_content_singular', 'blogun_content_singular' );


/**
 * Outputs the theme 404 page content.
 *
 * @since 1.0.0
 */
function blogun_404_page_content() {

	get_template_part( 'template-parts/content/content', '404' );
}
add_action( 'blogun_content_404', 'blogun_404_page_content' );

/**
 * Outputs the theme page content.
 *
 * @since 1.0.0
 */
function blogun_content_page() {

	get_template_part( 'template-parts/content/content', 'page' );
}
add_action( 'blogun_content_page', 'blogun_content_page' );

/**
 * Outputs the theme single post content.
 *
 * @since 1.0.0
 */
function blogun_content_single() {

	get_template_part( 'template-parts/content/content', 'single' );
}
add_action( 'blogun_content_single', 'blogun_content_single' );

/**
 * Outputs the comments template.
 *
 * @since 1.0.0
 */
function blogun_output_comments() {
	comments_template();
}
add_action( 'blogun_after_singular', 'blogun_output_comments' );

/**
 * Outputs the theme archive page info.
 *
 * @since 1.0.0
 */
function blogun_archive_info() {

	// Author info.
	if ( is_author() ) {
		get_template_part( 'template-parts/entry/entry', 'about-author' );
	}
}
add_action( 'blogun_before_content', 'blogun_archive_info' );

/**
 * Outputs more posts button to author description box.
 *
 * @since 1.0.0
 */
function blogun_add_author_posts_button() {
	if ( ! is_author() ) {
		get_template_part( 'template-parts/entry/entry', 'author-posts-button' );
	}
}
add_action( 'blogun_entry_after_author_description', 'blogun_add_author_posts_button' );

/**
 * Outputs Comments Toggle button.
 *
 * @since 1.0.0
 */
function blogun_comments_toggle() {

	if ( blogun_comments_toggle_displayed() ) {
		get_template_part( 'template-parts/entry/entry-show-comments' );
	}
}
add_action( 'blogun_before_comments', 'blogun_comments_toggle' );

/**
 * Outputs Pre-Footer area.
 *
 * @since 1.0.0
 */
function blogun_pre_footer() {

	if ( ! blogun_is_pre_footer_displayed() ) {
		return;
	}

	get_template_part( 'template-parts/pre-footer/base' );
}
add_action( 'blogun_before_colophon', 'blogun_pre_footer' );

/**
 * Outputs Page Preloader.
 *
 * @since 1.0.0
 */
function blogun_preloader() {

	if ( ! blogun_is_preloader_displayed() ) {
		return;
	}

	get_template_part( 'template-parts/preloader/base' );
}
add_action( 'blogun_before_page_wrapper', 'blogun_preloader' );

/**
 * Outputs breadcrumbs after header.
 *
 * @since  1.0.0
 * @return void
 */
function blogun_breadcrumb_after_header_output() {

	if ( 'below-header' === blogun_option( 'breadcrumbs_position' ) && blogun_has_breadcrumbs() ) {

		$alignment = 'blogun-text-align-' . blogun_option( 'breadcrumbs_alignment' );

		$args = array(
			'container_before' => '<div class="blogun-breadcrumbs"><div class="blogun-container ' . $alignment . '">',
			'container_after'  => '</div></div>',
		);

		blogun_breadcrumb( $args );
	}
}
add_action( 'blogun_main_start', 'blogun_breadcrumb_after_header_output' );

/**
 * Outputs breadcumbs in page header.
 *
 * @since  1.0.0
 * @return void
 */
function blogun_breadcrumb_page_header_output() {

	if ( blogun_page_header_has_breadcrumbs() ) {

		if ( is_singular( 'post' ) ) {
			$args = array(
				'container_before' => '<div class="blogun-container blogun-breadcrumbs">',
				'container_after'  => '</div>',
			);
		} else {
			$args = array(
				'container_before' => '<div class="blogun-breadcrumbs">',
				'container_after'  => '</div>',
			);
		}

		blogun_breadcrumb( $args );
	}
}
add_action( 'blogun_page_header_end', 'blogun_breadcrumb_page_header_output' );

/**
 * Replace tranparent header logo.
 *
 * @since  1.0.0
 * @param  string $output Current logo markup.
 * @return string         Update logo markup.
 */
function blogun_transparent_header_logo( $output ) {

	// Check if transparent header is displayed.
	if ( blogun_is_header_transparent() ) {

		// Check if transparent logo is set.
		$logo = blogun_option( 'tsp_logo' );
		$logo = isset( $logo['background-image-id'] ) ? $logo['background-image-id'] : false;

		$retina = blogun_option( 'tsp_logo_retina' );
		$retina = isset( $retina['background-image-id'] ) ? $retina['background-image-id'] : false;

		if ( $logo ) {
			$output = blogun_get_logo_img_output( $logo, $retina, 'blogun-tsp-logo' );
		}
	}

	return $output;
}
add_filter( 'blogun_logo_img_output', 'blogun_transparent_header_logo' );
add_filter( 'blogun_site_title_markup', 'blogun_transparent_header_logo' );

/**
 * Output the main navigation template.
 */
function blogun_main_navigation_template() {
	get_template_part( 'template-parts/header/navigation' );
}

/**
 * Output the Header logo template.
 */
function blogun_header_logo_template() {
	get_template_part( 'template-parts/header/logo' );
}


if ( ! function_exists( 'blogun_display_customizer_shortcut' ) ) {
	/**
	 * This function display a shortcut to a customizer control.
	 *
	 * @param string $class_name The name of control we want to link this shortcut with.
	 * @param bool   $is_section_toggle Tells function to display eye icon if it's true.
	 */
	function blogun_display_customizer_shortcut( $class_name, $is_section_toggle = false, $should_return = false ) {
		if ( ! is_customize_preview() ) {
			return;
		}
		$icon = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
				<path d="M13.89 3.39l2.71 2.72c.46.46.42 1.24.03 1.64l-8.01 8.02-5.56 1.16 1.16-5.58s7.6-7.63 7.99-8.03c.39-.39 1.22-.39 1.68.07zm-2.73 2.79l-5.59 5.61 1.11 1.11 5.54-5.65zm-2.97 8.23l5.58-5.6-1.07-1.08-5.59 5.6z"></path>
			</svg>';
		if ( $is_section_toggle ) {
			$icon = '<i class="far fa-eye"></i>';
		}

		$data = '<span class="blogun-hide-section-shortcut customize-partial-edit-shortcut customize-partial-edit-shortcut-' . esc_attr( $class_name ) . '">
		<button class="customize-partial-edit-shortcut-button">
			' . $icon . '
		</button>
	</span>';
		if ( $should_return === true ) {
			return $data;
		}
		echo $data;
	}
}

function blogun_about_button() {
	$button_widgets = blogun_option( 'about_widgets' );

	if ( empty( $button_widgets ) ) {
		return;
	}
	foreach ( $button_widgets as $widget ) {
		call_user_func( 'blogun_about_widget_' . $widget['type'], $widget['values'] );
	}
}

function blogun_cta_widgets() {
	$widgets = blogun_option( 'cta_widgets' );

	if ( empty( $widgets ) ) {
		return;
	}
	foreach ( $widgets as $widget ) {
		call_user_func( 'blogun_cta_widget_' . $widget['type'], $widget['values'] );
	}
}

/**
 * Outputs the content of theme Service.
 *
 * @since 1.0.0
 */
function blogun_service_content_output( $args ) {
	$args = (object) $args;
	// Get the selected service layout from Customizer.
	$services_style = blogun_option( 'services_style' );

	// Load service layout template.
	get_template_part( 'template-parts/components/service/service-layout', $services_style, $args );

}
add_action( 'blogun_service_content', 'blogun_service_content_output', 10, 1 );

/**
 * Outputs the content of theme Info.
 *
 * @since 1.0.0
 */
function blogun_info_content_output( $args ) {
	$args = (object) $args;
	// Get the selected info layout from Customizer.
	$info_style = blogun_option( 'info_style' );

	// Load info layout template.
	get_template_part( 'template-parts/components/info/info-layout', $info_style, $args );

}
add_action( 'blogun_info_content', 'blogun_info_content_output', 10, 1 );

/**
 * Outputs the content of theme Team.
 *
 * @since 1.0.0
 */
function blogun_team_content_output( $args ) {
	$args = (object) $args;
	// Get the selected team layout from Customizer.
	$team_style = blogun_option( 'team_style' );

	// Load team layout template.
	get_template_part( 'template-parts/components/team/team-layout', $team_style, $args );

}
add_action( 'blogun_team_content', 'blogun_team_content_output', 10, 1 );

/**
 * Outputs the content of theme Features.
 *
 * @since 1.0.0
 */
function blogun_features_content_output( $args ) {
	$args = (object) $args;
	// Get the selected features layout from Customizer.
	$features_style = blogun_option( 'features_style' );

	// Load features layout template.
	get_template_part( 'template-parts/components/features/features-layout', $features_style, $args );

}
add_action( 'blogun_features_content', 'blogun_features_content_output', 10, 1 );
