/**
 * Update Customizer settings live.
 *
 * @since 1.0.0
 */
( function( $ ) {
	'use strict';

	// Declare variables
	var api = wp.customize,
		$body = $( 'body' ),
		$head = $( 'head' ),
		$style_tag,
		$link_tag,
		blogun_visibility_classes = 'blogun-hide-mobile blogun-hide-tablet blogun-hide-mobile-tablet',
		blogun_style_tag_collection = [],
		blogun_link_tag_collection = [];

	/**
	 * Helper function to get style tag with id.
	 */
	function blogun_get_style_tag( id ) {
		if ( blogun_style_tag_collection[id]) {
			return blogun_style_tag_collection[id];
		}

		$style_tag = $( 'head' ).find( '#blogun-dynamic-' + id );

		if ( ! $style_tag.length ) {
			$( 'head' ).append( '<style id="blogun-dynamic-' + id + '" type="text/css" href="#"></style>' );
			$style_tag = $( 'head' ).find( '#blogun-dynamic-' + id );
		}

		blogun_style_tag_collection[id] = $style_tag;

		return $style_tag;
	}

	/**
	 * Helper function to get link tag with id.
	 */
	function blogun_get_link_tag( id, url ) {
		if ( blogun_link_tag_collection[id]) {
			return blogun_link_tag_collection[id];
		}

		$link_tag = $( 'head' ).find( '#blogun-dynamic-link-' + id );

		if ( ! $link_tag.length ) {
			$( 'head' ).append( '<link id="blogun-dynamic-' + id + '" type="text/css" rel="stylesheet" href="' + url + '"/>' );
			$link_tag = $( 'head' ).find( '#blogun-dynamic-link-' + id );
		} else {
			$link_tag.attr( 'href', url );
		}

		blogun_link_tag_collection[id] = $link_tag;

		return $link_tag;
	}

	/*
	 * Helper function to print visibility classes.
	 */
	function blogun_print_visibility_classes( $element, newval ) {
		if ( ! $element.length ) {
			return;
		}

		$element.removeClass( blogun_visibility_classes );

		if ( 'all' !== newval ) {
			$element.addClass( 'blogun-' + newval );
		}
	}

	/*
	 * Helper function to convert hex to rgba.
	 */
	function blogun_hex2rgba( hex, opacity ) {
		if ( 'rgba' === hex.substring( 0, 4 ) ) {
			return hex;
		}

		// Expand shorthand form (e.g. "03F") to full form (e.g. "0033FF").
		var shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;

		hex = hex.replace( shorthandRegex, function( m, r, g, b ) {
			return r + r + g + g + b + b;
		});

		var result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec( hex );

		if ( opacity ) {
			if ( 1 < opacity ) {
				opacity = 1;
			}

			opacity = ',' + opacity;
		}

		if ( result ) {
			return 'rgba(' + parseInt( result[1], 16 ) + ',' + parseInt( result[2], 16 ) + ',' + parseInt( result[3], 16 ) + opacity + ')';
		}

		return false;
	}

	/**
	 * Helper function to lighten or darken the provided hex color.
	 */
	function blogun_luminance( hex, percent ) {

		// Convert RGB color to HEX.
		if ( hex.includes( 'rgb' ) ) {
			hex = blogun_rgba2hex( hex );
		}

		// Expand shorthand form (e.g. "03F") to full form (e.g. "0033FF").
		var shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;

		hex = hex.replace( shorthandRegex, function( m, r, g, b ) {
			return r + r + g + g + b + b;
		});

		var result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec( hex );

		var isColor = /^#[0-9A-F]{6}$/i.test( hex );

		if ( ! isColor ) {
			return hex;
		}

		var from, to;

		for ( var i = 1; 3 >= i; i++ ) {
			result[i] = parseInt( result[i], 16 );
			from = 0 > percent ? 0 : result[i];
			to = 0 > percent ? result[i] : 255;
			result[i] = result[i] + Math.ceil( ( to - from ) * percent );
		}

		result = '#' + blogun_dec2hex( result[1]) + blogun_dec2hex( result[2]) + blogun_dec2hex( result[3]);

		return result;
	}

	/**
	 * Convert dec to hex.
	 */
	function blogun_dec2hex( c ) {
		var hex = c.toString( 16 );
		return 1 == hex.length ? '0' + hex : hex;
	}

	/**
	 * Convert rgb to hex.
	 */
	function blogun_rgba2hex( c ) {
		var a, x;

		a = c.split( '(' )[1].split( ')' )[0].trim();
		a = a.split( ',' );

		var result = '';

		for ( var i = 0; 3 > i; i++ ) {
			x = parseInt( a[i]).toString( 16 );
			result += 1 === x.length ? '0' + x : x;
		}

		if ( result ) {
			return '#' + result;
		}

		return false;
	}

	/**
	 * Check if is light color.
	 */
	function blogun_is_light_color( color = '' ) {
		var r, g, b, brightness;

		if ( color.match( /^rgb/ ) ) {
			color = color.match( /^rgba?\((\d+),\s*(\d+),\s*(\d+)(?:,\s*(\d+(?:\.\d+)?))?\)$/ );
			r = color[1];
			g = color[2];
			b = color[3];
		} else {
			color = +( '0x' + color.slice( 1 ).replace( 5 > color.length && /./g, '$&$&' ) );
			r = color >> 16;
			g = ( color >> 8 ) & 255;
			b = color & 255;
		}

		brightness = ( r * 299 + g * 587 + b * 114 ) / 1000;

		return 137 < brightness;
	}

	/**
	 * Detect if we should use a light or dark color on a background color.
	 */
	function blogun_light_or_dark( color, dark = '#000000', light = '#FFFFFF' ) {
		return blogun_is_light_color( color ) ? dark : light;
	}

	/**
	 * Spacing field CSS.
	 */
	function blogun_spacing_field_css( selector, property, setting, responsive ) {
		if ( ! Array.isArray( setting ) && 'object' !== typeof setting ) {
			return;
		}

		// Set up unit.
		var unit = 'px',
			css = '';

		if ( 'unit' in setting ) {
			unit = setting.unit;
		}

		var before = '',
			after = '';

		Object.keys( setting ).forEach( function( index, el ) {
			if ( 'unit' === index ) {
				return;
			}

			if ( responsive ) {
				if ( 'tablet' === index ) {
					before = '@media only screen and (max-width: 768px) {';
					after = '}';
				} else if ( 'mobile' === index ) {
					before = '@media only screen and (max-width: 480px) {';
					after = '}';
				} else {
					before = '';
					after = '';
				}

				css += before + selector + '{';

				Object.keys( setting[index]).forEach( function( position ) {
					if ( 'border' === property ) {
						position += '-width';
					}

					if ( setting[index][position]) {
						css += property + '-' + position + ': ' + setting[index][position] + unit + ';';
					}
				});

				css += '}' + after;
			} else {
				if ( 'border' === property ) {
					index += '-width';
				}

				css += property + '-' + index + ': ' + setting[index] + unit + ';';
			}
		});

		if ( ! responsive ) {
			css = selector + '{' + css + '}';
		}

		return css;
	}

	/**
	 * Range field CSS.
	 */
	function blogun_range_field_css( selector, property, setting, responsive, unit ) {
		var css = '',
			before = '',
			after = '';

		if ( responsive && ( Array.isArray( setting ) || 'object' === typeof setting ) ) {
			Object.keys( setting ).forEach( function( index, el ) {
				if ( setting[index]) {
					if ( 'tablet' === index ) {
						before = '@media only screen and (max-width: 768px) {';
						after = '}';
					} else if ( 'mobile' === index ) {
						before = '@media only screen and (max-width: 480px) {';
						after = '}';
					} else if ( 'desktop' === index ) {
						before = '';
						after = '';
					} else {
						return;
					}

					css += before + selector + '{' + property + ': ' + setting[index] + unit + '; }' + after;
				}
			});
		}

		if ( ! responsive ) {
			if ( setting.value ) {
				setting = setting.value;
			} else {
				setting = 0;
			}

			css = selector + '{' + property + ': ' + setting + unit + '; }';
		}

		return css;
	}

	/**
	 * Typography field CSS.
	 */
	function blogun_typography_field_css( selector, setting ) {
		var css = '';

		css += selector + '{';

		if ( 'default' === setting['font-family']) {
			css += 'font-family: ' + blogun_customizer_preview.default_system_font + ';';
		} else if ( setting['font-family'] in blogun_customizer_preview.fonts.standard_fonts.fonts ) {
			css += 'font-family: ' + blogun_customizer_preview.fonts.standard_fonts.fonts[setting['font-family']].fallback + ';';
		} else if ( 'inherit' !== setting['font-family']) {
			css += 'font-family: "' + setting['font-family'] + '";';
		}

		css += 'font-weight:' + setting['font-weight'] + ';';
		css += 'font-style:' + setting['font-style'] + ';';
		css += 'text-transform:' + setting['text-transform'] + ';';

		if ( 'text-decoration' in setting ) {
			css += 'text-decoration:' + setting['text-decoration'] + ';';
		}

		if ( 'letter-spacing' in setting ) {
			css += 'letter-spacing:' + setting['letter-spacing'] + setting['letter-spacing-unit'] + ';';
		}

		if ( 'line-height-desktop' in setting ) {
			css += 'line-height:' + setting['line-height-desktop'] + ';';
		}

		if ( 'font-size-desktop' in setting && 'font-size-unit' in setting ) {
			css += 'font-size:' + setting['font-size-desktop'] + setting['font-size-unit'] + ';';
		}

		css += '}';

		if ( 'font-size-tablet' in setting && setting['font-size-tablet']) {
			css += '@media only screen and (max-width: 768px) {' + selector + '{' + 'font-size: ' + setting['font-size-tablet'] + setting['font-size-unit'] + ';' + '}' + '}';
		}

		if ( 'line-height-tablet' in setting && setting['line-height-tablet']) {
			css += '@media only screen and (max-width: 768px) {' + selector + '{' + 'line-height:' + setting['line-height-tablet'] + ';' + '}' + '}';
		}

		if ( 'font-size-mobile' in setting && setting['font-size-mobile']) {
			css += '@media only screen and (max-width: 480px) {' + selector + '{' + 'font-size: ' + setting['font-size-mobile'] + setting['font-size-unit'] + ';' + '}' + '}';
		}

		if ( 'line-height-mobile' in setting && setting['line-height-mobile']) {
			css += '@media only screen and (max-width: 480px) {' + selector + '{' + 'line-height:' + setting['line-height-mobile'] + ';' + '}' + '}';
		}

		return css;
	}

	/**
	 * Load google font.
	 */
	function blogun_enqueue_google_font( font ) {
		if ( blogun_customizer_preview.fonts.google_fonts.fonts[font]) {
			var id = 'google-font-' + font.trim().toLowerCase().replace( ' ', '-' );
			var url = blogun_customizer_preview.google_fonts_url + '/css?family=' + font + ':' + blogun_customizer_preview.google_font_weights;

			var tag = blogun_get_link_tag( id, url );
		}
	}

	/**
	 * Design Options field CSS.
	 */
	function blogun_design_options_css( selector, setting, type ) {
		var css = '',
			before = '',
			after = '';

		if ( 'background' === type ) {
			var bg_type = setting['background-type'];

			css += selector + '{';

			if ( 'color' === bg_type ) {
				setting['background-color'] = setting['background-color'] ? setting['background-color'] : 'inherit';
				css += 'background: ' + setting['background-color'] + ';';
			} else if ( 'gradient' === bg_type ) {
				css += 'background: ' + setting['gradient-color-1'] + ';';

				if ( 'linear' === setting['gradient-type']) {
					css +=
						'background: -webkit-linear-gradient(' +
						setting['gradient-linear-angle'] +
						'deg, ' +
						setting['gradient-color-1'] +
						' ' +
						setting['gradient-color-1-location'] +
						'%, ' +
						setting['gradient-color-2'] +
						' ' +
						setting['gradient-color-2-location'] +
						'%);' +
						'background: -o-linear-gradient(' +
						setting['gradient-linear-angle'] +
						'deg, ' +
						setting['gradient-color-1'] +
						' ' +
						setting['gradient-color-1-location'] +
						'%, ' +
						setting['gradient-color-2'] +
						' ' +
						setting['gradient-color-2-location'] +
						'%);' +
						'background: linear-gradient(' +
						setting['gradient-linear-angle'] +
						'deg, ' +
						setting['gradient-color-1'] +
						' ' +
						setting['gradient-color-1-location'] +
						'%, ' +
						setting['gradient-color-2'] +
						' ' +
						setting['gradient-color-2-location'] +
						'%);';
				} else if ( 'radial' === setting['gradient-type']) {
					css +=
						'background: -webkit-radial-gradient(' +
						setting['gradient-position'] +
						', circle, ' +
						setting['gradient-color-1'] +
						' ' +
						setting['gradient-color-1-location'] +
						'%, ' +
						setting['gradient-color-2'] +
						' ' +
						setting['gradient-color-2-location'] +
						'%);' +
						'background: -o-radial-gradient(' +
						setting['gradient-position'] +
						', circle, ' +
						setting['gradient-color-1'] +
						' ' +
						setting['gradient-color-1-location'] +
						'%, ' +
						setting['gradient-color-2'] +
						' ' +
						setting['gradient-color-2-location'] +
						'%);' +
						'background: radial-gradient(circle at ' +
						setting['gradient-position'] +
						', ' +
						setting['gradient-color-1'] +
						' ' +
						setting['gradient-color-1-location'] +
						'%, ' +
						setting['gradient-color-2'] +
						' ' +
						setting['gradient-color-2-location'] +
						'%);';
				}
			} else if ( 'image' === bg_type ) {
				css +=
					'' +
					'background-image: url(' +
					setting['background-image'] +
					');' +
					'background-size: ' +
					setting['background-size'] +
					';' +
					'background-attachment: ' +
					setting['background-attachment'] +
					';' +
					'background-position: ' +
					setting['background-position-x'] +
					'% ' +
					setting['background-position-y'] +
					'%;' +
					'background-repeat: ' +
					setting['background-repeat'] +
					';';
			}

			css += '}';

			// Background image color overlay.
			if ( 'image' === bg_type && setting['background-color-overlay'] && setting['background-image']) {
				css += selector + '::after { background-color: ' + setting['background-color-overlay'] + '; }';
			} else {
				css += selector + '::after { background-color: initial; }';
			}
		} else if ( 'color' === type ) {
			setting['text-color'] = setting['text-color'] ? setting['text-color'] : 'inherit';
			setting['link-color'] = setting['link-color'] ? setting['link-color'] : 'inherit';
			setting['link-hover-color'] = setting['link-hover-color'] ? setting['link-hover-color'] : 'inherit';

			css += selector + ' { color: ' + setting['text-color'] + '; }';
			css += selector + ' a { color: ' + setting['link-color'] + '; }';
			css += selector + ' a:hover { color: ' + setting['link-hover-color'] + ' !important; }';
		} else if ( 'border' === type ) {
			setting['border-color'] = setting['border-color'] ? setting['border-color'] : 'inherit';
			setting['border-style'] = setting['border-style'] ? setting['border-style'] : 'solid';
			setting['border-left-width'] = setting['border-left-width'] ? setting['border-left-width'] : 0;
			setting['border-top-width'] = setting['border-top-width'] ? setting['border-top-width'] : 0;
			setting['border-right-width'] = setting['border-right-width'] ? setting['border-right-width'] : 0;
			setting['border-bottom-width'] = setting['border-bottom-width'] ? setting['border-bottom-width'] : 0;

			css += selector + '{';
			css += 'border-color: ' + setting['border-color'] + ';';
			css += 'border-style: ' + setting['border-style'] + ';';
			css += 'border-left-width: ' + setting['border-left-width'] + 'px;';
			css += 'border-top-width: ' + setting['border-top-width'] + 'px;';
			css += 'border-right-width: ' + setting['border-right-width'] + 'px;';
			css += 'border-bottom-width: ' + setting['border-bottom-width'] + 'px;';
			css += '}';
		} else if ( 'separator_color' === type ) {
			css += selector + ':after{ background-color: ' + setting['separator-color'] + '; }';
		}

		return css;
	}

	/**
	 * Logo max height.
	 */
	api( 'blogun_logo_max_height', function( value ) {
		value.bind( function( newval ) {
			var $logo = $( '.blogun-logo' );

			if ( ! $logo.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_logo_max_height' );
			var style_css = '';

			style_css += blogun_range_field_css( '.blogun-logo img', 'max-height', newval, true, 'px' );
			style_css += blogun_range_field_css( '.blogun-logo img.blogun-svg-logo', 'height', newval, true, 'px' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Logo text font size.
	 */
	api( 'blogun_logo_text_font_size', function( value ) {
		value.bind( function( newval ) {
			var $logo = $( '#blogun-header .blogun-logo .site-title' );

			if ( ! $logo.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_logo_text_font_size' );
			var style_css = '';

			style_css += blogun_range_field_css( '#blogun-header .blogun-logo .site-title', 'font-size', newval, true, newval.unit );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Logo margin.
	 */
	api( 'blogun_logo_margin', function( value ) {
		value.bind( function( newval ) {
			var $logo = $( '.blogun-logo' );

			if ( ! $logo.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_logo_margin' );

			var style_css = blogun_spacing_field_css( '.blogun-logo .logo-inner', 'margin', newval, true );
			$style_tag.html( style_css );
		});
	});

	/**
	 * Tagline.
	 */
	api( 'blogdescription', function( value ) {
		value.bind( function( newval ) {
			if ( $( '.blogun-logo' ).find( '.site-description' ).length ) {
				$( '.blogun-logo' ).find( '.site-description' ).html( newval );
			}
		});
	});

	/**
	 * Site Title.
	 */
	api( 'blogname', function( value ) {
		value.bind( function( newval ) {
			if ( $( '.blogun-logo' ).find( '.site-title' ).length ) {
				$( '.blogun-logo' ).find( '.site-title' ).find( 'a' ).html( newval );
			}
		});
	});

	/**
	 * Site Layout.
	 */
	api( 'blogun_site_layout', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-layout__(?!boxed-separated)\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-layout__' + newval );
		});
	});

	/**
	 * Sticky Sidebar.
	 */
	api( 'blogun_sidebar_sticky', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-sticky-\S+/g ) || []).join( ' ' );
			});

			if ( newval ) {
				$body.addClass( 'blogun-sticky-' + newval );
			}
		});
	});

	/**
	 * Sidebar width.
	 */
	api( 'blogun_sidebar_width', function( value ) {
		value.bind( function( newval ) {
			var $sidebar = $( '#secondary' );

			if ( ! $sidebar.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_sidebar_width' );
			var style_css = '#secondary { width: ' + newval.value + '%; }';
			style_css += 'body:not(.blogun-no-sidebar) #primary { ' + 'max-width: ' + ( 100 - parseInt( newval.value ) ) + '%;' + '};';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Sidebar style.
	 */
	api( 'blogun_sidebar_style', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-sidebar-style-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-sidebar-style-' + newval );
		});
	});

	/**
	 * Responsive sidebar position.
	 */
	api( 'blogun_sidebar_responsive_position', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-sidebar-r__\S+/g ) || []).join( ' ' );
			});

			if ( newval ) {
				$body.addClass( 'blogun-sidebar-r__' + newval );
			}
		});
	});

	/**
	 * Featured Image Position (Horizontal Blog layout)
	 */
	api( 'blogun_blog_image_position', function( value ) {
		value.bind( function( newval ) {
			$( '.blogun-blog-entry-wrapper' ).removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-thumb-\S+/g ) || []).join( ' ' );
			});

			$( '.blogun-blog-entry-wrapper' ).addClass( 'blogun-thumb-' + newval );
		});
	});

	/**
	 * Single page - title in header alignment.
	 */
	api( 'blogun_single_title_alignment', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-page-title-align-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-page-title-align-' + newval );
		});
	});

	/**
	 * Single Page title spacing.
	 */
	api( 'blogun_single_title_spacing', function( value ) {
		value.bind( function( newval ) {
			var $page_header = $( '.page-header' );

			if ( ! $page_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_single_title_spacing' );

			var style_css = blogun_spacing_field_css( '.blogun-single-title-in-page-header #page .page-header .blogun-page-header-wrapper', 'padding', newval, true );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Single post narrow container width.
	 */
	api( 'blogun_single_narrow_container_width', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_single_narrow_container_width' );
			var style_css = '';

			style_css +=
				'.single-post.narrow-content .entry-content > :not([class*="align"]):not([class*="gallery"]):not(.wp-block-image):not(.quote-inner):not(.quote-post-bg), ' +
				'.single-post.narrow-content .mce-content-body:not([class*="page-template-full-width"]) > :not([class*="align"]):not([data-wpview-type*="gallery"]):not(blockquote):not(.mceTemp), ' +
				'.single-post.narrow-content .entry-footer, ' +
				'.single-post.narrow-content .post-nav, ' +
				'.single-post.narrow-content .entry-content > .alignwide, ' +
				'.single-post.narrow-content p.has-background:not(.alignfull):not(.alignwide)' +
				'.single-post.narrow-content #blogun-comments-toggle, ' +
				'.single-post.narrow-content #comments, ' +
				'.single-post.narrow-content .entry-content .aligncenter, ' +
				'.single-post.narrow-content .blogun-narrow-element, ' +
				'.single-post.narrow-content.blogun-single-title-in-content .entry-header, ' +
				'.single-post.narrow-content.blogun-single-title-in-content .entry-meta, ' +
				'.single-post.narrow-content.blogun-single-title-in-content .post-category, ' +
				'.single-post.narrow-content.blogun-no-sidebar .blogun-page-header-wrapper, ' +
				'.single-post.narrow-content.blogun-no-sidebar .blogun-breadcrumbs > .blogun-container > nav {' +
				'max-width: ' +
				parseInt( newval.value ) +
				'px; margin-left: auto; margin-right: auto; ' +
				'}';

			style_css += '.single-post.narrow-content .author-box, ' + '.single-post.narrow-content .entry-content > .alignwide { ' + 'max-width: ' + ( parseInt( newval.value ) + 70 ) + 'px;' + '}';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Single post content font size.
	 */
	api( 'blogun_single_content_font_size', function( value ) {
		value.bind( function( newval ) {
			var $content = $( '.single-post' );

			if ( ! $content.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_single_content_font_size' );
			var style_css = '';

			style_css += blogun_range_field_css( '.single-post .entry-content', 'font-size', newval, true, newval.unit );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Header container width.
	 */
	api( 'blogun_header_container_width', function( value ) {
		value.bind( function( newval ) {
			var $header = $( '#blogun-header' );

			if ( ! $header.length ) {
				return;
			}

			if ( 'full-width' === newval ) {
				$header.addClass( 'blogun-container__wide' );
			} else {
				$header.removeClass( 'blogun-container__wide' );
			}
		});
	});

	/**
	 * Main navigation disply breakpoint.
	 */
	api( 'blogun_main_nav_mobile_breakpoint', function( value ) {
		value.bind( function( newval ) {
			var $nav = $( '#blogun-header-inner .blogun-nav' );

			if ( ! $nav.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_main_nav_mobile_breakpoint' );
			var style_css = '';

			style_css += '@media screen and (min-width: ' + parseInt( newval ) + 'px) {#blogun-header-inner .blogun-nav {display:flex} .blogun-mobile-nav,.blogun-mobile-toggen,#blogun-header-inner .blogun-nav .menu-item-has-children>a > .blogun-icon,#blogun-header-inner .blogun-nav .page_item_has_children>a > .blogun-icon {display:none;} }';
			style_css += '@media screen and (max-width: ' + parseInt( newval ) + 'px) {#blogun-header-inner .blogun-nav {display:none} .blogun-mobile-nav,.blogun-mobile-toggen {display:inline-flex;} }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Mobile Menu Button Label.
	 */
	api( 'blogun_main_nav_mobile_label', function( value ) {
		value.bind( function( newval ) {
			if ( $( '.blogun-hamburger-blogun-primary-nav' ).find( '.hamburger-label' ).length ) {
				$( '.blogun-hamburger-blogun-primary-nav' ).find( '.hamburger-label' ).html( newval );
			}
		});
	});

	/**
	 * Main Nav Font color.
	 */
	api( 'blogun_main_nav_font_color', function( value ) {
		value.bind( function( newval ) {
			var $navigation = $( '#blogun-header-inner .blogun-nav' );

			if ( ! $navigation.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_main_nav_font_color' );
			var style_css = '';

			// Link color.
			newval['link-color'] = newval['link-color'] ? newval['link-color'] : 'inherit';
			style_css += '#blogun-header-inner .blogun-nav > ul > li > a { color: ' + newval['link-color'] + '; }';

			// Link hover color.
			newval['link-hover-color'] = newval['link-hover-color'] ? newval['link-hover-color'] : api.value( 'blogun_accent_color' )();
			style_css +=
				'#blogun-header-inner .blogun-nav > ul > li > a:hover, ' +
				'#blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a, ' +
				'#blogun-header-inner .blogun-nav > ul > li.current-menu-item > a, ' +
				'#blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a ' +
				'#blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a, ' +
				'#blogun-header-inner .blogun-nav > ul > li.current_page_item > a, ' +
				'#blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a ' +
				'{ color: ' +
				newval['link-hover-color'] +
				'; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Main Nav Background.
	 */
	api( 'blogun_main_nav_background', function( value ) {
		value.bind( function( newval ) {
			var $navigation = $( '.blogun-header-layout-3 .blogun-nav-container' );

			if ( ! $navigation.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_main_nav_background' );
			var style_css = blogun_design_options_css( '.blogun-header-layout-3 .blogun-nav-container', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Main Nav Border.
	 */
	api( 'blogun_main_nav_border', function( value ) {
		value.bind( function( newval ) {
			var $navigation = $( '.blogun-header-layout-3 .blogun-nav-container' );

			if ( ! $navigation.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_main_nav_border' );
			var style_css = blogun_design_options_css( '.blogun-header-layout-3 .blogun-nav-container', newval, 'border' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Main Nav font size.
	 */
	api( 'blogun_main_nav_font_size', function( value ) {
		value.bind( function( newval ) {
			var $nav = $( '#blogun-header-inner' );

			if ( ! $nav.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_main_nav_font_size' );
			var style_css = '';

			style_css += blogun_range_field_css( '.blogun-nav.blogun-header-element, .blogun-header-layout-1 .blogun-header-widgets, .blogun-header-layout-2 .blogun-header-widgets', 'font-size', newval, false, newval.unit );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Top Bar container width.
	 */
	api( 'blogun_top_bar_container_width', function( value ) {
		value.bind( function( newval ) {
			var $topbar = $( '#blogun-topbar' );

			if ( ! $topbar.length ) {
				return;
			}

			if ( 'full-width' === newval ) {
				$topbar.addClass( 'blogun-container__wide' );
			} else {
				$topbar.removeClass( 'blogun-container__wide' );
			}
		});
	});

	/**
	 * Top Bar visibility.
	 */
	api( 'blogun_top_bar_visibility', function( value ) {
		value.bind( function( newval ) {
			var $topbar = $( '#blogun-topbar' );

			blogun_print_visibility_classes( $topbar, newval );
		});
	});

	/**
	 * Top Bar widgets separator.
	 */
	api( 'blogun_top_bar_widgets_separator', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-topbar__separators-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-topbar__separators-' + newval );
		});
	});

	/**
	 * Top Bar background.
	 */
	api( 'blogun_top_bar_background', function( value ) {
		value.bind( function( newval ) {
			var $topbar = $( '#blogun-topbar' );

			if ( ! $topbar.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_top_bar_background' );
			var style_css = blogun_design_options_css( '#blogun-topbar', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Top Bar color.
	 */
	api( 'blogun_top_bar_text_color', function( value ) {
		value.bind( function( newval ) {
			var $topbar = $( '#blogun-topbar' );

			if ( ! $topbar.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_top_bar_text_color' );
			var style_css = '';

			newval['text-color'] = newval['text-color'] ? newval['text-color'] : 'inherit';
			newval['link-color'] = newval['link-color'] ? newval['link-color'] : 'inherit';
			newval['link-hover-color'] = newval['link-hover-color'] ? newval['link-hover-color'] : 'inherit';

			// Text color.
			style_css += '#blogun-topbar { color: ' + newval['text-color'] + '; }';

			// Link color.
			style_css += '.blogun-topbar-widget__text a, ' + '.blogun-topbar-widget .blogun-nav > ul > li > a, ' + '.blogun-topbar-widget__socials .blogun-social-nav > ul > li > a, ' + '#blogun-topbar .blogun-topbar-widget__text .blogun-icon { color: ' + newval['link-color'] + '; }';

			// Link hover color.
			style_css +=
				'#blogun-topbar .blogun-nav > ul > li > a:hover, ' +
				'#blogun-topbar .blogun-nav > ul > li.menu-item-has-children:hover > a,  ' +
				'#blogun-topbar .blogun-nav > ul > li.current-menu-item > a, ' +
				'#blogun-topbar .blogun-nav > ul > li.current-menu-ancestor > a, ' +
				'#blogun-topbar .blogun-topbar-widget__text a:hover, ' +
				'#blogun-topbar .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon { color: ' +
				newval['link-hover-color'] +
				'; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Top Bar border.
	 */
	api( 'blogun_top_bar_border', function( value ) {
		value.bind( function( newval ) {
			var $topbar = $( '#blogun-topbar' );

			if ( ! $topbar.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_top_bar_border' );
			var style_css = blogun_design_options_css( '#blogun-topbar', newval, 'border' );

			style_css += blogun_design_options_css( '#blogun-topbar .blogun-topbar-widget', newval, 'separator_color' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Header menu item hover animation.
	 */
	api( 'blogun_main_nav_hover_animation', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-menu-animation-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-menu-animation-' + newval );
		});
	});

	/**
	 * Header widgets separator.
	 */
	api( 'blogun_header_widgets_separator', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-header__separators-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-header__separators-' + newval );
		});
	});

	/**
	 * Header background.
	 */
	api( 'blogun_header_background', function( value ) {
		value.bind( function( newval ) {
			var $header = $( '#blogun-header-inner' );

			if ( ! $header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_header_background' );
			var style_css = blogun_design_options_css( '#blogun-header-inner', newval, 'background' );

			if ( 'color' === newval['background-type'] && newval['background-color']) {
				style_css += '.blogun-header-widget__cart .blogun-cart .blogun-cart-count { border: 2px solid ' + newval['background-color'] + '; }';
			} else {
				style_css += '.blogun-header-widget__cart .blogun-cart .blogun-cart-count { border: none; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Header font color.
	 */
	api( 'blogun_header_text_color', function( value ) {
		value.bind( function( newval ) {
			var $header = $( '#blogun-header' );

			if ( ! $header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_header_text_color' );
			var style_css = '';

			// Text color.
			style_css += '.blogun-logo .site-description { color: ' + newval['text-color'] + '; }';

			// Link color.
			if ( newval['link-color']) {
				style_css += '#blogun-header, ' + '.blogun-header-widgets a:not(.blogun-btn), ' + '.blogun-logo a,' + '.blogun-hamburger { color: ' + newval['link-color'] + '; }';
				style_css += '.hamburger-inner,' + '.hamburger-inner::before,' + '.hamburger-inner::after { background-color: ' + newval['link-color'] + '; }';
			}

			// Link hover color.
			if ( newval['link-hover-color']) {
				style_css +=
					'.blogun-header-widgets a:not(.blogun-btn):hover, ' +
					'#blogun-header-inner .blogun-header-widgets .blogun-active,' +
					'.blogun-logo .site-title a:hover, ' +
					'.blogun-hamburger:hover .hamburger-label, ' +
					'.is-mobile-menu-active .blogun-hamburger .hamburger-label,' +
					'#blogun-header-inner .blogun-nav > ul > li > a:hover,' +
					'#blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,' +
					'#blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,' +
					'#blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,' +
					'#blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,' +
					'#blogun-header-inner .blogun-nav > ul > li.current_page_item > a,' +
					'#blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a { color: ' +
					newval['link-hover-color'] +
					'; }';

				style_css +=
					'.blogun-hamburger:hover .hamburger-inner,' +
					'.blogun-hamburger:hover .hamburger-inner::before,' +
					'.blogun-hamburger:hover .hamburger-inner::after,' +
					'.is-mobile-menu-active .blogun-hamburger .hamburger-inner,' +
					'.is-mobile-menu-active .blogun-hamburger .hamburger-inner::before,' +
					'.is-mobile-menu-active .blogun-hamburger .hamburger-inner::after { background-color: ' +
					newval['link-hover-color'] +
					'; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Header border.
	 */
	api( 'blogun_header_border', function( value ) {
		value.bind( function( newval ) {
			var $header = $( '#blogun-header-inner' );

			if ( ! $header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_header_border' );
			var style_css = blogun_design_options_css( '#blogun-header-inner', newval, 'border' );

			// Separator color.
			newval['separator-color'] = newval['separator-color'] ? newval['separator-color'] : 'inherit';
			style_css += '.blogun-header-widget:after { background-color: ' + newval['separator-color'] + '; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Hero container width.
	 */
	api( 'blogun_hero_hover_slider_container', function( value ) {
		value.bind( function( newval ) {
			var $hero_container = $( '#hero .blogun-hero-container' );

			if ( ! $hero_container.length ) {
				return;
			}

			if ( 'full-width' === newval ) {
				$hero_container.addClass( 'blogun-container__wide' );
			} else {
				$hero_container.removeClass( 'blogun-container__wide' );
			}
		});
	});

	/**
	 * Hero overlay style.
	 */
	api( 'blogun_hero_hover_slider_overlay', function( value ) {
		value.bind( function( newval ) {
			var $hero = $( '#hero .blogun-hover-slider' );

			if ( ! $hero.length ) {
				return;
			}

			$hero
				.removeClass( function( index, className ) {
					return ( className.match( /(^|\s)slider-overlay-\S+/g ) || []).join( ' ' );
				})
				.addClass( 'slider-overlay-' + newval );
		});
	});

	/**
	 * Hero height.
	 */
	api( 'blogun_hero_hover_slider_height', function( value ) {
		value.bind( function( newval ) {
			var $hero = $( '#hero' );

			if ( ! $hero.length ) {
				return;
			}

			$hero.find( '.hover-slide-item' ).css( 'height', newval.value + 'px' );
		});
	});

	/**
	 * Hero visibility.
	 */
	api( 'blogun_hero_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#hero' ), newval );
		});
	});

	/**
	 * PYML title.
	 */
	api( 'blogun_pyml_title', function( value ) {
		value.bind( function( newval ) {
			$( '#pyml .widget-title' ).text( newval );
		});
	});

	/**
	 * PYML container width.
	 */
	api( 'blogun_pyml_container', function( value ) {
		value.bind( function( newval ) {
			var $pyml_container = $( '#pyml .blogun-pyml-container' );

			if ( ! $pyml_container.length ) {
				return;
			}

			if ( 'full-width' === newval ) {
				$pyml_container.addClass( 'blogun-container__wide' );
			} else {
				$pyml_container.removeClass( 'blogun-container__wide' );
			}
		});
	});

	/**
	 * PYML visibility.
	 */
	api( 'blogun_pyml_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#pyml' ), newval );
		});
	});

	/**
	 * Ticker News title.
	 */
	api( 'blogun_ticker_title', function( value ) {
		value.bind( function( newval ) {
			$( '#ticker .ticker-title .title' ).text( newval );
		});
	});

	/**
	 * Ticker News container width.
	 */
	api( 'blogun_ticker_container', function( value ) {
		value.bind( function( newval ) {
			var $ticker_container = $( '#ticker .blogun-ticker-container' );

			if ( ! $ticker_container.length ) {
				return;
			}

			if ( 'full-width' === newval ) {
				$ticker_container.addClass( 'blogun-container__wide' );
			} else {
				$ticker_container.removeClass( 'blogun-container__wide' );
			}
		});
	});

	/**
	 * Ticker News visibility.
	 */
	api( 'blogun_ticker_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#ticker' ), newval );
		});
	});

	/**
	 * Custom input style.
	 */
	api( 'blogun_custom_input_style', function( value ) {
		value.bind( function( newval ) {
			if ( newval ) {
				$body.addClass( 'blogun-input-supported' );
			} else {
				$body.removeClass( 'blogun-input-supported' );
			}
		});
	});

	/**
	 * Pre Footer Call to Action Enable.
	 */
	api( 'blogun_enable_pre_footer_cta', function( value ) {
		value.bind( function( newval ) {
			if ( newval ) {
				$body.addClass( 'blogun-pre-footer-cta-style-' + api.value( 'blogun_pre_footer_cta_style' )() );
			} else {
				$body.removeClass( function( index, className ) {
					return ( className.match( /(^|\s)blogun-pre-footer-cta-style-\S+/g ) || []).join( ' ' );
				});
			}
		});
	});

	/**
	 * Pre Footer Call to Action visibility.
	 */
	api( 'blogun_pre_footer_cta_visibility', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '.blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			blogun_print_visibility_classes( $cta, newval );
		});
	});

	/**
	 * Pre Footer Call to Action Text.
	 */
	api( 'blogun_pre_footer_cta_text', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '#blogun-pre-footer .blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			$cta.find( 'p.h3' ).html( newval );
		});
	});

	/**
	 * Pre Footer Call to Action Style.
	 */
	api( 'blogun_pre_footer_cta_style', function( value ) {
		value.bind( function( newval ) {
			$body
				.removeClass( function( index, className ) {
					return ( className.match( /(^|\s)blogun-pre-footer-cta-style-\S+/g ) || []).join( ' ' );
				})
				.addClass( 'blogun-pre-footer-cta-style-' + api.value( 'blogun_pre_footer_cta_style' )() );
		});
	});

	/**
	 * Pre Footer Call to Action Button Text.
	 */
	api( 'blogun_pre_footer_cta_btn_text', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '#blogun-pre-footer .blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			if ( newval ) {
				$cta.find( 'a' ).css( 'display', 'inline-flex' ).html( newval );
			} else {
				$cta.find( 'a' ).css( 'display', 'none' ).html( '' );
			}
		});
	});

	/**
	 * Pre Footer Call to Action Background.
	 */
	api( 'blogun_pre_footer_cta_background', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '#blogun-pre-footer .blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_pre_footer_cta_background' );
			var style_css = '';

			if ( 'color' === newval['background-type']) {
				style_css += blogun_design_options_css( '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::before, .blogun-pre-footer-cta-style-2 #blogun-pre-footer::before', newval, 'background' );
				style_css += '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::after,' + '.blogun-pre-footer-cta-style-2 #blogun-pre-footer::after' + '{ background-image: none; }';
			} else {
				style_css += blogun_design_options_css( '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::after', newval, 'background' );
				style_css += blogun_design_options_css( '.blogun-pre-footer-cta-style-2 #blogun-pre-footer::after', newval, 'background' );
			}

			if ( 'image' === newval['background-type'] && newval['background-color-overlay'] && newval['background-image']) {
				style_css += '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::before,' + '.blogun-pre-footer-cta-style-2 #blogun-pre-footer::before' + '{ background-color: ' + newval['background-color-overlay'] + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Pre Footer Call to Action Text Color.
	 */
	api( 'blogun_pre_footer_cta_text_color', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '#blogun-pre-footer .blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_pre_footer_cta_text_color' );
			var style_css = '';

			style_css += blogun_design_options_css( '#blogun-pre-footer .h2', newval, 'color' );
			style_css += blogun_design_options_css( '#blogun-pre-footer .h3', newval, 'color' );
			style_css += blogun_design_options_css( '#blogun-pre-footer .h4', newval, 'color' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Pre Footer Call to Action Border.
	 */
	api( 'blogun_pre_footer_cta_border', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '#blogun-pre-footer .blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_pre_footer_cta_border' );
			var style_css = blogun_design_options_css( '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::before, .blogun-pre-footer-cta-style-2 #blogun-pre-footer::before', newval, 'border' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Pre Footer CTA font size.
	 */
	api( 'blogun_pre_footer_cta_font_size', function( value ) {
		value.bind( function( newval ) {
			var $cta = $( '#blogun-pre-footer .blogun-pre-footer-cta' );

			if ( ! $cta.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_pre_footer_cta_font_size' );
			var style_css = blogun_range_field_css( '#blogun-pre-footer .h3', 'font-size', newval, true, newval.unit );

			$style_tag.html( style_css );
		});
	});

	/**
	 * WooCommerce sale badge text.
	 */
	api( 'blogun_product_sale_badge_text', function( value ) {
		value.bind( function( newval ) {
			var $badge = $( '.woocommerce ul.products li.product .onsale, .woocommerce span.onsale' ).not( '.sold-out' );

			if ( ! $badge.length ) {
				return;
			}

			$badge.html( newval );
		});
	});

	/**
	 * Accent color.
	 */
	api( 'blogun_accent_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_accent_color' );
			var style_css;

			// Colors.
			style_css =
				':root { ' +
				'--blogun-primary: ' + newval + ';' +
				'--blogun-primary_15: ' + blogun_luminance( newval, 0.15 ) + ';' +
				'--blogun-primary_09: ' + blogun_hex2rgba( newval, 0.09 ) + ';' +
				'--blogun-primary_04: ' + blogun_hex2rgba( newval, 0.04 ) + ';' +
				'}';

			// Gradient.
			style_css +=
				'.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::before,' +
				'.blogun-pre-footer-cta-style-2 #blogun-pre-footer::before { ' +
				'background: linear-gradient(to right, ' +
				blogun_hex2rgba( newval, 0.9 ) +
				' 0%, ' +
				blogun_hex2rgba( newval, 0.82 ) +
				' 35%, ' +
				blogun_hex2rgba( newval, 0.4 ) +
				' 100% );' +
				'-webkit-gradient(linear, left top, right top, from(' +
				blogun_hex2rgba( newval, 0.9 ) +
				'), color-stop(35%, ' +
				blogun_hex2rgba( newval, 0.82 ) +
				'), to(' +
				blogun_hex2rgba( newval, 0.4 ) +
				')); }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Content background color.
	 */
	api( 'blogun_boxed_content_background_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_boxed_content_background_color' );
			var style_css = '';

			if ( newval ) {
				style_css =
					'.blogun-layout__boxed #page, ' +
					'.blogun-layout__boxed-separated .ticker-slider-items, ' +
					'.blogun-layout__boxed-separated .pyml-slider-items, ' +
					'.blogun-layout__boxed-separated.author .author-box, ' +
					'.blogun-layout__boxed-separated #content, ' +
					'.blogun-layout__boxed-separated.blogun-sidebar-style-3 #secondary .blogun-widget, ' +
					'.blogun-layout__boxed-separated.blogun-sidebar-style-3 .elementor-widget-sidebar .blogun-widget, ' +
					'.blogun-layout__boxed-separated.archive .blogun-article, ' +
					'.blogun-layout__boxed-separated.blog .blogun-article, ' +
					'.blogun-layout__boxed-separated.search-results .blogun-article, ' +
					'.blogun-layout__boxed-separated.category .blogun-article { background-color: ' +
					newval +
					'; }';

				// style_css += '@media screen and (max-width: 960px) { ' + '.blogun-layout__boxed-separated #page { background-color: ' + newval + '; }' + '}';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Content text color.
	 */
	api( 'blogun_content_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_content_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css =
					'body { ' +
					'color: ' +
					newval +
					';' +
					'}' +
					':root { ' +
					'--blogun-secondary_38: ' +
					newval +
					';' +
					'}' +
					'.comment-form .comment-notes, ' +
					'#comments .no-comments, ' +
					'#page .wp-caption .wp-caption-text,' +
					'#comments .comment-meta,' +
					'.comments-closed,' +
					'.entry-meta,' +
					'.blogun-entry cite,' +
					'legend,' +
					'.blogun-page-header-description,' +
					'.page-links em,' +
					'.site-content .page-links em,' +
					'.single .entry-footer .last-updated,' +
					'.single .post-nav .post-nav-title,' +
					'#main .widget_recent_comments span,' +
					'#main .widget_recent_entries span,' +
					'#main .widget_calendar table > caption,' +
					'.post-thumb-caption, ' +
					'.wp-block-image figcaption, ' +
					'.blogun-cart-item .blogun-x,' +
					'.woocommerce form.login .lost_password a,' +
					'.woocommerce form.register .lost_password a,' +
					'.woocommerce a.remove,' +
					'#add_payment_method .cart-collaterals .cart_totals .woocommerce-shipping-destination, ' +
					'.woocommerce-cart .cart-collaterals .cart_totals .woocommerce-shipping-destination, ' +
					'.woocommerce-checkout .cart-collaterals .cart_totals .woocommerce-shipping-destination,' +
					'.woocommerce ul.products li.product .blogun-loop-product__category-wrap a,' +
					'.woocommerce ul.products li.product .blogun-loop-product__category-wrap,' +
					'.woocommerce .woocommerce-checkout-review-order table.shop_table thead th,' +
					'#add_payment_method #payment div.payment_box, ' +
					'.woocommerce-cart #payment div.payment_box, ' +
					'.woocommerce-checkout #payment div.payment_box,' +
					'#add_payment_method #payment ul.payment_methods .about_paypal, ' +
					'.woocommerce-cart #payment ul.payment_methods .about_paypal, ' +
					'.woocommerce-checkout #payment ul.payment_methods .about_paypal,' +
					'.woocommerce table dl,' +
					'.woocommerce table .wc-item-meta,' +
					'.widget.woocommerce .reviewer,' +
					'.woocommerce.widget_shopping_cart .cart_list li a.remove:before,' +
					'.woocommerce .widget_shopping_cart .cart_list li a.remove:before,' +
					'.woocommerce .widget_shopping_cart .cart_list li .quantity, ' +
					'.woocommerce.widget_shopping_cart .cart_list li .quantity,' +
					'.woocommerce div.product .woocommerce-product-rating .woocommerce-review-link,' +
					'.woocommerce div.product .woocommerce-tabs table.shop_attributes td,' +
					'.woocommerce div.product .product_meta > span span:not(.blogun-woo-meta-title), ' +
					'.woocommerce div.product .product_meta > span a,' +
					'.woocommerce .star-rating::before,' +
					'.woocommerce div.product #reviews #comments ol.commentlist li .comment-text p.meta,' +
					'.ywar_review_count,' +
					'.woocommerce .add_to_cart_inline del, ' +
					'.woocommerce div.product p.price del, ' +
					'.woocommerce div.product span.price del { color: ' +
					blogun_hex2rgba( newval, 0.75 ) +
					'; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Content link hover color.
	 */
	api( 'blogun_content_link_hover_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_content_link_hover_color' );
			var style_css = '';

			if ( newval ) {

				// Content link hover.
				style_css +=
					'.content-area a:not(.blogun-btn):not(.wp-block-button__link):hover, ' +
					'.blogun-woo-before-shop select.custom-select-loaded:hover ~ #blogun-orderby, ' +
					'#add_payment_method #payment ul.payment_methods .about_paypal:hover, ' +
					'.woocommerce-cart #payment ul.payment_methods .about_paypal:hover, ' +
					'.woocommerce-checkout #payment ul.payment_methods .about_paypal:hover, ' +
					'.blogun-breadcrumbs a:hover, ' +
					'.woocommerce div.product .woocommerce-product-rating .woocommerce-review-link:hover, ' +
					'.woocommerce ul.products li.product .meta-wrap .woocommerce-loop-product__link:hover, ' +
					'.woocommerce ul.products li.product .blogun-loop-product__category-wrap a:hover { ' +
					'color: ' +
					newval +
					';' +
					'}';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Content text color.
	 */
	api( 'blogun_headings_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_headings_color' );
			var style_css = '';

			if ( newval ) {
				style_css = 'h1, h2, h3, h4, h5, h6, .h1, .h2, .h3, .h4, .blogun-logo .site-title, .error-404 .page-header h1 { ' + 'color: ' + newval + ';' + '} :root { ' + '--blogun-secondary: ' + newval + ';' + '}';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Scroll Top visibility.
	 */
	api( 'blogun_scroll_top_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#blogun-scroll-top' ), newval );
		});
	});

	/**
	 * Page Preloader visibility.
	 */
	api( 'blogun_preloader_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#blogun-preloader' ), newval );
		});
	});

	/**
	 * Footer visibility.
	 */
	api( 'blogun_footer_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#blogun-footer' ), newval );
		});
	});

	/**
	 * Footer Widget Heading Style Enable.
	 */
	 api( 'blogun_footer_widget_heading_style', function( value ) {
		value.bind( function( newval ) {
			$body
				.removeClass( function( index, className ) {
					return ( className.match( /(^|\s)is-footer-heading-init-s\S+/g ) || []).join( ' ' );
				})
				.addClass( 'is-footer-heading-init-s' + api.value( 'blogun_footer_widget_heading_style' )() );
		});
	});

	/**
	 * Footer background.
	 */
	api( 'blogun_footer_background', function( value ) {
		value.bind( function( newval ) {
			var $footer = $( '#colophon' );

			if ( ! $footer.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_footer_background' );
			var style_css = blogun_design_options_css( '#colophon', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Footer font color.
	 */
	api( 'blogun_footer_text_color', function( value ) {
		var $footer = $( '#blogun-footer' ),
			copyright_separator_color,
			style_css;

		value.bind( function( newval ) {
			if ( ! $footer.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_footer_text_color' );

			style_css = '';

			newval['text-color'] = newval['text-color'] ? newval['text-color'] : 'inherit';
			newval['link-color'] = newval['link-color'] ? newval['link-color'] : 'inherit';
			newval['link-hover-color'] = newval['link-hover-color'] ? newval['link-hover-color'] : 'inherit';
			newval['widget-title-color'] = newval['widget-title-color'] ? newval['widget-title-color'] : 'inherit';

			// Text color.
			style_css += '#colophon { color: ' + newval['text-color'] + '; }';

			// Link color.
			style_css += '#colophon a { color: ' + newval['link-color'] + '; }';

			// Link hover color.
			style_css += '#colophon a:hover, #colophon li.current_page_item > a, #colophon .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon ' + '{ color: ' + newval['link-hover-color'] + '; }';

			// Widget title color.
			style_css += '#colophon .widget-title, #colophon .wp-block-heading { color: ' + newval['widget-title-color'] + '; }';

			// Copyright separator color.
			copyright_separator_color = blogun_light_or_dark( newval['text-color'], 'rgba(255,255,255,0.1)', 'rgba(0,0,0,0.1)' );

			// copyright_separator_color = blogun_luminance( newval['text-color'], 0.8 );

			style_css += '#blogun-copyright.contained-separator > .blogun-container:before { background-color: ' + copyright_separator_color + '; }';
			style_css += '#blogun-copyright.fw-separator { border-top-color: ' + copyright_separator_color + '; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Footer border.
	 */
	api( 'blogun_footer_border', function( value ) {
		value.bind( function( newval ) {
			var $footer = $( '#blogun-footer' );

			if ( ! $footer.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_footer_border' );
			var style_css = '';

			if ( newval['border-top-width']) {
				style_css += '#colophon { ' + 'border-top-width: ' + newval['border-top-width'] + 'px;' + 'border-top-style: ' + newval['border-style'] + ';' + 'border-top-color: ' + newval['border-color'] + ';' + '}';
			}

			if ( newval['border-bottom-width']) {
				style_css += '#colophon { ' + 'border-bottom-width: ' + newval['border-bottom-width'] + 'px;' + 'border-bottom-style: ' + newval['border-style'] + ';' + 'border-bottom-color: ' + newval['border-color'] + ';' + '}';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Copyright layout.
	 */
	api( 'blogun_copyright_layout', function( value ) {
		value.bind( function( newval ) {
			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-copyright-layout-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-copyright-' + newval );
		});
	});

	/**
	 * Copyright separator.
	 */
	api( 'blogun_copyright_separator', function( value ) {
		value.bind( function( newval ) {
			var $copyright = $( '#blogun-copyright' );

			if ( ! $copyright.length ) {
				return;
			}

			$copyright.removeClass( 'fw-separator contained-separator' );

			if ( 'none' !== newval ) {
				$copyright.addClass( newval );
			}
		});
	});

	/**
	 * Copyright visibility.
	 */
	api( 'blogun_copyright_visibility', function( value ) {
		value.bind( function( newval ) {
			blogun_print_visibility_classes( $( '#blogun-copyright' ), newval );
		});
	});

	/**
	 * Copyright background.
	 */
	api( 'blogun_copyright_background', function( value ) {
		value.bind( function( newval ) {
			var $copyright = $( '#blogun-copyright' );

			if ( ! $copyright.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_copyright_background' );
			var style_css = blogun_design_options_css( '#blogun-copyright', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Copyright text color.
	 */
	api( 'blogun_copyright_text_color', function( value ) {
		value.bind( function( newval ) {
			var $copyright = $( '#blogun-copyright' );

			if ( ! $copyright.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_copyright_text_color' );
			var style_css = '';

			newval['text-color'] = newval['text-color'] ? newval['text-color'] : 'inherit';
			newval['link-color'] = newval['link-color'] ? newval['link-color'] : 'inherit';
			newval['link-hover-color'] = newval['link-hover-color'] ? newval['link-hover-color'] : 'inherit';

			// Text color.
			style_css += '#blogun-copyright { color: ' + newval['text-color'] + '; }';

			// Link color.
			style_css += '#blogun-copyright a { color: ' + newval['link-color'] + '; }';

			// Link hover color.
			style_css +=
				'#blogun-copyright a:hover, #blogun-copyright .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon, #blogun-copyright li.current_page_item > a, #blogun-copyright .blogun-nav > ul > li.current-menu-item > a, #blogun-copyright .blogun-nav > ul > li.current-menu-ancestor > a #blogun-copyright .blogun-nav > ul > li:hover > a, #blogun-copyright .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon { color: ' +
				newval['link-hover-color'] +
				'; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Container width.
	 */
	api( 'blogun_container_width', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_container_width' );
			var style_css;

			style_css = '.blogun-container,' + '.alignfull > div { ' + 'max-width: ' + newval.value + 'px;' + '}';

			style_css +=
				'.blogun-layout__boxed #page, .blogun-layout__boxed.blogun-sticky-header.blogun-is-mobile #blogun-header-inner, ' +
				'.blogun-layout__boxed.blogun-sticky-header:not(.blogun-header-layout-3) #blogun-header-inner, ' +
				'.blogun-layout__boxed.blogun-sticky-header:not(.blogun-is-mobile).blogun-header-layout-3 #blogun-header-inner .blogun-nav-container > .blogun-container { max-width: ' +
				( parseInt( newval.value ) + 100 ) +
				'px; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Transparent Header Logo max height.
	 */
	api( 'blogun_tsp_logo_max_height', function( value ) {
		value.bind( function( newval ) {
			var $logo = $( '.blogun-tsp-header .blogun-logo' );

			if ( ! $logo.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_tsp_logo_max_height' );
			var style_css = '';

			style_css += blogun_range_field_css( '.blogun-tsp-header .blogun-logo img', 'max-height', newval, true, 'px' );
			style_css += blogun_range_field_css( '.blogun-tsp-header .blogun-logo img.blogun-svg-logo', 'height', newval, true, 'px' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Transparent Header Logo margin.
	 */
	api( 'blogun_tsp_logo_margin', function( value ) {
		value.bind( function( newval ) {
			var $logo = $( '.blogun-tsp-header .blogun-logo' );

			if ( ! $logo.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_tsp_logo_margin' );

			var style_css = blogun_spacing_field_css( '.blogun-tsp-header .blogun-logo .logo-inner', 'margin', newval, true );
			$style_tag.html( style_css );
		});
	});

	/**
	 * Transparent header - Main Header & Topbar background.
	 */
	api( 'blogun_tsp_header_background', function( value ) {
		value.bind( function( newval ) {
			var $tsp_header = $( '.blogun-tsp-header' );

			if ( ! $tsp_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_tsp_header_background' );

			var style_css = '';
			style_css += blogun_design_options_css( '.blogun-tsp-header #blogun-header-inner', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Transparent header - Main Header & Topbar font color.
	 */
	api( 'blogun_tsp_header_font_color', function( value ) {
		value.bind( function( newval ) {
			var $tsp_header = $( '.blogun-tsp-header' );

			if ( ! $tsp_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_tsp_header_font_color' );

			var style_css = '';

			newval['text-color'] = newval['text-color'] ? newval['text-color'] : 'inherit';
			newval['link-color'] = newval['link-color'] ? newval['link-color'] : 'inherit';
			newval['link-hover-color'] = newval['link-hover-color'] ? newval['link-hover-color'] : 'inherit';

			/** Header **/

			// Text color.
			style_css += '.blogun-tsp-header .blogun-logo .site-description { color: ' + newval['text-color'] + '; }';

			// Link color.
			if ( newval['link-color']) {
				style_css += '.blogun-tsp-header #blogun-header, ' + '.blogun-tsp-header .blogun-header-widgets a:not(.blogun-btn), ' + '.blogun-tsp-header .blogun-logo a,' + '.blogun-tsp-header .blogun-hamburger, ' + '.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li > a { color: ' + newval['link-color'] + '; }';
				style_css += '.blogun-tsp-header .hamburger-inner,' + '.blogun-tsp-header .hamburger-inner::before,' + '.blogun-tsp-header .hamburger-inner::after { background-color: ' + newval['link-color'] + '; }';
			}

			// Link hover color.
			if ( newval['link-hover-color']) {
				style_css +=
					'.blogun-tsp-header .blogun-header-widgets a:not(.blogun-btn):hover, ' +
					'.blogun-tsp-header #blogun-header-inner .blogun-header-widgets .blogun-active,' +
					'.blogun-tsp-header .blogun-logo .site-title a:hover, ' +
					'.blogun-tsp-header .blogun-hamburger:hover .hamburger-label, ' +
					'.is-mobile-menu-active .blogun-tsp-header .blogun-hamburger .hamburger-label,' +
					'.blogun-tsp-header.using-keyboard .site-title a:focus,' +
					'.blogun-tsp-header.using-keyboard .blogun-header-widgets a:not(.blogun-btn):focus,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.hovered > a,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li > a:hover,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current_page_item > a,' +
					'.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a { color: ' +
					newval['link-hover-color'] +
					'; }';

				style_css +=
					'.blogun-tsp-header .blogun-hamburger:hover .hamburger-inner,' +
					'.blogun-tsp-header .blogun-hamburger:hover .hamburger-inner::before,' +
					'.blogun-tsp-header .blogun-hamburger:hover .hamburger-inner::after,' +
					'.is-mobile-menu-active .blogun-tsp-header .blogun-hamburger .hamburger-inner,' +
					'.is-mobile-menu-active .blogun-tsp-header .blogun-hamburger .hamburger-inner::before,' +
					'.is-mobile-menu-active .blogun-tsp-header .blogun-hamburger .hamburger-inner::after { background-color: ' +
					newval['link-hover-color'] +
					'; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Transparent header - Main Header & Topbar border.
	 */
	api( 'blogun_tsp_header_border', function( value ) {
		value.bind( function( newval ) {
			var $tsp_header = $( '.blogun-tsp-header' );

			if ( ! $tsp_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_tsp_header_border' );

			var style_css = '';

			style_css += blogun_design_options_css( '.blogun-tsp-header #blogun-header-inner', newval, 'border' );

			// Separator color.
			newval['separator-color'] = newval['separator-color'] ? newval['separator-color'] : 'inherit';
			style_css += '.blogun-tsp-header .blogun-header-widget:after { background-color: ' + newval['separator-color'] + '; }';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Page Header layout.
	 */
	api( 'blogun_page_header_alignment', function( value ) {
		value.bind( function( newval ) {
			if ( $body.hasClass( 'single-post' ) ) {
				return;
			}

			$body.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-page-title-align-\S+/g ) || []).join( ' ' );
			});

			$body.addClass( 'blogun-page-title-align-' + newval );
		});
	});

	/**
	 * Page Header spacing.
	 */
	api( 'blogun_page_header_spacing', function( value ) {
		value.bind( function( newval ) {
			var $page_header = $( '.page-header' );

			if ( ! $page_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_page_header_spacing' );

			var style_css = blogun_spacing_field_css( '.blogun-page-title-align-left .page-header.blogun-has-page-title, .blogun-page-title-align-right .page-header.blogun-has-page-title, .blogun-page-title-align-center .page-header .blogun-page-header-wrapper', 'padding', newval, true );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Page Header background.
	 */
	api( 'blogun_page_header_background', function( value ) {
		value.bind( function( newval ) {
			var $page_header = $( '.page-header' );

			if ( ! $page_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_page_header_background' );

			var style_css = '';
			style_css += blogun_design_options_css( '.page-header', newval, 'background' );
			style_css += blogun_design_options_css( '.blogun-tsp-header:not(.blogun-tsp-absolute) #masthead', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Header Text color.
	 */
	api( 'blogun_page_header_text_color', function( value ) {
		value.bind( function( newval ) {
			var $page_header = $( '.page-header' );

			if ( ! $page_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_page_header_text_color' );
			var style_css = '';

			newval['text-color'] = newval['text-color'] ? newval['text-color'] : 'inherit';
			newval['link-color'] = newval['link-color'] ? newval['link-color'] : 'inherit';
			newval['link-hover-color'] = newval['link-hover-color'] ? newval['link-hover-color'] : 'inherit';

			// Text color.
			style_css += '.page-header .page-title { color: ' + newval['text-color'] + '; }';
			style_css += '.page-header .blogun-page-header-description' + '{ color: ' + blogun_hex2rgba( newval['text-color'], 0.75 ) + '}';

			// Link color.
			style_css += '.page-header .blogun-breadcrumbs a' + '{ color: ' + newval['link-color'] + '; }';

			style_css += '.page-header .blogun-breadcrumbs span,' + '.page-header .breadcrumb-trail .trail-items li::after, .page-header .blogun-breadcrumbs .separator' + '{ color: ' + blogun_hex2rgba( newval['link-color'], 0.75 ) + '}';

			$style_tag.html( style_css );
		});
	});

	/**
	 * Page Header border.
	 */
	api( 'blogun_page_header_border', function( value ) {
		value.bind( function( newval ) {
			var $page_header = $( '.page-header' );

			if ( ! $page_header.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_page_header_border' );
			var style_css = blogun_design_options_css( '.page-header', newval, 'border' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Breadcrumbs alignment.
	 */
	api( 'blogun_breadcrumbs_alignment', function( value ) {
		value.bind( function( newval ) {
			var $breadcrumbs = $( '#main > .blogun-breadcrumbs > .blogun-container' );

			if ( ! $breadcrumbs.length ) {
				return;
			}

			$breadcrumbs.removeClass( function( index, className ) {
				return ( className.match( /(^|\s)blogun-text-align\S+/g ) || []).join( ' ' );
			});

			$breadcrumbs.addClass( 'blogun-text-align-' + newval );
		});
	});

	/**
	 * Breadcrumbs spacing.
	 */
	api( 'blogun_breadcrumbs_spacing', function( value ) {
		value.bind( function( newval ) {
			var $breadcrumbs = $( '.blogun-breadcrumbs' );

			if ( ! $breadcrumbs.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_breadcrumbs_spacing' );

			var style_css = blogun_spacing_field_css( '.blogun-breadcrumbs', 'padding', newval, true );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Breadcrumbs Background.
	 */
	api( 'blogun_breadcrumbs_background', function( value ) {
		value.bind( function( newval ) {
			var $breadcrumbs = $( '.blogun-breadcrumbs' );

			if ( ! $breadcrumbs.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_breadcrumbs_background' );
			var style_css = blogun_design_options_css( '.blogun-breadcrumbs', newval, 'background' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Breadcrumbs Text Color.
	 */
	api( 'blogun_breadcrumbs_text_color', function( value ) {
		value.bind( function( newval ) {
			var $breadcrumbs = $( '.blogun-breadcrumbs' );

			if ( ! $breadcrumbs.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_breadcrumbs_text_color' );
			var style_css = blogun_design_options_css( '.blogun-breadcrumbs', newval, 'color' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Breadcrumbs Border.
	 */
	api( 'blogun_breadcrumbs_border', function( value ) {
		value.bind( function( newval ) {
			var $breadcrumbs = $( '.blogun-breadcrumbs' );

			if ( ! $breadcrumbs.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_breadcrumbs_border' );
			var style_css = blogun_design_options_css( '.blogun-breadcrumbs', newval, 'border' );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Base HTML font size.
	 */
	api( 'blogun_html_base_font_size', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_html_base_font_size' );
			var style_css = blogun_range_field_css( 'html', 'font-size', newval, true, '%' );
			$style_tag.html( style_css );
		});
	});

	/**
	 * Font smoothing.
	 */
	api( 'blogun_font_smoothing', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_font_smoothing' );

			if ( newval ) {
				$style_tag.html( '*,' + '*::before,' + '*::after {' + '-moz-osx-font-smoothing: grayscale;' + '-webkit-font-smoothing: antialiased;' + '}' );
			} else {
				$style_tag.html( '*,' + '*::before,' + '*::after {' + '-moz-osx-font-smoothing: auto;' + '-webkit-font-smoothing: auto;' + '}' );
			}

			$style_tag = blogun_get_style_tag( 'blogun_html_base_font_size' );
			var style_css = blogun_range_field_css( 'html', 'font-size', newval, true, '%' );
			$style_tag.html( style_css );
		});
	});

	/**
	 * Body font.
	 */
	api( 'blogun_body_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_body_font' );
			var style_css = blogun_typography_field_css( 'body', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Headings font.
	 */
	api( 'blogun_headings_font', function( value ) {
		var style_css, selector;
		value.bind( function( newval ) {
			selector = 'h1, .h1, .blogun-logo .site-title, .page-header h1.page-title';
			selector += ', h2, .h2, .woocommerce div.product h1.product_title';
			selector += ', h3, .h3, .woocommerce #reviews #comments h2';
			selector += ', h4, .h4, .woocommerce .cart_totals h2, .woocommerce .cross-sells > h4, .woocommerce #reviews #respond .comment-reply-title';
			selector += ', h5, h6, .h5, .h6';

			style_css = blogun_typography_field_css( selector, newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag = blogun_get_style_tag( 'blogun_headings_font' );
			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading 1 font.
	 */
	api( 'blogun_h1_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_h1_font' );

			var style_css = blogun_typography_field_css( 'h1, .h1, .blogun-logo .site-title, .page-header h1.page-title', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading 2 font.
	 */
	api( 'blogun_h2_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_h2_font' );

			var style_css = blogun_typography_field_css( 'h2, .h2, .woocommerce div.product h1.product_title', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading 3 font.
	 */
	api( 'blogun_h3_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_h3_font' );

			var style_css = blogun_typography_field_css( 'h3, .h3, .woocommerce #reviews #comments h2', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading 4 font.
	 */
	api( 'blogun_h4_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_h4_font' );

			var style_css = blogun_typography_field_css( 'h4, .h4, .woocommerce .cart_totals h2, .woocommerce .cross-sells > h4, .woocommerce #reviews #respond .comment-reply-title', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading 5 font.
	 */
	api( 'blogun_h5_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_h5_font' );
			var style_css = blogun_typography_field_css( 'h5, .h5', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading 6 font.
	 */
	api( 'blogun_h6_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_h6_font' );
			var style_css = blogun_typography_field_css( 'h6, .h6', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Heading emphasized font.
	 */
	api( 'blogun_heading_em_font', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_heading_em_font' );
			var style_css = blogun_typography_field_css( 'h1 em, h2 em, h3 em, h4 em, h5 em, h6 em, .h1 em, .h2 em, .h3 em, .h4 em, .h5 em, .h6 em, .blogun-logo .site-title em, .error-404 .page-header h1 em', newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Sidebar widget title font size.
	 */
	api( 'blogun_sidebar_widget_title_font_size', function( value ) {
		value.bind( function( newval ) {
			var $widget_title = $( '#main .widget-title, .widget-area .wp-block-heading' );

			if ( ! $widget_title.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_sidebar_widget_title_font_size' );
			var style_css = '';

			style_css += blogun_range_field_css( '#main .widget-title, .widget-area .wp-block-heading', 'font-size', newval, true, newval.unit );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Footer widget title font size.
	 */
	api( 'blogun_footer_widget_title_font_size', function( value ) {
		value.bind( function( newval ) {
			var $widget_title = $( '#colophon .widget-title, #colophon .wp-block-heading' );

			if ( ! $widget_title.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_footer_widget_title_font_size' );
			var style_css = '';

			style_css += blogun_range_field_css( '#colophon .widget-title, #colophon .wp-block-heading', 'font-size', newval, true, newval.unit );

			$style_tag.html( style_css );
		});
	});

	/**
	 * Page title font size.
	 */
	api( 'blogun_page_header_font_size', function( value ) {
		value.bind( function( newval ) {
			var $page_title = $( '.page-header .page-title' );

			if ( ! $page_title.length ) {
				return;
			}

			$style_tag = blogun_get_style_tag( 'blogun_page_header_font_size' );
			var style_css = '';

			style_css += blogun_range_field_css( '#page .page-header .page-title', 'font-size', newval, true, newval.unit );

			$style_tag.html( style_css );
		});
	});

	var $btn_selectors =
		'.blogun-btn, ' +
		'body:not(.wp-customizer) input[type=submit], ' +
		'.site-main .woocommerce #respond input#submit, ' +
		'.site-main .woocommerce a.button, ' +
		'.site-main .woocommerce button.button, ' +
		'.site-main .woocommerce input.button, ' +
		'.woocommerce ul.products li.product .added_to_cart, ' +
		'.woocommerce ul.products li.product .button, ' +
		'.woocommerce div.product form.cart .button, ' +
		'.woocommerce #review_form #respond .form-submit input, ' +
		'#infinite-handle span';

	var $btn_hover_selectors =
		'.blogun-btn:hover, ' +
		'.blogun-btn:focus, ' +
		'body:not(.wp-customizer) input[type=submit]:hover, ' +
		'body:not(.wp-customizer) input[type=submit]:focus, ' +
		'.site-main .woocommerce #respond input#submit:hover, ' +
		'.site-main .woocommerce #respond input#submit:focus, ' +
		'.site-main .woocommerce a.button:hover, ' +
		'.site-main .woocommerce a.button:focus, ' +
		'.site-main .woocommerce button.button:hover, ' +
		'.site-main .woocommerce button.button:focus, ' +
		'.site-main .woocommerce input.button:hover, ' +
		'.site-main .woocommerce input.button:focus, ' +
		'.woocommerce ul.products li.product .added_to_cart:hover, ' +
		'.woocommerce ul.products li.product .added_to_cart:focus, ' +
		'.woocommerce ul.products li.product .button:hover, ' +
		'.woocommerce ul.products li.product .button:focus, ' +
		'.woocommerce div.product form.cart .button:hover, ' +
		'.woocommerce div.product form.cart .button:focus, ' +
		'.woocommerce #review_form #respond .form-submit input:hover, ' +
		'.woocommerce #review_form #respond .form-submit input:focus, ' +
		'#infinite-handle span:hover';

	/**
	 * Primary button background color.
	 */
	api( 'blogun_primary_button_bg_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_bg_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_selectors + '{ background-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button hover background color.
	 */
	api( 'blogun_primary_button_hover_bg_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_hover_bg_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_hover_selectors + ' { background-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button text color.
	 */
	api( 'blogun_primary_button_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_selectors + ' { color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button hover text color.
	 */
	api( 'blogun_primary_button_hover_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_hover_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_hover_selectors + ' { color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button border width.
	 */
	api( 'blogun_primary_button_border_width', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_border_width' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_selectors + ' { border-width: ' + newval.value + 'rem; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button border radius.
	 */
	api( 'blogun_primary_button_border_radius', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_border_radius' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_selectors + ' { ' + 'border-top-left-radius: ' + newval['top-left'] + 'rem;' + 'border-top-right-radius: ' + newval['top-right'] + 'rem;' + 'border-bottom-left-radius: ' + newval['bottom-left'] + 'rem;' + 'border-bottom-right-radius: ' + newval['bottom-right'] + 'rem; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button border color.
	 */
	api( 'blogun_primary_button_border_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_border_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_selectors + ' { border-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button hover border color.
	 */
	api( 'blogun_primary_button_hover_border_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_hover_border_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_hover_selectors + ' { border-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Primary button typography.
	 */
	api( 'blogun_primary_button_typography', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_primary_button_typography' );
			var style_css = blogun_typography_field_css( $btn_selectors, newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	// Secondary button.
	var $btn_sec_selectors = '.btn-secondary, .blogun-btn.btn-secondary';

	var $btn_sec_hover_selectors = '.btn-secondary:hover, ' + '.btn-secondary:focus, ' + '.blogun-btn.btn-secondary:hover, ' + '.blogun-btn.btn-secondary:focus';

	/**
	 * Secondary button background color.
	 */
	api( 'blogun_secondary_button_bg_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_bg_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_selectors + '{ background-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button hover background color.
	 */
	api( 'blogun_secondary_button_hover_bg_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_hover_bg_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_hover_selectors + '{ background-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button text color.
	 */
	api( 'blogun_secondary_button_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_selectors + '{ color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button hover text color.
	 */
	api( 'blogun_secondary_button_hover_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_hover_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_hover_selectors + '{ color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button border width.
	 */
	api( 'blogun_secondary_button_border_width', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_border_width' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_selectors + ' { border-width: ' + newval.value + 'rem; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button border radius.
	 */
	api( 'blogun_secondary_button_border_radius', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_border_radius' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_selectors + ' { ' + 'border-top-left-radius: ' + newval['top-left'] + 'rem;' + 'border-top-right-radius: ' + newval['top-right'] + 'rem;' + 'border-bottom-left-radius: ' + newval['bottom-left'] + 'rem;' + 'border-bottom-right-radius: ' + newval['bottom-right'] + 'rem; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button border color.
	 */
	api( 'blogun_secondary_button_border_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_border_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_selectors + ' { border-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button hover border color.
	 */
	api( 'blogun_secondary_button_hover_border_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_hover_border_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_sec_hover_selectors + ' { border-color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Secondary button typography.
	 */
	api( 'blogun_secondary_button_typography', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_secondary_button_typography' );
			var style_css = blogun_typography_field_css( $btn_sec_selectors, newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	// Text button.
	var $btn_text_selectors = '.blogun-btn.btn-text-1, .btn-text-1';

	var $btn_text_hover_selectors = '.blogun-btn.btn-text-1:hover, .blogun-btn.btn-text-1:focus, .btn-text-1:hover, .btn-text-1:focus';

	/**
	 * Text button text color.
	 */
	api( 'blogun_text_button_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_text_button_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_text_selectors + '{ color: ' + newval + '; }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Text button hover text color.
	 */
	api( 'blogun_text_button_hover_text_color', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_text_button_hover_text_color' );
			var style_css = '';

			if ( newval ) {
				style_css = $btn_text_hover_selectors + '{ color: ' + newval + '; }';
				style_css += '.blogun-btn.btn-text-1 > span::before { background-color: ' + newval + ' }';
			}

			$style_tag.html( style_css );
		});
	});

	/**
	 * Text button typography.
	 */
	api( 'blogun_text_button_typography', function( value ) {
		value.bind( function( newval ) {
			$style_tag = blogun_get_style_tag( 'blogun_text_button_typography' );
			var style_css = blogun_typography_field_css( $btn_text_selectors, newval );

			blogun_enqueue_google_font( newval['font-family']);

			$style_tag.html( style_css );
		});
	});

	/**
	 * Section Heading Style Enable.
	 */
	api( 'blogun_section_heading_style', function( value ) {
		value.bind( function( newval ) {
			$body
				.removeClass( function( index, className ) {
					return ( className.match( /(^|\s)is-section-heading-init-s\S+/g ) || []).join( ' ' );
				})
				.addClass( 'is-section-heading-init-s' + api.value( 'blogun_section_heading_style' )() );
		});
	});


	// Selective refresh.
	if ( api.selectiveRefresh ) {

		// Bind partial content rendered event.
		api.selectiveRefresh.bind( 'partial-content-rendered', function( placement ) {

			// Hero Hover Slider.
			if ( 'blogun_hero_hover_slider_post_number' === placement.partial.id || 'blogun_hero_hover_slider_elements' === placement.partial.id ) {
				document.querySelectorAll( placement.partial.params.selector ).forEach( ( item ) => {
					blogunHoverSlider( item );
				});

				// Force refresh height.
				api( 'blogun_hero_hover_slider_height', function( newval ) {
					newval.callbacks.fireWith( newval, [ newval.get() ]);
				});
			}

			// Preloader style.
			if ( 'blogun_preloader_style' === placement.partial.id ) {
				$body.removeClass( 'blogun-loaded' );

				setTimeout( function() {
					window.blogun.preloader();
				}, 300 );
			}
		});
	}

	// Custom Customizer Preview class (attached to the Customize API)
	api.blogunCustomizerPreview = {

		// Init
		init: function() {
			var self = this; // Store a reference to "this"
			var previewBody = self.preview.body;

			previewBody.on( 'click', '.blogun-set-widget', function() {
				self.preview.send( 'set-footer-widget', $( this ).data( 'sidebar-id' ) );
			});
		}
	};

	/**
	 * Capture the instance of the Preview since it is private (this has changed in WordPress 4.0)
	 *
	 * @see https://github.com/WordPress/WordPress/blob/5cab03ab29e6172a8473eb601203c9d3d8802f17/wp-admin/js/customize-controls.js#L1013
	 */
	var blogunOldPreview = api.Preview;
	api.Preview = blogunOldPreview.extend({
		initialize: function( params, options ) {

			// Store a reference to the Preview
			api.blogunCustomizerPreview.preview = this;

			// Call the old Preview's initialize function
			blogunOldPreview.prototype.initialize.call( this, params, options );
		}
	});

	// Document ready
	$( function() {

		// Initialize our Preview
		api.blogunCustomizerPreview.init();
	});
}( jQuery ) );
