<?php
/**
 * Enqueue scripts & styles.
 *
 * @package     Blogun
 * @author      Peregrine Themes
 * @since       1.0.0
 */

/**
 * Enqueue and register scripts and styles.
 *
 * @since 1.0.0
 */
class Blogun_Enqueue_Scripts {

	/**
	 * Check if debug is on
	 *
	 * @var boolean
	 */
	private $is_debug;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->is_debug = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG;
		add_action( 'wp_enqueue_scripts', array( $this, 'blogun_enqueues' ) );
		add_action( 'wp_print_footer_scripts', array( $this, 'blogun_skip_link_focus_fix' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'blogun_block_editor_assets' ) );
	}

	/**
	 * Enqueue styles and scripts.
	 *
	 * @since 1.0.0
	 */
	public function blogun_enqueues() {
		// Script debug.
		$blogun_dir    = $this->is_debug ? 'dev/' : '';
		$blogun_suffix = $this->is_debug ? '' : '.min';

		// fontawesome enqueue.
		wp_enqueue_style(
			'FontAwesome',
			BLOGUN_THEME_URI . '/assets/css/all' . $blogun_suffix . '.css',
			false,
			'5.15.4',
			'all'
		);
		// Enqueue theme stylesheet.
		wp_enqueue_style(
			'blogun-styles',
			BLOGUN_THEME_URI . '/assets/css/style' . $blogun_suffix . '.css',
			false,
			BLOGUN_THEME_VERSION,
			'all'
		);

		// Enqueue IE specific styles.
		wp_enqueue_style(
			'blogun-ie',
			BLOGUN_THEME_URI . '/assets/css/compatibility/ie' . $blogun_suffix . '.css',
			false,
			BLOGUN_THEME_VERSION,
			'all'
		);

		wp_style_add_data( 'blogun-ie', 'conditional', 'IE' );

		// Enqueue HTML5 shiv.
		wp_register_script(
			'html5shiv',
			BLOGUN_THEME_URI . '/assets/js/' . $blogun_dir . 'vendors/html5' . $blogun_suffix . '.js',
			array(),
			'3.7.3',
			true
		);

		// Load only on < IE9.
		wp_script_add_data(
			'html5shiv',
			'conditional',
			'lt IE 9'
		);

		// Flexibility.js for crossbrowser flex support.
		wp_enqueue_script(
			'blogun-flexibility',
			BLOGUN_THEME_URI . '/assets/js/' . $blogun_dir . 'vendors/flexibility' . $blogun_suffix . '.js',
			array(),
			BLOGUN_THEME_VERSION,
			false
		);

		wp_add_inline_script(
			'blogun-flexibility',
			'flexibility(document.documentElement);'
		);

		wp_script_add_data(
			'blogun-flexibility',
			'conditional',
			'IE'
		);

		// Register ImagesLoaded library.
		// wp_register_script(
		// 'imagesloaded',
		// BLOGUN_THEME_URI . '/assets/js/' . $blogun_dir . 'vendors/imagesloaded' . $blogun_suffix . '.js',
		// array(),
		// '4.1.4',
		// true
		// );

		// Register Blogun slider.
		wp_register_script(
			'blogun-slider',
			BLOGUN_THEME_URI . '/assets/js/' . $blogun_dir . 'blogun-slider' . $blogun_suffix . '.js',
			array( 'imagesloaded' ),
			BLOGUN_THEME_VERSION,
			true
		);

		wp_register_script(
			'blogun-marquee',
			BLOGUN_THEME_URI . '/assets/js/' . $blogun_dir . 'vendors/jquery.marquee' . $blogun_suffix . '.js',
			array( 'imagesloaded' ),
			BLOGUN_THEME_VERSION,
			true
		);

		if ( blogun()->options->get( 'blogun_blog_layout' ) == 'blog-masonry' ) {
			wp_enqueue_script( 'masonry' );
		}

		// Load comment reply script if comments are open.
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}

		// Enqueue main theme script.
		wp_enqueue_script(
			'blogun',
			BLOGUN_THEME_URI . '/assets/js/' . $blogun_dir . 'blogun' . $blogun_suffix . '.js',
			array( 'jquery', 'imagesloaded' ),
			BLOGUN_THEME_VERSION,
			true
		);

		// Comment count used in localized strings.
		$comment_count = get_comments_number();

		// Localized variables so they can be used for translatable strings.
		$localized = array(
			'ajaxurl'               => esc_url( admin_url( 'admin-ajax.php' ) ),
			'nonce'                 => wp_create_nonce( 'blogun-nonce' ),
			'responsive-breakpoint' => intval( blogun_option( 'main_nav_mobile_breakpoint' ) ),
			'sticky-header'         => array(
				'enabled' => blogun_option( 'sticky_header' ),
				'hide_on' => blogun_option( 'sticky_header_hide_on' ),
			),
			'strings'               => array(
				/* translators: %s Comment count */
				'comments_toggle_show' => $comment_count > 0 ? esc_html( sprintf( _n( 'Show %s Comment', 'Show %s Comments', $comment_count, 'blogun' ), $comment_count ) ) : esc_html__( 'Leave a Comment', 'blogun' ),
				'comments_toggle_hide' => esc_html__( 'Hide Comments', 'blogun' ),
			),
		);

		wp_localize_script(
			'blogun',
			'blogun_vars',
			apply_filters( 'blogun_localized', $localized )
		);

		// Enqueue google fonts.
		blogun()->fonts->enqueue_google_fonts();

		// Add additional theme styles.
		do_action( 'blogun_enqueue_scripts' );
	}

	/**
	 * Skip link focus fix for IE11.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function blogun_skip_link_focus_fix() {
		?>
		<script>
			! function() {
				var e = -1 < navigator.userAgent.toLowerCase().indexOf("webkit"),
					t = -1 < navigator.userAgent.toLowerCase().indexOf("opera"),
					n = -1 < navigator.userAgent.toLowerCase().indexOf("msie");
				(e || t || n) && document.getElementById && window.addEventListener && window.addEventListener("hashchange", function() {
					var e, t = location.hash.substring(1);
					/^[A-z0-9_-]+$/.test(t) && (e = document.getElementById(t)) && (/^(?:a|select|input|button|textarea)$/i.test(e.tagName) || (e.tabIndex = -1), e.focus())
				}, !1)
			}();
		</script>
		<?php
	}

	/**
	 * Enqueue assets for the Block Editor.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function blogun_block_editor_assets() {

		// RTL version.
		$rtl = is_rtl() ? '-rtl' : '';

		// Minified version.
		$min = $this->is_debug ? '' : '.min';
		// Enqueue block editor styles.
		wp_enqueue_style(
			'blogun-block-editor-styles',
			BLOGUN_THEME_URI . '/inc/admin/assets/css/blogun-block-editor-styles' . $rtl . $min . '.css',
			false,
			BLOGUN_THEME_VERSION,
			'all'
		);

		// Enqueue google fonts.
		blogun()->fonts->enqueue_google_fonts();

		// Add dynamic CSS as inline style.
		wp_add_inline_style(
			'blogun-block-editor-styles',
			apply_filters( 'blogun_block_editor_dynamic_css', blogun_dynamic_styles()->get_block_editor_css() )
		);
	}
}
