<?php

/**
 * Dynamically generate CSS code.
 * The code depends on options set in the Highend Options and Post/Page metaboxes.
 *
 * If possible, write the dynamically generated code into a .css file, otherwise return the code. The file is refreshed on each modification of metaboxes & theme options.
 *
 * @package     Blogun
 * @author      Peregrine Themes
 * @since       1.0.0
 */

/**
 * Do not allow direct script access.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Blogun_Dynamic_Styles' ) ) :
	/**
	 * Dynamically generate CSS code.
	 */
	class Blogun_Dynamic_Styles {

		/**
		 * Singleton instance of the class.
		 *
		 * @since 1.0.0
		 * @var object
		 */
		private static $instance;

		/**
		 * URI for Dynamic CSS file.
		 *
		 * @since 1.0.0
		 * @var object
		 */
		private $dynamic_css_uri;

		/**
		 * Path for Dynamic CSS file.
		 *
		 * @since 1.0.0
		 * @var object
		 */
		private $dynamic_css_path;

		/**
		 * Main Blogun_Dynamic_Styles Instance.
		 *
		 * @since 1.0.0
		 * @return Blogun_Dynamic_Styles
		 */
		public static function instance() {

			if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Blogun_Dynamic_Styles ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Primary class constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {

			$upload_dir = wp_upload_dir();

			$this->dynamic_css_uri  = trailingslashit( set_url_scheme( $upload_dir['baseurl'] ) ) . 'blogun/';
			$this->dynamic_css_path = trailingslashit( set_url_scheme( $upload_dir['basedir'] ) ) . 'blogun/';

			if ( ! is_customize_preview() && wp_is_writable( trailingslashit( $upload_dir['basedir'] ) ) ) {
				add_action( 'blogun_enqueue_scripts', array( $this, 'enqueue_dynamic_style' ), 20 );
			} else {
				add_action( 'blogun_enqueue_scripts', array( $this, 'print_dynamic_style' ), 99 );
			}

			// Include button styles.
			add_filter( 'blogun_dynamic_styles', array( $this, 'get_button_styles' ), 6 );

			// Remove Customizer Custom CSS from wp_head, we will include it in our dynamic file.
			if ( ! is_customize_preview() ) {
				remove_action( 'wp_head', 'wp_custom_css_cb', 101 );
			}

			// Generate new styles on Customizer Save action.
			add_action( 'customize_save_after', array( $this, 'update_dynamic_file' ) );

			// Generate new styles on theme activation.
			add_action( 'after_switch_theme', array( $this, 'update_dynamic_file' ) );

			// Delete the css stye on theme deactivation.
			add_action( 'switch_theme', array( $this, 'delete_dynamic_file' ) );

			// Generate initial dynamic css.
			add_action( 'init', array( $this, 'init' ) );
		}

		/**
		 * Init.
		 *
		 * @since 1.0.0
		 */
		public function init() {

			// Ensure we have dynamic stylesheet generated.
			if ( false === get_transient( 'blogun_has_dynamic_css' ) ) {
				$this->update_dynamic_file();
			}
		}

		/**
		 * Enqueues dynamic styles file.
		 *
		 * @since 1.0.0
		 */
		public function enqueue_dynamic_style() {

			$exists = file_exists( $this->dynamic_css_path . 'dynamic-styles.css' );
			// Generate the file if it's missing.
			if ( ! $exists ) {
				$exists = $this->update_dynamic_file();
			}

			// Enqueue the file if available.
			if ( $exists ) {
				wp_enqueue_style(
					'blogun-dynamic-styles',
					$this->dynamic_css_uri . 'dynamic-styles.css',
					false,
					filemtime( $this->dynamic_css_path . 'dynamic-styles.css' ),
					'all'
				);
			}
		}

		/**
		 * Prints inline dynamic styles if writing to file is not possible.
		 *
		 * @since 1.0.0
		 */
		public function print_dynamic_style() {
			$dynamic_css = $this->get_css();
			wp_add_inline_style( 'blogun-styles', $dynamic_css );
		}

		/**
		 * Generates dynamic CSS code, minifies it and cleans cache.
		 *
		 * @param  boolean $custom_css - should we include the wp_get_custom_css.
		 * @return string, minifed code
		 * @since  1.0.0
		 */
		public function get_css( $custom_css = false ) {

			// Refresh options.
			blogun()->options->refresh();

			// Delete google fonts enqueue transients.
			delete_transient( 'blogun_google_fonts_enqueue' );

			// Add our theme custom CSS.
			$css = '';

			// Dark Mode.
			if ( is_array( $header_active_widgets = blogun_option( 'header_widgets' ) ) ) {
				array_walk(
					$header_active_widgets,
					function ( $value, $key ) use ( &$css ) {
						$darkMode = blogun_option( 'dark_mode' );
						if ( $value['type'] == 'darkmode' || $darkMode ) {
							$css = '
							[data-theme=dark] {
								--blogun-white: ' . blogun_sanitize_color(blogun_option( 'headings_color' )) . ';
								--blogun-secondary: #ffffff !important;
							}
							[data-theme=dark] select option {
								background: rgba(0, 0, 0, 0.3);
  								color: #fff;
							}
							[data-theme=dark] .entry-media > a .entry-media-icon .blogun-icon,
							[data-theme=dark] .entry-media > a .entry-media-icon svg,
							[data-theme=dark] #blogun-scroll-top svg,
							[data-theme=dark] .navigation .nav-links .page-numbers svg,
							[data-theme=dark] .navigation .nav-links .page-numbers:hover svg,
							[data-theme=dark] .using-keyboard .navigation .nav-links .page-numbers:focus svg {
								fill: #fff;
							}
							[data-theme=dark] .wp-block-search .wp-block-search__button {
								--blogun-white: #ffffff;
							}
							[data-theme=dark] #colophon .search-form .search-submit,
							[data-theme=dark] #main .search-form .search-submit,
							[data-theme=dark] .content-area a:not(.blogun-btn,.showcoupon,.wp-block-button__link):hover,
							[data-theme=dark] #secondary .hester-core-custom-list-widget .blogun-entry a:not(.blogun-btn):hover,
							[data-theme=dark] .blogun-breadcrumbs a:hover,
							[data-theme=dark] #add_payment_method table.cart td.actions .coupon .input-text:focus,
							[data-theme=dark] .woocommerce-cart table.cart td.actions .coupon .input-text:focus,
							[data-theme=dark] .woocommerce-checkout table.cart td.actions .coupon .input-text:focus,
							[data-theme=dark] input[type="date"]:focus,
							[data-theme=dark] input[type="email"]:focus,
							[data-theme=dark] input[type="password"]:focus,
							[data-theme=dark] input[type="search"]:focus,
							[data-theme=dark] input[type="tel"]:focus,
							[data-theme=dark] input[type="text"]:focus,
							[data-theme=dark] input[type="url"]:focus,
							[data-theme=dark] textarea:focus,
							[data-theme=dark] .entry-media > a .entry-media-icon .blogun-icon,
							[data-theme=dark] .entry-media > a .entry-media-icon svg,
							[data-theme=dark] .navigation .nav-links .page-numbers:hover button,
							[data-theme=dark] .using-keyboard .navigation .nav-links .page-numbers:focus button,
							[data-theme=dark] .navigation .nav-links .page-numbers:not(.prev, .next).current,
							[data-theme=dark] .navigation .nav-links .page-numbers:not(.prev, .next):hover,
							[data-theme=dark] .using-keyboard .navigation .nav-links .page-numbers:not(.prev, .next):focus,
							[data-theme=dark] .page-links a:hover span,
							[data-theme=dark] .using-keyboard .page-links a:focus span, .page-links > span,
							[data-theme=dark] .blogun-btn.btn-text-1:hover,
							[data-theme=dark] .blogun-btn.btn-text-1:focus,
							[data-theme=dark] .btn-text-1:hover, .btn-text-1:focus,
							[data-theme=dark] .blogun-header-widgets .blogun-search-simple .blogun-search-form button:not(.blogun-search-close),
							[data-theme=dark] #blogun-header,
							[data-theme=dark] .blogun-header-widgets a:not(.blogun-btn),
							[data-theme=dark] .blogun-logo a,
							[data-theme=dark] .blogun-hamburger,
							[data-theme=dark] h1,
							[data-theme=dark] h2,
							[data-theme=dark] h3,
							[data-theme=dark] h4,
							[data-theme=dark] h5,
							[data-theme=dark] h6,
							[data-theme=dark] .h1,
							[data-theme=dark] .h2,
							[data-theme=dark] .h3,
							[data-theme=dark] .h4,
							[data-theme=dark] .blogun-logo .site-title,
							[data-theme=dark] .error-404 .page-header h1,
							[data-theme=dark] body,
							[data-theme=dark] #blogun-header .blogun-nav li > a {
								color: #ffffff;
							}
							[data-theme=dark] .woocommerce table.my_account_orders thead th,
							[data-theme=dark] .woocommerce table.woocommerce-table--order-downloads thead th,
							[data-theme=dark] .woocommerce table.woocommerce-table--order-details thead th,
							[data-theme=dark] .blogun-cart-item .blogun-x,
							[data-theme=dark] .woocommerce form.login .lost_password a,
							[data-theme=dark] .woocommerce form.register .lost_password a,
							[data-theme=dark] .woocommerce a.remove,
							[data-theme=dark] #add_payment_method .cart-collaterals .cart_totals .woocommerce-shipping-destination,
							[data-theme=dark] .woocommerce-cart .cart-collaterals .cart_totals .woocommerce-shipping-destination,
							[data-theme=dark] .woocommerce-checkout .cart-collaterals .cart_totals .woocommerce-shipping-destination,
							[data-theme=dark] .woocommerce ul.products li.product .blogun-loop-product__category-wrap a,
							[data-theme=dark] .woocommerce ul.products li.product .blogun-loop-product__category-wrap,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table thead th,
							[data-theme=dark] #add_payment_method #payment div.payment_box, .woocommerce-cart #payment div.payment_box,
							[data-theme=dark] .woocommerce-checkout #payment div.payment_box,
							[data-theme=dark] #add_payment_method #payment ul.payment_methods .about_paypal,
							[data-theme=dark] .woocommerce-cart #payment ul.payment_methods .about_paypal,
							[data-theme=dark] .woocommerce-checkout #payment ul.payment_methods .about_paypal,
							[data-theme=dark] .woocommerce table dl, .woocommerce table .wc-item-meta,
							[data-theme=dark] .widget.woocommerce .reviewer, .woocommerce.widget_shopping_cart .cart_list li a.remove::before,
							[data-theme=dark] .woocommerce .widget_shopping_cart .cart_list li a.remove::before,
							[data-theme=dark] .woocommerce .widget_shopping_cart .cart_list li .quantity,
							[data-theme=dark] .woocommerce.widget_shopping_cart .cart_list li .quantity,
							[data-theme=dark] .woocommerce div.product .woocommerce-product-rating .woocommerce-review-link,
							[data-theme=dark] .woocommerce div.product .woocommerce-tabs table.shop_attributes td,
							[data-theme=dark] .woocommerce div.product .product_meta > span span:not(.blogun-woo-meta-title),
							[data-theme=dark] .woocommerce div.product .product_meta > span a,
							[data-theme=dark] .woocommerce .star-rating::before,
							[data-theme=dark] .woocommerce div.product #reviews #comments ol.commentlist li .comment-text p.meta,
							[data-theme=dark] .ywar_review_count, .woocommerce .add_to_cart_inline del,
							[data-theme=dark] .woocommerce div.product p.price del,
							[data-theme=dark] .woocommerce div.product span.price del,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table thead,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table thead,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table thead,
							[data-theme=dark] .woocommerce div.product .woocommerce-tabs ul.tabs li a,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table,
							[data-theme=dark] .blogun-btn.btn-text-1,
							[data-theme=dark] .btn-text-1,
							[data-theme=dark] .comment-form .comment-notes,
							[data-theme=dark] #comments .no-comments,
							[data-theme=dark] #page .wp-caption .wp-caption-text, #comments .comment-meta,
							[data-theme=dark] .comments-closed,
							[data-theme=dark] .entry-meta,
							[data-theme=dark] .blogun-entry cite,
							[data-theme=dark] legend,
							[data-theme=dark] .blogun-page-header-description,
							[data-theme=dark] .page-links em,
							[data-theme=dark] .site-content .page-links em,
							[data-theme=dark] .single .entry-footer .last-updated,
							[data-theme=dark] .single .post-nav .post-nav-title,
							[data-theme=dark] #main .widget_recent_comments span,
							[data-theme=dark] #main .widget_recent_entries span,
							[data-theme=dark] #main .widget_calendar table > caption,
							[data-theme=dark] .post-thumb-caption,
							[data-theme=dark] .wp-block-image figcaption,
							[data-theme=dark] .wp-block-embed figcaption {
								color: rgba(255,255,255,0.7);
							}
							[data-theme=dark] #blogun-header .blogun-nav .children li.current_page_ancestor > a,
							[data-theme=dark] #blogun-header .blogun-nav .children li.current_page_item > a,
							[data-theme=dark] #blogun-header .blogun-nav .children li:hover > a,
							[data-theme=dark] #blogun-header .blogun-nav .sub-menu li.current-menu-ancestor > a,
							[data-theme=dark] #blogun-header .blogun-nav .sub-menu li.current-menu-item > a,
							[data-theme=dark] #blogun-header .blogun-nav .sub-menu li:hover > a {
								color: rgba(255,255,255,0.7) !important;
							}
							[data-theme=dark] .entry-meta .entry-meta-elements > span::before {
								background-color: rgba(255, 255, 255, 0.25);
							}

							[data-theme=dark] .blogun-post-gallery .swiper-button-prev,
							[data-theme=dark] .blogun-post-gallery .swiper-button-next,
							[data-theme=dark] .blogun-vertical-slider .swiper-button-prev,
							[data-theme=dark] .blogun-vertical-slider .swiper-button-next,
							[data-theme=dark] .blogun-horizontal-slider .swiper-button-prev,
							[data-theme=dark] .blogun-horizontal-slider .swiper-button-next,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table th:first-child,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table td:first-child,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table th:first-child,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table td:first-child,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table th:first-child,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table td:first-child,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table td,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table td,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table td,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table tr:nth-last-child(2) td,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table tr:nth-last-child(2) td,
							[data-theme=dark] .woocommerce .cart_totals table.shop_table,
							[data-theme=dark] .woocommerce .cart_totals table.shop_table th,
							[data-theme=dark] .woocommerce .cart_totals table.shop_table td,
							[data-theme=dark] .blogun-header-layout-5 #masthead+#main .blogun-breadcrumbs,
							[data-theme=dark] #blogun-topbar,
							[data-theme=dark] #blogun-header-inner,
							[data-theme=dark] .page-header,
							[data-theme=dark] .blogun-header-layout-3 .blogun-nav-container,
							[data-theme=dark] .blogun-header-layout-4 .blogun-nav-container {
								border-color: rgba(255,255,255,0.08);
							}
							html[data-theme=dark] body,
							[data-theme=dark] .select2-dropdown,
							[data-theme=dark] .blogun-header-layout-5 #masthead+#main .blogun-breadcrumbs,
							[data-theme=dark] #add_payment_method #payment ul.payment_methods li:not(.woocommerce-notice),
							[data-theme=dark] .woocommerce-cart #payment ul.payment_methods li:not(.woocommerce-notice),
							[data-theme=dark] .woocommerce-checkout #payment ul.payment_methods li:not(.woocommerce-notice),
							html[data-theme=dark] .woocommerce div.product .woocommerce-tabs table.shop_attributes,
							[data-theme=dark] .blogun-header-layout-4 .blogun-nav-container,
							[data-theme=dark] .blogun-header-layout-3 .blogun-nav-container,
							[data-theme=dark] #blogun-header-inner {
								background: ' . blogun_sanitize_color(blogun_option( 'headings_color' )) . ';
							}
							[data-theme=dark] .blogun-hover-slider,
							[data-theme=dark] .select2-container--default .select2-selection--single,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table,
							[data-theme=dark] .woocommerce #yith-wcwl-form table.shop_table thead th,
							[data-theme=dark] .woocommerce .woocommerce-cart-form table.shop_table thead th,
							[data-theme=dark] .woocommerce .woocommerce-checkout-review-order table.shop_table thead th,
							[data-theme=dark] .woocommerce .cart_totals table.shop_table .order-total th,
							[data-theme=dark] .woocommerce .cart_totals table.shop_table .order-total td,
							[data-theme=dark] .woocommerce div.product .woocommerce-tabs .wc-tab,
							[data-theme=dark] #page .woocommerce-error, #page .woocommerce-info,
							[data-theme=dark] #page .woocommerce-message,
							[data-theme=dark] .woocommerce div.product .woocommerce-tabs ul.tabs::before,
							[data-theme=dark] .woocommerce div.product .woocommerce-tabs ul.tabs::after,
							[data-theme=dark] .blogun-layout__boxed-separated .ticker-slider-items,
							[data-theme=dark] .blogun-layout__boxed-separated .pyml-slider-items,
							[data-theme=dark] .blogun-layout__framed #page,
							[data-theme=dark] .blogun-layout__boxed #page,
							[data-theme=dark] .blogun-layout__boxed-separated:not(.blog, .archive, .category, .search-results) #content,
							[data-theme=dark] .blogun-layout__boxed-separated.author .author-box,
							[data-theme=dark] .blogun-layout__boxed-separated.single #content,
							[data-theme=dark] .blogun-layout__boxed-separated.blogun-sidebar-style-3 #secondary .blogun-widget,
							[data-theme=dark] .blogun-layout__boxed-separated.blogun-sidebar-style-3 .elementor-widget-sidebar .blogun-widget,
							[data-theme=dark] .blogun-layout__boxed-separated.archive .blogun-article,
							[data-theme=dark] .blogun-layout__boxed-separated.blog .blogun-article,
							[data-theme=dark] .blogun-layout__boxed-separated.search-results .blogun-article,
							[data-theme=dark] .blogun-layout__boxed-separated.category .blogun-article {
								background-color: rgba(0,0,0,0.3);
							}
							[data-theme=dark] .woocommerce ul.products li.product:hover,
							[data-theme=dark] .woocommerce ul.products li.product:focus-within,
							[data-theme=dark] .blogun-layout__framed #page,
							[data-theme=dark] .blogun-layout__boxed #page {
								-webkit-box-shadow: 0 0 3.5rem rgba(0, 0, 0, 0.4);
								box-shadow: 0 0 3.5rem rgba(0, 0, 0, 0.4);
							}
							[data-theme=dark] .blogun-btn.btn-text-1 > span::before {
								background-color: #fff;
							}
							[data-theme=dark] .woocommerce .quantity .blogun-woo-minus:not(:hover, :focus),
							[data-theme=dark] .woocommerce .quantity .blogun-woo-plus:not(:hover, :focus) {
								color: ' . blogun_sanitize_color(blogun_option( 'headings_color' )) . ' !important;
							}
							[data-theme=dark] .blogun-layout__boxed-separated .ticker-slider-items,
							[data-theme=dark] .blogun-layout__boxed-separated .pyml-slider-items,
							[data-theme=dark] .blogun-layout__boxed-separated.blogun-sidebar-style-3 #secondary .blogun-widget,
							[data-theme=dark] .blogun-layout__boxed-separated.archive article.blogun-article,
							[data-theme=dark] .blogun-layout__boxed-separated.blog article.blogun-article,
							[data-theme=dark] .blogun-layout__boxed-separated.category article.blogun-article,
							[data-theme=dark] .blogun-layout__boxed-separated.search-results article.blogun-article {
								border-color: rgba(190,190,190,0.30);
							}
							[data-theme=dark] .blogun-social-nav.rounded > ul > li > a > span:not(.screen-reader-text) {
								background-color: rgba(190,190,190,0.30);
							}
							[data-theme=dark] .blogun-blog-layout-1 .blogun-article .entry-thumb-image,
							[data-theme=dark] .pyml-slide-item .pyml-slider-backgrounds .pyml-slide-bg {
								background-color: rgba(39,39,39,.75);
							}
							@media screen and (max-width: ' . intval( blogun_option( 'main_nav_mobile_breakpoint' ) ) . 'px) {
								[data-theme=dark] .blogun-layout__boxed-separated #page {
									background-color: rgba(0,0,0,0.3);
								}
								[data-theme=dark] #blogun-header-inner .site-navigation > ul li {
									border-bottom-color: rgba(255,255,255,0.08);
								}
								[data-theme=dark] #blogun-header-inner .site-navigation {
									background: ' . blogun_sanitize_color(blogun_option( 'headings_color' )) . ';
								}
								[data-theme=dark] .blogun-mobile-toggen,
								[data-theme=dark] #blogun-header-inner .blogun-nav {
									color: rgba(255,255,255,0.7);
								}
								[data-theme=dark] #blogun-header-inner .blogun-nav .menu-item-has-children > a > span,
								[data-theme=dark] #blogun-header-inner .blogun-nav .page_item_has_children > a > span {
									border-right-color: rgba(255,255,255,0.08);
								}
								[data-theme=dark] #blogun-header-inner .site-navigation > ul .sub-menu {
									background: rgba(0,0,0,0.3);
								}
							}
							';
						}
					}
				);
			}

			// Accent color.
			$accent_color = blogun_option( 'accent_color' );

			$css .= '
				:root {
					--blogun-primary: ' . blogun_sanitize_color( $accent_color ) . ';
					--blogun-primary_15: ' . blogun_sanitize_color( blogun_luminance( $accent_color, .15 ) ) . ';
					--blogun-primary_09: ' . blogun_sanitize_color( blogun_hex2rgba( $accent_color, .09 ) ) . ';
					--blogun-primary_04: ' . blogun_sanitize_color( blogun_hex2rgba( $accent_color, .04 ) ) . ';
				}
			';

			$header_layout_3_additional_css = '';

			if ( 'layout-3' === blogun_option( 'header_layout' ) || is_customize_preview() ) {
				$header_layout_3_additional_css = '

					.blogun-header-layout-3 .blogun-logo-container > .blogun-container {
						flex-wrap: wrap;
					}

					.blogun-header-layout-3 .blogun-logo-container .blogun-logo > .logo-inner {
						align-items: flex-start;
					}
					
					.blogun-header-layout-3 .blogun-logo-container .blogun-logo {
						order: 0;
						align-items: flex-start;
						flex-basis: auto;
						margin-left: 0;
					}

					.blogun-header-layout-3 .blogun-logo-container .blogun-header-element {
						flex-basis: auto;
					}

					.blogun-header-layout-3 .blogun-logo-container .blogun-mobile-nav {
						order: 5;
					}

				';
			}

			$header_layout_4_additional_css = '';

			if ( 'layout-4' === blogun_option( 'header_layout' ) || is_customize_preview() ) {
				$header_layout_4_additional_css = '

					.blogun-header-layout-4 .blogun-logo-container > .blogun-container {
						flex-wrap: wrap;
					}

					.blogun-header-layout-4 .blogun-logo-container .blogun-logo > .logo-inner {
						align-items: flex-start;
					}
					
					.blogun-header-layout-4 .blogun-logo-container .blogun-logo {
						order: 0;
						align-items: flex-start;
						flex-basis: auto;
						margin-left: 0;
					}

					.blogun-header-layout-4 .blogun-logo-container .blogun-header-element {
						flex-basis: auto;
					}

					.blogun-header-layout-4 .blogun-logo-container .blogun-mobile-nav {
						order: 5;
					}

					.blogun-header-layout-4 .blogun-widget-location-left .dropdown-item {
						left: auto;
						right: -7px;
					}

					.blogun-header-layout-4 .blogun-widget-location-left .dropdown-item::after {
						left: auto;
						right: 8px;
					}

					.blogun-header-layout-4 .blogun-logo-container .blogun-widget-location-right:not(.blogun-header-widgets-two) {
						-js-display: flex;
						display: -webkit-box;
						display: -ms-flexbox;
						display: flex;
					}

					.blogun-header-layout-4 .blogun-nav-container .blogun-header-element,
					.blogun-header-layout-4 .blogun-header-widgets.blogun-header-widgets-two {
						display: none;
					}

				';
			}

			$css .= '
						@media screen and (max-width: ' . intval( blogun_option( 'main_nav_mobile_breakpoint' ) ) . 'px) {
							
							' . $header_layout_4_additional_css . '
						}';

			if ( 'layout-4' === blogun_option( 'header_layout' ) ) {

				// Background.
				$css .= $this->get_design_options_field_css( '.blogun-header-layout-4 .blogun-nav-container', 'main_nav_background', 'background' );

				// Border.
				$css .= $this->get_design_options_field_css( '.blogun-header-layout-4 .blogun-nav-container', 'main_nav_border', 'border' );
			}

			/**
			 * Top Bar.
			 */

			// Background.
			$css .= $this->get_design_options_field_css( '#blogun-topbar', 'top_bar_background', 'background' );

			// Border.
			$css .= $this->get_design_options_field_css( '#blogun-topbar', 'top_bar_border', 'border' );
			$css .= $this->get_design_options_field_css( '.blogun-topbar-widget', 'top_bar_border', 'separator_color' );

			// Top Bar colors.
			$topbar_color = blogun_option( 'top_bar_text_color' );

			// Top Bar text color.
			if ( isset( $topbar_color['text-color'] ) && $topbar_color['text-color'] ) {
				$css .= '#blogun-topbar { color: ' . blogun_sanitize_color( $topbar_color['text-color'] ) . '; }';
			}

			// Top Bar link color.
			if ( isset( $topbar_color['link-color'] ) && $topbar_color['link-color'] ) {
				$css .= '
					.blogun-topbar-widget__text a,
					.blogun-topbar-widget .blogun-nav > ul > li > a,
					.blogun-topbar-widget__socials .blogun-social-nav > ul > li > a,
					#blogun-topbar .blogun-topbar-widget__text .blogun-icon { 
						color: ' . blogun_sanitize_color( $topbar_color['link-color'] ) . '; }
				';
			}

			// Top Bar link hover color.
			if ( isset( $topbar_color['link-hover-color'] ) && $topbar_color['link-hover-color'] ) {
				$css .= '
					#blogun-topbar .blogun-nav > ul > li > a:hover,
					#blogun-topbar .blogun-nav > ul > li.menu-item-has-children:hover > a,
					#blogun-topbar .blogun-nav > ul > li.current-menu-item > a,
					#blogun-topbar .blogun-nav > ul > li.current-menu-ancestor > a,
					#blogun-topbar .blogun-topbar-widget__text a:hover,
					#blogun-topbar .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon { 
						color: ' . blogun_sanitize_color( $topbar_color['link-hover-color'] ) . '; }
				';
			}

			/**
			 * Header.
			 */

			// Background.
			$css .= $this->get_design_options_field_css( '#blogun-header-inner', 'header_background', 'background' );

			// Font colors.
			$header_color = blogun_option( 'header_text_color' );

			// Header text color.
			if ( isset( $header_color['text-color'] ) && $header_color['text-color'] ) {
				$css .= '.blogun-logo .site-description { color: ' . blogun_sanitize_color( $header_color['text-color'] ) . '; }';
			}

			// Header link color.
			if ( isset( $header_color['link-color'] ) && $header_color['link-color'] ) {
				$css .= '
					#blogun-header,
					.blogun-header-widgets a:not(.blogun-btn),
					.blogun-logo a,
					.blogun-hamburger { 
						color: ' . blogun_sanitize_color( $header_color['link-color'] ) . '; }
				';
			}

			// Header link hover color.
			if ( isset( $header_color['link-hover-color'] ) && $header_color['link-hover-color'] ) {
				$css .= '
					.blogun-header-widgets a:not(.blogun-btn):hover, 
					#blogun-header-inner .blogun-header-widgets .blogun-active,
					.blogun-logo .site-title a:hover, 
					.blogun-hamburger:hover, 
					.is-mobile-menu-active .blogun-hamburger,
					#blogun-header-inner .blogun-nav > ul > li > a:hover,
					#blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,
					#blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,
					#blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,
					#blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,
					#blogun-header-inner .blogun-nav > ul > li.current_page_item > a,
					#blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a {
						color: ' . blogun_sanitize_color( $header_color['link-hover-color'] ) . ';
					}
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li > a:hover,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current_page_item > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a {
						color: ' . blogun_sanitize_color( $header_color['link-hover-color'] ) . ';
						border-color: ' . blogun_sanitize_color( $header_color['link-hover-color'] ) . ';
					}
				';
			}

			// Header border.
			$css .= $this->get_design_options_field_css( '#blogun-header-inner', 'header_border', 'border' );

			// Header separator color.
			$css .= $this->get_design_options_field_css( '.blogun-header-widget', 'header_border', 'separator_color' );

			// Main navigation breakpoint.
			$css .= '
				@media screen and (max-width: ' . intval( blogun_option( 'main_nav_mobile_breakpoint' ) ) . 'px) {

					#blogun-header-inner .blogun-nav {
						display: none;
						color: #000;
					}
					.blogun-mobile-toggen,
					.blogun-mobile-nav {
						display: inline-flex;
					}

					#blogun-header-inner {
						position: relative;
					}

					#blogun-header-inner .blogun-nav > ul > li > a {
						color: inherit;
					}

					#blogun-header-inner .blogun-nav-container {
						position: static;
						border: none;
					}

					#blogun-header-inner .site-navigation {
						display: none;
						position: absolute;
						top: 100%;
						width: 100%;
						height: 100%;
						min-height: 100vh;
						left: 0;
						right: 0;
						margin: -1px 0 0;
						background: #FFF;
						border-top: 1px solid #eaeaea;
						box-shadow: 0 15px 25px -10px  rgba(50, 52, 54, 0.125);
						z-index: 999;
						font-size: 1.7rem;
						padding: 0;
					}

					.blogun-header-layout-5 #blogun-header-inner .site-navigation {
						min-height: unset;
						border-radius: 15px;
						height: unset;
					}

					#blogun-header-inner .site-navigation > ul {
						overflow-y: auto;
						max-height: 68vh;
						display: block;
					}

					#blogun-header-inner .site-navigation > ul > li > a {
						padding: 0 !important;
					}

					#blogun-header-inner .site-navigation > ul li {
						display: block;
						width: 100%;
						padding: 0;
						margin: 0;
						margin-left: 0 !important;
					}

					#blogun-header-inner .site-navigation > ul .sub-menu {
						position: static;
						display: none;
						border: none;
						box-shadow: none;
						border: 0;
						opacity: 1;
						visibility: visible;
						font-size: 1.7rem;
						transform: none;
						background: #f8f8f8;
						pointer-events: all;
						min-width: initial;
						left: 0;
						padding: 0;
						margin: 0;
						border-radius: 0;
						line-height: inherit;
					}

					#blogun-header-inner .site-navigation > ul .sub-menu > li > a > span {
						padding-left: 50px !important;
					}

					#blogun-header-inner .site-navigation > ul .sub-menu .sub-menu > li > a > span {
						padding-left: 70px !important;
					}

					#blogun-header-inner .site-navigation > ul .sub-menu a > span {
						padding: 10px 30px 10px 50px;
					}

					#blogun-header-inner .site-navigation > ul a {
						padding: 0;
						position: relative;
						background: none;
					}

					#blogun-header-inner .site-navigation > ul li {
						border-bottom: 1px solid #eaeaea;
					}

					#blogun-header-inner .site-navigation > ul > li:last-child {
						border-bottom: 0;
					}

					#blogun-header-inner .site-navigation > ul a > span {
						padding: 10px 30px !important;
						width: 100%;
						display: block;
					}

					#blogun-header-inner .site-navigation > ul a > span::after,
					#blogun-header-inner .site-navigation > ul a > span::before {
						display: none !important;
					}

					#blogun-header-inner .site-navigation > ul a > span.description {
						display: none;
					}

					#blogun-header-inner .site-navigation > ul .menu-item-has-children > a {
						display: inline-flex;
    					width: 100%;
						max-width: calc(100% - 50px);
					}

					#blogun-header-inner .blogun-nav .menu-item-has-children>a > span, 
					#blogun-header-inner .blogun-nav .page_item_has_children>a > span {
					    border-right: 1px solid rgba(0,0,0,.09);
					}

					#blogun-header-inner .blogun-nav .menu-item-has-children>a > .blogun-icon, 
					#blogun-header-inner .blogun-nav .page_item_has_children>a > .blogun-icon {
						transform: none;
						width: 50px;
					    margin: 0;
					    position: absolute;
					    right: 0;
					    pointer-events: none;
					    height: 1em;
						display: none;
					}

					.blogun-header-layout-3 .blogun-widget-location-left .dropdown-item {
						left: auto;
						right: -7px;
					}

					.blogun-header-layout-3 .blogun-widget-location-left .dropdown-item::after {
						left: auto;
						right: 8px;
					}

					.blogun-nav .sub-menu li.current-menu-item > a {
						font-weight: 500;
					}

					.blogun-mobile-toggen {
						width: 50px;
						height: 1em;
						background: none;
						border: none;
						cursor: pointer;
					}

					.blogun-mobile-toggen .blogun-icon {
						transform: none;
						width: 50px;
						margin: 0;
						position: absolute;
						right: 0;
						pointer-events: none;
						height: 1em;
					}

					#blogun-header-inner .site-navigation > ul .menu-item-has-children.blogun-open > .blogun-mobile-toggen > .blogun-icon {
						transform: rotate(180deg);
					}

					' . $header_layout_3_additional_css . '
				}
			';

			/**
			 * Main Navigation.
			 */

			// Font Color.
			$main_nav_font_color = blogun_option( 'main_nav_font_color' );

			if ( $main_nav_font_color['link-color'] ) {
				$css .= '#blogun-header-inner .blogun-nav > ul > li > a { color: ' . $main_nav_font_color['link-color'] . '; }';
			}

			if ( $main_nav_font_color['link-hover-color'] ) {
				$css .= '
					#blogun-header-inner .blogun-nav > ul > li > a:hover,
					#blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,
					#blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,
					#blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,
					#blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,
					#blogun-header-inner .blogun-nav > ul > li.current_page_item > a,
					#blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a {
						color: ' . blogun_sanitize_color( $main_nav_font_color['link-hover-color'] ) . ';
					}
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li > a:hover,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current_page_item > a,
					.blogun-menu-animation-squareborder:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a {
						color: ' . blogun_sanitize_color( $main_nav_font_color['link-hover-color'] ) . ';
						border-color: ' . blogun_sanitize_color( $main_nav_font_color['link-hover-color'] ) . ';
					}
				';
			}

			if ( 'layout-3' === blogun_option( 'header_layout' ) ) {

				// Background.
				$css .= $this->get_design_options_field_css( '.blogun-header-layout-3 .blogun-nav-container', 'main_nav_background', 'background' );

				// Border.
				$css .= $this->get_design_options_field_css( '.blogun-header-layout-3 .blogun-nav-container', 'main_nav_border', 'border' );
			}

			// Font size.
			$css .= $this->get_range_field_css( '.blogun-nav.blogun-header-element, .blogun-header-layout-1 .blogun-header-widgets, .blogun-header-layout-2 .blogun-header-widgets', 'font-size', 'main_nav_font_size', false );

			/**
			 * Hero Section.
			 */
			if ( blogun_option( 'enable_hero' ) ) {
				// Hero height.
				$css .= '#hero .blogun-hover-slider .hover-slide-item { height: ' . intval( blogun_option( 'hero_hover_slider_height' ) ) . 'px; }';
			}

			/**
			 * Pre Footer.
			 */
			if ( blogun_option( 'enable_pre_footer_cta' ) ) {

				// Call to Action.
				if ( blogun_option( 'enable_pre_footer_cta' ) ) {

					$cta_style = absint( blogun_option( 'pre_footer_cta_style' ) );

					// Background.
					$cta_background = blogun_option( 'pre_footer_cta_background' );

					if ( 1 === $cta_style || is_customize_preview() ) {
						$css .= $this->get_design_options_field_css( '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::after', 'pre_footer_cta_background', 'background' );
					}

					if ( 2 === $cta_style || is_customize_preview() ) {
						$css .= $this->get_design_options_field_css( '.blogun-pre-footer-cta-style-2 #blogun-pre-footer::after', 'pre_footer_cta_background', 'background' );
					}

					if ( 'image' === $cta_background['background-type'] && isset( $cta_background['background-color-overlay'] ) && $cta_background['background-color-overlay'] ) {
						$css .= '
							.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::before,
			 				.blogun-pre-footer-cta-style-2 #blogun-pre-footer::before {
			 					background-color: ' . blogun_sanitize_color( $cta_background['background-color-overlay'] ) . ';
			 				}
			 				';
					}

					// Text color.
					$css .= $this->get_design_options_field_css( '#blogun-pre-footer .h2, #blogun-pre-footer .h3, #blogun-pre-footer .h4', 'pre_footer_cta_text_color', 'color' );

					// Border.
					if ( 1 === $cta_style || is_customize_preview() ) {
						$css .= $this->get_design_options_field_css( '.blogun-pre-footer-cta-style-1 #blogun-pre-footer .blogun-flex-row::before', 'pre_footer_cta_border', 'border' );
					}

					if ( 2 === $cta_style || is_customize_preview() ) {
						$css .= $this->get_design_options_field_css( '.blogun-pre-footer-cta-style-2 #blogun-pre-footer::before', 'pre_footer_cta_border', 'border' );
					}

					// Font size.
					$css .= $this->get_range_field_css( '#blogun-pre-footer .h3', 'font-size', 'pre_footer_cta_font_size', true );
				}
			}

			// Footer Background.
			if ( blogun_option( 'enable_footer' ) || blogun_option( 'enable_copyright' ) ) {

				// Background.
				$css .= $this->get_design_options_field_css( '#colophon', 'footer_background', 'background' );

				// Footer font color.
				$footer_font_color = blogun_option( 'footer_text_color' );

				// Footer text color.
				if ( isset( $footer_font_color['text-color'] ) && $footer_font_color['text-color'] ) {
					$css .= '
						#colophon { 
							color: ' . blogun_sanitize_color( $footer_font_color['text-color'] ) . ';
						}
					';
				}

				// Footer link color.
				if ( isset( $footer_font_color['link-color'] ) && $footer_font_color['link-color'] ) {
					$css .= '
						#colophon a { 
							color: ' . blogun_sanitize_color( $footer_font_color['link-color'] ) . '; 
						}
					';
				}

				// Footer link hover color.
				if ( isset( $footer_font_color['link-hover-color'] ) && $footer_font_color['link-hover-color'] ) {
					$css .= '
						#colophon a:not(.blogun-btn):hover,
						.using-keyboard #colophon a:not(.blogun-btn):focus,
						#colophon li.current_page_item > a,
						#colophon .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon { 
							color: ' . blogun_sanitize_color( $footer_font_color['link-hover-color'] ) . ';
						}
					';
				}

				// Footer widget title.
				if ( isset( $footer_font_color['widget-title-color'] ) && $footer_font_color['widget-title-color'] ) {
					$css .= '
						#colophon .widget-title, #colophon .wp-block-heading { 
							color: ' . blogun_sanitize_color( $footer_font_color['widget-title-color'] ) . ';
						}
					';
				}
			}

			// Main Footer border.
			if ( blogun_option( 'enable_footer' ) ) {

				// Border.
				$footer_border = blogun_option( 'footer_border' );

				if ( $footer_border['border-top-width'] ) {
					$css .= '
						#colophon {
							border-top-width: ' . intval( $footer_border['border-top-width'] ) . 'px;
							border-top-style: ' . sanitize_text_field( $footer_border['border-style'] ) . ';
							border-top-color: ' . blogun_sanitize_color( $footer_border['border-color'] ) . ';
						}
					';
				}

				if ( $footer_border['border-bottom-width'] ) {
					$css .= '
						#colophon {
							border-bottom-width: ' . intval( $footer_border['border-bottom-width'] ) . 'px;
							border-bottom-style: ' . sanitize_text_field( $footer_border['border-style'] ) . ';
							border-bottom-color: ' . blogun_sanitize_color( $footer_border['border-color'] ) . ';
						}
					';
				}
			}

			// Sidebar.
			$css .= '
				#secondary {
					width: ' . intval( blogun_option( 'sidebar_width' ) ) . '%;
				}

				body:not(.blogun-no-sidebar) #primary {
					max-width: ' . intval( 100 - intval( blogun_option( 'sidebar_width' ) ) ) . '%;
				}
			';

			// Content background.
			$boxed_content_background_color = blogun_sanitize_color( blogun_option( 'boxed_content_background_color' ) );

			// Boxed Separated Layout specific CSS.
			$css .= '
				.blogun-layout__boxed-separated .ticker-slider-items,
				.blogun-layout__boxed-separated .pyml-slider-items,
				.blogun-layout__boxed-separated.author .author-box,
				.blogun-layout__boxed-separated #content, 
				.blogun-layout__boxed-separated.blogun-sidebar-style-3 #secondary .blogun-widget, 
				.blogun-layout__boxed-separated.blogun-sidebar-style-3 .elementor-widget-sidebar .blogun-widget, 
				.blogun-layout__boxed-separated.archive .blogun-article,
				.blogun-layout__boxed-separated.blog .blogun-article, 
				.blogun-layout__boxed-separated.search-results .blogun-article, 
				.blogun-layout__boxed-separated.category .blogun-article {
					background-color: ' . $boxed_content_background_color . ';
				}				
			';

			$css .= '
				.blogun-layout__boxed #page {
					background-color: ' . $boxed_content_background_color . ';
				}
			';

			// Content text color.
			$content_text_color = blogun_sanitize_color( blogun_option( 'content_text_color' ) );

			$css .= '
				body {
					color: ' . $content_text_color . ';
				}

				:root {
					--blogun-secondary_38: ' . $content_text_color . ';
				}

				.comment-form .comment-notes,
				#comments .no-comments,
				#page .wp-caption .wp-caption-text,
				#comments .comment-meta,
				.comments-closed,
				.entry-meta,
				.blogun-entry cite,
				legend,
				.blogun-page-header-description,
				.page-links em,
				.site-content .page-links em,
				.single .entry-footer .last-updated,
				.single .post-nav .post-nav-title,
				#main .widget_recent_comments span,
				#main .widget_recent_entries span,
				#main .widget_calendar table > caption,
				.post-thumb-caption,
				.wp-block-image figcaption,
				.wp-block-embed figcaption {
					color: ' . $content_text_color . ';
				}
			';

			// blogun_hex2rgba( $content_text_color, 0.73 )
			// Lightened or darkened background color for backgrounds, borders & inputs.
			$background_color = blogun_sanitize_color( blogun_get_background_color() );

			$content_text_color_offset = blogun_sanitize_color( blogun_light_or_dark( $background_color, blogun_luminance( $background_color, -0.045 ), blogun_luminance( $background_color, 0.2 ) ) );
			// Only add for dark background color.
			if ( ! blogun_is_light_color( $background_color ) ) {
				$css .= '
					#content textarea,
					#content input[type="text"],
					#content input[type="number"],
					#content input[type="email"],
					#content input[type=password],
					#content input[type=tel],
					#content input[type=url],
					#content input[type=search],
					#content input[type=date] {
						background-color: ' . $background_color . ';
					}
				';

				// Offset border color.
				$css .= '
					.blogun-sidebar-style-3 #secondary .blogun-widget {
						border-color: ' . $content_text_color_offset . ';
					}
				';

				// Offset background color.
				$css .= '
					.entry-meta .entry-meta-elements > span:before {
						background-color: ' . $content_text_color_offset . ';
					}
				';
			}

			// Content link hover color.
			$css .= '
				.content-area a:not(.blogun-btn):not(.wp-block-button__link):hover,
				#secondary .hester-core-custom-list-widget .blogun-entry a:not(.blogun-btn):hover,
				.blogun-breadcrumbs a:hover {
					color: ' . blogun_sanitize_color( blogun_option( 'content_link_hover_color' ) ) . ';
				}
			';

			// Headings Color.
			$css .= '
				h1, h2, h3, h4, h5, h6,
				.h1, .h2, .h3, .h4,
				.blogun-logo .site-title,
				.error-404 .page-header h1 {
					color: ' . blogun_sanitize_color(blogun_option( 'headings_color' )) . ';
				}
				:root {
					--blogun-secondary: ' . blogun_sanitize_color(blogun_option( 'headings_color' )) . ';
				}
			';

			// Container width.
			$css .= '
				.blogun-container,
				.alignfull.blogun-wrap-content > div {
					max-width: ' . intval( blogun_option( 'container_width' ) ) . 'px;
				}

				.blogun-layout__boxed #page,
				.blogun-layout__boxed.blogun-sticky-header.blogun-is-mobile #blogun-header-inner,
				.blogun-layout__boxed.blogun-sticky-header:not(.blogun-header-layout-3, .blogun-header-layout-4) #blogun-header-inner,
				.blogun-layout__boxed.blogun-sticky-header:not(.blogun-is-mobile).blogun-header-layout-4 #blogun-header-inner .blogun-nav-container > .blogun-container,
				.blogun-layout__boxed.blogun-sticky-header:not(.blogun-is-mobile).blogun-header-layout-3 #blogun-header-inner .blogun-nav-container > .blogun-container {
					max-width: ' . ( intval( blogun_option( 'container_width' ) ) + 100 ) . 'px;
				}
			';

			// Adjust fullwidth sections for boxed layouts.
			if ( 'boxed' === blogun_option( 'site_layout' ) || is_customize_preview() ) {
				$css .= '
					@media screen and (max-width: ' . intval( blogun_option( 'container_width' ) ) . 'px) {
						body.blogun-layout__boxed.blogun-no-sidebar .elementor-section.elementor-section-stretched,
						body.blogun-layout__boxed.blogun-no-sidebar .blogun-fw-section,
						body.blogun-layout__boxed.blogun-no-sidebar .entry-content .alignfull {
							margin-left: -5rem !important;
							margin-right: -5rem !important;
						}
					}
				';
			}

			// Logo max height.
			$css .= $this->get_range_field_css( '.blogun-logo img', 'max-height', 'logo_max_height' );
			$css .= $this->get_range_field_css( '.blogun-logo img.blogun-svg-logo', 'height', 'logo_max_height' );

			// Logo margin.
			$css .= $this->get_spacing_field_css( '.blogun-logo .logo-inner', 'margin', 'logo_margin' );

			/**
			 * Transparent header.
			 */

			// Logo max height.
			$css .= $this->get_range_field_css( '.blogun-tsp-header .blogun-logo img', 'max-height', 'tsp_logo_max_height' );
			$css .= $this->get_range_field_css( '.blogun-tsp-header .blogun-logo img.blogun-svg-logo', 'height', 'tsp_logo_max_height' );

			// Logo margin.
			$css .= $this->get_spacing_field_css( '.blogun-tsp-header .blogun-logo .logo-inner', 'margin', 'tsp_logo_margin' );

			// Main Header custom background.
			$css .= $this->get_design_options_field_css( '.blogun-tsp-header #blogun-header-inner', 'tsp_header_background', 'background' );

			/** Font Colors */

			$tsp_font_color = blogun_option( 'tsp_header_font_color' );

			// Header text color.
			if ( isset( $tsp_font_color['text-color'] ) && $tsp_font_color['text-color'] ) {
				$css .= '
					.blogun-tsp-header .blogun-logo .site-description {
						color: ' . blogun_sanitize_color( $tsp_font_color['text-color'] ) . ';
					}
				';
			}

			// Header link color.
			if ( isset( $tsp_font_color['link-color'] ) && $tsp_font_color['link-color'] ) {
				$css .= '
					.blogun-tsp-header #blogun-header,
					.blogun-tsp-header .blogun-header-widgets a:not(.blogun-btn),
					.blogun-tsp-header .blogun-logo a,
					.blogun-tsp-header .blogun-hamburger,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li > a { 
						color: ' . blogun_sanitize_color( $tsp_font_color['link-color'] ) . ';
					}
				';
			}

			// Header link hover color.
			if ( isset( $tsp_font_color['link-hover-color'] ) && $tsp_font_color['link-hover-color'] ) {
				$css .= '
					.blogun-tsp-header .blogun-header-widgets a:not(.blogun-btn):hover, 
					.blogun-tsp-header #blogun-header-inner .blogun-header-widgets .blogun-active,
					.blogun-tsp-header .blogun-logo .site-title a:hover, 
					.blogun-tsp-header .blogun-hamburger:hover, 
					.is-mobile-menu-active .blogun-tsp-header .blogun-hamburger,
					.blogun-tsp-header.using-keyboard .site-title a:focus,
					.blogun-tsp-header.using-keyboard .blogun-header-widgets a:not(.blogun-btn):focus,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.hovered > a,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li > a:hover,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current_page_item > a,
					.blogun-tsp-header #blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a {
						color: ' . blogun_sanitize_color( $tsp_font_color['link-hover-color'] ) . ';
					}
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li > a:hover,
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.menu-item-has-children:hover > a,
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current-menu-item > a,
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current-menu-ancestor > a,
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.page_item_has_children:hover > a,
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current_page_item > a,
					.blogun-menu-animation-squareborder.blogun-tsp-header:not(.blogun-is-mobile) #blogun-header-inner .blogun-nav > ul > li.current_page_ancestor > a {
						color: ' . blogun_sanitize_color( $tsp_font_color['link-hover-color'] ) . ';
						border-color: ' . blogun_sanitize_color( $tsp_font_color['link-hover-color'] ) . ';
					}
				';
			}

			/** Border Color */
			$css .= $this->get_design_options_field_css( '.blogun-tsp-header #blogun-header-inner', 'tsp_header_border', 'border' );

			/** Separator Color */
			$css .= $this->get_design_options_field_css( '.blogun-tsp-header .blogun-header-widget', 'tsp_header_border', 'separator_color' );

			/**
			 * Page Header.
			 */
			if ( blogun_option( 'page_header_enable' ) ) {

				// Font size.
				$css .= $this->get_range_field_css( '#page .page-header .page-title', 'font-size', 'page_header_font_size', true );

				// Page Title spacing.
				$css .= $this->get_spacing_field_css( '.blogun-page-title-align-left .page-header.blogun-has-page-title, .blogun-page-title-align-right .page-header.blogun-has-page-title, .blogun-page-title-align-center .page-header .blogun-page-header-wrapper', 'padding', 'page_header_spacing' );

				// Page Header background.
				$css .= $this->get_design_options_field_css( '.blogun-tsp-header:not(.blogun-tsp-absolute) #masthead', 'page_header_background', 'background' );
				$css .= $this->get_design_options_field_css( '.page-header', 'page_header_background', 'background' );

				// Page Header font color.
				$page_header_color = blogun_option( 'page_header_text_color' );

				// Page Header text color.
				if ( isset( $page_header_color['text-color'] ) && $page_header_color['text-color'] ) {
					$css .= '
						.page-header .page-title { 
							color: ' . blogun_sanitize_color( $page_header_color['text-color'] ) . '; }

						.page-header .blogun-page-header-description {
							color: ' . blogun_sanitize_color( blogun_hex2rgba( $page_header_color['text-color'], 0.75 ) ) . '; 
						}
					';
				}

				// Page Header link color.
				if ( isset( $page_header_color['link-color'] ) && $page_header_color['link-color'] ) {
					$css .= '
						.page-header .blogun-breadcrumbs a { 
							color: ' . blogun_sanitize_color( $page_header_color['link-color'] ) . '; }

						.page-header .blogun-breadcrumbs span,
						.page-header .breadcrumb-trail .trail-items li::after, .page-header .blogun-breadcrumbs .separator {
							color: ' . blogun_sanitize_color( blogun_hex2rgba( $page_header_color['link-color'], 0.75 ) ) . '; 
						}
					';
				}

				// Page Header link hover color.
				if ( isset( $page_header_color['link-hover-color'] ) && $page_header_color['link-hover-color'] ) {
					$css .= '
						.page-header .blogun-breadcrumbs a:hover { 
							color: ' . blogun_sanitize_color( $page_header_color['link-hover-color'] ). '; }
					';
				}

				// Page Header border color.
				$page_header_border = blogun_option( 'page_header_border' );

				$css .= $this->get_design_options_field_css( '.page-header', 'page_header_border', 'border' );
			}

			/**
			 * Breadcrumbs.
			 */
			if ( blogun_option( 'breadcrumbs_enable' ) ) {

				// Spacing.
				$css .= $this->get_spacing_field_css( '.blogun-breadcrumbs', 'padding', 'breadcrumbs_spacing' );

				if ( 'below-header' === blogun_option( 'breadcrumbs_position' ) ) {

					// Background.
					$css .= $this->get_design_options_field_css( '.blogun-breadcrumbs', 'breadcrumbs_background', 'background' );

					// Border.
					$css .= $this->get_design_options_field_css( '.blogun-breadcrumbs', 'breadcrumbs_border', 'border' );

					// Text Color.
					$css .= $this->get_design_options_field_css( '.blogun-breadcrumbs', 'breadcrumbs_text_color', 'color' );
				}
			}

			/**
			 * Copyright Bar.
			 */
			if ( blogun_option( 'enable_copyright' ) ) {
				$css .= $this->get_design_options_field_css( '#blogun-copyright', 'copyright_background', 'background' );

				// Copyright font color.
				$copyright_color = blogun_option( 'copyright_text_color' );

				// Copyright text color.
				if ( isset( $copyright_color['text-color'] ) && $copyright_color['text-color'] ) {
					$css .= '
						#blogun-copyright { 
							color: ' . blogun_sanitize_color( $copyright_color['text-color'] ) . '; }
					';
				}

				// Copyright link color.
				if ( isset( $copyright_color['link-color'] ) && $copyright_color['link-color'] ) {
					$css .= '
						#blogun-copyright a { 
							color: ' . blogun_sanitize_color( $copyright_color['link-color'] ) . '; }
					';
				}

				// Copyright link hover color.
				if ( isset( $copyright_color['link-hover-color'] ) && $copyright_color['link-hover-color'] ) {
					$css .= '
						#blogun-copyright a:hover,
						.using-keyboard #blogun-copyright a:focus,
						#blogun-copyright .blogun-social-nav > ul > li > a .blogun-icon.bottom-icon,
						#blogun-copyright .blogun-nav > ul > li.current-menu-item > a,
						#blogun-copyright .blogun-nav > ul > li.current-menu-ancestor > a,
						#blogun-copyright .blogun-nav > ul > li:hover > a { 
							color: ' . blogun_sanitize_color( $copyright_color['link-hover-color'] ) . '; }
					';
				}

				// Copyright separator color.
				$footer_text_color = blogun_option( 'footer_text_color' );
				$footer_text_color = $footer_text_color['text-color'];

				$copyright_separator_color = blogun_light_or_dark( $footer_text_color, 'rgba(255,255,255,0.1)', 'rgba(0,0,0,0.1)' );

				$css .= '
					#blogun-copyright.contained-separator > .blogun-container::before {
						background-color: ' . blogun_sanitize_color( $copyright_separator_color ) . ';
					}

					#blogun-copyright.fw-separator {
						border-top-color: ' . blogun_sanitize_color( $copyright_separator_color ) . ';
					}
				';
			}

			/**
			 * Typography.
			 */

			// Base HTML font size.
			$css .= $this->get_range_field_css( 'html', 'font-size', 'html_base_font_size', true, '%' );

			// Font smoothing.
			if ( blogun_option( 'font_smoothing' ) ) {
				$css .= '
					* {
						-moz-osx-font-smoothing: grayscale;
						-webkit-font-smoothing: antialiased;
					}
				';
			}

			// Body.
			$css .= $this->get_typography_field_css( 'body', 'body_font' );

			// Headings.
			$css .= $this->get_typography_field_css( 'h1, .h1, .blogun-logo .site-title, .page-header .page-title, h2, .h2, h3, .h3, h4, .h4, h5, .h5, h6, .h6', 'headings_font' );

			$css .= $this->get_typography_field_css( 'h1, .h1, .blogun-logo .site-title, .page-header .page-title', 'h1_font' );
			$css .= $this->get_typography_field_css( 'h2, .h2', 'h2_font' );
			$css .= $this->get_typography_field_css( 'h3, .h3', 'h3_font' );
			$css .= $this->get_typography_field_css( 'h4, .h4', 'h4_font' );
			$css .= $this->get_typography_field_css( 'h5, .h5', 'h5_font' );
			$css .= $this->get_typography_field_css( 'h6, .h6', 'h6_font' );
			$css .= $this->get_typography_field_css( 'h1 em, h2 em, h3 em, h4 em, h5 em, h6 em, .h1 em, .h2 em, .h3 em, .h4 em, .h5 em, .h6 em, .blogun-logo .site-title em, .error-404 .page-header h1 em', 'heading_em_font' );

			// Emphasized Heading.
			$css .= $this->get_typography_field_css( 'h1 em, h2 em, h3 em, h4 em, h5 em, h6 em, .h1 em, .h2 em, .h3 em, .h4 em, .h5 em, .h6 em, .blogun-logo .site-title em, .error-404 .page-header h1 em', 'heading_em_font' );

			// Site Title font size.
			$css .= $this->get_range_field_css( '#blogun-header .blogun-logo .site-title', 'font-size', 'logo_text_font_size', true );

			// Sidebar widget title.
			$css .= $this->get_range_field_css( '#main .widget-title, .widget-area .wp-block-heading', 'font-size', 'sidebar_widget_title_font_size', true );

			// Footer widget title.
			$css .= $this->get_range_field_css( '#colophon .widget-title, #colophon .wp-block-heading', 'font-size', 'footer_widget_title_font_size', true );

			// Blog Single Post - Title Spacing.
			$css .= $this->get_spacing_field_css( '.blogun-single-title-in-page-header #page .page-header .blogun-page-header-wrapper', 'padding', 'single_title_spacing', true );

			// Blog Single Post - Content Font Size.
			$css .= $this->get_range_field_css( '.single-post .entry-content', 'font-size', 'single_content_font_size', true );

			// Blog Single Post - narrow container.
			if ( 'narrow' === blogun_option( 'single_content_width' ) ) {
				$css .= '
					.single-post.narrow-content .entry-content > :not([class*="align"]):not([class*="gallery"]):not(.wp-block-image):not(.quote-inner):not(.quote-post-bg), 
					.single-post.narrow-content .mce-content-body:not([class*="page-template-full-width"]) > :not([class*="align"]):not([data-wpview-type*="gallery"]):not(blockquote):not(.mceTemp), 
					.single-post.narrow-content .entry-footer, 
					.single-post.narrow-content .entry-content > .alignwide,
					.single-post.narrow-content p.has-background:not(.alignfull):not(.alignwide),
					.single-post.narrow-content .post-nav, 
					.single-post.narrow-content #blogun-comments-toggle, 
					.single-post.narrow-content #comments, 
					.single-post.narrow-content .entry-content .aligncenter, .single-post.narrow-content .blogun-narrow-element, 
					.single-post.narrow-content.blogun-single-title-in-content .entry-header, 
					.single-post.narrow-content.blogun-single-title-in-content .entry-meta, 
					.single-post.narrow-content.blogun-single-title-in-content .post-category,
					.single-post.narrow-content.blogun-no-sidebar .blogun-page-header-wrapper,
					.single-post.narrow-content.blogun-no-sidebar .blogun-breadcrumbs nav {
						max-width: ' . intval( blogun_option( 'single_narrow_container_width' ) ) . 'px;
						margin-left: auto;
						margin-right: auto;
					}

					.single-post.narrow-content .author-box,
					.single-post.narrow-content .entry-content > .alignwide,
					.single.blogun-single-title-in-page-header .page-header.blogun-align-center .blogun-page-header-wrapper {
						max-width: ' . ( intval( blogun_option( 'single_narrow_container_width' ) ) + 70 ) . 'px;
					}
				';
			}

			// Allow CSS to be filtered.
			$css = apply_filters( 'blogun_dynamic_styles', $css );

			// Add user custom CSS.
			if ( $custom_css || ! is_customize_preview() ) {
				$css .= wp_get_custom_css();
			}

			// Minify the CSS code.
			$css = $this->minify( $css );

			return $css;
		}

		/**
		 * Update dynamic css file with new CSS. Cleans caches after that.
		 *
		 * @return [Boolean] returns true if successfully updated the dynamic file.
		 */
		public function update_dynamic_file() {

			$css = $this->get_css( true );

			if ( empty( $css ) || '' === trim( $css ) ) {
				return;
			}

			// Load file.php file.
			require_once ABSPATH . 'wp-admin' . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR . 'file.php'; // phpcs:ignore

			global $wp_filesystem;

			// Check if the the global filesystem isn't setup yet.
			if ( is_null( $wp_filesystem ) ) {
				WP_Filesystem();
			}

			$wp_filesystem->mkdir( $this->dynamic_css_path );

			if ( $wp_filesystem->put_contents( $this->dynamic_css_path . 'dynamic-styles.css', $css ) ) {
				$this->clean_cache();
				set_transient( 'blogun_has_dynamic_css', true, 0 );
				return true;
			}

			return false;
		}

		/**
		 * Delete dynamic css file.
		 *
		 * @return void
		 */
		public function delete_dynamic_file() {

			// Load file.php file.
			require_once ABSPATH . 'wp-admin' . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR . 'file.php'; // phpcs:ignore

			global $wp_filesystem;

			// Check if the the global filesystem isn't setup yet.
			if ( is_null( $wp_filesystem ) ) {
				WP_Filesystem();
			}

			$wp_filesystem->delete( $this->dynamic_css_path . 'dynamic-styles.css' );

			delete_transient( 'blogun_has_dynamic_css' );
		}

		/**
		 * Simple CSS code minification.
		 *
		 * @param  string $css code to be minified.
		 * @return string, minifed code
		 * @since  1.0.0
		 */
		private function minify( $css ) {
			$css = preg_replace( '/\s+/', ' ', $css );
			$css = preg_replace( '/\/\*[^\!](.*?)\*\//', '', $css );
			$css = preg_replace( '/(,|:|;|\{|}) /', '$1', $css );
			$css = preg_replace( '/ (,|;|\{|})/', '$1', $css );
			$css = preg_replace( '/(:| )0\.([0-9]+)(%|em|ex|px|in|cm|mm|pt|pc)/i', '${1}.${2}${3}', $css );
			$css = preg_replace( '/(:| )(\.?)0(%|em|ex|px|in|cm|mm|pt|pc)/i', '${1}0', $css );

			return trim( $css );
		}

		/**
		 * Cleans various caches. Compatible with cache plugins.
		 *
		 * @since 1.0.0
		 */
		private function clean_cache() {

			// If W3 Total Cache is being used, clear the cache.
			if ( function_exists( 'w3tc_pgcache_flush' ) ) {
				w3tc_pgcache_flush();
			}

			// if WP Super Cache is being used, clear the cache.
			if ( function_exists( 'wp_cache_clean_cache' ) ) {
				global $file_prefix;
				wp_cache_clean_cache( $file_prefix );
			}

			// If SG CachePress is installed, reset its caches.
			if ( class_exists( 'SG_CachePress_Supercacher' ) ) {
				if ( method_exists( 'SG_CachePress_Supercacher', 'purge_cache' ) ) {
					SG_CachePress_Supercacher::purge_cache();
				}
			}

			// Clear caches on WPEngine-hosted sites.
			if ( class_exists( 'WpeCommon' ) ) {

				if ( method_exists( 'WpeCommon', 'purge_memcached' ) ) {
					WpeCommon::purge_memcached();
				}

				if ( method_exists( 'WpeCommon', 'clear_maxcdn_cache' ) ) {
					WpeCommon::clear_maxcdn_cache();
				}

				if ( method_exists( 'WpeCommon', 'purge_varnish_cache' ) ) {
					WpeCommon::purge_varnish_cache();
				}
			}

			// Clean OpCache.
			if ( function_exists( 'opcache_reset' ) ) {
				opcache_reset(); // phpcs:ignore PHPCompatibility.FunctionUse.NewFunctions.opcache_resetFound
			}

			// Clean WordPress cache.
			if ( function_exists( 'wp_cache_flush' ) ) {
				wp_cache_flush();
			}
		}

		/**
		 * Prints spacing field CSS based on passed params.
		 *
		 * @since  1.0.0
		 *
		 * @param  string $css_selector CSS selector.
		 * @param  string $css_property CSS property, such as 'margin', 'padding' or 'border'.
		 * @param  string $setting_id The ID of the customizer setting containing all information about the setting.
		 * @param  bool   $responsive Has responsive values.
		 * @return string  Generated CSS.
		 */
		public function get_spacing_field_css( $css_selector, $css_property, $setting_id, $responsive = true ) {

			// Get the saved setting.
			$setting = blogun_option( $setting_id );

			// If setting doesn't exist, return.
			if ( ! is_array( $setting ) ) {
				return;
			}

			// Get the unit. Defaults to px.
			$unit = 'px';

			if ( isset( $setting['unit'] ) ) {
				if ( $setting['unit'] ) {
					$unit = $setting['unit'];
				}

				unset( $setting['unit'] );
			}

			// CSS buffer.
			$css_buffer = '';

			// Loop through options.
			foreach ( $setting as $key => $value ) {

				// Check if responsive options are available.
				if ( is_array( $value ) ) {

					if ( 'desktop' === $key ) {
						$mq_open  = '';
						$mq_close = '';
					} elseif ( 'tablet' === $key ) {
						$mq_open  = '@media only screen and (max-width: 768px) {';
						$mq_close = '}';
					} elseif ( 'mobile' === $key ) {
						$mq_open  = '@media only screen and (max-width: 480px) {';
						$mq_close = '}';
					} else {
						$mq_open  = '';
						$mq_close = '';
					}

					// Add media query prefix.
					$css_buffer .= $mq_open . $css_selector . '{';

					// Loop through all choices.
					foreach ( $value as $pos => $val ) {

						if ( empty( $val ) ) {
							continue;
						}

						if ( 'border' === $css_property ) {
							$pos .= '-width';
						}

						$css_buffer .= $css_property . '-' . $pos . ': ' . intval( $val ) . $unit . ';';
					}

					$css_buffer .= '}' . $mq_close;
				} else {

					if ( 'border' === $css_property ) {
						$key .= '-width';
					}

					$css_buffer .= $css_property . '-' . $key . ': ' . intval( $value ) . $unit . ';';
				}
			}

			// Check if field is has responsive values.
			if ( ! $responsive ) {
				$css_buffer = $css_selector . '{' . $css_buffer . '}';
			}

			// Finally, return the generated CSS code.
			return $css_buffer;
		}

		/**
		 * Prints range field CSS based on passed params.
		 *
		 * @since  1.0.0
		 *
		 * @param  string $css_selector CSS selector.
		 * @param  string $css_property CSS property, such as 'margin', 'padding' or 'border'.
		 * @param  string $setting_id The ID of the customizer setting containing all information about the setting.
		 * @param  bool   $responsive Has responsive values.
		 * @param  string $unit Unit.
		 * @return string  Generated CSS.
		 */
		public function get_range_field_css( $css_selector, $css_property, $setting_id, $responsive = true, $unit = 'px' ) {

			// Get the saved setting.
			$setting = blogun_option( $setting_id );

			// If just a single value option.
			if ( ! is_array( $setting ) ) {
				return $css_selector . ' { ' . $css_property . ': ' . $setting . $unit . '; }';
			}

			// Resolve units.
			if ( isset( $setting['unit'] ) ) {
				if ( $setting['unit'] ) {
					$unit = $setting['unit'];
				}

				unset( $setting['unit'] );
			}

			// CSS buffer.
			$css_buffer = '';

			if ( is_array( $setting ) && ! empty( $setting ) ) {

				// Media query syntax wrap.
				$mq_open  = '';
				$mq_close = '';

				// Loop through options.
				foreach ( $setting as $key => $value ) {

					if ( empty( $value ) ) {
						continue;
					}

					if ( 'desktop' === $key ) {
						$mq_open  = '';
						$mq_close = '';
					} elseif ( 'tablet' === $key ) {
						$mq_open  = '@media only screen and (max-width: 768px) {';
						$mq_close = '}';
					} elseif ( 'mobile' === $key ) {
						$mq_open  = '@media only screen and (max-width: 480px) {';
						$mq_close = '}';
					} else {
						$mq_open  = '';
						$mq_close = '';
					}

					// Add media query prefix.
					$css_buffer .= $mq_open . $css_selector . '{';
					$css_buffer .= $css_property . ': ' . floatval( $value ) . $unit . ';';
					$css_buffer .= '}' . $mq_close;
				}
			}

			// Finally, return the generated CSS code.
			return $css_buffer;
		}

		/**
		 * Prints design options field CSS based on passed params.
		 *
		 * @since 1.0.0
		 * @param string       $css_selector CSS selector.
		 * @param string|mixed $setting The ID of the customizer setting containing all information about the setting.
		 * @param string       $type Design options field type.
		 * @return string      Generated CSS.
		 */
		public function get_design_options_field_css( $css_selector, $setting, $type ) {

			if ( is_string( $setting ) ) {
				// Get the saved setting.
				$setting = blogun_option( $setting );
			}

			// Setting has to be array.
			if ( ! is_array( $setting ) || empty( $setting ) ) {
				return;
			}

			// CSS buffer.
			$css_buffer = '';

			// Background.
			if ( 'background' === $type ) {

				// Background type.
				$background_type = $setting['background-type'];
				
				if ( 'color' === $background_type ) {
					if ( isset( $setting['background-color'] ) && ! empty( $setting['background-color'] ) ) {
						$css_buffer .= 'background: ' . blogun_sanitize_color( $setting['background-color'] ) . ';';
					}
				} elseif ( 'gradient' === $background_type ) {

					$css_buffer .= 'background: ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ';';

					if ( 'linear' === $setting['gradient-type'] ) {
						$css_buffer .= '
							background: -webkit-linear-gradient(' . intval( $setting['gradient-linear-angle'] ) . 'deg, ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ' ' . intval( $setting['gradient-color-1-location'] ) . '%, ' . blogun_sanitize_color( $setting['gradient-color-2'] ) . ' ' . intval( $setting['gradient-color-2-location'] ) . '%);
							background: -o-linear-gradient(' . intval( $setting['gradient-linear-angle'] ) . 'deg, ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ' ' . intval( $setting['gradient-color-1-location'] ) . '%, ' . blogun_sanitize_color( $setting['gradient-color-2'] ) . ' ' . intval( $setting['gradient-color-2-location'] ) . '%);
							background: linear-gradient(' . intval( $setting['gradient-linear-angle'] ) . 'deg, ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ' ' . intval( $setting['gradient-color-1-location'] ) . '%, ' . blogun_sanitize_color( $setting['gradient-color-2'] ) . ' ' . intval( $setting['gradient-color-2-location'] ) . '%);

						';
					} elseif ( 'radial' === $setting['gradient-type'] ) {
						$css_buffer .= '
							background: -webkit-radial-gradient(' . sanitize_text_field( $setting['gradient-position'] ) . ', circle, ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ' ' . intval( $setting['gradient-color-1-location'] ) . '%, ' . blogun_sanitize_color( $setting['gradient-color-2'] ) . ' ' . intval( $setting['gradient-color-2-location'] ) . '%);
							background: -o-radial-gradient(' . sanitize_text_field( $setting['gradient-position'] ) . ', circle, ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ' ' . intval( $setting['gradient-color-1-location'] ) . '%, ' . blogun_sanitize_color( $setting['gradient-color-2'] ) . ' ' . intval( $setting['gradient-color-2-location'] ) . '%);
							background: radial-gradient(circle at ' . sanitize_text_field( $setting['gradient-position'] ) . ', ' . blogun_sanitize_color( $setting['gradient-color-1'] ) . ' ' . intval( $setting['gradient-color-1-location'] ) . '%, ' . blogun_sanitize_color( $setting['gradient-color-2'] ) . ' ' . intval( $setting['gradient-color-2-location'] ) . '%);
						';
					}
				} elseif ( 'image' === $background_type ) {
					$css_buffer .= '
						background-image: url(' . esc_url( $setting['background-image'] ) . ');
						background-size: ' . sanitize_text_field( $setting['background-size'] ) . ';
						background-attachment: ' . sanitize_text_field( $setting['background-attachment'] ) . ';
						background-position: ' . intval( $setting['background-position-x'] ) . '% ' . intval( $setting['background-position-y'] ) . '%;
						background-repeat: ' . sanitize_text_field( $setting['background-repeat'] ) . ';
					';
				}

				$css_buffer = ! empty( $css_buffer ) ? $css_selector . '{' . $css_buffer . '}' : '';

				if ( 'image' === $background_type && isset( $setting['background-color-overlay'] ) && $setting['background-color-overlay'] && isset( $setting['background-image'] ) && $setting['background-image'] ) {
					$css_buffer .= $css_selector . '::after { background-color: ' . blogun_sanitize_color( $setting['background-color-overlay'] ) . '; }';
				}
			} elseif ( 'color' === $type ) {

				// Text color.
				if ( isset( $setting['text-color'] ) && ! empty( $setting['text-color'] ) ) {
					$css_buffer .= $css_selector . ' { color: ' . blogun_sanitize_color( $setting['text-color'] ) . '; }';
				}

				// Link Color.
				if ( isset( $setting['link-color'] ) && ! empty( $setting['link-color'] ) ) {
					$css_buffer .= $css_selector . ' a { color: ' . blogun_sanitize_color( $setting['link-color'] ) . '; }';
				}

				// Link Hover Color.
				if ( isset( $setting['link-hover-color'] ) && ! empty( $setting['link-hover-color'] ) ) {
					$css_buffer .= $css_selector . ' a:hover { color: ' . blogun_sanitize_color( $setting['link-hover-color'] ) . ' !important; }';
				}
			} elseif ( 'border' === $type ) {

				// Color.
				if ( isset( $setting['border-color'] ) && ! empty( $setting['border-color'] ) ) {
					$css_buffer .= 'border-color:' . blogun_sanitize_color( $setting['border-color'] ) . ';';
				}

				// Style.
				if ( isset( $setting['border-style'] ) && ! empty( $setting['border-style'] ) ) {
					$css_buffer .= 'border-style: ' . sanitize_text_field( $setting['border-style'] ) . ';';
				}

				// Width.
				$positions = array( 'top', 'right', 'bottom', 'left' );

				foreach ( $positions as $position ) {
					if ( isset( $setting[ 'border-' . $position . '-width' ] ) && ! empty( $setting[ 'border-' . $position . '-width' ] ) ) {
						$css_buffer .= 'border-' . sanitize_text_field( $position ) . '-width: ' . $setting[ 'border-' . sanitize_text_field( $position ) . '-width' ] . 'px;';
					}
				}

				$css_buffer = ! empty( $css_buffer ) ? $css_selector . '{' . $css_buffer . '}' : '';
			} elseif ( 'separator_color' === $type && isset( $setting['separator-color'] ) && ! empty( $setting['separator-color'] ) ) {

				// Separator Color.
				$css_buffer .= $css_selector . '::after { background-color:' . blogun_sanitize_color( $setting['separator-color'] ) . '; }';
			}

			// Finally, return the generated CSS code.
			return $css_buffer;
		}

		/**
		 * Prints typography field CSS based on passed params.
		 *
		 * @since  1.0.0
		 * @param  string       $css_selector CSS selector.
		 * @param  string|mixed $setting The ID of the customizer setting containing all information about the setting.
		 * @return string       Generated CSS.
		 */
		public function get_typography_field_css( $css_selector, $setting ) {

			if ( is_string( $setting ) ) {
				// Get the saved setting.
				$setting = blogun_option( $setting );
			}

			// Setting has to be array.
			if ( ! is_array( $setting ) || empty( $setting ) ) {
				return;
			}

			// CSS buffer.
			$css_buffer = '';

			// Properties.
			$properties = array(
				'font-weight',
				'font-style',
				'text-transform',
				'text-decoration',
			);

			foreach ( $properties as $property ) {

				if ( 'inherit' !== $setting[ $property ] ) {
					$css_buffer .= $property . ':' . $setting[ $property ] . ';';
				}
			}

			// Font family.
			if ( 'inherit' !== $setting['font-family'] ) {
				$font_family = blogun()->fonts->get_font_family( $setting['font-family'] );

				$css_buffer .= 'font-family: ' . sanitize_text_field( $font_family ) . ';';
			}

			// Letter spacing.
			if ( ! empty( $setting['letter-spacing'] ) ) {
				$css_buffer .= 'letter-spacing:' . floatval( $setting['letter-spacing'] ) . sanitize_text_field( $setting['letter-spacing-unit'] ) . ';';
			}

			// Font size.
			if ( ! empty( $setting['font-size-desktop'] ) ) {
				$css_buffer .= 'font-size:' . floatval( $setting['font-size-desktop'] ) . sanitize_text_field( $setting['font-size-unit'] ) . ';';
			}

			// Line Height.
			if ( ! empty( $setting['line-height-desktop'] ) ) {
				$css_buffer .= 'line-height:' . floatval( $setting['line-height-desktop'] ) . ';';
			}

			$css_buffer = $css_buffer ? $css_selector . '{' . $css_buffer . '}' : '';

			// Responsive options - tablet.
			$tablet = '';

			if ( ! empty( $setting['font-size-tablet'] ) ) {
				$tablet .= 'font-size:' . floatval( $setting['font-size-tablet'] ) . sanitize_text_field( $setting['font-size-unit'] ) . ';';
			}

			if ( ! empty( $setting['line-height-tablet'] ) ) {
				$tablet .= 'line-height:' . floatval( $setting['line-height-tablet'] ) . ';';
			}

			$tablet = ! empty( $tablet ) ? '@media only screen and (max-width: 768px) {' . $css_selector . '{' . $tablet . '} }' : '';

			$css_buffer .= $tablet;

			// Responsive options - mobile.
			$mobile = '';

			if ( ! empty( $setting['font-size-mobile'] ) ) {
				$mobile .= 'font-size:' . floatval( $setting['font-size-mobile'] ) . sanitize_text_field( $setting['font-size-unit'] ) . ';';
			}

			if ( ! empty( $setting['line-height-mobile'] ) ) {
				$mobile .= 'line-height:' . floatval( $setting['line-height-mobile'] ) . ';';
			}

			$mobile = ! empty( $mobile ) ? '@media only screen and (max-width: 480px) {' . $css_selector . '{' . $mobile . '} }' : '';

			$css_buffer .= $mobile;

			// Equeue google fonts.
			if ( blogun()->fonts->is_google_font( $setting['font-family'] ) ) {

				$params = array();

				if ( 'inherit' !== $setting['font-weight'] ) {
					$params['weight'] = $setting['font-weight'];
				}

				if ( 'inherit' !== $setting['font-style'] ) {
					$params['style'] = $setting['font-style'];
				}

				if ( $setting['font-subsets'] && ! empty( $setting['font-subsets'] ) ) {
					$params['subsets'] = $setting['font-subsets'];
				}


				blogun()->fonts->enqueue_google_font(
					$setting['font-family'],
					$params
				);
			}

			// Finally, return the generated CSS code.
			return $css_buffer;
		}

		/**
		 * Filters the dynamic styles to include button styles and makes sure it has the highest priority.
		 *
		 * @since  1.0.0
		 * @param  string $css The dynamic CSS.
		 * @return string Filtered dynamic CSS.
		 */
		public function get_button_styles( $css ) {

			/**
			 * Primary Button.
			 */

			$primary_button_selector = '
				.blogun-btn,
				body:not(.wp-customizer) input[type=submit], 
				.site-main .woocommerce #respond input#submit, 
				.site-main .woocommerce a.button, 
				.site-main .woocommerce button.button, 
				.site-main .woocommerce input.button, 
				.woocommerce ul.products li.product .added_to_cart, 
				.woocommerce ul.products li.product .button, 
				.woocommerce div.product form.cart .button, 
				.woocommerce #review_form #respond .form-submit input, 
				#infinite-handle span';

			/*
			$primary_button_bg_color      = blogun_option( 'primary_button_bg_color' );
			$primary_button_border_radius = blogun_option( 'primary_button_border_radius' );

			if ( '' !== $primary_button_bg_color ) {
				$css .= $primary_button_selector . ' {
					background-color: ' . $primary_button_bg_color . ';
				}';
			}
			*/
			// Primary button text color, border color & border width.
			/*
			$css .= $primary_button_selector . ' {
				color: ' . blogun_option( 'primary_button_text_color' ) . ';
				border-color: ' . blogun_option( 'primary_button_border_color' ) . ';
				border-width: ' . blogun_option( 'primary_button_border_width' ) . 'rem;
				border-top-left-radius: ' . $primary_button_border_radius['top-left'] . 'rem;
				border-top-right-radius: ' . $primary_button_border_radius['top-right'] . 'rem;
				border-bottom-right-radius: ' . $primary_button_border_radius['bottom-right'] . 'rem;
				border-bottom-left-radius: ' . $primary_button_border_radius['bottom-left'] . 'rem;
			}';
			*/
			$css .= $primary_button_selector . ' {
				color: ' . blogun_sanitize_color( blogun_option( 'primary_button_text_color' ) ) . ';
				border-color: ' . blogun_sanitize_color( blogun_option( 'primary_button_border_color' ) ) . ';
				border-width: ' . floatval( blogun_option( 'primary_button_border_width' ) ) . 'rem;
			}';
			// Primary button hover.
			$primary_button_hover_selector = '';
			/*
			$primary_button_hover_selector = '
				.blogun-btn:hover,
				.blogun-btn:focus,
				body:not(.wp-customizer) input[type=submit]:hover,
				body:not(.wp-customizer) input[type=submit]:focus,
				.site-main .woocommerce #respond input#submit:hover,
				.site-main .woocommerce #respond input#submit:focus,
				.site-main .woocommerce a.button:hover,
				.site-main .woocommerce a.button:focus,
				.site-main .woocommerce button.button:hover,
				.site-main .woocommerce button.button:focus,
				.site-main .woocommerce input.button:hover,
				.site-main .woocommerce input.button:focus,
				.woocommerce ul.products li.product .added_to_cart:hover,
				.woocommerce ul.products li.product .added_to_cart:focus,
				.woocommerce ul.products li.product .button:hover,
				.woocommerce ul.products li.product .button:focus,
				.woocommerce div.product form.cart .button:hover,
				.woocommerce div.product form.cart .button:focus,
				.woocommerce #review_form #respond .form-submit input:hover,
				.woocommerce #review_form #respond .form-submit input:focus,
				#infinite-handle span:hover';

			$primary_button_hover_bg_color = blogun_option( 'primary_button_hover_bg_color' );
			*/
			// Primary button hover bg color.
			/*
			if ( '' !== $primary_button_hover_bg_color ) {
				$css .= $primary_button_hover_selector . ' {
					background-color: ' . $primary_button_hover_bg_color . ';
				}';
			}
			*/
			// Primary button hover color & border.
			$css .= $primary_button_hover_selector . '{
				color: ' . blogun_sanitize_color( blogun_option( 'primary_button_hover_text_color' ) ) . ';
				border-color: ' . blogun_sanitize_color( blogun_option( 'primary_button_hover_border_color' ) ) . ';
			}';

			// Primary button typography.
			$css .= $this->get_typography_field_css( $primary_button_selector, 'primary_button_typography' );

			/**
			 * Secondary Button.
			 */

			$secondary_button_selector = '
				input[type="reset"],
				.btn-secondary,
				.blogun-btn.btn-secondary';

			$secondary_button_bg_color      = blogun_option( 'secondary_button_bg_color' );
			$secondary_button_border_radius = blogun_option( 'secondary_button_border_radius' );

			// Secondary button text color, border color & border width.
			$css .= $secondary_button_selector . ' {
				color: ' . blogun_sanitize_color( blogun_option( 'secondary_button_text_color' ) ) . ';
				border-color: ' . blogun_sanitize_color( blogun_option( 'secondary_button_border_color' ) ) . ';
				border-width: ' . floatval( blogun_option( 'secondary_button_border_width' ) ) . 'rem;
				background-color: ' . blogun_sanitize_color( $secondary_button_bg_color ) . ';
				border-top-left-radius: ' . floatval( $secondary_button_border_radius['top-left'] ) . 'rem;
				border-top-right-radius: ' . floatval( $secondary_button_border_radius['top-right'] ) . 'rem;
				border-bottom-right-radius: ' . floatval( $secondary_button_border_radius['bottom-right'] ) . 'rem;
				border-bottom-left-radius: ' . floatval( $secondary_button_border_radius['bottom-left'] ) . 'rem;
			}';

			// Secondary button hover.
			$secondary_button_hover_selector = '
				.btn-secondary:hover, 
				.btn-secondary:focus, 
				.blogun-btn.btn-secondary:hover, 
				.blogun-btn.btn-secondary:focus';

			$secondary_button_hover_bg_color = blogun_option( 'secondary_button_hover_bg_color' );

			// Secondary button hover color & border.
			$css .= $secondary_button_hover_selector . '{
				color: ' . blogun_sanitize_color( blogun_option( 'secondary_button_hover_text_color' ) ) . ';
				border-color: ' . blogun_sanitize_color( blogun_option( 'secondary_button_hover_border_color' ) ) . ';
				background-color: ' . blogun_sanitize_color( $secondary_button_hover_bg_color ) . ';
			}';

			// Secondary button typography.
			$css .= $this->get_typography_field_css( $secondary_button_selector, 'secondary_button_typography' );

			// Text Button.
			$css .= '
				.blogun-btn.btn-text-1, .btn-text-1 {
					color: ' . blogun_sanitize_color( blogun_option( 'text_button_text_color' ) ) . ';
				}
			';

			$css .= '
				.blogun-btn.btn-text-1:hover, .blogun-btn.btn-text-1:focus, .btn-text-1:hover, .btn-text-1:focus {
					color: ' . blogun_sanitize_color( blogun_option( 'accent_color' ) ) . ';
				}
			';

			$css .= '
				.blogun-btn.btn-text-1 > span::before {
					background-color: ' . blogun_sanitize_color( blogun_option( 'accent_color' ) ) . ';
				}
			';

			if ( blogun_option( 'text_button_hover_text_color' ) ) {
				$css .= '
					.blogun-btn.btn-text-1:hover, .blogun-btn.btn-text-1:focus, .btn-text-1:hover, .btn-text-1:focus {
						color: ' . blogun_sanitize_color( blogun_option( 'text_button_hover_text_color' ) ) . ';
					}

					.blogun-btn.btn-text-1 > span::before {
						background-color: ' . blogun_sanitize_color( blogun_option( 'text_button_hover_text_color' ) ) . ';
					}
				';
			}

			// Secondary button typography.
			$css .= $this->get_typography_field_css( '.blogun-btn.btn-text-1, .btn-text-1', 'text_button_typography' );

			// Return the filtered CSS.
			return $css;
		}

		/**
		 * Generate dynamic Block Editor styles.
		 *
		 * @since  1.0.0
		 * @return string
		 */
		public function get_block_editor_css() {

			// Current post.
			$post_id   = get_the_ID();
			$post_type = get_post_type( $post_id );

			// Layout.
			$site_layout          = blogun_get_site_layout( $post_id );
			$sidebar_position     = blogun_get_sidebar_position( $post_id );
			$container_width      = blogun_option( 'container_width' );
			$single_content_width = blogun_option( 'single_content_width' );

			$container_width = $container_width - 100;

			if ( blogun_is_sidebar_displayed( $post_id ) ) {

				$sidebar_width   = blogun_option( 'sidebar_width' );
				$container_width = $container_width * ( 100 - intval( $sidebar_width ) ) / 100;
				$container_width = $container_width - 50;

				if ( 'boxed-separated' === $site_layout ) {
					if ( 3 === intval( blogun_option( 'sidebar_style' ) ) ) {
						$container_width += 15;
					}
				}
			}

			if ( 'boxed-separated' === $site_layout ) {
				$container_width += 16;
			}

			if ( 'boxed' === $site_layout ) {
				$container_width = $container_width + 200;
			}

			$background_color = blogun_sanitize_color( get_background_color() );
			$accent_color     = blogun_sanitize_color( blogun_option( 'accent_color' ) );
			$content_color    = blogun_sanitize_color( blogun_option( 'boxed_content_background_color' ) );
			$text_color       = blogun_sanitize_color( blogun_option( 'content_text_color' ) );
			$link_hover_color = blogun_sanitize_color( blogun_option( 'content_link_hover_color' ) );
			$headings_color   = blogun_sanitize_color( blogun_option( 'headings_color' ) );
			$font_smoothing   = sanitize_text_field( blogun_option( 'font_smoothing' ) );

			$css = '';

			// Base HTML font size.
			$css .= $this->get_range_field_css( 'html', 'font-size', 'html_base_font_size', true, '%' );

			// Accent color.
			$css .= '
				.editor-styles-wrapper .block-editor-rich-text__editable mark,
				.editor-styles-wrapper .block-editor-rich-text__editable span.highlight,
				.editor-styles-wrapper .block-editor-rich-text__editable code,
				.editor-styles-wrapper .block-editor-rich-text__editable kbd,
				.editor-styles-wrapper .block-editor-rich-text__editable var,
				.editor-styles-wrapper .block-editor-rich-text__editable samp,
				.editor-styles-wrapper .block-editor-rich-text__editable tt {
					background-color: ' . blogun_sanitize_color( blogun_hex2rgba( $accent_color, .09 ) ) . ';
				}

				.editor-styles-wrapper .wp-block code.block,
				.editor-styles-wrapper .block code {
					background-color: ' . blogun_sanitize_color( blogun_hex2rgba( $accent_color, .075 ) ) . ';
				}

				.editor-styles-wrapper .wp-block .block-editor-rich-text__editable a,
				.editor-styles-wrapper .block-editor-rich-text__editable code,
				.editor-styles-wrapper .block-editor-rich-text__editable kbd,
				.editor-styles-wrapper .block-editor-rich-text__editable var,
				.editor-styles-wrapper .block-editor-rich-text__editable samp,
				.editor-styles-wrapper .block-editor-rich-text__editable tt {
					color: ' . $accent_color . ';
				}

				#editor .editor-styles-wrapper ::-moz-selection { background-color: ' . $accent_color . '; color: #FFF; }
				#editor .editor-styles-wrapper ::selection { background-color: ' . $accent_color . '; color: #FFF; }

				
				.editor-styles-wrapper blockquote,
				.editor-styles-wrapper .wp-block-quote {
					border-color: ' . $accent_color . ';
				}
			';

			// Container width.
			/*
			if ( 'fw-stretched' === $site_layout ) {
				$css .= '
					.editor-styles-wrapper .wp-block {
						max-width: none;
					}
				';
			} elseif ( 'boxed-separated' === $site_layout || 'boxed' === $site_layout ) {

				$css .= '
					.editor-styles-wrapper {
						max-width: ' . $container_width . 'px;
						margin: 0 auto;
					}

					.editor-styles-wrapper .wp-block {
						max-width: none;
					}
				';

				if ( 'boxed' === $site_layout ) {
					$css .= '
						.editor-styles-wrapper {
							-webkit-box-shadow: 0 0 30px rgba(50, 52, 54, 0.06);
							box-shadow: 0 0 30px rgba(50, 52, 54, 0.06);
							padding-left: 42px;
							padding-right: 42px;
						}
					';
				} else {
					$css .= '
						.editor-styles-wrapper {
							border-radius: 3px;
							border: 1px solid rgba(190, 190, 190, 0.30);
						}
					';
				}
			}
			else {
				$css .= '
					.editor-styles-wrapper .wp-block {
						max-width: ' . $container_width . 'px;
					}
				';
			} */

			if ( 'boxed-separated' === $site_layout || 'boxed' === $site_layout ) {

				if ( 'boxed' === $site_layout ) {
					$css .= '
						.editor-styles-wrapper {
							-webkit-box-shadow: 0 0 30px rgba(50, 52, 54, 0.06);
							box-shadow: 0 0 30px rgba(50, 52, 54, 0.06);
							padding-left: 42px;
							padding-right: 42px;
						}
					';
				} else {
					$css .= '
						.editor-styles-wrapper {
							border-radius: 0;
							border: 1px solid rgba(190, 190, 190, 0.30);
						}
					';
				}
			}

			if ( 'post' === $post_type && 'narrow' === $single_content_width ) {

				$narrow_container_width = intval( blogun_option( 'single_narrow_container_width' ) );

				$css .= '
					.editor-styles-wrapper .wp-block {
						max-width: ' . $narrow_container_width . 'px;
					}
				';
			}

			// Background color.
			if ( 'boxed-separated' === $site_layout || 'boxed' === $site_layout ) {
				$css .= '
					:root .edit-post-layout .interface-interface-skeleton__content {
						background-color: #' . trim( $background_color, '#' ) . ';
					}

					:root .editor-styles-wrapper {
						background-color: ' . $content_color . ';
					}
				';
			} else {
				$css .= '
					:root .editor-styles-wrapper {
						background-color: #' . trim( $background_color, '#' ) . ';
					}
				';
			}

			// Body.
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper, .editor-styles-wrapper .wp-block, .block-editor-default-block-appender textarea.block-editor-default-block-appender__content', 'body_font' );
			$css .= '
				:root .editor-styles-wrapper {
					color: ' . $text_color . ';
				}
			';

			// If single post, use single post font size settings.
			if ( 'post' === $post_type ) {
				$css .= $this->get_range_field_css( ':root .editor-styles-wrapper .wp-block', 'font-size', 'single_content_font_size', true );
			}

			// Headings typography.
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h1.wp-block, :root .editor-styles-wrapper h2.wp-block, :root .editor-styles-wrapper h3.wp-block, :root .editor-styles-wrapper h4.wp-block, :root .editor-styles-wrapper h5.wp-block, :root .editor-styles-wrapper h6.wp-block, :root .editor-styles-wrapper .editor-post-title__block .editor-post-title__input', 'headings_font' );

			// Heading em.
			$css .= $this->get_typography_field_css( '.editor-styles-wrapper h1.wp-block em, .editor-styles-wrapper h2.wp-block em, .editor-styles-wrapper h3.wp-block em, .editor-styles-wrapper h4.wp-block em, .editor-styles-wrapper h5.wp-block em, .editor-styles-wrapper h6.wp-block em', 'heading_em_font' );

			// Headings (H1-H6).
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h1.wp-block, :root .editor-styles-wrapper .h1, :root .editor-styles-wrapper .editor-post-title__block .editor-post-title__input', 'h1_font' );
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h2.wp-block, :root .editor-styles-wrapper .h2', 'h2_font' );
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h3.wp-block, :root .editor-styles-wrapper .h3', 'h3_font' );
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h4.wp-block', 'h4_font' );
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h5.wp-block', 'h5_font' );
			$css .= $this->get_typography_field_css( ':root .editor-styles-wrapper h6.wp-block', 'h6_font' );

			$css .= '
				:root .editor-styles-wrapper h1,
				:root .editor-styles-wrapper h2,
				:root .editor-styles-wrapper h3,
				:root .editor-styles-wrapper h4,
				:root .editor-styles-wrapper .h4,
				:root .editor-styles-wrapper h5,
				:root .editor-styles-wrapper h6,
				:root .editor-post-title__block .editor-post-title__input {
					color: ' . $headings_color . ';
				}
			';

			// Page header font size.
			$css .= $this->get_range_field_css( ':root .editor-styles-wrapper .editor-post-title__block .editor-post-title__input', 'font-size', 'page_header_font_size', true );

			// Link hover color.
			$css .= '
				.editor-styles-wrapper .wp-block .block-editor-rich-text__editable a:hover { 
					color: ' . $link_hover_color . '; 
				}
			';

			// Font smoothing.
			if ( $font_smoothing ) {
				$css .= '
					.editor-styles-wrapper {
						-moz-osx-font-smoothing: grayscale;
						-webkit-font-smoothing: antialiased;
					}
				';
			}

			return $css;
		}
	}
endif;

/**
 * The function which returns the one Blogun_Dynamic_Styles instance.
 *
 * Use this function like you would a global variable, except without needing
 * to declare the global.
 *
 * Example: <?php $dynamic_styles = blogun_dynamic_styles(); ?>
 *
 * @since 1.0.0
 * @return object
 */
function blogun_dynamic_styles() {
	return Blogun_Dynamic_Styles::instance();
}

blogun_dynamic_styles();
