<?php
/**
 * SVG icons related functions and filters
 *
 * @package     Bloglo
 * @author      Peregrine Themes
 * @since       1.0.0
 */

/**
 * Do not allow direct script access.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filters the arguments for a single nav menu item to include dropdown indicators.
 *
 * @since 1.0.0
 *
 * @param stdClass $args  An object of wp_nav_menu() arguments.
 * @param WP_Post  $item  Menu item data object.
 * @param int      $depth Depth of menu item. Used for padding.
 */
function bloglo_nav_menu_item_args( $args, $item, $depth ) {

	if ( bloglo_option( 'main_nav_sub_indicators' ) ) {
		$dropdown_indicator = bloglo()->icons->get_svg( 'chevron-down' );
		if ( in_array( 'menu-item-has-children', $item->classes, true ) ) {
			if ( false === strpos( $args->link_after, $dropdown_indicator ) ) {
				$args->link_after .= $dropdown_indicator;
				$args->after       = '<button type="button" class="bloglo-mobile-toggen">' . $dropdown_indicator . '</button>';
			}
		} else {
			$args->link_after = str_replace( $dropdown_indicator, '', $args->link_after );
			$args->after      = '';
		}
	}

	return $args;
}
add_filter( 'nav_menu_item_args', 'bloglo_nav_menu_item_args', 10, 3 );


/**
 * Display social icons in social links menu.
 *
 * @param  string  $item_output The menu item output.
 * @param  WP_Post $item        Menu item object.
 * @param  int     $depth       Depth of the menu.
 * @param  array   $args        wp_nav_menu() arguments.
 * @return string  $item_output The menu item output with social icon.
 */
function bloglo_nav_menu_social_icons( $item_output, $item, $depth, $args ) {

	// Get supported social icons.
	$social_icons = bloglo_social_links_icons();

	// Change SVG icon inside social links menu if there is supported URL.
	if ( false !== strpos( $args->menu_class, 'bloglo-socials-menu' ) ) {

		foreach ( $social_icons as $attr => $value ) {
			if ( false !== strpos( $item_output, $attr ) ) {
				$item_output = str_replace(
					$args->link_after,
					'</span><span class="' . $value . '">' . bloglo()->icons->get_svg( $value, array( 'aria-hidden' => 'true' ) ) . bloglo()->icons->get_svg(
						$value,
						array(
							'class'       => 'bottom-icon',
							'aria-hidden' => 'true',
						)
					),
					$item_output
				);
			}
		}
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'bloglo_nav_menu_social_icons', 10, 4 );

/**
 * Returns an array of supported social links (URL and icon name).
 *
 * @return array $social_links_icons
 */
function bloglo_social_links_icons() {

	// Supported social links icons.
	$social_links_icons = array(
		'500px.com'         => '500px',
		'amazon.com'        => 'amazon',
		'behance.net'       => 'behance',
		'digg.com'          => 'digg',
		'dribbble.com'      => 'dribbble',
		'deviantart'        => 'deviantart',
		'etsy.com'          => 'etsy',
		'facebook.com'      => 'facebook',
		'flipboard.com'     => 'flipboard',
		'flickr.com'        => 'flickr',
		'foursquare.com'    => 'foursquare',
		'github.com'        => 'github',
		'plus.google.com'   => 'google-plus',
		'instagram.com'     => 'instagram',
		'linkedin.com'      => 'linkedin',
		'mailto:'           => 'mail',
		'tel:'           	=> 'phone',
		'medium.com'        => 'medium',
		'pinterest.com'     => 'pinterest',
		'reddit.com'        => 'reddit',
		'skype.com'         => 'skype',
		'skype:'            => 'skype',
		'snapchat.com'      => 'snapchat',
		'soundcloud.com'    => 'soundcloud',
		'spotify.com'       => 'spotify',
		'tumblr.com'        => 'tumblr',
		'twitch.tv'         => 'twitch',
		'twitter.com'       => 'twitter',
		'x.com'       		=> 'twitter',
		'vimeo.com'         => 'vimeo',
		'xing.com'          => 'xing',
		'vk.com'            => 'vkontakte',
		'youtube.com'       => 'youtube',
		'yelp.com'          => 'yelp',
		'wa.me'             => 'whatsapp',
		'tiktok.com'        => 'tiktok',
		'stackoverflow.com' => 'stackoverflow',
		'rss.com'           => 'rss',
		't.me'           	=> 'telegram',
		'discord.com'       => 'discord',
		'wechat.com'     	=> 'wechat',
		'qq.com'       		=> 'qq',
		'bilibili.tv'       => 'bilibili',
		'threads.net'       => 'threads',
		'linktr.ee'       	=> 'linktree',
		'rumble.com'       	=> 'rumble',
		'mastodon.social'   => 'mastodon',
        'bsky.app'          => 'bluesky',
        'bluesky.app'       => 'bluesky',
        'bsky.social'       => 'bluesky',
        'gemspace.com'      =>  'gemspace',
        'lifegroupchat.com' =>  'lifegroupchat',
	);

	/**
	 * Filter Bloglo social links icons.
	 *
	 * @since 1.0.0
	 * @param array $social_links_icons Array of social links icons.
	 */
	return apply_filters( 'bloglo_social_links_icons', $social_links_icons );
}

if ( ! class_exists( 'Bloglo_Icons' ) ) :
	/**
	 * Bloglo Icons class.
	 *
	 * @since 1.0.0
	 */
	class Bloglo_Icons {
		/**
		 * Primary class constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
		}

		/**
		 * GET SVG CODE
		 * Get the SVG code for the specified icon
		 *
		 * @param string $icon Icon name.
		 * @param array  $args Parameters needed to display an SVG.
		 */
		public function get_svg( $icon = '', $args = array() ) {

			$arr = self::get();

			if ( ! array_key_exists( $icon, $arr ) ) {
				return null;
			}

			$args = wp_parse_args(
				$args,
				array(
					'class' => '',
				)
			);

			// .bloglo-icon is a required class.
			if ( false === strpos( $args['class'], 'bloglo-icon' ) ) {
				$args['class'] = trim( 'bloglo-icon ' . $args['class'] );
			}

			$repl = '<svg ';

			if ( ! empty( $args ) ) {
				foreach ( $args as $key => $value ) {
					if ( ! empty( $value ) ) {
						$repl .= esc_html( $key ) . '="' . esc_attr( $value ) . '" ';
					}
				}
			}

			$svg = preg_replace( '/^<svg /', $repl, trim( $arr[ $icon ] ) ); // Add extra attributes to SVG code.
			$svg = preg_replace( "/([\n\t]+)/", ' ', $svg ); // Remove newlines & tabs.
			$svg = preg_replace( '/>\s*</', '><', $svg );    // Remove whitespace between SVG tags.

			// Make sure that only our allowed tags and attributes are included.
			$svg = wp_kses(
				$svg,
				array(
					'svg'     => array(
						'class'       => true,
						'xmlns'       => true,
						'width'       => true,
						'height'      => true,
						'viewbox'     => true,
						'aria-hidden' => true,
						'aria-label'  => true,
						'role'        => true,
						'focusable'   => true,
					),
					'path'    => array(
						'fill'      => true,
						'fill-rule' => true,
						'd'         => true,
						'transform' => true,
					),
					'polygon' => array(
						'fill'      => true,
						'fill-rule' => true,
						'points'    => true,
						'transform' => true,
						'focusable' => true,
					),
				)
			);

			if ( ! $svg ) {
				return null;
			}

			return $svg;
		}

		/**
		 * Get SVG icons array.
		 *
		 * @return array
		 */
		public function get() {
			return apply_filters( 'bloglo_icons_svg', self::$icons );
		}

		/**
		 * Get all available SVG icons
		 *
		 * @return array
		 */
		public function get_all_svg_icons() {

			$arr = self::get();

			$return = array();

			foreach ( $arr as $icon => $svg ) {

				$svg = self::get_svg( $icon );

				if ( $svg ) {
					$return[ $icon ] = self::get_svg( $icon );
				}
			}

			return $return;
		}

		/**
		 * Get icon for post entry meta.
		 *
		 * @since  1.0.0
		 * @param  string $slug    Icon slug.
		 * @param  string $icon    Icon HTML markup.
		 * @param  string $post_id Current post ID.
		 * @return string          Icon HTML markup.
		 */
		public function get_meta_icon( $slug = '', $icon = '', $post_id = '' ) {

			$return = '';

			if ( is_single( $post_id ) ) {
				if ( bloglo_option( 'single_entry_meta_icons' ) ) {
					$return = $icon;
				}
			} elseif ( ! is_single() ) {
				if ( bloglo_option( 'entry_meta_icons' ) ) {
					$return = $icon;
				}
			}

			return apply_filters( 'bloglo_' . $slug . '_meta_icon', $return, $post_id );
		}

		/**
		 * Icons.
		 * Store the code for all SVGs in an array.
		 *
		 * @var array
		 */
		public static $icons = array(
            '500px'           => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.912 21.451c0 .032.16.48.256.705.512 1.249 1.281 2.37 2.242 3.331s2.082 1.73 3.331 2.242c1.281.545 2.659.833 4.068.833s2.787-.288 4.068-.833c1.249-.512 2.37-1.281 3.331-2.242s1.73-2.082 2.242-3.331c.545-1.281.833-2.659.833-4.068s-.288-2.787-.833-4.068c-.512-1.249-1.281-2.37-2.242-3.331s-2.082-1.73-3.331-2.242a10.376 10.376 0 00-4.068-.833c-1.441 0-2.851.288-4.132.833-1.025.416-2.755 1.537-3.748 2.595v-8.68h14.51c.512 0 .512-.737.512-.993s0-.993-.545-.993L8.775.408a.705.705 0 00-.705.673v12.14c0 .384.48.673.929.769.897.192 1.089-.096 1.313-.384l.032-.032c.32-.48 1.377-1.537 1.377-1.537 1.602-1.602 3.78-2.498 6.086-2.498s4.452.897 6.054 2.498c1.634 1.634 2.498 3.78 2.498 6.054s-.897 4.42-2.498 6.054c-1.602 1.602-3.812 2.498-6.086 2.498a8.428 8.428 0 01-4.324-1.185v-7.399c0-.993.416-2.05 1.153-2.851.833-.929 1.954-1.441 3.171-1.441 1.185 0 2.306.448 3.139 1.281a4.188 4.188 0 011.281 3.043c0 2.466-1.954 4.388-4.42 4.388-.48 0-1.345-.224-1.377-.224-.512-.16-.705.545-.769.769-.256.865.128 1.057.224 1.089.801.256 1.313.288 2.018.288 3.491 0 6.342-2.851 6.342-6.342 0-3.459-2.851-6.31-6.342-6.31-1.698 0-3.331.641-4.516 1.858-1.153 1.121-1.826 2.627-1.826 4.132v6.086c-.673-.737-1.313-1.858-1.73-2.979-.16-.448-.545-.352-1.057-.192-.288.096-.993.288-.833.801zm7.463-1.73c0 .224.192.384.32.512v.032c.192.192.384.288.577.288.16 0 .224-.064.256-.096l1.185-1.185 1.121 1.089c.096.128.224.16.352.192.192 0 .384-.096.577-.32.48-.48.224-.737.128-.865l-1.121-1.121 1.153-1.185c.256-.288.032-.577-.192-.801-.32-.32-.641-.416-.833-.224l-1.153 1.153-1.185-1.185a.29.29 0 00-.224-.096c-.16 0-.352.096-.545.32-.352.352-.448.609-.256.801l1.185 1.185-1.153 1.185a.356.356 0 00-.192.32zm2.563-15.887c-1.89 0-3.876.384-5.349 1.025-.16.064-.256.192-.256.352-.032.16.032.384.128.673.096.224.32.865.801.673 1.505-.577 3.171-.897 4.677-.897 1.73 0 3.363.32 4.933.993 1.249.513 2.402 1.281 3.684 2.402a.39.39 0 00.288.128c.256 0 .48-.256.705-.48.352-.384.577-.705.224-1.025-1.217-1.153-2.563-2.018-4.196-2.723a14.747 14.747 0 00-5.637-1.121zm10.089 23.191c-.224-.224-.416-.352-.577-.416s-.32 0-.448.096l-.096.128a12.823 12.823 0 01-4.036 2.723c-1.57.673-3.235.993-4.933.993s-3.363-.32-4.933-.993a12.66 12.66 0 01-4.036-2.723c-1.217-1.217-2.114-2.563-2.723-4.036-.577-1.409-.769-2.498-.833-2.851v-.064c-.064-.416-.448-.448-.993-.352-.224.032-.929.128-.865.641.192 1.153.512 2.274.961 3.363a14.653 14.653 0 003.107 4.613c1.345 1.345 2.883 2.402 4.612 3.107 1.794.769 3.716 1.153 5.67 1.153s3.876-.384 5.67-1.153a14.653 14.653 0 004.613-3.107l.128-.128c.192-.128.32-.416-.288-.993z"/></svg>',
            'amazon'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M27.724 27.076c.536-.268.947.143.393.858s-5.005 4.79-12.476 4.79S2.45 27.612.698 25.503c-.483-.554.072-.804.393-.59 5.237 3.182 13.424 8.419 26.633 2.163zm3.7-2.056c.268.358 0 1.93-.465 3.074-.465 1.126-1.144 1.913-1.519 2.216-.393.322-.679.197-.465-.268s1.376-3.325.912-3.932c-.465-.59-2.645-.304-3.432-.232-.769.072-.929.143-1.001-.018-.161-.411 1.555-1.108 2.681-1.251 1.126-.125 2.931-.054 3.289.411zm-7.043-7.918c0 1.966 2.306 3.772 2.306 3.772l-4.058 4.004c-1.591-1.501-2.788-2.753-2.788-2.753a3.397 3.397 0 01-.447-.59c-3.235 5.058-13.12 4.737-13.12-3.092 0-7.293 8.633-8.276 12.119-8.401v-2.27c0-.483.179-2.681-2.538-2.681 0 0-2.717 0-3.879 3.539l-5.255-.483c0-3.521 3.343-7.454 9.634-7.454 6.274 0 8.026 4.075 8.026 5.881v10.528zm-12.012.375c0 3.629 6.024 4.487 6.024-1.233v-2.896c-2.413.072-6.024.751-6.024 4.129z"/></svg>',
            'arrow-up'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M17.9137 25.3578L17.9137 9.8758L24.9877 16.9498C25.5217 17.4838 26.3227 17.4838 26.8557 16.9498C27.3887 16.4158 27.3897 15.6148 26.8557 15.0818L17.5137 5.7398C17.3807 5.6068 17.2467 5.4728 17.1137 5.4728C16.8467 5.3398 16.4467 5.3398 16.0457 5.4728C15.9127 5.6058 15.7787 5.6058 15.6457 5.7398L6.30373 15.0818C6.03673 15.3488 5.90373 15.7488 5.90373 16.0158C5.90373 16.2828 6.03673 16.6828 6.30373 16.9498C6.42421 17.0763 6.56912 17.1769 6.72967 17.2457C6.89022 17.3145 7.06307 17.35 7.23773 17.35C7.4124 17.35 7.58525 17.3145 7.7458 17.2457C7.90635 17.1769 8.05125 17.0763 8.17173 16.9498L15.2457 9.8758L15.2457 25.3578C15.2457 26.1588 15.7797 26.6928 16.5807 26.6928C17.3817 26.6928 17.9157 26.1588 17.9157 25.3578L17.9137 25.3578Z"/></svg>',
            'arrow-left'      => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.358 14.087H9.876l7.074-7.074c.534-.534.534-1.335 0-1.868s-1.335-.534-1.868 0L5.74 14.487c-.133.133-.267.267-.267.4-.133.267-.133.667 0 1.068.133.133.133.267.267.4l9.342 9.342c.267.267.667.4.934.4s.667-.133.934-.4a1.29 1.29 0 000-1.868l-7.074-7.074h15.482c.801 0 1.335-.534 1.335-1.335s-.534-1.335-1.335-1.335z"/></svg>',
            'arrow-right'     => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M26.559 15.955c.133-.267.133-.667 0-1.068-.133-.133-.133-.267-.267-.4L16.95 5.145c-.534-.534-1.335-.534-1.868 0s-.534 1.335 0 1.868l7.074 7.074H6.674c-.801 0-1.335.534-1.335 1.335s.534 1.335 1.335 1.335h15.482l-7.074 7.074a1.29 1.29 0 000 1.868c.267.267.667.4.934.4s.667-.133.934-.4l9.342-9.342c.133-.133.267-.267.267-.4z"/></svg>',
            'behance'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.841 9.666h-8.008V7.789h8.008zm-13.263 7.57c.626.876.938 1.877.938 3.128 0 1.001-.188 1.877-.563 2.565a6.298 6.298 0 01-1.564 1.814c-.688.438-1.439.751-2.252 1.001-.813.188-1.689.313-2.565.313H0V6.163h9.259c.938 0 1.814.188 2.628.375.751.188 1.376.5 1.939.876s1.001.938 1.314 1.564c.313.563.438 1.376.438 2.315s-.188 1.814-.688 2.44c-.438.688-1.126 1.251-2.002 1.627 1.251.375 2.127 1.001 2.69 1.877zM4.192 14.17h4.504c.813 0 1.439-.063 1.939-.438s.751-.876.751-1.752c0-.438-.063-.813-.25-1.126-.188-.25-.375-.501-.688-.688-.25-.125-.563-.313-.938-.375s-.751-.125-1.126-.125H4.192zm8.007 5.943c0-1.001-.313-1.689-.876-2.127s-1.314-.626-2.252-.626h-4.88v5.443h4.755c.438 0 .876-.063 1.251-.125.375-.125.688-.25 1.001-.438s.563-.438.688-.813c.188-.313.313-.751.313-1.314zm19.832-.187H21.708c0 1.001.375 2.19 1.001 2.753.563.563 1.439.876 2.502.876.813 0 1.501-.25 2.065-.688.563-.375.876-.688 1.064-1.314h3.441c-.563 1.939-1.439 3.003-2.565 3.754s-2.502 1.189-4.129 1.189c-1.126 0-2.127-.188-3.066-.563a7.094 7.094 0 01-2.252-1.501 7.983 7.983 0 01-1.501-2.44 8.877 8.877 0 01-.5-3.003c0-1.064.188-2.065.563-3.003.313-.938.813-1.689 1.439-2.377.688-.688 1.439-1.189 2.315-1.627a8.014 8.014 0 013.003-.563c1.189 0 2.252.25 3.191.688.876.501 1.627 1.126 2.19 1.877.626.813 1.001 1.752 1.251 2.753s.375 2.252.313 3.191zm-3.941-2.565c-.125-.938-.438-1.689-.876-2.19-.5-.563-1.314-.813-2.252-.813-.626 0-1.126.125-1.564.375-.375.188-.751.438-1.001.751s-.375.626-.5 1.001c-.125.313-.188.563-.188.876z"/></svg>',
            'bookmark'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M22.689 2.075H9.343c-2.269 0-4.004 1.735-4.004 4.004v21.354c0 .534.267.934.667 1.201s.934.133 1.335-.133l8.542-6.139 8.542 6.139c.267.133.534.267.801.267s.4 0 .667-.133c.4-.267.667-.667.667-1.201V6.08c.133-2.269-1.602-4.004-3.87-4.004zm1.334 22.822l-7.207-5.205c-.267-.133-.534-.267-.801-.267s-.534.133-.801.267l-7.207 5.205V6.079c0-.801.534-1.335 1.335-1.335h13.346c.801 0 1.335.534 1.335 1.335v18.818z"/></svg>',
            'chat'            => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M512 160h-96V64c0-35.3-28.7-64-64-64H64C28.7 0 0 28.7 0 64v160c0 35.3 28.7 64 64 64h32v52c0 7.1 5.8 12 12 12 2.4 0 4.9-.7 7.1-2.4L224 288h128c35.3 0 64-28.7 64-64v-32h96c17.6 0 32 14.4 32 32v160c0 17.6-14.4 32-32 32h-64v49.6l-80.2-45.4-7.3-4.2H256c-17.6 0-32-14.4-32-32v-96l-32 18.1V384c0 35.3 28.7 64 64 64h96l108.9 61.6c2.2 1.6 4.7 2.4 7.1 2.4 6.2 0 12-4.9 12-12v-52h32c35.3 0 64-28.7 64-64V224c0-35.3-28.7-64-64-64zm-128 64c0 17.6-14.4 32-32 32H215.6l-7.3 4.2-80.3 45.4V256H64c-17.6 0-32-14.4-32-32V64c0-17.6 14.4-32 32-32h288c17.6 0 32 14.4 32 32v160z"/></svg>',
            'chevron-down'    => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.958 10.483a1.29 1.29 0 00-1.868 0l-7.074 7.074-7.074-7.074c-.534-.534-1.335-.534-1.868 0s-.534 1.335 0 1.868l8.008 8.008c.267.267.667.4.934.4s.667-.133.934-.4l8.008-8.008a1.29 1.29 0 000-1.868z"/></svg>',
            'chevron-up'      => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.958 18.491l-8.008-8.008a1.29 1.29 0 00-1.868 0l-8.008 8.008c-.534.534-.534 1.335 0 1.868s1.335.534 1.868 0l7.074-7.074 7.074 7.074c.267.267.667.4.934.4s.667-.133.934-.4a1.29 1.29 0 000-1.868z"/></svg>',
            'clock'           => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 29.36 29.36"><g data-name="Layer 2"><path d="M14.68 0a14.68 14.68 0 1014.68 14.68A14.64 14.64 0 0014.68 0zm0 26.69a12 12 0 1112-12 12 12 0 01-12 12zm5.87-10.54L16 13.88V6.67a1.25 1.25 0 00-1.33-1.33 1.26 1.26 0 00-1.34 1.33v8a1.28 1.28 0 00.81 1.2l5.33 2.67c.14.13.27.13.54.13a1.28 1.28 0 001.2-.8 1.41 1.41 0 00-.67-1.73z" data-name="Layer 1"/></g></svg>',
            'calendar'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><!--!Font Awesome Free 6.5.1 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2024 Fonticons, Inc.--><path d="M400 64h-48V12c0-6.6-5.4-12-12-12h-40c-6.6 0-12 5.4-12 12v52H160V12c0-6.6-5.4-12-12-12h-40c-6.6 0-12 5.4-12 12v52H48C21.5 64 0 85.5 0 112v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V112c0-26.5-21.5-48-48-48zm-6 400H54c-3.3 0-6-2.7-6-6V160h352v298c0 3.3-2.7 6-6 6z"/></svg>',
            'deviantart'      => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.217 17.548c-1.501-1.814-3.378-2.94-5.505-3.816-2.753-1.126-5.63-1.564-8.571-1.627h-.751V9.102c-1.314.125-2.565.313-4.192.438v2.69c.063.25-.063.375-.313.438-2.127.438-4.129 1.126-6.006 2.252-1.439.876-2.69 1.939-3.629 3.378C.124 19.987-.314 21.864.186 23.803h15.202v-9.196h1.939c1.627.125 3.253.375 4.755 1.001 2.002.938 3.378 2.377 3.941 4.504.063.125.063.5.125.5H20.83v-.438-3.691c0-.25-.063-.375-.313-.375a29.814 29.814 0 00-2.315-.438c-.5-.063-1.251-.063-1.564-.125v8.258H31.84c.063-.626.188-.938.188-1.376.063-1.877-.626-3.441-1.814-4.88zm-19.019 3.066H5.943c.125-2.252 2.69-5.067 5.255-5.38v5.38z"/></svg>',
            'digg'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M9.634 11.292h3.191v10.886H9.634zM5.13 6.788h3.191v15.39H0V11.292h5.13zm0 7.069H3.191v5.756H5.13zm4.504-3.816h3.191V6.788H9.634zm4.442 1.251h8.321v14.764h-8.321v-2.565h5.13v-1.314h-5.13zm3.191 8.321h1.939v-5.756h-1.939zm14.764-8.321v14.764H23.71v-2.565h5.13v-1.314h-5.13V11.291zm-3.19 2.565h-1.939v5.756h1.939z"/></svg>',
            'dribbble'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M15.953.532C7.132.532-.063 7.727-.063 16.548s7.195 16.016 16.016 16.016c8.884 0 16.016-7.195 16.016-16.016S24.837.532 15.953.532zm10.635 7.382c1.877 2.315 3.066 5.255 3.066 8.508-.438-.125-4.942-1.001-9.509-.438-.125-.25-.188-.501-.313-.688a19.291 19.291 0 00-.876-2.002c5.005-2.065 7.32-5.005 7.632-5.38zM15.953 2.847a13.58 13.58 0 019.071 3.441c-.188.375-2.252 3.128-7.132 4.942-2.252-4.066-4.755-7.445-5.13-8.008a13.905 13.905 0 013.191-.375zM10.197 4.16c.313.5 2.753 3.879 5.067 7.945-6.444 1.689-12.074 1.627-12.7 1.627.938-4.254 3.754-7.758 7.632-9.572zM2.315 16.547v-.438c.563.063 7.195.125 14.076-1.939.375.751.751 1.564 1.064 2.315-.188.063-.313.125-.5.188-7.069 2.252-10.823 8.508-11.136 9.009a13.602 13.602 0 01-3.503-9.134zm13.638 13.701c-3.128 0-6.068-1.126-8.383-2.878.25-.563 3.003-5.881 10.761-8.571 0 0 .063 0 .063-.063 1.939 5.005 2.753 9.196 2.94 10.385-1.627.688-3.441 1.126-5.38 1.126zm7.695-2.377c-.188-.813-.876-4.817-2.69-9.76 4.317-.688 8.07.438 8.508.626-.563 3.816-2.753 7.069-5.818 9.134z"/></svg>',
            'edit-3'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M4.004 23.429h5.339c.4 0 .667-.133.934-.4L24.958 8.348a1.29 1.29 0 000-1.868l-5.339-5.339a1.29 1.29 0 00-1.868 0L3.07 15.822c-.267.267-.4.534-.4.934v5.339c0 .801.534 1.335 1.335 1.335zm1.335-6.139L18.685 3.944l3.47 3.47L8.809 20.76h-3.47v-3.47zm22.688 10.143H4.004c-.801 0-1.335.534-1.335 1.335s.534 1.335 1.335 1.335h24.023c.801 0 1.335-.534 1.335-1.335s-.534-1.335-1.335-1.335z"/></svg>',
            'etsy'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M3.503 1.658c-.063-.438.125-.751.751-.751.563 0 6.631.188 10.698.188 8.383 0 10.448-.688 11.887-.688.751.063.938.626.938 1.001s-.375 1.126-.375 2.127.25 2.002.25 2.94c0 .876-.751 1.126-1.189 1.126-.375 0-.626-.125-.626-.876-.063-1.439 1.001-3.503-9.259-3.503-5.193 0-5.318.626-5.318 2.252v7.945c0 .876.25 1.126.938 1.126h7.445c1.439 0 2.94-.751 3.128-2.815.063-1.064.313-1.126.876-1.126s1.001.438.938 1.439c-.125 1.064-.313 2.127-.313 3.566 0 1.501.313 2.94.313 4.004s-1.439 1.314-1.689.125c-.25-1.064-.688-3.003-3.941-3.003h-6.631c-.751 0-1.064.438-1.064 1.627v8.446c0 1.501.438 3.066 3.378 3.066 3.003 0 8.383.313 10.886-1.627 1.001-.813 1.376-2.065 1.689-4.066.188-.876 2.19-.876 2.065 1.251-.063 1.001-1.001 3.066-1.001 4.442 0 1.064.25 2.628-.751 2.628-.876 0-.5-.5-4.379-.5s-17.204.125-19.456.125c-.813 0-1.001-.25-1.001-.751s.313-.876 1.001-.876 3.691 0 3.691-3.754V7.039c0-2.002-.438-4.066-2.94-4.317-1.001-.125-.938-.5-.938-1.064z"/></svg>',
            'external-link'   => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.023 15.421c-.801 0-1.335.534-1.335 1.335v8.008c0 .801-.534 1.335-1.335 1.335H6.672c-.801 0-1.335-.534-1.335-1.335V10.083c0-.801.534-1.335 1.335-1.335h8.008c.801 0 1.335-.534 1.335-1.335s-.534-1.335-1.335-1.335H6.672c-2.269 0-4.004 1.735-4.004 4.004v14.681c0 2.269 1.735 4.004 4.004 4.004h14.681c2.269 0 4.004-1.735 4.004-4.004v-8.008c0-.801-.534-1.335-1.335-1.335zm5.206-12.545a1.6 1.6 0 00-.667-.667c-.133-.133-.4-.133-.534-.133H20.02c-.801 0-1.335.534-1.335 1.335s.534 1.335 1.335 1.335h4.805L12.413 17.158a1.29 1.29 0 000 1.868c.267.267.534.4.934.4s.667-.133.934-.4L26.693 6.614v4.805c0 .801.534 1.335 1.335 1.335s1.335-.534 1.335-1.335V3.411c0-.133 0-.4-.133-.534z"/></svg>',
            'facebook'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M18.518 32.438V17.799h4.88l.751-5.693h-5.631V8.477c0-1.627.438-2.753 2.815-2.753h3.003V.657c-.5-.125-2.315-.25-4.379-.25-4.379 0-7.32 2.628-7.32 7.507v4.192H7.695v5.693h4.942v14.639z"/></svg>',
            'flipboard'       => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M4.317 2.721v23.398h7.82v-7.82h7.758v-7.758h7.82v-7.82z"/></svg>',
            'flickr'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.257 23.679C3.253 23.679 0 20.426 0 16.422s3.253-7.257 7.257-7.257 7.257 3.253 7.257 7.257-3.253 7.257-7.257 7.257zm17.517 0c-4.004 0-7.257-3.253-7.257-7.257s3.253-7.257 7.257-7.257 7.257 3.253 7.257 7.257-3.253 7.257-7.257 7.257z"/></svg>',
            'foursquare'      => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.649.407H7.57c-2.315 0-3.003 1.752-3.003 2.878v27.151c0 1.251.626 1.752 1.001 1.877.438.188 1.439.313 2.065-.438 0 0 8.07-9.384 8.195-9.509.25-.188.25-.188.438-.188h5.255c2.19 0 2.502-1.564 2.753-2.502.188-.751 2.315-11.699 3.066-15.202.5-2.628-.125-4.066-2.69-4.066zm-.375 19.268c.188-.751 2.315-11.699 3.066-15.202zm-.626-14.639L22.96 8.79c-.125.375-.626.813-1.064.813h-6.694c-.751 0-1.251.501-1.251 1.251v.813c0 .751.501 1.251 1.251 1.251h5.693c.5 0 1.001.626.938 1.189-.125.563-.688 3.378-.751 3.691-.063.25-.375.813-1.064.813h-4.567c-.876 0-1.126.125-1.689.813s-5.568 6.757-5.568 6.757c-.063.063-.125.063-.125 0V4.973c0-.5.438-1.064 1.064-1.064h13.701c.5 0 .938.501.813 1.126z"/></svg>',
            'folder-open'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><!--!Font Awesome Free 6.5.1 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2024 Fonticons, Inc.--><path d="M527.9 224H480v-48c0-26.5-21.5-48-48-48H272l-64-64H48C21.5 64 0 85.5 0 112v288c0 26.5 21.5 48 48 48h400c16.5 0 31.9-8.5 40.7-22.6l79.9-128c20-31.9-3-73.4-40.7-73.4zM48 118c0-3.3 2.7-6 6-6h134.1l64 64H426c3.3 0 6 2.7 6 6v42H152c-16.8 0-32.4 8.8-41.1 23.2L48 351.4zm400 282H72l77.2-128H528z"/></svg>',
            'github'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.28 11.543c0-2.252-.751-4.129-2.127-5.568.25-.5.938-2.628-.188-5.505 0 0-1.752-.563-5.693 2.127a20.865 20.865 0 00-5.255-.688c-1.752 0-3.503.25-5.193.688C7.82-.093 6.131.47 6.131.47c-1.189 2.878-.438 5.005-.25 5.505-1.314 1.439-2.127 3.316-2.127 5.568 0 8.008 4.88 9.76 9.509 10.323-.626.5-1.189 1.439-1.376 2.753-1.189.563-4.192 1.439-6.006-1.752 0 0-1.126-1.939-3.191-2.127 0 0-2.065 0-.188 1.251 0 0 1.376.688 2.315 3.066 0 0 1.251 3.691 7.007 2.44.063 1.752.063 4.442.063 4.942H22.21v-6.757c0-1.939-.626-3.191-1.376-3.879 4.63-.5 9.447-2.252 9.447-10.26zM4.129 21.74c-.063.063-.25.125-.375.063s-.25-.25-.188-.313c.063-.125.188-.188.375-.063a.345.345 0 01.188.313zm.813.938c-.063.063-.313 0-.438-.125s-.125-.313-.063-.438a.33.33 0 01.438.125c.125.125.188.313.063.438zm.814 1.189c-.125.063-.313 0-.438-.188s-.125-.438 0-.5c.125-.063.313 0 .438.188s.125.375 0 .5zm1.126 1.126a.33.33 0 01-.5-.063c-.188-.188-.25-.375-.125-.5.125-.188.375-.125.5.063.188.125.25.375.125.5zm1.564.688c-.063.188-.313.25-.5.188-.25-.063-.438-.25-.375-.438s.313-.25.563-.188c.188.063.375.25.313.438zm1.251.438c-.25 0-.438-.125-.438-.313s.188-.313.438-.313.438.125.438.313-.188.313-.438.313zm1.627-.187c-.25.063-.438-.063-.501-.25 0-.125.188-.313.438-.375s.438.063.501.25c0 .125-.188.313-.438.375z"/></svg>',
            'google-plus'     => '<svg xmlns="http://www.w3.org/2000/svg" width="41" height="32" viewBox="0 0 32 32"><path d="M25.686 17.012c0 7.489-5.023 12.798-12.584 12.798C5.863 29.81 0 23.947 0 16.708S5.863 3.606 13.102 3.606c3.539 0 6.489 1.287 8.776 3.432l-3.557 3.414c-.965-.929-2.663-2.02-5.219-2.02-4.469 0-8.115 3.7-8.115 8.276s3.646 8.276 8.115 8.276c5.184 0 7.132-3.736 7.436-5.648h-7.436v-4.504h12.369c.125.661.215 1.323.215 2.181zm15.497-2.181v3.754h-3.736v3.736h-3.754v-3.736h-3.736v-3.754h3.736v-3.736h3.754v3.736h3.736z"/></svg>',
            'instagram'       => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M16.016 3.284c4.317 0 4.755.063 6.444.125 1.627.063 2.44.313 3.003.5.751.313 1.314.688 1.814 1.189.563.563.938 1.126 1.189 1.814.25.626.5 1.439.563 3.003.063 1.752.125 2.19.125 6.506s-.063 4.755-.125 6.444c-.063 1.627-.313 2.44-.5 3.003-.313.751-.688 1.314-1.189 1.814-.563.563-1.126.938-1.814 1.189-.626.25-1.439.5-3.003.563-1.752.063-2.19.125-6.506.125s-4.755-.063-6.444-.125c-1.627-.063-2.44-.313-3.003-.5-.751-.313-1.314-.688-1.814-1.189-.563-.563-.938-1.126-1.189-1.814-.25-.626-.5-1.439-.563-3.003-.063-1.752-.125-2.19-.125-6.506s.063-4.755.125-6.444c.063-1.627.313-2.44.5-3.003.313-.751.688-1.314 1.189-1.814.563-.563 1.126-.938 1.814-1.189.626-.25 1.439-.5 3.003-.563 1.752-.063 2.19-.125 6.506-.125m0-2.877c-4.379 0-4.88.063-6.569.125-1.752.063-2.94.313-3.879.688-1.064.438-2.002 1.001-2.878 1.877S1.251 4.911.813 5.975C.438 6.976.187 8.102.125 9.854.062 11.543 0 12.044 0 16.423s.063 4.88.125 6.569c.063 1.752.313 2.94.688 3.879.438 1.064 1.001 2.002 1.877 2.878s1.814 1.439 2.878 1.877c1.001.375 2.127.626 3.879.688 1.689.063 2.19.125 6.569.125s4.88-.063 6.569-.125c1.752-.063 2.94-.313 3.879-.688 1.064-.438 2.002-1.001 2.878-1.877s1.439-1.814 1.877-2.878c.375-1.001.626-2.127.688-3.879.063-1.689.125-2.19.125-6.569s-.063-4.88-.125-6.569c-.063-1.752-.313-2.94-.688-3.879-.438-1.064-1.001-2.002-1.877-2.878s-1.814-1.439-2.878-1.877C25.463.845 24.337.594 22.585.532c-1.689-.063-2.19-.125-6.569-.125zm0 7.757c-4.567 0-8.258 3.691-8.258 8.258s3.691 8.258 8.258 8.258c4.567 0 8.258-3.691 8.258-8.258s-3.691-8.258-8.258-8.258zm0 13.639c-2.94 0-5.38-2.44-5.38-5.38s2.44-5.38 5.38-5.38 5.38 2.44 5.38 5.38-2.44 5.38-5.38 5.38zM26.463 7.851c0 1.064-.813 1.939-1.877 1.939s-1.939-.876-1.939-1.939c0-1.064.876-1.877 1.939-1.877s1.877.813 1.877 1.877z"/></svg>',
            'linkedin'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M23.961 16.798c-.438-.5-1.189-.751-2.252-.751-1.376 0-2.315.438-2.878 1.189-.563.813-.813 1.939-.813 3.378v10.948c0 .188-.063.375-.25.563-.125.125-.375.25-.563.25h-5.756c-.188 0-.375-.125-.563-.25-.125-.188-.25-.375-.25-.563V11.355c0-.188.125-.375.25-.563.188-.125.375-.25.563-.25h5.568c.188 0 .375.063.5.125s.188.25.25.438c0 .188.063.375.063.438v.501c1.439-1.376 3.378-2.002 5.693-2.002 2.69 0 4.755.626 6.256 1.939 1.501 1.376 2.252 3.316 2.252 5.943v13.638c0 .188-.063.375-.25.563-.125.125-.375.25-.563.25h-5.881c-.188 0-.375-.125-.563-.25-.125-.188-.25-.375-.25-.563V19.237c0-1.064-.188-1.877-.563-2.44zM6.631 7.101c-.751.751-1.689 1.126-2.753 1.126s-2.002-.375-2.753-1.126C.374 6.35-.001 5.474-.001 4.348c0-1.064.375-1.939 1.126-2.69S2.814.469 3.878.469 5.88.907 6.631 1.658s1.126 1.627 1.126 2.69c0 1.126-.375 2.002-1.126 2.753zm.939 4.254v20.207c0 .188-.125.375-.25.563-.188.125-.375.25-.563.25H1.001c-.188 0-.375-.125-.563-.25-.125-.188-.25-.375-.25-.563V11.355c0-.188.125-.375.25-.563.188-.125.375-.25.563-.25h5.756c.188 0 .375.125.563.25.125.188.25.375.25.563z"/></svg>',
            'mail'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M26.693 3.41H5.339c-2.269 0-4.004 1.735-4.004 4.004V23.43c0 2.269 1.735 4.004 4.004 4.004h21.354c2.269 0 4.004-1.735 4.004-4.004V7.414c0-2.269-1.735-4.004-4.004-4.004zM5.339 6.079h21.354c.534 0 .934.267 1.201.801l-11.878 8.275L4.138 6.88c.267-.534.667-.801 1.201-.801zm21.354 18.685H5.339c-.801 0-1.335-.534-1.335-1.335V9.949l11.211 7.874c.267.133.534.267.801.267s.534-.133.801-.267l11.211-7.874v13.48c0 .801-.534 1.335-1.335 1.335z"/></svg>',
            'phone'           => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M27.227 17.957c-1.201-.133-2.402-.4-3.47-.801-1.468-.534-3.07-.133-4.271.934l-.934.934c-2.402-1.602-4.538-3.604-6.139-6.139l.934-.934c1.068-1.068 1.468-2.803.934-4.271-.4-1.068-.667-2.269-.801-3.47C13.213 2.208 11.478.74 9.476.74H5.072c-1.068.133-2.002.667-2.669 1.468s-.934 1.868-.934 2.936c.4 4.271 2.002 8.542 4.271 12.145a28.055 28.055 0 0 0 8.408 8.408c3.604 2.402 7.874 3.87 12.145 4.271h.4c1.068 0 2.135-.4 2.803-1.201s1.201-1.735 1.201-2.803V21.96c0-2.002-1.468-3.737-3.47-4.004zm.8 4.004v4.004c0 .4-.133.667-.4.934s-.534.4-1.068.4c-3.87-.4-7.607-1.735-10.944-3.87-3.07-1.868-5.605-4.538-7.607-7.607-2.135-3.337-3.47-7.074-3.87-10.944 0-.4.133-.667.267-.934.267-.267.667-.534 1.068-.534h4.004c.667 0 1.201.534 1.335 1.201a13.644 13.644 0 0 0 1.068 4.004c.133.534 0 1.068-.267 1.468l-1.735 1.602c-.4.4-.534 1.068-.267 1.602a21.896 21.896 0 0 0 8.542 8.542c.534.267 1.201.267 1.602-.267l1.735-1.735c.4-.4.934-.534 1.468-.267 1.335.534 2.669.801 4.004 1.068.534.133 1.068.667 1.068 1.335z"/></svg>',
            'medium'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M31.781 7.726L21.646 2.659h-.125c-.125 0-.313.063-.375.125l-6.194 10.197 6.944 11.324 10.01-16.203c.063-.125 0-.313-.125-.375zM11.636 9.415v10.948l9.76 4.88zm10.573 16.204l8.946 4.504c.438.25.876 0 .876-.563V9.666zM10.51 7.789l-9.322-4.63-.626-.375c-.063 0-.125-.063-.188-.063-.125 0-.188.063-.25.125s-.125.188-.125.375v21.333c0 .375.25.813.563.938l9.196 4.63c.125.063.25.125.375.125.313 0 .501-.25.501-.688V7.975c0-.063 0-.125-.125-.188z"/></svg>',
            'message-square'  => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.358 2.075H6.673c-2.269 0-4.004 1.735-4.004 4.004v21.354c0 .534.267 1.068.801 1.201.133.133.4.133.534.133.4 0 .667-.133.934-.4l4.938-4.938h15.482c2.269 0 4.004-1.735 4.004-4.004V6.079c0-2.269-1.735-4.004-4.004-4.004zm1.335 17.35c0 .801-.534 1.335-1.335 1.335H9.342c-.4 0-.667.133-.934.4l-3.07 3.07V6.079c0-.801.534-1.335 1.335-1.335h18.685c.801 0 1.335.534 1.335 1.335v13.346z"/></svg>',
            'pinterest'       => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M10.635 12.669c0-1.376.313-2.502 1.064-3.441.688-.938 1.564-1.439 2.565-1.439.813 0 1.439.25 1.877.813s.688 1.251.688 2.065c0 .501-.125 1.126-.313 1.814-.188.751-.375 1.627-.688 2.565-.313 1.001-.563 1.752-.688 2.315-.25 1.001-.063 1.877.563 2.628.626.688 1.439 1.064 2.44 1.064 1.752 0 3.191-1.001 4.317-2.94 1.126-2.002 1.689-4.379 1.689-7.195 0-2.127-.688-3.879-2.065-5.255-1.376-1.314-3.316-2.002-5.818-2.002-2.815 0-5.067.876-6.757 2.69-1.752 1.752-2.628 3.879-2.628 6.381 0 1.501.438 2.753 1.251 3.754.313.313.375.688.313 1.064-.125.313-.25.813-.375 1.501-.063.25-.188.438-.375.5s-.375.063-.563 0c-1.314-.563-2.252-1.439-2.94-2.753s-1.001-2.815-1.001-4.504c0-1.126.188-2.19.563-3.316s.876-2.19 1.627-3.191c.751-1.064 1.689-1.939 2.69-2.753 1.064-.813 2.315-1.439 3.879-1.939a17.23 17.23 0 014.817-.688c1.814 0 3.441.313 4.942.938 1.501.563 2.815 1.376 3.816 2.377 1.064 1.064 1.814 2.19 2.44 3.503.563 1.314.876 2.69.876 4.066 0 3.754-.938 6.882-2.878 9.322-1.939 2.502-4.379 3.691-7.382 3.691-1.001 0-1.939-.188-2.815-.688-.876-.438-1.501-1.001-1.877-1.689-.688 2.94-1.126 4.692-1.314 5.255-.375 1.439-1.251 3.191-2.69 5.255H8.571c-.25-2.565-.188-4.692.188-6.381l2.44-10.323c-.375-.813-.563-1.814-.563-3.066z"/></svg>',
            'play'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M26.025 14.354L7.34 2.342a1.213 1.213 0 00-1.335 0c-.4.133-.667.534-.667 1.068v24.023c0 .534.267.934.667 1.201.267.133.4.133.667.133s.534-.133.667-.267l18.685-12.012c.4-.267.667-.667.667-1.068s-.267-.934-.667-1.068zM8.008 25.031V5.812l14.948 9.609L8.008 25.03z"/></svg>',
            'play-2'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path fill="currentColor" fill-rule="evenodd" d="M8.893.016A6.8 6.8 0 0 0 4.136 2.6c-1.254 1.58-1.958 3.694-2.11 6.34-.035.601-.035 13.524 0 14.126.17 2.935 1.023 5.234 2.535 6.822 1.084 1.139 2.36 1.806 3.923 2.05.333.052.504.062 1.051.063.684 0 .956-.022 1.582-.135.718-.13 1.585-.399 2.352-.73.8-.345 1.54-.777 6.78-3.956a1556.03 1556.03 0 0 0 5.445-3.313c2.643-1.688 4.382-3.969 4.86-6.372a8.38 8.38 0 0 0 .131-.953c.004-.082.013 1.29.02 3.046.008 2.045.013.745.013-3.612 0-4.374-.005-5.66-.013-3.6a315.685 315.685 0 0 1-.02 3.074 7.82 7.82 0 0 0-.055-.487c-.367-2.623-2.136-5.051-5.003-6.867-.438-.278-.575-.324-.92-.31-.248.009-.283.017-.468.111a1.224 1.224 0 0 0-.569.592 1.28 1.28 0 0 0-.002 1.14c.121.258.29.43.635.647 1.34.84 2.417 1.86 3.078 2.915.684 1.091.967 2.25.825 3.384-.217 1.745-1.379 3.404-3.345 4.778-.597.418-10.824 6.627-11.546 7.01-1.26.67-2.388 1.008-3.524 1.055-2 .085-3.556-.896-4.473-2.818-.414-.869-.665-1.847-.798-3.104-.044-.419-.047-.843-.047-7.52 0-7.793-.01-7.269.155-8.273.254-1.544.8-2.801 1.6-3.68 1.497-1.642 3.773-1.894 6.431-.711.611.272 1.005.497 3.078 1.759 1.093.665 2.044 1.236 2.115 1.27.53.256 1.197.035 1.527-.505.226-.37.237-.872.03-1.302-.145-.303-.198-.34-2.39-1.674C14.424 1.281 13.873.987 12.752.584a10.357 10.357 0 0 0-2.153-.53A11.868 11.868 0 0 0 8.893.016Zm-6.88 15.986c0 3.744.003 5.275.006 3.403.004-1.872.004-4.935 0-6.807-.003-1.872-.006-.34-.006 3.404Z" clip-rule="evenodd"/></svg>',
            'reddit'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M32.031 16.172a3.097 3.097 0 00-3.128-3.128c-1.001 0-1.877.438-2.44 1.189-2.502-1.814-6.006-2.94-9.947-3.003l2.315-6.882 5.756 1.376h.25c.063 1.314 1.126 2.377 2.502 2.377s2.44-1.126 2.44-2.502a2.405 2.405 0 00-2.44-2.44c-1.064 0-1.877.563-2.315 1.439 0 0-.063-.063-.125-.063L18.58 3.034c-.313-.063-.626.125-.688.375l-2.69 7.82c-3.816.125-7.195 1.189-9.697 2.94a3 3 0 00-2.377-1.126A3.097 3.097 0 000 16.171c0 1.251.751 2.377 1.877 2.878a5.336 5.336 0 00-.188 1.439c0 5.13 6.381 9.322 14.264 9.322s14.264-4.192 14.264-9.322c0-.438-.063-.938-.188-1.376 1.189-.5 2.002-1.627 2.002-2.94zM8.696 18.674c0-1.251 1.064-2.315 2.315-2.315s2.315 1.064 2.315 2.315-1.064 2.315-2.315 2.315-2.315-1.064-2.315-2.315zm12.512 6.382c-1.064 1.064-2.753 1.564-5.193 1.564s-4.129-.5-5.193-1.564c-.125-.125-.125-.375 0-.563.125-.125.375-.125.501 0 .938.938 2.44 1.376 4.692 1.376 2.19 0 3.754-.438 4.692-1.376a.38.38 0 01.5 0c.125.188.125.438 0 .563zm-.125-4.067c-1.314 0-2.377-1.001-2.377-2.315 0-1.251 1.064-2.315 2.377-2.315 1.251 0 2.252 1.064 2.252 2.315 0 1.314-1.001 2.315-2.252 2.315z"/></svg>',
            'search'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.962 26.499l-4.938-4.938c1.602-2.002 2.669-4.671 2.669-7.474 0-6.673-5.339-12.012-12.012-12.012S2.669 7.414 2.669 14.087a11.962 11.962 0 0012.012 12.012c2.803 0 5.472-.934 7.474-2.669l4.938 4.938c.267.267.667.4.934.4s.667-.133.934-.4a1.29 1.29 0 000-1.868zM5.339 14.087c0-5.205 4.137-9.342 9.342-9.342s9.342 4.137 9.342 9.342c0 2.536-1.068 4.938-2.669 6.54-1.735 1.735-4.004 2.669-6.54 2.669-5.339.133-9.476-4.004-9.476-9.209z"/></svg>',
            'share-2'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.023 19.425c-1.468 0-2.803.667-3.737 1.602l-7.074-4.137c0-.534.133-.934.133-1.468s-.133-.934-.267-1.468l7.074-4.137c1.068.934 2.402 1.602 3.87 1.602 2.936 0 5.339-2.402 5.339-5.339S26.959.741 24.022.741s-5.339 2.402-5.339 5.339c0 .534.133.934.267 1.468l-7.207 4.137c-.934-.934-2.269-1.602-3.737-1.602-2.936 0-5.339 2.402-5.339 5.339s2.402 5.339 5.339 5.339c1.468 0 2.803-.667 3.737-1.602l7.074 4.137c0 .534-.133.934-.133 1.468 0 2.936 2.402 5.339 5.339 5.339s5.339-2.402 5.339-5.339-2.402-5.339-5.339-5.339zm0-16.015c1.468 0 2.669 1.201 2.669 2.669s-1.201 2.669-2.669 2.669-2.669-1.201-2.669-2.669 1.201-2.669 2.669-2.669zM8.008 18.091c-1.468 0-2.669-1.201-2.669-2.669s1.201-2.669 2.669-2.669 2.669 1.201 2.669 2.669a2.677 2.677 0 01-2.669 2.669zm16.015 9.342a2.677 2.677 0 01-2.669-2.669c0-.534.133-.934.4-1.335.4-.801 1.335-1.335 2.269-1.335 1.468 0 2.669 1.201 2.669 2.669s-1.201 2.669-2.669 2.669z"/></svg>',
            'shopping-bag'    => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 20 20"><path d="M3 2h14a1 1 0 011 1v14a1 1 0 01-1 1H3a1 1 0 01-1-1V3a1 1 0 011-1zM0 3a3 3 0 013-3h14a3 3 0 013 3v14a3 3 0 01-3 3H3a3 3 0 01-3-3zm10 7c-2.761 0-5-2.686-5-6h2c0 2.566 1.669 4 3 4s3-1.434 3-4h2c0 3.314-2.239 6-5 6z"></path></svg>',
            'shopping-empty'  => '<svg xmlns="http://www.w3.org/2000/svg" width="52" height="32" viewBox="0 0 640 512"><path d="M632 416H128V8c0-4.4-3.6-8-8-8H8C3.6 0 0 3.6 0 8v16c0 4.4 3.6 8 8 8h88v408c0 4.4 3.6 8 8 8h58.9c-1.8 5-2.9 10.4-2.9 16 0 26.5 21.5 48 48 48s48-21.5 48-48c0-5.6-1.2-11-2.9-16H451c-1.8 5-2.9 10.4-2.9 16 0 26.5 21.5 48 48 48s48-21.5 48-48c0-5.6-1.2-11-2.9-16H632c4.4 0 8-3.6 8-8v-16c0-4.4-3.6-8-8-8zm-408 48c0 8.8-7.2 16-16 16s-16-7.2-16-16 7.2-16 16-16 16 7.2 16 16zm288 0c0 8.8-7.2 16-16 16s-16-7.2-16-16 7.2-16 16-16 16 7.2 16 16z"></path></svg>',
            'shopping-cart'   => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M14.681 27.433a2.67 2.67 0 11-5.338 0 2.67 2.67 0 015.338 0zm14.681 0a2.67 2.67 0 11-5.338 0 2.67 2.67 0 015.338 0zm2.402-20.82c-.267-.4-.667-.534-1.068-.534H9.075L8.007.474C7.874-.193 7.34-.594 6.672-.594H1.333C.532-.594-.002-.06-.002.741s.534 1.335 1.335 1.335h4.271l1.068 5.605v.133l2.269 11.077c.4 1.868 2.002 3.203 3.87 3.203H25.89c2.002 0 3.604-1.335 4.004-3.203L32.029 7.68c0-.4 0-.801-.267-1.068zm-4.537 11.745c-.133.667-.667 1.068-1.335 1.068H12.946c-.667 0-1.201-.4-1.335-1.068L9.609 8.749h19.486l-1.868 9.609z"/></svg>',
            'shopping-cart-2' => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none"><path fill="none" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-miterlimit="10" stroke-width="1.5" d="M2 2h1.74c1.08 0 1.93.93 1.84 2l-.83 9.96a2.796 2.796 0 0 0 2.79 3.03h10.65c1.44 0 2.7-1.18 2.81-2.61l.54-7.5c.12-1.66-1.14-3.01-2.81-3.01H5.82M16.25 22a1.25 1.25 0 1 0 0-2.5 1.25 1.25 0 0 0 0 2.5ZM8.25 22a1.25 1.25 0 1 0 0-2.5 1.25 1.25 0 0 0 0 2.5ZM9 8h12"/></svg>',
            'skype'           => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.905 19.801c.25-1.001.313-2.127.313-3.191 0-8.321-6.694-15.077-15.015-15.077-.876 0-1.752.063-2.565.188C12.262.908 10.698.407 8.946.407 4.129.407.25 4.348.25 9.166c0 1.564.438 3.128 1.189 4.379-.188 1.001-.313 2.002-.313 3.065 0 8.321 6.757 15.015 15.077 15.015.938 0 1.814-.063 2.753-.25 1.189.688 2.628 1.064 4.129 1.064 4.817 0 8.696-3.941 8.696-8.759 0-1.376-.313-2.69-.876-3.879zm-7.132 3.941c-.688 1.001-1.689 1.752-3.066 2.315-1.314.563-2.878.813-4.692.813-2.127 0-3.941-.375-5.38-1.126-1.001-.5-1.877-1.251-2.502-2.19s-.938-1.814-.938-2.69c0-.563.188-1.064.626-1.439s.938-.563 1.564-.563c.501 0 .938.125 1.314.438s.688.751.876 1.314c.313.563.563 1.126.876 1.501s.751.751 1.251 1.001c.563.25 1.314.375 2.19.375 1.251 0 2.315-.25 3.066-.813.751-.5 1.126-1.126 1.126-1.877 0-.563-.188-1.064-.626-1.439-.375-.375-.938-.626-1.564-.876a57.515 57.515 0 00-2.753-.688c-1.564-.313-2.94-.688-4.004-1.189-1.064-.438-1.939-1.064-2.565-1.877s-1.001-1.814-1.001-3.003c0-1.126.375-2.19 1.064-3.066s1.627-1.564 2.94-2.002c1.251-.5 2.69-.688 4.379-.688 1.376 0 2.565.125 3.503.438 1.001.313 1.877.751 2.502 1.251.688.501 1.189 1.064 1.501 1.627s.438 1.189.438 1.752c0 .501-.188 1.001-.626 1.439-.375.438-.938.626-1.564.626-.5 0-.938-.125-1.251-.375s-.563-.688-.938-1.189c-.375-.751-.813-1.314-1.314-1.689s-1.376-.563-2.502-.563-2.002.188-2.628.626c-.626.375-.938.876-.938 1.439 0 .313.063.626.313.876.188.25.501.501.876.688s.813.375 1.189.438c.438.125 1.126.313 2.127.563 1.189.25 2.315.501 3.316.813 1.001.375 1.877.751 2.565 1.189.751.501 1.314 1.064 1.689 1.814.438.688.626 1.627.626 2.628a5.847 5.847 0 01-1.064 3.378z"/></svg>',
            'snapchat'        => '<svg xmlns="http://www.w3.org/2000/svg" width="30" height="32" viewBox="0 0 32 32"><path d="M15.158 2.981c3.217-.036 5.881 1.77 7.221 4.665.411.876.483 2.234.483 3.2 0 1.144-.089 2.27-.161 3.414.143.072.34.125.5.125.643 0 1.18-.483 1.823-.483.608 0 1.484.429 1.484 1.144 0 1.716-3.593 1.394-3.593 2.896 0 .268.107.518.215.769.858 1.877 2.485 3.682 4.415 4.487.465.197.929.304 1.43.411.322.072.5.304.5.626 0 1.215-3.092 1.716-3.915 1.841-.358.554-.089 1.859-1.037 1.859-.733 0-1.466-.232-2.252-.232-.375 0-.751.018-1.108.089-2.127.358-2.842 2.645-6.328 2.645-3.36 0-4.165-2.288-6.238-2.645a5.938 5.938 0 00-1.126-.089c-.804 0-1.573.268-2.216.268-1.001 0-.697-1.323-1.072-1.895-.822-.125-3.915-.626-3.915-1.841 0-.322.179-.554.5-.626.5-.107.965-.215 1.43-.411 1.913-.787 3.575-2.61 4.415-4.487.107-.25.215-.5.215-.769 0-1.501-3.611-1.215-3.611-2.878 0-.697.822-1.144 1.448-1.144.554 0 1.108.465 1.805.465.197 0 .393-.036.572-.125-.072-1.126-.161-2.252-.161-3.396 0-.965.072-2.342.483-3.217 1.573-3.396 4.236-4.63 7.793-4.665z"/></svg>',
            'soundcloud'      => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M.626 21.552H0v-3.879h.626zm1.626 1.252h-.626v-6.068h.626zm1.251.375h-.626V16.11h.626zm1.314.313h-.626v-7.069h.626zm1.564 0h-.626V14.17h.626zm1.314 0h-.626V12.919h.626zm1.564 0h-.626V12.231h.626zm1.314 0h-.626V11.918h.626zm1.626 0h-.688V12.231h.688zm1.252 0h-.626V12.606h.626zm1.313 0h-.688V11.293h.688zm1.564 0h-.626V10.354h.626zm11.762-.125H17.142c-.25 0-.5-.188-.5-.438V10.417c0-.25.125-.375.438-.501.751-.313 1.627-.438 2.502-.438 3.629 0 6.631 2.753 6.944 6.319.5-.188 1.001-.313 1.564-.313 2.19 0 3.941 1.752 3.941 3.941s-1.752 3.941-3.941 3.941z"/></svg>',
            'spotify'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.838 26.369c-.375.626-1.251.876-1.877.438-5.318-3.191-11.949-3.941-19.769-2.127-.751.125-1.501-.313-1.689-1.064s.313-1.501 1.064-1.689c8.571-1.939 15.891-1.126 21.834 2.502.688.375.876 1.251.438 1.939zm2.752-6.131a1.726 1.726 0 01-2.377.563c-6.068-3.691-15.265-4.755-22.459-2.628-.876.313-1.877-.25-2.19-1.126-.25-.938.25-1.939 1.189-2.19 8.195-2.502 18.33-1.314 25.275 2.94.813.563 1.126 1.627.563 2.44zm3.128-7.131c-.563 1.001-1.877 1.314-2.878.751C21.583 9.541 9.634 9.166 2.689 11.23c-1.064.375-2.252-.25-2.628-1.376C-.252 8.728.374 7.539 1.5 7.226c7.945-2.44 21.146-1.939 29.529 3.003 1.001.626 1.314 1.877.688 2.878z"/></svg>',
            'star'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.697 11.417c-.133-.534-.534-.801-1.068-.934l-8.542-1.201-3.87-7.741c-.4-.934-2.002-.934-2.402 0l-3.87 7.607-8.408 1.335c-.534 0-.934.4-1.201.934-.133.534 0 1.068.4 1.335l6.139 6.006L6.407 27.3c-.133.534.133 1.068.534 1.335s.934.4 1.468.133l7.607-4.004 7.607 4.004h.667c.267 0 .534-.133.801-.267.4-.267.667-.801.534-1.335l-1.468-8.542 6.139-6.006c.4-.267.534-.801.4-1.201zm-8.942 5.873c-.267.4-.4.801-.4 1.201l1.068 6.54-5.872-3.07c-.4-.267-.801-.267-1.201 0l-5.872 3.07 1.201-6.54c0-.4-.133-.801-.4-1.201l-4.805-4.538 6.54-.934c.4 0 .801-.4 1.068-.667l2.936-6.006 2.936 5.872c.133.4.534.667 1.068.667l6.54.934-4.805 4.671z"/></svg>',
            'tag'             => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.428 12.619L16.95 1.141c-.267-.267-.534-.4-.934-.4H2.67c-.801 0-1.335.534-1.335 1.335v13.346c0 .4.133.667.4.934l11.478 11.478c.801.801 1.868 1.201 2.803 1.201 1.068 0 2.002-.4 2.803-1.201l9.609-9.609c1.602-1.602 1.602-4.004 0-5.605zm-1.869 3.737l-9.609 9.609a1.29 1.29 0 01-1.868 0L4.005 14.888V3.41h11.478L26.56 14.487a1.29 1.29 0 010 1.868zM8.408 7.814c-.267.267-.4.534-.4.934s.133.667.4.934.534.4.934.4.667-.133.934-.4.4-.534.4-.934-.133-.667-.4-.934a1.29 1.29 0 00-1.868 0z"/></svg>',
            'tumblr'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M6.131 13.544V8.977c1.251-.375 2.377-1.001 3.253-1.752s1.627-1.689 2.127-2.753c.563-1.064.938-2.44 1.126-4.066h4.567v8.07h7.632v5.067h-7.632v8.195c0 1.877.125 3.066.313 3.566.188.563.563.938 1.064 1.251.751.438 1.564.688 2.44.688 1.627 0 3.253-.563 4.88-1.627v5.067c-1.376.626-2.628 1.126-3.754 1.376-1.064.25-2.315.375-3.566.375-1.501 0-2.753-.188-3.879-.563s-2.127-.876-2.878-1.564c-.813-.688-1.376-1.439-1.689-2.19s-.438-1.877-.438-3.378V13.541z"/></svg>',
            'twitch'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M22.96 28.246h-6.256l-4.192 4.192H8.383v-4.192H.688V5.974L2.815.406h28.528v19.519zm5.568-9.759V3.16H5.568v20.207h6.256v4.192l4.192-4.192h7.632zM20.895 8.79v8.321h2.753V8.79zM13.2 17.11h2.815V8.789H13.2z"/></svg>',
            'twitter'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24"><path d="M14.258 10.152 23.176 0h-2.113l-7.747 8.813L7.133 0H0l9.352 13.328L0 23.973h2.113l8.176-9.309 6.531 9.309h7.133zm-2.895 3.293-.949-1.328L2.875 1.56h3.246l6.086 8.523.945 1.328 7.91 11.078h-3.246zm0 0"/></svg>',
            'vimeo'           => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M20.708 9.541c-.751 0-1.564.188-2.377.563 1.564-5.193 4.567-7.695 9.009-7.507 3.316.063 4.88 2.19 4.692 6.381-.188 3.128-2.315 7.382-6.569 12.825-4.317 5.63-8.008 8.508-11.011 8.508-1.877 0-3.441-1.752-4.755-5.193-.876-3.191-1.752-6.319-2.565-9.509-1.001-3.441-2.002-5.13-3.128-5.13-.25 0-1.064.501-2.502 1.501L.001 10.041c1.564-1.439 3.128-2.815 4.692-4.192 2.065-1.814 3.691-2.753 4.755-2.878 2.44-.25 4.004 1.439 4.567 5.13.626 3.941 1.064 6.381 1.314 7.32.688 3.253 1.501 4.88 2.377 4.88.688 0 1.689-1.064 3.003-3.128 1.314-2.127 2.065-3.754 2.19-4.88.188-1.814-.563-2.753-2.19-2.753z"/></svg>',
            'vkontakte'       => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.968 22.491c-.938-1.251-1.752-2.002-2.94-3.128-.5-.5-1.189-1.001-1.251-1.689-.063-.938.813-1.814 1.251-2.44l.188-.188.125-.125c0-.063.063-.063.063-.125.063-.063.063-.125.125-.125 0-.063.063-.063.063-.063.063-.063.063-.125.125-.188.813-1.064 1.627-2.002 2.44-3.253.25-.438 1.064-2.002.813-2.628-.375-.751-2.315-.501-3.378-.501-1.314 0-2.753-.25-3.691.125-.563.188-.938 1.439-1.251 2.127-.5 1.064-.938 1.939-1.439 2.753 0 .063-.063.125-.125.188 0 .063-.063.125-.125.188-.063.125-.125.25-.188.313 0 .063-.063.063-.063.063 0 .063-.063.063-.063.125-.063.063-.063.063-.063.125-.063 0-.063.063-.125.125 0 0-.063.063-.063.125-.063 0-.063.063-.063.063-.063.063-.125.125-.188.25-.438.626-.876 1.376-1.689 1.501-.063-.063-.063-.063-.125-.063v-.063h-.063c0-.063-.063-.063-.063-.063l-.063-.063s0-.063-.063-.063c0-.063-.063-.063-.063-.063v-.063s-.063 0-.063-.063 0-.063-.063-.063v-.063-.063h-.063v-.125l-.063-.063v-.063-.063s0-.063-.063-.063v-.063-.063-.063c0-.063 0-.063-.063-.063v-.063-.063-.063-.063-.063-.125-.125-.125s-.063 0-.063-.063v-.063-.063-.063-.063-.063c0-.063.063-.063.063-.063v-.063-.063-.125-.125-.501-.063c0-.125 0-.313.063-.438 0-1.376.438-3.191-.188-4.254-.563-.938-2.69-.813-4.192-.813-1.814 0-3.253.125-3.816 1.189.626.438 1.314.438 1.627 1.064.375.688.438 2.127.375 3.441v.188c0 1.126-.063 2.94-.813 3.066s-1.376-1.001-1.814-1.564c-.063-.125-.125-.188-.188-.313l-.063-.063c-.063-.063-.063-.125-.125-.25l-.063-.063c-.063-.063-.125-.125-.125-.188-.063-.063-.063-.063-.063-.125-.063-.063-.125-.188-.188-.25-.563-.938-1.126-1.877-1.627-3.066-.375-.688-.813-1.877-1.189-2.065-.751-.375-2.252-.188-3.253-.188-1.126 0-2.315-.188-3.003.375 0 1.251.626 2.315 1.126 3.316.688 1.376 1.376 2.69 2.127 3.941 0 .063.063.125.063.188.125.125.188.313.313.438 0 .063.063.125.063.188.125.125.188.313.313.438 0 .063.063.125.125.188l.375.563c.125.188.25.438.375.626.063.063.063.125.125.125.125.188.188.313.313.438 0 .063.063.125.125.188.063.125.188.313.313.438 0 .063.063.125.125.188.125.188.25.375.438.563 1.939 2.44 4.129 4.755 8.446 5.255 1.376.125 3.378.25 3.816-.563.626-1.001-.063-3.378 1.314-3.566.938-.188 1.752 1.251 2.315 1.814.938 1.064 2.065 2.19 3.691 2.502h.125c1.439-.626 5.756.688 6.006-1.126.063-.626-.688-1.627-1.064-2.065z"/></svg>',
            'x'               => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M17.884 15.421l7.074-7.074c.534-.534.534-1.335 0-1.868s-1.335-.534-1.868 0l-7.074 7.074-7.074-7.074c-.534-.534-1.335-.534-1.868 0s-.534 1.335 0 1.868l7.074 7.074-7.074 7.074a1.29 1.29 0 000 1.868c.267.267.534.4.934.4s.667-.133.934-.4l7.074-7.074 7.074 7.074c.267.267.667.4.934.4s.667-.133.934-.4a1.29 1.29 0 000-1.868l-7.074-7.074z"/></svg>',
            'xing'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.695 22.866h-4.63c-.25 0-.438-.125-.563-.313s-.125-.5 0-.751l4.88-8.633v-.063l-3.128-5.38c-.125-.25-.125-.5 0-.688s.313-.313.626-.313h4.63c.688 0 1.064.438 1.251.876l3.191 5.505c-.188.375-5.005 8.821-5.005 8.821-.25.438-.563.938-1.251.938zM29.529 1.408l-10.197 18.08v.063l6.506 11.887c.125.25.125.5 0 .688s-.313.313-.626.313h-4.63c-.688 0-1.001-.5-1.251-.876 0 0-6.569-12.012-6.569-12.074.313-.563 10.26-18.205 10.26-18.205.25-.438.563-.876 1.251-.876h4.692c.25 0 .5.125.563.313.125.188.125.438 0 .688z"/></svg>',
            'quote'           => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 123.961 123.961"><path d="M49.8 29.032c3.1-1.3 4.4-5 3-8l-4.9-10.3c-1.4-2.899-4.8-4.2-7.8-2.899-8.5 3.6-15.8 8.3-21.6 14C11.4 28.532 6.6 36.232 4 44.732c-2.6 8.601-4 20.3-4 35.2v30.7c0 3.3 2.7 6 6 6h39.3c3.3 0 6-2.7 6-6v-39.3c0-3.301-2.7-6-6-6H26.5c.2-10.101 2.6-18.2 7-24.301 3.6-4.898 9-8.898 16.3-11.999zM120.4 29.032c3.1-1.3 4.399-5 3-8l-4.9-10.199c-1.4-2.9-4.8-4.2-7.8-2.9-8.4 3.6-15.601 8.3-21.5 13.9-7.101 6.8-12 14.5-14.601 23-2.6 8.399-3.899 20.1-3.899 35.1v30.7c0 3.3 2.7 6 6 6H116c3.3 0 6-2.7 6-6v-39.3c0-3.301-2.7-6-6-6H97.1c.2-10.101 2.601-18.2 7-24.301 3.6-4.899 9-8.899 16.3-12z"/></svg>',
            'yelp'            => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M19.027 20.421c-.54.545-.083 1.535-.083 1.535l4.065 6.788s.668.895 1.246.895c.581 0 1.155-.477 1.155-.477l3.214-4.594s.324-.58.331-1.087c.012-.722-1.077-.92-1.077-.92l-7.609-2.444s-.746-.198-1.242.303zM18.642 17c.389.66 1.463.468 1.463.468l7.592-2.219s1.035-.421 1.182-.982c.145-.561-.171-1.238-.171-1.238L25.08 8.755s-.314-.54-.967-.595c-.719-.061-1.162.809-1.162.809l-4.29 6.75s-.379.672-.02 1.28zm-3.587-2.632c.894-.22 1.036-1.519 1.036-1.519L16.03 2.041S15.895.708 15.296.346c-.94-.57-1.218-.272-1.487-.233L7.506 2.455s-.617.204-.939.719c-.459.73.466 1.798.466 1.798l6.551 8.93s.647.669 1.47.465zm-1.557 4.374c.023-.834-1.001-1.334-1.001-1.334l-6.775-3.423s-1.004-.414-1.491-.126c-.373.221-.704.62-.737.973l-.441 5.432s-.066.941.178 1.369c.345.608 1.482.184 1.482.184l7.909-1.748c.307-.207.846-.225.876-1.328zm1.967 2.931c-.679-.349-1.491.373-1.491.373l-5.296 5.83s-.661.892-.493 1.439c.158.513.42.768.791.948l5.319 1.679s.645.134 1.133-.008c.693-.201.565-1.286.565-1.286l.12-7.894s-.027-.76-.648-1.082z"/></svg>',
            'youtube'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M31.718 10.041s-.313-2.252-1.251-3.191c-1.251-1.314-2.628-1.314-3.253-1.376-4.442-.313-11.198-.313-11.198-.313s-6.757 0-11.198.313c-.626.063-2.002.063-3.253 1.376-.938.938-1.251 3.191-1.251 3.191s-.313 2.565-.313 5.13v2.44c0 2.628.313 5.193.313 5.193s.313 2.19 1.251 3.191c1.251 1.251 2.878 1.251 3.566 1.376 2.565.25 10.886.313 10.886.313s6.757 0 11.198-.313c.626-.063 2.002-.125 3.253-1.376.938-1.001 1.251-3.191 1.251-3.191s.313-2.565.313-5.193v-2.44c0-2.565-.313-5.13-.313-5.13zm-11.386 6.632l-7.57 3.941v-9.009l8.633 4.504z"/></svg>',
            'whatsapp'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M27.281 4.65C24.287 1.65 20.306 0 16.062 0 7.324 0 .212 7.112.212 15.856c0 2.794.731 5.525 2.119 7.925L.081 32l8.406-2.206a15.858 15.858 0 007.575 1.931h.006c8.738 0 15.856-7.113 15.856-15.856 0-4.238-1.65-8.219-4.644-11.219zm-11.212 24.4a13.17 13.17 0 01-6.713-1.837l-.481-.288-4.987 1.306 1.331-4.863-.313-.5a13.086 13.086 0 01-2.019-7.012c0-7.269 5.912-13.181 13.188-13.181 3.519 0 6.831 1.375 9.319 3.862a13.108 13.108 0 013.856 9.325c-.006 7.275-5.919 13.188-13.181 13.188zm7.225-9.875c-.394-.2-2.344-1.156-2.706-1.288s-.625-.2-.894.2c-.262.394-1.025 1.288-1.256 1.556-.231.262-.462.3-.856.1s-1.675-.619-3.188-1.969c-1.175-1.05-1.975-2.35-2.206-2.744s-.025-.613.175-.806c.181-.175.394-.463.594-.694s.262-.394.394-.662c.131-.262.069-.494-.031-.694s-.894-2.15-1.219-2.944c-.319-.775-.65-.669-.894-.681-.231-.012-.494-.012-.756-.012s-.694.1-1.056.494c-.363.394-1.387 1.356-1.387 3.306s1.419 3.831 1.619 4.1c.2.262 2.794 4.269 6.769 5.981.944.406 1.681.65 2.256.837.95.3 1.813.256 2.494.156.762-.113 2.344-.956 2.675-1.881s.331-1.719.231-1.881c-.094-.175-.356-.275-.756-.475z"/></svg>',
            'tiktok'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24"><path d="M19.589 6.686a4.793 4.793 0 0 1-3.77-4.245V2h-3.445v13.672a2.896 2.896 0 0 1-5.201 1.743l-.002-.001.002.001a2.895 2.895 0 0 1 3.183-4.51v-3.5a6.329 6.329 0 0 0-5.394 10.692 6.33 6.33 0 0 0 10.857-4.424V8.687a8.182 8.182 0 0 0 4.773 1.526V6.79a4.831 4.831 0 0 1-1.003-.104z"/></svg>',
            'stackoverflow'   => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.75 28.765v-8.09H29V32H3V20.675h3.25v8.09h19.5ZM8.687 25.527v-3.235h14.626v3.235H8.688Zm.126-7.084.563-3.186 14.405 2.528-.566 3.186-14.402-2.528Zm2.655-8.902 1.625-2.801 12.665 7.278-1.625 2.801-12.665-7.278ZM17.02 2.08 19.51 0l9.4 11.152-2.489 2.078-9.4-11.15Z"/></svg>',
            'rss'             => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24.912 24.912"><path d="M3.692 17.517A3.696 3.696 0 0 0 0 21.211C0 23.244 1.656 24.9 3.692 24.9s3.694-1.657 3.694-3.689a3.697 3.697 0 0 0-3.694-3.694z"/><path d="M.384 8.142A.386.386 0 0 0 0 8.527v4.688c0 .211.173.383.384.383 6.02 0 10.919 4.898 10.919 10.92 0 .209.171.383.384.383h4.705a.385.385 0 0 0 .387-.383l-.018-.121C16.692 15.423 9.37 8.142.384 8.142z"/><path d="M24.89 24.397C24.825 10.936 13.854.011.384.011A.385.385 0 0 0 0 .397v4.824c0 .212.173.383.384.383 10.429 0 18.913 8.486 18.913 18.914 0 .209.172.383.382.383h4.845a.39.39 0 0 0 .388-.383l-.022-.121z"/></svg>',
            'pin'             => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M1.599 32c-.407 0-.813-.16-1.125-.472a1.579 1.579 0 0 1-.16-2.073l6.654-9.01-5.214-5.218a1.607 1.607 0 0 1-.245-1.937c.249-.394 2.395-3.526 7.166-2.344.08.01.165.016.256.024.196.017.415.037.652.075 1.01.164 2.796-.647 4.346-1.97 1.485-1.265 2.445-2.742 2.445-3.76 0-.237-.006-.489-.012-.736-.04-.957-.1-2.428.999-3.527a3.564 3.564 0 0 1 4.787-.245.902.902 0 0 1 .08.074c.767.751 8.637 8.62 8.72 8.702A3.574 3.574 0 0 1 32 12.12c0 .959-.372 1.858-1.047 2.534-1.094 1.096-2.572 1.033-3.547.992-.23-.005-.483-.01-.719-.01-.963 0-2.114.674-3.239 1.898-1.587 1.727-2.564 3.958-2.48 4.957.035.32.1.887.113.966 1.152 4.676-1.971 6.808-2.33 7.035a1.596 1.596 0 0 1-1.968-.226l-5.571-5.575a.799.799 0 1 1 1.13-1.13l5.57 5.575c.246-.16 2.502-1.708 1.603-5.368a32.445 32.445 0 0 1-.137-1.124c-.134-1.598 1.112-4.251 2.895-6.192 1.013-1.101 2.6-2.415 4.414-2.415.247 0 .512.006.77.011 1.012.043 1.813.03 2.367-.524a1.986 1.986 0 0 0-.005-2.81c-.082-.082-8.463-8.456-8.71-8.694-.758-.616-1.896-.559-2.617.163-.559.56-.572 1.36-.531 2.347.006.275.012.539.012.786 0 1.511-1.124 3.373-3.006 4.977-1.546 1.317-3.84 2.62-5.637 2.332a6.768 6.768 0 0 0-.537-.061 7.63 7.63 0 0 1-.29-.029.794.794 0 0 1-.225-.04c-3.678-.908-5.234 1.357-5.4 1.618l5.707 5.689a.8.8 0 0 1 .078 1.04L1.598 30.41l7.452-5.454a.8.8 0 0 1 .941 1.291l-7.458 5.441c-.28.209-.607.312-.934.312Z"/></svg>',
            'telegram'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="27" viewBox="0 0 32 27"><path fill-rule="evenodd" d="M24.512 26.372c.43.304.983.38 1.476.193.494-.188.857-.609.966-1.12C28.113 20 30.924 6.217 31.978 1.264a1.041 1.041 0 0 0-.347-1.01c-.293-.25-.7-.322-1.063-.187C24.979 2.136 7.762 8.596.724 11.2a1.102 1.102 0 0 0-.722 1.065c.016.472.333.882.79 1.019 3.156.944 7.299 2.257 7.299 2.257s1.936 5.847 2.945 8.82c.127.374.419.667.804.768.384.1.795-.005 1.082-.276l4.128-3.897s4.762 3.492 7.463 5.416Zm-14.68-11.57 2.24 7.385.497-4.676 13.58-12.248a.37.37 0 0 0 .043-.503.379.379 0 0 0-.5-.085L9.831 14.803Z" clip-rule="evenodd"/></svg>',
            'discord'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="25" viewBox="0 0 32 27"><path d="M27.129 2.038A26.392 26.392 0 0 0 20.515 0c-.291.501-.615 1.18-.841 1.714a25.285 25.285 0 0 0-7.327 0A21.038 21.038 0 0 0 11.506 0 26.019 26.019 0 0 0 4.89 2.038C.702 8.215-.43 14.247.136 20.198c2.781 2.021 5.466 3.25 8.103 4.06a19.716 19.716 0 0 0 1.73-2.799 16.972 16.972 0 0 1-2.732-1.31c.226-.161.452-.34.663-.517 5.271 2.41 10.982 2.41 16.189 0 .226.178.436.356.663.517a16.97 16.97 0 0 1-2.733 1.31 19.737 19.737 0 0 0 1.73 2.798 26.81 26.81 0 0 0 8.104-4.059c.695-6.889-1.102-12.872-4.724-18.16Zm-16.432 14.49c-1.585 0-2.878-1.44-2.878-3.203 0-1.762 1.261-3.202 2.878-3.202 1.601 0 2.911 1.44 2.879 3.202 0 1.763-1.278 3.202-2.879 3.202Zm10.627 0c-1.585 0-2.88-1.44-2.88-3.203 0-1.762 1.262-3.202 2.88-3.202 1.6 0 2.91 1.44 2.878 3.202 0 1.763-1.261 3.202-2.879 3.202Z"/></svg>',
            'bluesky'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.503 4.75c3.44 2.542 7.139 7.695 8.497 10.46 1.358-2.765 5.058-7.918 8.497-10.46C26.98 2.916 31 1.497 31 6.012c0 .902-.525 7.576-.833 8.66-1.071 3.766-4.973 4.727-8.444 4.145 6.067 1.017 7.61 4.383 4.277 7.75-6.33 6.394-9.099-1.604-9.808-3.654-.13-.375-.191-.551-.192-.402 0-.15-.062.027-.192.402-.71 2.05-3.477 10.048-9.808 3.654-3.333-3.367-1.79-6.734 4.277-7.75-3.47.582-7.373-.38-8.444-4.146C1.525 13.588 1 6.914 1 6.012 1 1.497 5.021 2.916 7.503 4.75Z"/></svg>',
            'wechat'          => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M21.655 10.747c.373 0 .733.017 1.094.063C21.783 6.33 16.89 3 11.328 3 5.091 3 0 7.16 0 12.45c0 3.05 1.705 5.57 4.532 7.518L3.41 23.316l3.957-1.949c1.42.275 2.548.555 3.968.555.36 0 .704-.018 1.065-.046a7.729 7.729 0 0 1-.361-2.331c-.006-4.85 4.241-8.798 9.617-8.798Zm-6.08-3.022c.844 0 1.408.554 1.408 1.394 0 .828-.564 1.382-1.408 1.382-.86 0-1.704-.554-1.704-1.382.005-.84.849-1.394 1.704-1.394ZM7.64 10.5c-.843 0-1.704-.554-1.704-1.382 0-.846.86-1.394 1.704-1.394.861 0 1.42.554 1.42 1.394 0 .834-.559 1.382-1.42 1.382ZM32 19.42c0-4.45-4.532-8.072-9.623-8.072-5.394 0-9.623 3.622-9.623 8.072 0 4.45 4.235 8.073 9.623 8.073 1.123 0 2.263-.292 3.41-.566l3.106 1.674-.861-2.777C30.312 24.143 32 21.922 32 19.42Zm-12.748-1.4c-.564 0-1.122-.554-1.122-1.12 0-.553.564-1.102 1.122-1.102.861 0 1.42.554 1.42 1.103 0 .571-.564 1.12-1.42 1.12Zm6.232 0c-.565 0-1.123-.554-1.123-1.12 0-.553.564-1.102 1.123-1.102.843 0 1.42.554 1.42 1.103.005.571-.577 1.12-1.42 1.12Z"/></svg>',
            'qq'         	  => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M29.57 26.275c-.718.088-2.806-3.294-2.806-3.294 0 1.956-1.006 4.513-3.193 6.363 1.05.325 3.425 1.2 2.862 2.15-.456.769-7.844.494-9.975.25-2.131.237-9.519.519-9.975-.25-.562-.956 1.806-1.825 2.863-2.15-2.182-1.844-3.194-4.4-3.194-6.363 0 0-2.081 3.381-2.806 3.294-.338-.044-.775-1.85.58-6.231.645-2.063 1.376-3.782 2.507-6.613C6.252 6.131 9.271 0 16.458 0c7.106 0 10.2 6.006 10.019 13.438 1.131 2.824 1.869 4.556 2.506 6.612 1.363 4.381.919 6.194.581 6.231l.007-.006Z"/></svg>',
            'bilibili'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 512 512"><path d="M488.6 104.1c16.7 18.1 24.4 39.7 23.3 65.7v202.4c-.4 26.4-9.2 48.1-26.5 65.1-17.2 17-39.1 25.9-65.5 26.7H92c-26.4-.8-48.2-9.8-65.3-27.2C9.7 419.4.8 396.5 0 368.2V169.8c.8-26 9.7-47.6 26.7-65.7C43.8 87.8 65.6 78.8 92 78h29.4L96.1 52.2c-5.8-5.7-8.7-13-8.7-21.8s2.9-16.1 8.7-21.8c5.7-5.7 13-8.6 21.8-8.6s16.1 2.9 21.9 8.6L213.1 78h88l74.5-69.4C381.7 2.9 389.2 0 398 0c8.8 0 16.1 2.9 21.9 8.6 5.7 5.7 8.6 13 8.6 21.8s-2.9 16.1-8.6 21.8L394.6 78h29.3c26.4.8 48 9.8 64.7 26.1zm-38.8 69.7c-.4-9.6-3.7-17.4-10.7-23.5-5.2-6.1-14-9.4-22.7-9.8H96.1c-9.6.4-17.5 3.7-23.6 9.8-6.2 6.1-9.4 13.9-9.8 23.5v194.4c0 9.2 3.3 17 9.8 23.5 6.5 6.5 14.4 9.8 23.6 9.8h320.3c9.2 0 17-3.3 23.3-9.8 6.3-6.5 9.7-14.3 10.1-23.5V173.8zm-264.3 42.7c6.3 6.3 9.7 14.1 10.1 23.2V273c-.4 9.2-3.7 16.9-9.8 23.2-6.2 6.3-14 9.5-23.6 9.5-9.6 0-17.5-3.2-23.6-9.5-6.1-6.3-9.4-14-9.8-23.2v-33.3c.4-9.1 3.8-16.9 10.1-23.2 6.3-6.3 13.2-9.6 23.3-10 9.2.4 17 3.7 23.3 10zm191.5 0c6.3 6.3 9.7 14.1 10.1 23.2V273c-.4 9.2-3.7 16.9-9.8 23.2-6.1 6.3-14 9.5-23.6 9.5-9.6 0-17.4-3.2-23.6-9.5-7-6.3-9.4-14-9.7-23.2v-33.3c.3-9.1 3.7-16.9 10-23.2 6.3-6.3 14.1-9.6 23.3-10 9.2.4 17 3.7 23.3 10z"/></svg>',
            'threads'         => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 448 512"><path d="M331.5 235.7c2.2.9 4.2 1.9 6.3 2.8 29.2 14.1 50.6 35.2 61.8 61.4 15.7 36.5 17.2 95.8-30.3 143.2-36.2 36.2-80.3 52.5-142.6 53h-.3c-70.2-.5-124.1-24.1-160.4-70.2-32.3-41-48.9-98.1-49.5-169.6v-.5c.5-71.5 17.1-128.6 49.4-169.6 36.3-46.1 90.3-69.7 160.5-70.2h.3c70.3.5 124.9 24 162.3 69.9 18.4 22.7 32 50 40.6 81.7l-40.4 10.8c-7.1-25.8-17.8-47.8-32.2-65.4-29.2-35.8-73-54.2-130.5-54.6-57 .5-100.1 18.8-128.2 54.4C72.1 146.1 58.5 194.3 58 256c.5 61.7 14.1 109.9 40.3 143.3 28 35.6 71.2 53.9 128.2 54.4 51.4-.4 85.4-12.6 113.7-40.9 32.3-32.2 31.7-71.8 21.4-95.9-6.1-14.2-17.1-26-31.9-34.9-3.7 26.9-11.8 48.3-24.7 64.8-17.1 21.8-41.4 33.6-72.7 35.3-23.6 1.3-46.3-4.4-63.9-16-20.8-13.8-33-34.8-34.3-59.3-2.5-48.3 35.7-83 95.2-86.4 21.1-1.2 40.9-.3 59.2 2.8-2.4-14.8-7.3-26.6-14.6-35.2-10-11.7-25.6-17.7-46.2-17.8h-.7c-16.6 0-39 4.6-53.3 26.3l-34.4-23.6c19.2-29.1 50.3-45.1 87.8-45.1h.8c62.6.4 99.9 39.5 103.7 107.7l-.2.2zm-156 68.8c1.3 25.1 28.4 36.8 54.6 35.3 25.6-1.4 54.6-11.4 59.5-73.2-13.2-2.9-27.8-4.4-43.4-4.4-4.8 0-9.6.1-14.4.4-42.9 2.4-57.2 23.2-56.2 41.8l-.1.1z"/></svg>',
            'linktree'        => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M2.05 8.123H8.1L3.775 4.098 6.15 1.742l4.1 4.098V0h3.55v5.84l4.1-4.098 2.375 2.356-4.325 4H22v3.313h-6.075l4.325 4.098-2.375 2.307L12 12.049 6.125 17.84 3.75 15.51l4.325-4.099H2V8.123h.05Zm8.225 8.024h3.55V24h-3.55v-7.853Z"/></svg>',
            'rumble'          => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M14.453 13.546a1.88 1.88 0 0 0 .275-2.645 1.88 1.88 0 0 0-.275-.275A21.213 21.213 0 0 0 10.14 7.85c-1.066-.51-2.256.2-2.426 1.414a23.523 23.523 0 0 0-.14 5.502c.116 1.23 1.292 1.964 2.372 1.492a19.628 19.628 0 0 0 4.506-2.704v-.008zm6.932-5.4a5.85 5.85 0 0 1 .014 7.872 26.149 26.149 0 0 1-13.104 7.828 5.116 5.116 0 0 1-6.167-3.578c-1.524-5.2-1.3-11.08.17-16.305C3.07 1.22 5.651-.503 8.308.131c4.925 1.174 9.545 4.196 13.077 8.013v.001z"/></svg>',
            'mastodon'        => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><path d="M11.954 3.023c-2.266.01-4.516.297-5.686.837 0 0-2.518 1.146-2.518 5.05 0 4.649-.004 10.486 4.172 11.614 1.599.43 2.974.523 4.08.459 2.006-.114 2.998-.729 2.998-.729l-.068-1.482s-1.3.458-2.91.405c-1.595-.056-3.276-.175-3.538-2.167a4.23 4.23 0 0 1-.034-.56c3.38.84 6.262.366 7.056.27 2.215-.27 4.143-1.658 4.39-2.928.385-2.001.352-4.881.352-4.881 0-3.905-2.514-5.051-2.514-5.051-1.233-.576-3.515-.846-5.78-.837ZM9.529 6.002c.776.02 1.543.367 2.016 1.101l.457.777.455-.777c.948-1.475 3.075-1.383 4.087-.242.933 1.086.725 1.787.725 6.639v.002h-1.835V9.28c0-1.976-2.52-2.053-2.52.274V12H11.09V9.554c0-2.327-2.518-2.252-2.518-.276V13.5h-1.84c0-4.855-.205-5.561.724-6.639.51-.574 1.295-.88 2.072-.86Z"/></svg>',
            'gemspace'        => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M23.498 4.427c-4.31-.59-10.685-.549-15 0C5.613 6.866 3.323 9.27 1 12.297 5.238 17.924 10.6 23.517 16 28c5.4-4.489 10.762-10.076 15-15.703-2.323-3.033-4.613-5.431-7.497-7.875l-.005.005ZM16 21.707v-5.194c-1.949-.04-3.902-.116-5.851-.312A48.53 48.53 0 0 1 7 12.307a29.468 29.468 0 0 1 3.374-3.542c3.039-.393 8.175-.433 11.213 0A30.139 30.139 0 0 1 25 12.347c-2.549 3.345-5.774 6.69-9 9.366v-.006Z"/></svg>',
            'lifegroupchat'   => '<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.701 27.088c.916-.717.877-2.08.04-2.89-.837-.812-2.154-.757-3.115-.103a9.461 9.461 0 0 1-10.549.089c-.27-.178-1.893 1.354-2.144 1.15-.294-.24.787-2.218.523-2.49a9.734 9.734 0 0 1-1.584-2.158 9.874 9.874 0 0 1-1.108-5.894 9.81 9.81 0 0 1 2.502-5.436A9.557 9.557 0 0 1 15.43 6.44a9.46 9.46 0 0 1 5.867.709 9.604 9.604 0 0 1 3.126 2.34c.773.874 2.037 1.257 3.044.68 1.008-.578 1.376-1.888.662-2.812a13.822 13.822 0 0 0-5.08-4.074 13.592 13.592 0 0 0-8.43-1.018 13.73 13.73 0 0 0-7.418 4.19 14.095 14.095 0 0 0-3.595 7.81 14.185 14.185 0 0 0 1.592 8.468 13.876 13.876 0 0 0 6.176 5.915 13.595 13.595 0 0 0 8.416 1.12 13.684 13.684 0 0 0 5.911-2.679Z"/><path d="M12.996 17.244c.677 0 1.226-.557 1.226-1.244 0-.687-.549-1.244-1.226-1.244-.676 0-1.225.557-1.225 1.244 0 .687.549 1.244 1.225 1.244ZM18.817 16c0 .687-.549 1.244-1.226 1.244A1.235 1.235 0 0 1 16.366 16c0-.687.549-1.244 1.226-1.244.676 0 1.225.557 1.225 1.244ZM22.186 17.244c.677 0 1.226-.557 1.226-1.244 0-.687-.549-1.244-1.226-1.244-.676 0-1.225.557-1.225 1.244 0 .687.549 1.244 1.226 1.244Z"/></svg>'
        );
	}
endif;
