<?php
/**
 * Blogfolio core. Direct questions to jon@planetjon.ca
 *
 */

// Load administrative panel functionality.
include( get_template_directory() . '/admin-functions.php' );

// Set WordPress content width magic global.
if (! isset( $content_width ) )
	$content_width = 600;

// Installation hooks
register_activation_hook( __FILE__, array( 'Blogfolio', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'Blogfolio', 'deactivate' ) );
register_uninstall_hook( __FILE__, array( 'Blogfolio', 'uninstall' ) );

// run theme setup after theme is loaded.
add_action( 'after_setup_theme', array( 'Blogfolio', 'after_setup_theme' ) );

// Run initializations.
add_action( 'init', array( 'Blogfolio', 'init' ) );

// Run widgets initializations.
add_action( 'widgets_init', array( 'Blogfolio', 'widgets_init' ) );

// Load scripts & styles
add_action( 'wp_enqueue_scripts', array( 'Blogfolio', 'wp_enqueue_scripts' ) );

add_filter( 'body_class', array( 'Blogfolio', 'body_class' ) );
/**
 * Provides functions that must be hooked into WordPress hooks.
 * 
 */
class Blogfolio {
	const options = 'Blogfolio';
	const version = 1.64;

	private static $config = null;
	private static $template = null;

	// Get the template instance. Use this if you need to unregister any of the hooked functions.
	static function template() {
		if(! self::$template)
			self::$template = new BlogfolioTemplate;

		return self::$template;
	}

	// Run when Blogfolio is activated.
	static function activate() {
		if( false == get_option( self::options ) )
			add_option( self::options, array( 'version' => self::version, 'search-in-menubar' => 1, 'feature-newest-post' => 1, 'credit-in-footer' => 1 ) );
	}

	// Run when Blogfolio is deactivated.
	static function deactivate() {}

	// Run when BLogfolio is uninstalled.
	static function uninstall() {
		delete_option( self::options );
	}

	// Registry for accessing theme settings.
	static function config( $key = null ) {
		if( null === self::$config )
			return null;

		if( $key !== null )
			return isset( self::$config[ $key ] ) ? self::$config[ $key ] : null;
		else
			return (array) self::$config;
	}

	// To be hooked into after_setup_theme
	static function after_setup_theme() {
		self::$config = get_option( self::options );

		// i18n support.
		load_theme_textdomain( self::options, get_template_directory() . '/resources/languages' );

		// Custom header support.
		add_theme_support( 'custom-header', array(
			'default-text-color' => '444',
			'default-image' => '',
			'width' => 718,
			'height' => 80,
			'flex-height' => true,
			'flex-width' => true
		) );

		// Custom background support.
		add_theme_support( 'custom-background', array(
			'default-color' => '000',
			'default-image' => get_template_directory_uri() . '/resources/images/darkwood.jpg',
		) );

		// Automatic feed links support.
		add_theme_support( 'automatic-feed-links' );

		// Post formats support.
		add_theme_support( 'post-formats', array( 'audio', 'video', 'image', 'gallery', 'link', 'chat', 'quote', 'status', 'aside' ) );

		// Post thumbnails support.
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 272, 204, true );

		// Custom menu support.
		register_nav_menu( 'primary', __( 'Primary Navigation', 'blogfolio' ) );

		// Hook into Blogfolio template hooks.
		add_action( 'blogfolio_topbar', array( self::template(), 'topbar' ) );
		add_action( 'blogfolio_site_header', array( self::template(), 'siteHeader' ) );
		add_action( 'blogfolio_before_content', array( self::template(), 'beforeContent' ) );
		add_action( 'blogfolio_post_feature', array( self::template(), 'postFeature' ) );
		add_action( 'blogfolio_loop_header', array( self::template(), 'loopHeader' ) );
		add_action( 'blogfolio_page_header', array( self::template(), 'pageHeader' ) );
		add_action( 'blogfolio_page_footer', array( self::template(), 'pageFooter' ) );
		add_action( 'blogfolio_single_header', array( self::template(), 'singleHeader' ) );
		add_action( 'blogfolio_single_footer', array( self::template(), 'singleFooter' ) );
		add_action( 'blogfolio_attachment_header', array( self::template(), 'attachmentHeader' ) );
		add_action( 'blogfolio_attachment_footer', array( self::template(), 'attachmentFooter' ) );
		add_action( 'blogfolio_after_content', array( self::template(), 'afterContent' ) );
		add_action( 'blogfolio_site_footer', array( self::template(), 'siteFooter' ) );
	}

	// To be hooked into init
	static function init() {
		self::_updateAsNeeded();

		add_editor_style( 'editor-style.css' );
		add_filter( 'wp_link_pages_args', array('BlogfolioTemplate', 'wpLinkPagesArgs') );
		remove_action( 'the_excerpt', 'sharing_display', 19 );
	}

	// To be hooked into widget_init
	static function widgets_init() {
		// Content Sidebar.
		register_sidebar( array(
			'name' => __( 'Content Sidebar', 'blogfolio' ),
			'id' => 'content-sidebar',
			'description' => __( 'For placing widgets alongside content', 'blogfolio' ),
			'before_widget' => '<div id="%1$s" class="widget-box %2$s">',
			'after_widget' => '</div>',
			'before_title' => '<h3 class="widget-title title"><span class="text">',
			'after_title' => '</span></h3>'
		) );

		// Site Header widget area.
		register_sidebar( array(
			'name' => __( 'Site Header Widget Area', 'blogfolio' ),
			'id' => 'site-header-widget-container',
			'description' => __( 'For placing widgets immediately after the site header', 'blogfolio' ),
			'before_widget' => '<div id="%1$s" class="site-header-widget widget-box %2$s">',
			'after_widget' => '</div>',
			'before_title' => '<h3 class="widget-title title"><span class="text">',
			'after_title' => '</span></h3>'
		) );

		// Homepage widget area.
		register_sidebar( array(
			'name' => __( 'Home Page Widget Area', 'blogfolio' ),
			'id' => 'home-page-widget-container',
			'description' => __( 'For placing widgets on the home page', 'Blogfolio' ),
			'before_widget' => '<div id="%1$s" class="home-widget widget-box %2$s">',
			'after_widget' => '</div>',
			'before_title' => '<h2 class="widget-title title"><span class="text">',
			'after_title' => '</span></h2>'
		) );

		// Pre content Widget area.
		register_sidebar( array(
			'name' => __( 'Before Content Widget Area', 'blogfolio' ),
			'id' => 'before-content-widget-container',
			'description' => __( 'For placing widgets before the content', 'blogfolio' ),
			'before_widget' => '<div id="%1$s" class="before-content-widget widget-box %2$s">',
			'after_widget' => '</div>',
			'before_title' => '<h3 class="widget-title title"><span class="text">',
			'after_title' => '</span></h3>'
		) );

		// Post content Widget area.
		register_sidebar( array(
			'name' => __( 'After Content Widget Area', 'blogfolio' ),
			'id' => 'after-content-widget-container',
			'description' => __( 'For placing widgets after the content', 'blogfolio' ),
			'before_widget' => '<div id="%1$s" class="after-content-widget widget-box %2$s">',
			'after_widget' => '</div>',
			'before_title' => '<h3 class="widget-title title"><span class="text">',
			'after_title' => '</span></h3>'
		) );
	}

	// To be hooked into wp_enqueue_scripts
	static function wp_enqueue_scripts() {
		wp_enqueue_style( 'css.normalize', get_template_directory_uri() . '/resources/css/normalize.css' );
		wp_enqueue_style( 'html.styles', get_template_directory_uri() . '/resources/css/html.styles.css' );
		wp_enqueue_style( 'wordpress.styles', get_template_directory_uri() . '/resources/css/wordpress.styles.css' );
		wp_enqueue_style( 'wordpress.comments', get_template_directory_uri() . '/resources/css/wordpress.comments.css' );
		wp_enqueue_style( 'Blogfolio.theme', get_stylesheet_directory_uri() . '/style.css' );
		wp_add_inline_style( 'Blogfolio.theme', self::_themeSettingStyles() );

		wp_enqueue_script( 'html5shiv', get_template_directory_uri() . '/resources/js/html5shiv/html5shiv.js' );
	}

	// add category nicenames in body and post class
	static function body_class( $classes ) {
		if( ! is_active_sidebar( 'content-sidebar' ) || is_page_template( 'fullwidthpage.php' ) )
			$classes[] = 'full-width-content';

		return $classes;
	}

	// Creates css style rules from theme styling options.
	private static function _themeSettingStyles() {
		$styles = array();

		if( $sitetitlecolour = esc_html( self::config( 'site-title-colour' ) ) )
			$styles[] = ".site-title .text { color: $sitetitlecolour; }";

		if( $sitesubtitlecolour = esc_html( self::config( 'site-subtitle-colour' ) ) )
			$styles[] = ".site-subtitle { color: $sitesubtitlecolour; }";

		return implode("\n", $styles);
	}

	private static function _updateAsNeeded() {
		$_config = self::config();
		$_version = isset( $_config[ 'version' ] ) ? $_config[ 'version' ] : 0; 

		if( $_version == self::version )
			return;

		$_config = array( 'version' => self::version ) + $_config;

		if( $_version <= 1.62 )
			$_config = array( 'feature-newest-post' => 1 ) + $_config;

		update_option( self::options, $_config );
	}

}

/**
 * Defines template fragment loader, an enhanced version of get_template_part().
 * Implements behaviour for displaying major sections of site, heavily leveraging template fragments.
 * Also Provides some utility functionality for showing core components of the site.
 *
 */
class BlogfolioTemplate {
	const fragments = 'fragments/';

	// Utility method for rendering a navigation menu.
	static function showNavigation( $themeposition ) {
		self::loadFragment( 'navigation', $themeposition, compact( 'themeposition' ) );
	}

	// Utility method for rendering a widget area.
	static function showWidgetContainer( $widgetcontainer ) {
		self::loadFragment( 'sidebar', $widgetcontainer, compact( 'widgetcontainer' ) );
	}

	// Renders a theme preview, customized based on the post format.
	static function showPostPreview( $context = false ) {
		$format = get_post_format() or $format = 'standard';

		switch( $format ) {
			case 'audio' :
				echo '<div class="audio-post post-format"></div>';
				break;
			case 'video' :
				echo '<div class="video-post post-format"></div>';
			case 'image' :
				echo '<div class="image-post post-format"></div>';
				break;
			case 'gallery' :
				echo '<div class="gallery-post post-format"></div>';
				break;
			case 'link' :
				echo '<div class="link-post post-format"></div>';
				break;
			case 'chat' :
				echo '<div class="chat-post post-format"></div>';
				break;
			case 'quote' :
				echo '<div class="quote-post post-format"></div>';
				break;
			case 'status' :
				echo '<div class="status-post post-format"></div>';
				break;
			case 'aside' :
				echo '<div class="aside-post post-format"></div>';
				break;
			case 'standard' :
			default :
				echo '<div class="excerpt standard-post post-format">';
				the_excerpt();
				echo '</div>';
		}
	}

	// Utility method for rendering a pagination navigation menu.
	static function paginate( $location ) {
		self::loadFragment( 'pagination', $location);
	}

	// For inserting a new render type 'next_and_number' in wp_link_pages(). Makes output consistent with paginate_links()
	static function wpLinkPagesArgs( $args ) {
		global $page, $numpages, $more, $pagenow;

		if ( ! $more || $args[ 'next_or_number' ] != 'next_and_number' ) 
			return $args;

		$args[ 'next_or_number' ] = 'number';

		if( $page - 1 )
			$args[ 'before' ] = $args[ 'before' ]
				. _wp_link_page( $page - 1 )
				. $args[ 'link_before' ]
				. $args[ 'previouspagelink' ]
				. $args[ 'link_after' ]
				. '</a>';

		if( $page < $numpages )
			$args[ 'after' ] = ' ' . _wp_link_page( $page + 1 )
				. $args[ 'link_before' ]
				. $args[ 'nextpagelink' ]
				. $args['link_after']
				. '</a>'
				.$args[ 'after' ];

		return $args;
	}

	/*
		Template Fragment loader, based loosely on get_template_part(). Looks in the child theme first and then the parent theme.
		The reasoning behind this over get_template_part() is to be able to pass variables to the fragments.
	*/
	static function loadFragment( $slug, $name = null, array $args = array() ) {
		$fragments = array();

		if( isset( $name ) )
			$fragments[] = self::fragments . "{$slug}-{$name}.php";

		$fragments[] = self::fragments . "{$slug}.php";

		if( $template = locate_template( $fragments ) )
			self::_loadFragment( $template, $args );

		return $template;
	}

	// Does the actual loading once a template is found.
	private static function _loadFragment( $template, array $args = array() ) {
		extract($args);
		include($template);
	}

	/*
		Following are implementations that have been hooked into the Blogfolio template hooks.
	*/

	function topbar() {
		if( Blogfolio::config( 'search-in-menubar' ) ) get_search_form();
		self::showNavigation( 'primary' );
	}

	function siteHeader() {
		self::showWidgetContainer( 'siteheader' );
	}

	function beforeContent() {
		self::showWidgetContainer( 'beforecontent' );
	}

	function postFeature() {
		if( is_archive() || is_search() )
			self::loadFragment( 'thumbnail', 'loop' );
		else
			self::loadFragment( 'thumbnail' );
	}

	function loopHeader() {
		self::loadFragment( 'header', 'loop' );
		self::loadFragment( 'postmeta', 'authorship' );
	}

	function pageHeader() {
		self::loadFragment( 'header', 'page' );
	}

	function pageFooter() {}

	function singleHeader() {
		self::loadFragment( 'header', 'single' );
		self::loadFragment( 'postmeta', 'authorship' );
		if( has_excerpt() ) {
			echo '<div class="excerpt">';
			the_excerpt();
			echo '</div>';
		}
	}

	function singleFooter() {
		self::loadFragment( 'postmeta', 'classification' );
	}

	function attachmentHeader() {
		self::loadFragment( 'header', 'attachment' );
		self::loadFragment( 'postmeta', 'authorship' );
		if( has_excerpt() ) {
			echo '<div class="excerpt">';
			the_excerpt();
			echo '</div>';
		}
	}

	function attachmentFooter() {
		global $post;
		$link = get_permalink( $post->post_parent );
		$linktext = sprintf( __( 'Return to %s', 'blogfolio' ), get_the_title( $post->post_parent ) );

		echo '<div class="post-meta"><a class="parent-link" href="' . $link . '">' . $linktext . '</a></div>';
	}

	function afterContent() {
		self::showWidgetContainer( null );
		self::showWidgetContainer( 'aftercontent' );
		self::loadFragment( 'comments' );
	}

	function siteFooter() {
		self::loadFragment( 'sitefooter' );
	}
}

?>