<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package Blog Postx Pro
 */
use BlogPostx\CustomizerDefault as BPX;
/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function blog_postx_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	// Adds a class of no-sidebar when there is no sidebar present.
	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$classes[] = 'no-sidebar';
	}
	
	$archive_post_layout = BPX\blog_postx_get_customizer_option( 'archive_post_layout' );
	if( is_archive() || is_home() || is_page_template() ) {
		$archive_sidebar_layout_meta = 'customizer-setting';
		$archive_layout_meta = 'customizer-layout';
		$current_id = get_queried_object_id();
		if( is_category() ) {
			$archive_sidebar_layout_meta = metadata_exists( 'term', $current_id, '_blog_postx_category_sidebar_custom_meta_field' ) ? get_term_meta( $current_id, '_blog_postx_category_sidebar_custom_meta_field', true ) : 'customizer-setting';
			$archive_layout_meta = metadata_exists( 'term', $current_id, '_blog_postx_category_archive_custom_meta_field' ) ? get_term_meta( $current_id, '_blog_postx_category_archive_custom_meta_field', true ) : 'customizer-layout';
		} else if( is_tag() ) {
			$archive_sidebar_layout_meta = metadata_exists( 'term', $current_id, '_blog_postx_post_tag_sidebar_custom_meta_field' ) ? get_term_meta( $current_id, '_blog_postx_post_tag_sidebar_custom_meta_field', true ) : 'customizer-setting';
			$archive_layout_meta = metadata_exists( 'term', $current_id, '_blog_postx_post_tag_archive_custom_meta_field' ) ? get_term_meta( $current_id, '_blog_postx_post_tag_archive_custom_meta_field', true ) : 'customizer-layout';
		}
		$archive_sidebar_layout = BPX\blog_postx_get_customizer_option( 'archive_sidebar_layout' );
		$archive_post_column = BPX\blog_postx_get_customizer_option( 'archive_post_column' );
		$classes[] = 'archive-desktop-column--' . esc_attr( blog_postx_convert_number_to_numeric_string( $archive_post_column['desktop'] ) );
		$classes[] = 'archive-tablet-column--' . esc_attr( blog_postx_convert_number_to_numeric_string( $archive_post_column['tablet'] ) );
		$classes[] = 'archive-mobile-column--' . esc_attr( blog_postx_convert_number_to_numeric_string( $archive_post_column['smartphone'] ) );
		$classes[] = 'archive--' . esc_attr( ( $archive_layout_meta == 'customizer-layout' ) ? $archive_post_layout : $archive_layout_meta )  . '-layout';
		$classes[] = 'archive--' . esc_attr( ( $archive_sidebar_layout_meta == 'customizer-setting' ) ? $archive_sidebar_layout : $archive_sidebar_layout_meta );
	}

	if( is_single() ) {
		$single_post_layout = BPX\blog_postx_get_customizer_option( 'single_post_layout' );
		$single_sidebar_layout = BPX\blog_postx_get_customizer_option( 'single_sidebar_layout' );
		$single_sidebar_post_meta = metadata_exists( 'post', get_the_ID(), 'post_sidebar_layout' ) ? get_post_meta( get_the_ID(), 'post_sidebar_layout', true ) : 'customizer-setting';
		$single_layout_post_meta = metadata_exists( 'post', get_the_ID(), 'single_layout' ) ? get_post_meta( get_the_ID(), 'single_layout', true ) : 'customizer-layout';
		$classes[] = 'single-post--' . esc_attr( ( $single_layout_post_meta == 'customizer-layout' ) ? $single_post_layout : $single_layout_post_meta );
		$classes[] = 'single--' . esc_attr( ( $single_sidebar_post_meta == 'customizer-setting' ) ? $single_sidebar_layout : $single_sidebar_post_meta );
	}

	if( is_search() ) {
		$archive_post_column = BPX\blog_postx_get_customizer_option( 'archive_post_column' );
		$classes[] = 'archive-desktop-column--' . esc_attr( blog_postx_convert_number_to_numeric_string( $archive_post_column['desktop'] ) );
		$classes[] = 'archive-tablet-column--' . esc_attr( blog_postx_convert_number_to_numeric_string( $archive_post_column['tablet'] ) );
		$classes[] = 'archive-mobile-column--' . esc_attr( blog_postx_convert_number_to_numeric_string( $archive_post_column['smartphone'] ) );
		$classes[] = 'search-page--right-sidebar';
		$classes[] = 'archive--' . esc_attr( $archive_post_layout ) . '-layout';
	}

	if( is_page() ) {
		$page_settings_sidebar_layout = BPX\blog_postx_get_customizer_option( 'page_settings_sidebar_layout' );
		$page_sidebar_post_meta = metadata_exists( 'post', get_the_ID(), 'page_sidebar_layout' ) ? get_post_meta( get_the_ID(), 'page_sidebar_layout', true ) : 'customizer-setting';
		$classes[] = 'page--' . esc_attr( ( $page_sidebar_post_meta == 'customizer-setting' ) ? $page_settings_sidebar_layout : $page_sidebar_post_meta);
	}

	$classes[] = 'blog-postx-light-mode';

	$website_layout = BPX\blog_postx_get_customizer_option ('website_layout');
	if( $website_layout ) $classes[] = $website_layout;

	$classes[] = 'block-title--five';
	
	$title_hover = BPX\blog_postx_get_customizer_option( 'post_title_hover_effects' );
	$classes[] = 'title-hover--' . esc_attr( $title_hover );

	$image_hover = BPX\blog_postx_get_customizer_option( 'site_image_hover_effects' );
	$classes[] = 'image-hover--' . esc_attr( $image_hover );

	$canvas_menu_position = BPX\blog_postx_get_customizer_option( 'canvas_menu_position' );
	$classes[] = 'blog-postx-canvas-position--' . esc_attr( $canvas_menu_position );

	$classes[] = ' blog-postx-variables';
	$classes[] = ' is-desktop';
	
	$site_background_animation = BPX\blog_postx_get_customizer_option( 'site_background_animation' );
	$classes[] = 'background-animation--' . $site_background_animation;
	if( $site_background_animation !== 'none' ) $classes[] = 'background-animation--enabled';

	return $classes;
}
add_filter( 'body_class', 'blog_postx_body_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function blog_postx_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'blog_postx_pingback_header' );

if( ! function_exists( 'blog_postx_get_categories_html' ) ) :
	/**
	 * Return categories in <ul> <li> form
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_get_categories_html() {
		$blog_postx_categoies = get_categories( [ 'object_ids' => get_the_ID() ] );
		$post_cagtegories_html = '<ul class="post-categories">';
		foreach( $blog_postx_categoies as $category_key => $category_value ) :
			$post_cagtegories_html .= '<li class="cat-item item-'. ( $category_key + 1 ) .'">'. esc_html( $category_value->name ) .'</li>';
		endforeach;
		$post_cagtegories_html .= '</ul>';
		return $post_cagtegories_html;
	}
endif;

if( ! function_exists( 'blog_postx_post_order_args' ) ) :
	/**
	 * Return post order args
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_post_order_args() {
		return [
			'date-desc' =>  esc_html__( 'Newest - Oldest', 'blog-postx' ),
			'date-asc' =>  esc_html__( 'Oldest - Newest', 'blog-postx' ),
			'rand-desc' =>  esc_html__( 'Random', 'blog-postx' )
		];
	}
endif;

if( ! function_exists( 'blog_postx_get_image_sizes_option_array' ) ) :
	/**
	 * Get list of image sizes
	 * 
	 * @since 1.0.0
	 * @package Blog Postx Pro
	 */
	function blog_postx_get_image_sizes_option_array() {
		$image_sizes = get_intermediate_image_sizes();
		foreach( $image_sizes as $image_size ) :
			$sizes[$image_size] = $image_size;
		endforeach;
		return $sizes;
	}
endif;

add_filter( 'get_the_archive_title_prefix', 'blog_postx_prefix_string' );
function blog_postx_prefix_string($prefix) {
	return apply_filters( 'blog_postx_archive_page_title_prefix', false );
}

if( ! function_exists( 'blog_postx_widget_control_get_tags_options' ) ) :
	/**
	 * @since 1.0.0
	 * @package Blog Postx Pro
	 */
	function blog_postx_widget_control_get_tags_options() {
        check_ajax_referer( 'blog_postx_widget_nonce', 'security' );
        $searchKey = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ): '';
        $to_exclude = isset( $_POST['exclude'] ) ? sanitize_text_field( wp_unslash( $_POST['exclude'] ) ): '';
        $type = isset( $_POST['type'] ) ? sanitize_text_field( $_POST['type'] ): '';
		if( $type == 'category' ) :
			$posts_list = get_categories( [ 'number' => 4, 'search' => esc_html( $searchKey ), 'exclude' => explode( ',', $to_exclude ) ] );
		elseif( $type == 'tag' ) :
			$posts_list = get_tags( [ 'number' => 4, 'search' => esc_html( $searchKey ), 'exclude' => explode( ',', $to_exclude ) ] );
		elseif( $type == 'user' ):
			$posts_list = new \WP_User_Query([ 'number' => 4, 'search' => esc_html( $searchKey ), 'exclude' => explode( ',', $to_exclude ) ]);
			if( ! empty( $posts_list->get_results() ) ):
				foreach( $posts_list->get_results() as $user ) :
					$user_array[] = [
						'id'	=>	$user->ID,
						'text'	=>	$user->display_name
					];
				endforeach;
				wp_send_json_success( $user_array );
			else:
				wp_send_json_success( '' );
			endif;
		else:
			$post_args = [
				'post_type' =>  'post',
				'post_status'=>  'publish',
				'posts_per_page'    =>  6,
				'post__not_in' => explode( ',', $to_exclude ),
				's' => esc_html( $searchKey )
			];
			$posts_query = new \WP_Query( apply_filters( 'blog_postx_query_args_filter', $post_args ) );
			if( $posts_query->have_posts() ) :
				while( $posts_query->have_posts() ) :
					$posts_query->the_post();
					$post_array[] =	[
						'id'	=>	get_the_ID(),
						'text'	=>	get_the_title()
					];
				endwhile;
				wp_send_json_success( $post_array );
			endif;
		endif;
		if( ! empty( $posts_list ) ) :
			foreach( $posts_list as $postItem ) :
				$posts_array[] = [	
					'id'	=> esc_html( $postItem->term_taxonomy_id ),
					'text'	=> esc_html( $postItem->name .'('. $postItem->count .')' )
				];
			endforeach;
			wp_send_json_success( $posts_array );
		endif;
        wp_die();
    }
	add_action( 'wp_ajax_blog_postx_widget_control_get_tags_options', 'blog_postx_widget_control_get_tags_options' );
	
endif;

require get_template_directory() . '/inc/extras/helpers.php';
require get_template_directory() . '/inc/hooks/frontpage-sections-hook.php';
require get_template_directory() . '/inc/extras/extras.php';
require get_template_directory() . '/inc/extras/extend-api.php';
require get_template_directory() . '/inc/widgets/widgets.php'; // widget handlers
require get_template_directory() . '/inc/hooks/hooks.php'; // hooks handlers
require get_template_directory() . '/inc/metabox/metabox.php'; // metabox

/**
 * GEt appropriate color value
 * 
 * @since 1.0.0
 */
if(! function_exists('blog_postx_get_color_format')):
    function blog_postx_get_color_format($color) {
		if( ! is_string( $color ) ) return;
      if( str_contains( $color, '--blog-postx-global-preset' ) ) {
        return( 'var( ' .esc_html( $color ). ' )' );
      } else {
        return $color;
      }
    }
endif;

/**
 * Minify dynamic css
 * 
 * @since 1.0.0
 */
if( ! function_exists( 'blog_postx_minifyCSS' ) ) :
	function blog_postx_minifyCSS( $css ) {
		// Remove comments
		$css = preg_replace( '!/\*.*?\*/!s', '', $css );
		// Remove space after colons
		$css = preg_replace( '/\s*:\s*/', ':', $css );
		// Remove whitespace
		$css = preg_replace( '/\s+/', ' ', $css );
		// Remove space before/after brackets and semicolons
		$css = preg_replace( '/\s*{\s*/', '{', $css );
		$css = preg_replace( '/\s*}\s*/', '}', $css );
		$css = preg_replace( '/\s*;\s*/', ';', $css );
		// Remove final semicolon in a block
		$css = preg_replace( '/;}/', '}', $css );
		// Trim the final output
		return trim( $css );
	}
endif;

/**
 * Check if a specific widget is being used
 * 
 * @since 1.0.0
 */
if( ! function_exists( 'blog_postx_is_widget_being_used' ) ) :
	function blog_postx_is_widget_being_used( $widget_id ) {
		$sidebar_widgets = wp_get_sidebars_widgets();
		if( ! empty( $sidebar_widgets ) && is_array( $sidebar_widgets ) ) :
			foreach( $sidebar_widgets as $sidebar => $widgets_array ) :
				if( ! empty( $widgets_array ) ) :
					foreach( $widgets_array as $widget ) :
						if (strpos( $widget, $widget_id ) === 0) return true;
					endforeach;
				endif;
			endforeach;
		endif;
		return false;
	}
endif;

if( ! function_exists( 'blog_postx_current_styles' ) ) :
	/**
	 * Generates the current changes in styling of the theme.
	 * 
	 * @package Blog Postx Pro
	 * @since 1.0.0
	 */
	function blog_postx_current_styles() {
		/**
		 * Abbreviation
		 * 
		 * hr = header responsive
		 * stt = scroll to top
		 */
		$main_banner_option = BPX\blog_postx_get_customizer_option( 'main_banner_option' );
		$category_collection_option = BPX\blog_postx_get_customizer_option( 'category_collection_option' );
		$carousel_option = BPX\blog_postx_get_customizer_option( 'carousel_option' );
		$preloader_option = BPX\blog_postx_get_customizer_option( 'preloader_option' );
		$website_layout = BPX\blog_postx_get_customizer_option( 'website_layout' );
		$archive_content_reorder = blog_postx_get_reorder_value( BPX\blog_postx_get_customizer_option( 'archive_content_reorder' ) );
		$site_breadcrumb_option = BPX\blog_postx_get_customizer_option( 'site_breadcrumb_option' );
		$ticker_news_option = BPX\blog_postx_get_customizer_option( 'ticker_news_option' );
		$archive_pagination_type = BPX\blog_postx_get_customizer_option( 'archive_pagination_type' );
		$custom_logo_option = get_theme_mod( 'custom_logo' );
		$site_background_animation = BPX\blog_postx_get_customizer_option( 'site_background_animation' );
		$web_stories_option = BPX\blog_postx_get_customizer_option( 'web_stories_option' );
		$you_may_have_missed_option = BPX\blog_postx_get_customizer_option( 'you_may_have_missed_section_option' );
		// Header Builder
		$header_instagram_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'instagram' );
		$date_time_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'date-time' );
		$custom_button_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'button' );
		$theme_mode_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'theme-mode' );
		$search_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'search' );
		$off_canvas_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'off-canvas' );
		$site_logo_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'site-logo' );
		$menu_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'menu' );
		$secondary_menu_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'header_builder', 'secondary-menu' );
		// Footer Builder
		$footer_instagram_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'footer_builder', 'instagram' );
		$footer_secondary_menu_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'footer_builder', 'menu' );
		$copyright_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'footer_builder', 'copyright' );
		$stt_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'footer_builder', 'scroll-to-top' );
		// Responsive Header Builder
		$hr_date_time_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'date-time' );
		$hr_header_instagram_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'instagram' );
		$hr_custom_button_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'button' );
		$hr_theme_mode_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'theme-mode' );
		$hr_search_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'search' );
		$hr_off_canvas_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'off-canvas' );
		$hr_site_logo_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'site-logo' );
		$hr_menu_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'menu' );
		$hr_secondary_menu_option = \Blog_Postx_Builder\Builder_Base::widget_exists( 'responsive_header_builder', 'secondary-menu' );

		ob_start();
			blog_postx_preset_color_control( 'solid_color_preset', '--blog-postx-global-preset-color-' );
			blog_postx_preset_color_control( 'gradient_color_preset', '--blog-postx-global-preset-gradient-' );
			blog_postx_typography_preset();

			/** Value Change With Responsive **/
			blog_postx_value_change_responsive('body footer .footer-logo img', 'bottom_footer_logo_width','width');
			blog_postx_spacing_control( 'body header.site-header .row-one', 'header_first_row_padding', 'padding' );
			blog_postx_spacing_control( 'body header.site-header .row-two', 'header_second_row_padding', 'padding' );
			blog_postx_spacing_control( 'body header.site-header .row-three', 'header_third_row_padding', 'padding' );
			blog_postx_spacing_control( 'body footer.site-footer .bb-bldr-row.row-one', 'footer_first_row_padding', 'padding' );
			blog_postx_spacing_control( 'body footer.site-footer .bb-bldr-row.row-two', 'footer_second_row_padding', 'padding' );
			blog_postx_spacing_control( 'body footer.site-footer .bb-bldr-row.row-three', 'footer_third_row_padding', 'padding' );
			
			/** Value Change **/
			if( is_page() || is_404() ) blog_postx_value_change('.page #blog-postx-main-wrap #primary article, .page .blog-postx-table-of-content.display--inline .toc-wrapper, .error404 .error-404','page_border_radius','border-radius');
			blog_postx_value_change('.widget .post-thumb-image, .widget .post-thumb, .widget_blog_postx_carousel_widget .post-thumb-wrap, .widget.widget_media_image, .widget_blog_postx_category_collection_widget .categories-wrap .category-item .category-name, body .widget_blog_postx_post_list_widget .post-list-wrap .post-thumb-image .post-thumb, body .widget_blog_postx_posts_grid_two_column_widget .posts-wrap .post-thumb, body .widget_blog_postx_author_info_widget .bmm-author-thumb-wrap .post-thumb img','sidebar_image_border_radius','border-radius');
			blog_postx_value_change('body .site .widget, body #widget_block, body .widget.widget_media_image figure.wp-block-image img, body .widget-area .blog-postx-table-of-content .toc-wrapper, body.error404 #blog-postx-main-wrap #primary .not-found','sidebar_border_radius','border-radius');

			/** Color Group (no Gradient) (Variable) **/
			$bcColorAssign = function($var,$id) {
				blog_postx_assign_var($var,$id);
			};
			$bcColorAssign( '--blog-postx-global-preset-theme-color', 'theme_color' );
			$bcColorAssign( '--blog-postx-global-preset-gradient-theme-color', 'gradient_theme_color' );
			/** Text Color (Variable) **/
			blog_postx_variable_color('--blog-postx-mobile-canvas-icon-color', 'mobile_canvas_icon_color');

			// Category Bk Color
			blog_postx_category_bk_colors_styles();
			if( blog_postx_is_widget_being_used( 'blog_postx_tags_collection_widget' ) ) blog_postx_tags_bk_colors_styles();

			/* Typography (Variable) */
			$bTypoCode = function( $selector, $control, $is_variable = false ) {
				blog_postx_typography_control( $selector, $control, $is_variable );
			};
			$bTypoCode("--blog-postx-widget-title-font","sidebar_post_title_typography", true );
			$bTypoCode('body .blog-postx-widget-loader .load-more','sidebar_pagination_button_typo');
			$bTypoCode('body footer .widget_block .wp-block-group__inner-container .wp-block-heading, body footer section.widget .widget-title, body footer .wp-block-heading', 'footer_title_typography');
			$bTypoCode('body footer ul.wp-block-latest-posts a, body footer ol.wp-block-latest-comments li footer, body footer ul.wp-block-archives a, body footer ul.wp-block-categories a, body footer ul.wp-block-page-list a, body footer .widget_blog_postx_post_grid_widget .post-grid-wrap .post-title, body footer .menu .menu-item a, body footer .widget_blog_postx_category_collection_widget .categories-wrap .category-item .category-name, body footer .widget_blog_postx_post_list_widget .post-list-wrap .post-title a', 'footer_text_typography');
			$bTypoCode('body aside h1.wp-block-heading','sidebar_heading_one_typography');
			$bTypoCode('body aside h2.wp-block-heading','sidebar_heading_two_typo');
			$bTypoCode('body aside h3.wp-block-heading','sidebar_heading_three_typo');
			$bTypoCode('body aside h4.wp-block-heading','sidebar_heading_four_typo');
			$bTypoCode('body aside h5.wp-block-heading','sidebar_heading_five_typo');
			$bTypoCode('body aside h6.wp-block-heading','sidebar_heading_six_typo');
			$bTypoCode('body .blog-postx-sticky-posts .label-wrapper .label','sticky_posts_label_typography');
			$bTypoCode('body .blog-postx-sticky-posts .post-list article .post-content .post-title','sticky_posts_title_typography');
			$bTypoCode('body .blog-postx-web-stories .stories-wrap .preview .story-count','web_stories_preview_count_typo');
			$bTypoCode('body .blog-postx-web-stories .stories-wrap .story-title','web_stories_preview_title_typo');
			$bTypoCode('body .blog-postx-web-stories .inner-stories-wrap .content-wrap .title','web_stories_title_typo');
			$bTypoCode("--blog-postx-block-font", "site_block_title_typo", true );

			$bTypoCode("--blog-postx-category-font", "global_category_typography", true );
			$bTypoCode("--blog-postx-date-font", "global_date_typography", true );
			$bTypoCode("--blog-postx-author-font", "global_author_typography", true );
			$bTypoCode("--blog-postx-readtime-font", "global_read_time_typography", true );
			$bTypoCode("--blog-postx-comment-font", "global_comments_typography", true );

			/* background color */
			blog_postx_background_control('body.blog-postx-light-mode .site-header','header_builder_background');

			/* Main banner background color group */
			blog_postx_background_control('body:before, body.single-post #primary article .post-card .bmm-author-thumb-wrap .post-thumb','site_background_color');

			if( is_single() || is_page() ) :
				$bTypoCode('body article .post-inner h1','heading_one_typo');
				$bTypoCode('body article .post-inner h2','heading_two_typo');
				$bTypoCode('body article .post-inner h3','heading_three_typo');
				$bTypoCode('body article .post-inner h4','heading_four_typo');
				$bTypoCode('body article .post-inner h5','heading_five_typo');
				$bTypoCode('body article .post-inner h6','heading_six_typo');
			endif;

			// Main banner
			if( $main_banner_option && ( is_front_page() || is_home() ) ) :
				blog_postx_value_change('body .blog-postx-main-banner-section .main-banner-wrap .slick-list .post-thumb, body .blog-postx-main-banner-section .main-banner-wrap .slick-list .post-elements, body .blog-postx-main-banner-section .main-banner-swiper-thumbs .slick-slide','main_banner_image_border_radius','border-radius');
				$bTypoCode("--blog-postx-banner-title-font", "main_banner_design_post_title_typography", true );
				$bTypoCode("--blog-postx-banner-excerpt-font", "main_banner_design_post_excerpt_typography", true );
				$bTypoCode("--blog-postx-banner-sidebar-title-font", "main_banner_sidebar_post_typography", true );
				blog_postx_value_change_responsive('body .blog-postx-main-banner-section .main-banner-sidebar .scrollable-post, body .blog-postx-main-banner-section .main-banner-sidebar .scrollable-post .count-image-wrapper .post-thumb','main_banner_border_radius','border-radius');
			endif;

			// Category Collection
			if( $category_collection_option && ( is_front_page() || is_home() ) ) :
				$bTypoCode("--blog-postx-category-collection-font","category_collection_typo", true );
				blog_postx_value_change_responsive('.blog-postx-category-collection-section .category-wrap .category-thumb a','category_collection_image_radius','border-radius');
			endif;

			// Carousel
			if( $carousel_option && ( is_front_page() || is_home() ) ) :
				$bTypoCode('body .blog-postx-carousel-section .carousel-wrap .post-elements .post-title', 'carousel_design_post_title_typography');
				blog_postx_spacing_control( 'body .blog-postx-carousel-section article.post-item .post-thumb', 'carousel_image_border_radius', 'border-radius' );
				blog_postx_value_change('body .blog-postx-carousel-section.carousel-layout--three article.post-item','carousel_section_border_radius','border-radius');
			endif;

			// Single
			if( is_single() ) :
				$bTypoCode('body.single-post.blog-postx-variables .site-main article .entry-content p','single_content_typo');
				$bTypoCode('body.single-post.blog-postx-variables .site-main article .entry-title, body.single-post.blog-postx-variables .single-header-content-wrap .entry-title, body.single-post #primary .post-navigation .nav-links .nav-title, .single-related-posts-section-wrap.layout--list .single-related-posts-wrap article .post-element .post-title','single_title_typo');
				blog_postx_value_change('body.single-post #blog-postx-main-wrap .blog-postx-container .row #primary .blog-postx-inner-content-wrap article > div, body.single-post #blog-postx-main-wrap .blog-postx-container .row #primary nav.navigation, body.single-post #blog-postx-main-wrap .blog-postx-container .row #primary .single-related-posts-section-wrap.layout--list, body.single-post #primary article .post-card .bmm-author-thumb-wrap, .single-related-posts-section-wrap article .post-thumbnail, body.single-post .comment-respond .comment-form-comment textarea, body.single-post form.comment-form p input, body.single-post #primary .post-navigation .nav-links figure.nav-thumb, .single .blog-postx-table-of-content.display--inline .toc-wrapper, body.single .wp-block-embed-soundcloud iframe, body.single .wp-block-embed-youtube iframe, .single .blog-postx-advertisement img','single_page_border_radius','border-radius');
				blog_postx_value_change('body.single .site .blog-postx-main-wrap .entry-header .post-thumbnail, body.single .site .post-inner .post-thumbnail','single_image_border_radius','border-radius');
			endif;

			// Page
			if( is_page() && ! is_page_template()) :
				$bTypoCode('body.page.blog-postx-variables #blog-postx-main-wrap #primary article .entry-title','page_title_typo');
				$bTypoCode('body.page.blog-postx-variables article .entry-content','page_content_typo');
				blog_postx_value_change('body.page-template-default.blog-postx-variables #primary article .post-thumbnail, body.page-template-default.blog-postx-variables #primary article .post-thumbnail img','page_image_border_radius','border-radius');
			endif;

			// Category Archive
			if( is_category() || is_archive() ) :
				$bTypoCode('body.blog-postx-variables.archive.category .page-header .page-title, .archive.date .page-header .page-title, .tax-post_format #blog-postx-main-wrap .page-header .page-title','archive_category_info_box_title_typo');
				$bTypoCode('body.blog-postx-variables.archive.category .page-header .archive-description','archive_category_info_box_description_typo');
			endif;

			// Tags Archive
			if( is_tag() ) :
				$bTypoCode('body.blog-postx-variables.archive.tag .page-header .archive-description','archive_tag_info_box_description_typo');
				$bTypoCode('body.blog-postx-variables.archive.tag .page-header .page-title','archive_tag_info_box_title_typo');
			endif;

			// Authors Archive
			if( is_author() ) :
				$bTypoCode('body.blog-postx-variables.archive.author .page-header .page-title','archive_author_info_box_title_typo');
				$bTypoCode('body.blog-postx-variables.archive.author .page-header .archive-description','archive_author_info_box_description_typo');
			endif;

			// You may have missed
			if( $you_may_have_missed_option ) :
				blog_postx_value_change('body .blog-postx-you-may-have-missed-section .blog-postx-you-may-missed-inner-wrap, body .blog-postx-you-may-have-missed-section .blog-postx-you-may-missed-inner-wrap .post-item .post-thumbnail-wrapper','you_may_have_missed_border_radius', 'border-radius' );
				$bTypoCode("--blog-postx-youmaymissed-title-font", "you_may_have_missed_design_post_title_typography", true );
			endif;

			// Date Time
			if( $date_time_option || $hr_date_time_option ) :
				$bTypoCode("body .top-date-time .top-date-time-inner", 'date_time_typography' );
				blog_postx_background_control('body .top-date-time .top-date-time-inner .separator i', 'date_time_separator_color', 'color' );
				blog_postx_background_control('body header.site-header .top-date-time .top-date-time-inner .time', 'time_color', 'color');
				blog_postx_background_control('body header.site-header .top-date-time .top-date-time-inner .date', 'date_color', 'color');
			endif;

			// Archive continue reading button
			if( $archive_content_reorder[ 'button' ] && ( is_archive() || is_home() || is_search() ) ) :
				$bTypoCode("--blog-postx-readmore-font", "global_button_typo", true );
			endif;

			// Breadcrumb
			if( $site_breadcrumb_option && ( ! is_home() && ! is_front_page() ) ) :
				$bTypoCode('body .blog-postx-breadcrumb-wrap ul li span[itemprop="name"]','breadcrumb_typo');
			endif;

			// Ticker
			if( $ticker_news_option && ( is_front_page() || is_home() ) ) {
				$bTypoCode('body .blog-postx-ticker-news .ticker-news-wrap .ticker-item .title-wrap .post-title','ticker_news_post_title_typo');
				blog_postx_value_change_responsive('.blog-postx-ticker-news .ticker-news-wrap .ticker-item .post-thumb, body .blog-postx-ticker-news .row','ticker_news_border_radius','border-radius');
			}

			// Archive  || Home || Search
			if( is_archive() || is_home() || is_search() ) :
				blog_postx_value_change('body .site #blog-postx-main-wrap > .blog-postx-container > .row #primary .blog-postx-inner-content-wrap article .blog-postx-article-inner, body #blog-postx-main-wrap > .blog-postx-container > .row #primary .blog-postx-inner-content-wrap article .blog-postx-article-inner .post-thumbnail-wrapper, body.search.search-results .site #blog-postx-main-wrap .blog-postx-container .page-header, #primary .blog-postx-inner-content-wrap .blog-postx-advertisement-block img, body.search section.no-results.not-found','archive_section_border_radius','border-radius');
				$bTypoCode("--blog-postx-post-title-font","archive_title_typo", true );
				$bTypoCode("--blog-postx-post-content-font","archive_excerpt_typo", true );
			endif;

			// Custom Button
			if( $custom_button_option || $hr_custom_button_option ) :
				$bTypoCode("--blog-postx-custom-button", 'custom_button_text_typography', true );
				blog_postx_variable_bk_color('--blog-postx-custom-button-bk-color','header_custom_button_background_color_group');
				blog_postx_value_change_responsive('body .site-header .header-custom-button','header_custom_button_border_radius','border-radius');
			endif;

			// Theme mode
			if( $theme_mode_option || $hr_theme_mode_option ) :
				blog_postx_value_change_responsive('body .site-header .mode-toggle i','theme_mode_icon_size','font-size');
				blog_postx_value_change_responsive('body .site-header .mode-toggle img', 'theme_mode_icon_size','width');
				blog_postx_variable_color('--blog-postx-theme-darkmode-color', 'theme_mode_dark_icon_color');
				blog_postx_variable_color('--blog-postx-theme-mode-color', 'theme_mode_light_icon_color');
			endif;

			// Search
			if( $search_option || $hr_search_option ) :
				blog_postx_value_change_responsive('body .site-header .search-trigger i', 'search_icon_size', 'font-size');
				blog_postx_variable_color('--blog-postx-search-icon-color', 'search_icon_color');
			endif;

			// Off Canvas
			if( $off_canvas_option || $hr_off_canvas_option ) :
				blog_postx_variable_color('--blog-postx-canvas-icon-color', 'canvas_menu_icon_color');
			endif;

			// Pagination Ajax Type
			if( $archive_pagination_type && ( is_archive() || is_home() || is_search() ) ) :
				blog_postx_variable_color('--blog-postx-ajax-pagination-color', 'pagination_button_text_color');
				blog_postx_variable_bk_color('--blog-postx-ajax-pagination-bk-color','pagination_button_background_color');
			endif;

			// Site logo Width
			if( $site_logo_option || $hr_site_logo_option ) :
				if( $custom_logo_option ) blog_postx_value_change_responsive('body .bb-bldr-row .site-branding img', 'site_logo_width','width');
				$bTypoCode( "--blog-postx-site-title", 'site_title_typo' , true );
				$bTypoCode( "--blog-postx-site-description", 'site_description_typo' , true );
			endif;

			// Header Menu
			if( $menu_option || $hr_menu_option ) :
				$bTypoCode("--blog-postx-menu", 'main_menu_typo', true );
				$bTypoCode("--blog-postx-submenu", 'main_menu_sub_menu_typo', true );
				blog_postx_variable_color('--blog-postx-menu-color', 'header_menu_color');
				blog_postx_variable_color('--blog-postx-menu-color-submenu', 'header_sub_menu_color');
			endif;

			// Secondary Header Menu
			if( $secondary_menu_option || $hr_secondary_menu_option ) :
				$bTypoCode("--blog-postx-secondary-menu", 'header_secondary_menu_typo', true );
				blog_postx_variable_color_single('--blog-postx-secondary-menu-color', 'header_secondary_menu_color');
			endif;

			// Copyright
			if( $copyright_option ) :
				$bTypoCode('body footer .site-info', 'bottom_footer_text_typography');
			endif;

			// Scroll to top
			if( $stt_option ) :
				blog_postx_variable_color('--blog-postx-scroll-text-color','stt_color_group');
			endif;

			// Footer Secondary Menu
			if( $footer_secondary_menu_option ) :
				$bTypoCode("--blog-postx-footer-menu", 'footer_menu_typography', true );
				blog_postx_variable_color('--blog-postx-footer-menu-color', 'footer_menu_color');
			endif;

			// Web Stories
			if( $web_stories_option ) :
				blog_postx_value_change('body .blog-postx-web-stories .stories-wrap .preview','web_stories_image_radius','border-radius');
			endif;
			
		$current_styles = ob_get_clean();
		return apply_filters( 'blog_postx_current_styles', wp_strip_all_tags( blog_postx_minifyCSS( $current_styles ) ) );
	}
endif;

if( ! function_exists( 'blog_postx_custom_excerpt_more' ) ) :
	/**
	 * Filters the excerpt content
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_custom_excerpt_more($more) {
		if( is_admin() ) return $more;
		return '';
	}
	add_filter('excerpt_more', 'blog_postx_custom_excerpt_more');
endif;

if( ! function_exists( 'blog_postx_check_youtube_api_key' ) ) :
	/**
	 * function to check whether the api key is valid or not
	 * 
	 * @since 1.0.0
	 * @package Blog Postx Pro
	 */
	function blog_postx_check_youtube_api_key( $api_key ) {
		$api_url = "https://www.googleapis.com/youtube/v3/videos?key=" . $api_key . "&part=snippet,contentDetails";
        $remote_get_video_info = wp_remote_get( $api_url );
		return $remote_get_video_info;
	}
endif;

if( ! function_exists( 'blog_postx_random_post_archive_advertisement_part' ) ) :
    /**
     * Blog Postx main banner element
     * 
     * @since 1.0.0
     */
    function blog_postx_random_post_archive_advertisement_part( $ads_rendered ) {
		if( is_null( $ads_rendered ) ) return;
        $advertisement_repeater = BPX\blog_postx_get_customizer_option( 'advertisement_repeater' );
        $advertisement_repeater_decoded = json_decode( $advertisement_repeater );
        $random_post_archive_advertisement = array_values(array_filter( $advertisement_repeater_decoded, function( $element ) {
            if( property_exists( $element, 'item_checkbox_random_post_archives' ) ) return ( $element->item_checkbox_random_post_archives == true && $element->item_option == 'show' ) ? $element : ''; 
        }));
        if( empty( $random_post_archive_advertisement ) ) return;
        $image_option = array_column( $random_post_archive_advertisement, 'item_image_option' );
        $alignment = array_column( $random_post_archive_advertisement, 'item_alignment' );
        $elementClass = 'alignment--' . $alignment[0];
        $elementClass .= ' image-option--' . ( ( $image_option[0] == 'full_width' ) ? 'full-width' : 'original' );
        ?>
            <div class="blog-postx-advertisement-block post <?php echo esc_html( $elementClass ); ?>">
                <a href="<?php echo esc_url( $random_post_archive_advertisement[$ads_rendered]->item_url ); ?>" target="<?php echo esc_attr( $random_post_archive_advertisement[$ads_rendered]->item_target ); ?>" rel="<?php echo esc_attr( $random_post_archive_advertisement[$ads_rendered]->item_rel_attribute ); ?>">
                    <img src="<?php echo esc_url( wp_get_attachment_image_url( $random_post_archive_advertisement[$ads_rendered]->item_image, 'full' ) ); ?>" loading="lazy">
                </a>
            </div>
        <?php
    }
 endif;

 if( ! function_exists( 'blog_postx_random_post_archive_advertisement_number' ) ) :
    /**
     * Blog Postx archive ads number
     * 
     * @since 1.0.0
     */
    function blog_postx_random_post_archive_advertisement_number() {
        $advertisement_repeater = BPX\blog_postx_get_customizer_option( 'advertisement_repeater' );
        $advertisement_repeater_decoded = json_decode( $advertisement_repeater );
        $random_post_archive_advertisement = array_filter( $advertisement_repeater_decoded, function( $element ) {
            if( property_exists( $element, 'item_checkbox_random_post_archives' ) ) return ( $element->item_checkbox_random_post_archives == true && $element->item_option == 'show' ) ? $element : ''; 
        });
        return sizeof( $random_post_archive_advertisement );
    }
 endif;

 if( ! function_exists( 'blog_postx_algorithm_to_push_ads_in_archive' ) ) :
	/**
	 * Algorithm to push ads into archive
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_algorithm_to_push_ads_in_archive( $args = [] ) {
		global $wp_query;
		$archive_ads_number = blog_postx_random_post_archive_advertisement_number();
		if( $archive_ads_number <= 0 ) return;
		if( empty( $args ) ) :
			$max_number_of_pages = absint( $wp_query->max_num_pages );
			$paged = absint( ( get_query_var( 'paged' ) == 0 ) ? 0 : ( get_query_var( 'paged' ) - 1 ) );
		else:
			if( ( $args['paged'] - 1 ) == $archive_ads_number ) return;
			$max_number_of_pages = absint( $args['max_number_of_pages'] );
			$paged = absint( $args['paged'] - 1 );
		endif;
		$count = 1;
		$ads_id = 0;
		$loop_var = 0;
		for( $i = $archive_ads_number ; $i > 0; $i-- ) :
			if( $count <= $max_number_of_pages ):
				$ads_to_render_in_a_single_page = ceil( $i / $max_number_of_pages );
				$ads_to_render = [];
				if( $ads_to_render_in_a_single_page > 1 ) :
					$to_loop = $ads_id + $ads_to_render_in_a_single_page;
					for( $j = $ads_id; $j < $to_loop; $j++ ) :
						if( ! in_array( $ads_id, $ads_to_render ) ) $ads_to_render[] = $ads_id;
						$ads_id++;
					endfor;
					$ads_to_render_in_current_page[$loop_var] = $ads_to_render;
				else:
					$ads_to_render_in_current_page[$loop_var] = $ads_id;
					$ads_id++;
				endif;
				$count++;
				$loop_var++;
			endif;
		endfor;
		$current_page_count = empty( $args ) ? absint( $wp_query->post_count ) : absint( $args['post_count'] );
		$ads_of_current_page = array_key_exists( $paged, $ads_to_render_in_current_page ) ? $ads_to_render_in_current_page[$paged] : null;
		$ads_count = is_array( $ads_of_current_page ) ? sizeof( $ads_of_current_page ) : 1;
		$random_numbers = [];
		for( $i = 0; $i < $ads_count; $i++ ) :
			if( ! in_array( $i, $random_numbers ) ) :
				$random_numbers[] = rand( 0, ( $current_page_count - 1 ) );
			else:
				$random_numbers[] = rand( 0, ( $current_page_count - 1 ) );
			endif;
		endfor;
		return [
			'random_numbers'	=>	$random_numbers,
			'ads_to_render'	=>	$ads_of_current_page
		];
	}
 endif;

 if( ! function_exists( 'blog_postx_get_all_menus' ) ) :
	/**
	 * Get all menus
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_get_all_menus() {
		$menus_array = get_terms( 'nav_menu', array( 'hide_empty' => true ) );
		$value = [
			'none'	=>	esc_html__( 'None', 'blog-postx' ),
		];
		if( ! empty( $menus_array ) && is_array( $menus_array ) ) :
			foreach( $menus_array as $menu ) :
				$value[ $menu->slug ] = $menu->name;
			endforeach;
			return $value;
		endif;
	}
 endif;

 if( ! function_exists( 'blog_postx_get_archive_layout_meta' ) ) :
	/**
	 * Get archive layout
	 * 
	 * @since 1.0.0
	 */
 	function blog_postx_get_archive_layout_meta() {
		$current_id = get_queried_object_id();
		if( is_category() ) :
			$archive_meta_key = '_blog_postx_category_archive_custom_meta_field';
		elseif ( is_tag() ) :
			$archive_meta_key = '_blog_postx_post_tag_archive_custom_meta_field';
		else: 
			return 'customizer-layout';
		endif;
		$archive_layout_meta = metadata_exists( 'term', $current_id, $archive_meta_key ) ? get_term_meta( $current_id, $archive_meta_key, true ) : 'customizer-layout';

		return $archive_layout_meta;
	}
 endif;

 if( ! function_exists( 'blog_postx_get_sidebar_meta' ) ) :
	/**
	 * Get archive meta
	 * 
	 * @since 1.0.0
	 */
 	function blog_postx_get_current_page_sidebar() {
		$current_id = get_queried_object_id();
		$sidebar_meta_key = '';
		if( is_category() ) :
			$sidebar_meta_key = '_blog_postx_category_sidebar_custom_meta_field';
		elseif( is_tag() ) :
			$sidebar_meta_key = '_blog_postx_post_tag_sidebar_custom_meta_field';
		elseif( is_single() ) :
			$sidebar_meta_key = 'post_sidebar_layout';
		elseif( is_page() ) :
			$sidebar_meta_key = 'page_sidebar_layout';
		else:
			return 'customizer-setting';
		endif;

		if( is_category() || is_tag() ) $sidebar_layout_meta = metadata_exists( 'term', $current_id, $sidebar_meta_key ) ? get_term_meta( $current_id, $sidebar_meta_key, true ) : 'customizer-setting';
		if( is_single() || is_page() ) $sidebar_layout_meta = metadata_exists( 'post', $current_id, $sidebar_meta_key ) ? get_post_meta( $current_id, $sidebar_meta_key, true ) : 'customizer-setting';
		return $sidebar_layout_meta;
	}
 endif;

 if( ! function_exists( 'news_event_add_menu_description' ) ) :
	// merge menu description element to the menu 
	function news_event_add_menu_description( $item_output, $item, $depth, $args ) {
		if($args->theme_location != 'menu-1') return $item_output;
		
		if ( !empty( $item->description ) ) {
			$item_output = str_replace( $args->link_after . '</a>', '<span class="menu-item-description"><span class="description-wrap">' . $item->description . '</span></span>' . $args->link_after . '</a>', $item_output );
		}
		return $item_output;
	}
	add_filter( 'walker_nav_menu_start_el', 'news_event_add_menu_description', 10, 4 );
endif;

if( ! function_exists( 'blog_postx_stories_ajax_call' ) ) :
	/**
	 * Add stories to web stories according to current active story
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_stories_ajax_call() {
		check_ajax_referer( 'blog-postx-security-nonce', '_wpnonce' );
		$story_ids = ( isset( $_POST[ 'storyIds' ] ) ) ? $_POST[ 'storyIds' ] : '';
		$count = ( isset( $_POST[ 'count' ] ) ) ? $_POST[ 'count' ] : '';
		if( $story_ids ) :
			$query_args = [
				'post_type'	=>	'post',
				'post_status'	=>	'publish',
				'no_found_rows'	=>	true
			];
			$show_title = $show_categories = $show_date = $show_author = true;
			$post_per_page = BPX\blog_postx_get_customizer_option( 'web_stories_max_no_of_inner_stories' );
			$query_args[ 'posts_per_page' ] = absint( $post_per_page );
			$success_flag = false;
			ob_start();
			foreach( $story_ids as $cat_id ) :
				$query_args[ 'cat' ] = $cat_id;
				$query_instance = new WP_Query( apply_filters( 'blog_postx_query_args_filter', $query_args ) );
				if( $query_instance->have_posts() ) :
					$success_flag = true;
						echo '<div class="inner-story">';
	
							echo '<div class="inner-story-wrap" data-id="', esc_attr( $cat_id ) ,'">';
	
								while( $query_instance->have_posts() ) : $query_instance->the_post();
									?>
										<div class="story-item<?php if( ! has_post_thumbnail() ) echo ' no-thumb'; ?>">
	
											<div class="story-cover-wrap">
	
												<figure class="story-cover">
	
													<?php 
														// thumbnail
														the_post_thumbnail( 'post-thumbnail', [
															'loading'	=>	'lazy'
														]);
	
													?>
	
												</figure>
	
											</div>
	
											<?php
	
												echo '<div class="content-wrap">';
														
													// categories
													if( $show_categories ) blog_postx_get_post_categories( get_the_ID(), 2 );
	
													// title
													if( $show_title ) the_title( '<h2 class="title"><a href="'. get_the_permalink() .'">', '</a></h2>' );
	
													// Meta
													if( $show_date || $show_author ) :
	
														echo '<div class="meta-wrap">';
			
															// author
															if( $show_author ) blog_postx_posted_by();
			
															// date
															if( $show_date ) blog_postx_posted_on();
			
														echo '</div>';
														
													endif;
	
												echo '</div>';
	
											?>
	
										</div>
	
									<?php
								endwhile;
	
							echo '</div>';
	
						echo '</div>';
						wp_reset_postdata();
					endif;
			endforeach;
			$res = ob_get_clean();
		endif;
		$success_flag ? wp_send_json_success( $res ) : wp_send_json_error( esc_html__( 'No stories found', 'blog-postx' ) );
		wp_die();
	}
	add_action( 'wp_ajax_blog_postx_stories_ajax_call', 'blog_postx_stories_ajax_call' );
	add_action( 'wp_ajax_nopriv_blog_postx_stories_ajax_call', 'blog_postx_stories_ajax_call' );
endif;

if( ! function_exists( 'blog_postx_get_reorder_value' ) ) :
	/**
	 * Get Reorder value
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_get_reorder_value( $reorder ) {
		if( is_array( $reorder ) ) :
			$filtered_value = [];
			foreach( $reorder as $section ) :
				$filtered_value[ $section[ 'value' ] ] = $section[ 'option' ];
			endforeach;
			return $filtered_value;
		endif;
		return $reorder;
	}
endif;

if( ! function_exists( 'blog_postx_sticky_posts_ajax_call' ) ) :
	/**
	 * Fetch append posts for sticky posts
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_sticky_posts_ajax_call() {
		check_ajax_referer( 'blog-postx-security-nonce', '_wpnonce' );
		$args[ 'sticky_posts_show_author' ] = true;
		$args[ 'sticky_posts_show_date' ] = true;
		$args[ 'classes' ] = 'append';
		$posts_to_append = BPX\blog_postx_get_customizer_option( 'sticky_posts_posts_to_append' );
		$total_posts = BPX\blog_postx_get_customizer_option( 'sticky_posts_to_show' );
		$query_args = blog_postx_get_query_args( 'sticky' );
		$query_args[ 'posts_per_page' ] = absint( $posts_to_append );
		$query_args[ 'offset' ] = absint( $total_posts - $posts_to_append );
		$success_flag = false;
		ob_start();
		$query_instance = new WP_Query( $query_args );
		if( $query_instance->have_posts() ) :
			$success_flag = true;
			$count = $total_posts - $posts_to_append;
			while( $query_instance->have_posts() ) : $query_instance->the_post();
				$count++;
				$args[ 'count' ] = $count;
				get_template_part( 'template-parts/content', 'sticky-posts', $args );
			endwhile;
		endif;
		$res = ob_get_clean();
		$success_flag ? wp_send_json_success( $res ) : wp_send_json_error( esc_html__( 'No posts to append.', 'blog-postx' ) );
		wp_die();
	}
	add_action( 'wp_ajax_blog_postx_sticky_posts_ajax_call', 'blog_postx_sticky_posts_ajax_call' );
	add_action( 'wp_ajax_nopriv_blog_postx_sticky_posts_ajax_call', 'blog_postx_sticky_posts_ajax_call' );
endif;

if( ! function_exists( 'blog_postx_get_query_args' ) ) :
	/**
	 * Get Query Parameters of WP Query
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_get_query_args( $prefix = '' ) {
		$category_ids = BPX\blog_postx_get_customizer_option( $prefix . '_posts_categories' );
		$posts_to_include = BPX\blog_postx_get_customizer_option( $prefix . '_posts_to_include' );
		$post_order = BPX\blog_postx_get_customizer_option( $prefix . '_posts_order' );
		$exploded_order = explode( '-', $post_order );
		$posts_to_show = BPX\blog_postx_get_customizer_option( $prefix . '_posts_to_show' );
		$hide_empty = BPX\blog_postx_get_customizer_option( $prefix . '_hide_empty' );

		$posts_args = [
            'post_type' =>  'post',
            'post_status'  =>  'publish',
            'posts_per_page'    =>  absint( $posts_to_show ),
            'order' =>  $exploded_order[ 1 ],
            'orderby'   =>  $exploded_order[ 0 ],
            'ignore_sticky_posts'   =>  true,
            'fields'    =>  'ids',
            'no_found_rows' =>  true
        ];

		$post_categories_id_args = ( ! empty( $category_ids ) ) ? implode( ",", array_column( $category_ids, 'value' ) ) : '';
        $post_to_include_id_args = ( ! empty( $posts_to_include ) ) ? array_column( $posts_to_include, 'value' ) : '';

		if( isset( $category_ids ) ) $posts_args[ 'cat' ] = $post_categories_id_args;
        if( isset( $posts_to_include ) ) $posts_args[ 'post__in' ] = $post_to_include_id_args;

		if( $hide_empty ) :
            $posts_args[ 'meta_query' ] = [
                [
                    'key'   =>  '_thumbnail_id',
                    'compare'   =>  'EXISTS'
                ]
            ];
        endif;
		return $posts_args;
	}
endif;

if( ! function_exists( 'blog_postx_search_page_ajax_call' ) ) :
	/**
	 * Search page ajax call
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_search_page_ajax_call() {
		check_ajax_referer( 'blog-postx-security-nonce', '_wpnonce' );
		$query_args = ( isset( $_POST[ 'query' ] ) ) ? $_POST[ 'query' ] : [];
		$clicked_button = ( isset( $_POST[ 'clickedButton' ] ) ) ? $_POST[ 'clickedButton' ] : 'load-more';
		$archive_meta_reorder = blog_postx_get_reorder_value( BPX\blog_postx_get_customizer_option( 'archive_meta_reorder' ) );
		$archive_content_reorder = blog_postx_get_reorder_value( BPX\blog_postx_get_customizer_option( 'archive_content_reorder' ) );
		$search_nothing_found_content = esc_html__( 'Sorry, but nothing matched your search terms. Please try again with some different keywords.', 'blog-postx' );
		$pagination_no_more_reults_text = esc_html__( 'No more results', 'blog-postx' );
		$args[ 'archive_meta_reorder' ] = $archive_meta_reorder;
		$args[ 'archive_content_reorder' ] = $archive_content_reorder;
		$args[ 'archive_post_layout' ] = BPX\blog_postx_get_customizer_option( 'archive_post_layout' );
		$args[ 'show_archive_category_in_mobile' ] = true;
		$args[ 'show_archive_date_in_mobile' ] = true;
		$query_args[ 'posts_per_page' ] =  get_option( 'posts_per_page' );
		$instance = new WP_Query( $query_args );
		$success_flag = false;
		if( $instance->have_posts() ) :
			$success_flag = true;
			ob_start();
			if( $instance->max_num_pages === ( $query_args[ 'paged' ] )  ) $res[ 'completed' ] = true;
			while( $instance->have_posts() ) : $instance->the_post();

				get_template_part( 'template-parts/archive/layout', '', $args );
				
			endwhile;
			wp_reset_postdata();
			$res = ob_get_clean();
		endif;
		if( ! $success_flag ) $res[ 'message' ] = '<p class="failure-message">'. esc_html( ( $clicked_button === 'load-more' ) ? $pagination_no_more_reults_text : $search_nothing_found_content ) . '</p>';
		$success_flag ? wp_send_json_success( $res ) : wp_send_json_error( $res );
		wp_die();
	}
	add_action( 'wp_ajax_blog_postx_search_page_ajax_call', 'blog_postx_search_page_ajax_call' );
	add_action( 'wp_ajax_nopriv_blog_postx_search_page_ajax_call', 'blog_postx_search_page_ajax_call' );
endif;

if( ! function_exists( 'blog_postx_search_query_section' ) ) :
	/**
	 * Search Query Section
	 * 
	 * @since 1.0.0
	 */
	function blog_postx_search_query_section() {
		$term_args = [
			'fields'	=>	'id=>name',
			'hide_empty'	=>	true
		];
		$categories = get_categories( $term_args );
		$tags = get_tags( $term_args );
		$user_query = new \WP_User_Query([
			'fields'    =>  [ 'ID', 'display_name' ]
		]);
		$post_types = [
			'post'	=>	esc_html__( 'Posts', 'blog-postx' ),
			'page'	=>	esc_html__( 'Pages', 'blog-postx' ),
		];
		?>
			<div class="filter-wrapper">
				<div class="filter has-dropdown type">
					<div class="head">
						<span class="label"><?php esc_html_e( 'Types', 'blog-postx' ); ?></span>
						<span class="icon"><i class="fa-solid fa-angle-down"></i></span>
					</div>
					<div class="body">
						<?php
							if( is_array( $post_types ) && ! empty( $post_types ) ) :
								echo '<div class="items">';
								foreach( $post_types as $key => $type ) :
									?>
										<div class="item">
											<input type="checkbox" name="type-<?php echo esc_attr( $key ); ?>" id="type-<?php echo esc_attr( $key ); ?>" value="<?php echo esc_attr( $key ); ?>">
											<label for="type-<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $type ); ?></label>
										</div>
									<?php
								endforeach;
								echo '</div>';
							endif;
						?>
						<button class="clear"><?php esc_html_e( 'Clear', 'blog-postx' ); ?></button>
					</div>
				</div>
				<div class="filter has-dropdown authors">
					<div class="head">
						<span class="label"><?php esc_html_e( 'Authors', 'blog-postx' ); ?></span>
						<span class="icon"><i class="fa-solid fa-angle-down"></i></span>
					</div>
					<div class="body">
						<?php
							if( is_array( $user_query->get_results() ) && ! empty( $user_query->get_results() ) ) :
								echo '<div class="items">';
								foreach( $user_query->get_results() as $user ) :
									$user_id = $user->ID;
									?>
										<div class="item">
											<input type="checkbox" name="user-<?php echo esc_attr( $user_id ); ?>" id="user-<?php echo esc_attr( $user_id ); ?>" value="<?php echo esc_attr( $user_id ); ?>">
											<label for="user-<?php echo esc_attr( $user_id ); ?>"><?php echo esc_html( $user->display_name ); ?></label>
										</div>
									<?php
								endforeach;
								echo '</div>';
							endif;
						?>
						<button class="clear"><?php esc_html_e( 'Clear', 'blog-postx' ); ?></button>
					</div>
				</div>
				<button class="action-btn filter-button disabled"><?php esc_html_e( 'Filter', 'blog-postx' ); ?></button>
				<button class="action-btn clear-button disabled"><?php esc_html_e( 'Clear', 'blog-postx' ); ?></button>
			</div>
		<?php
	}
endif;