<?php
/**
 * Includes helper hooks and function of the theme
 * 
 * @package Blog Postx Pro
 * @since 1.0.0
 */
use BlogPostx\CustomizerDefault as BPX;

if( ! function_exists( 'blog_postx_get_post_format' ) ) :
    /**
     * Gets the post format string
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    function blog_postx_get_post_format( $id = null ) {
        $post_format = ( $id ) ? get_post_format($id): get_post_format();
        return apply_filters( 'blog_postx_post_format_string_filter', $post_format );
    }
endif;

if( ! function_exists( 'blog_postx_progress_bar' ) ) :
    /**
     * Display a progress bar
     * 
     * @since 1.0.0
     */
    function blog_postx_progress_bar() {
        if( is_single() || is_page() || is_archive() || is_search() ) :
            echo '<div class="single-progress"></div>';
        endif;
    }
    add_action( 'blog_postx_header_after_hook', 'blog_postx_progress_bar', 10 );
endif;

if( ! function_exists( 'blog_postx_sticky_posts' ) ) :
    function blog_postx_sticky_posts(){
        if( ! BPX\blog_postx_get_customizer_option( 'sticky_posts_option' ) || is_paged() || ! is_home() || ! is_front_page() ) return;
        $sticky_posts_position = BPX\blog_postx_get_customizer_option( 'sticky_posts_position' );
        $posts_to_append = BPX\blog_postx_get_customizer_option( 'sticky_posts_posts_to_append' );
        $posts_to_show = BPX\blog_postx_get_customizer_option( 'sticky_posts_to_show' );
        $args[ 'sticky_posts_show_author' ] = true;
        $args[ 'sticky_posts_show_date' ] = true;
        $sectionClass = 'blog-postx-sticky-posts position-' . esc_html( $sticky_posts_position );
        $posts_args = blog_postx_get_query_args( 'sticky' );
        $posts_args[ 'posts_per_page' ] = absint( $posts_to_append - $posts_to_show );
        ?>
            <div
                class="<?php echo esc_attr( $sectionClass ); ?>"
                data-author="<?php echo esc_attr( $args[ 'sticky_posts_show_author' ] ); ?>"
                data-date="<?php echo esc_attr( $args[ 'sticky_posts_show_date' ] ); ?>"
            >
				<div class="label-wrapper">
                    <h2 class="label"><?php esc_html_e( 'Popular Posts', 'blog-postx' ); ?></h2>
                    <span class="icon"></span>
                </div>
				<div class="post-list">
					<?php
						$query_instance = new \WP_Query( $posts_args );
						if( $query_instance->have_posts() ) :
							$count = 0;
							while( ( $query_instance->have_posts() ) ) : $query_instance->the_post();
                                $count++;
                                $args[ 'count' ] = $count;
								get_template_part( 'template-parts/content', 'sticky-posts', $args );
							endwhile;
						endif;
					?>
					<div class="more-less-indicator">
                        <span class="indicator more active"></span>
                        <span class="indicator less"></span>
					</div>
				</div>
			</div><!-- .nekit-sticky-posts -->
        <?php
    }
    add_action( 'blog_postx_header_after_hook', 'blog_postx_sticky_posts', 10 );
endif;

if( ! function_exists( 'blog_postx_get_icon_control_html' ) ) :
    /**
     * Generates output for icon control
     * 
     * @since 1.0.0
     */
    function blog_postx_get_icon_control_html($icon) {
        if( $icon['type'] == 'none' ) return;
        switch($icon['type']) {
            case 'svg' : $output = '<img src="' .esc_url( wp_get_attachment_url( $icon['value'] ) ). '" loading="lazy" />';
                break;
            default: $output = '<i class="' .esc_attr( $icon['value'] ). '"></i>';
        }
        return $output;
    }
endif;

if( ! function_exists( 'blog_postx_convert_number_to_numeric_string' )) :
    /**
     * Function to convert int parameter to numeric string
     * 
     * @return string
     */
    function blog_postx_convert_number_to_numeric_string( $int ) {
        switch( $int ){
            case 2:
                return "two";
                break;
            case 3:
                return "three";
                break;
            case 4:
                return "four";
                break;
            case 5:
                return "five";
                break;
            case 6:
                return "six";
                break;
            case 7:
                return "seven";
                break;
            case 8:
                return "eight";
                break;
            case 9:
                return "nine";
                break;
            case 10:
                return "ten";
                break;
            default:
                return "one";
        }
    }
 endif;

 if( ! function_exists( 'blog_postx_post_read_time' ) ) :
    /**
     * Function derives the read time
     * @return float
     */
    function blog_postx_post_read_time( $string = '' ) {
    	$read_time = 0;
        if( empty( $string ) ) {
            return 0 . esc_html__( ' min', 'blog-postx' );
        } else {
            $read_time = apply_filters( 'blog_postx_content_read_time', ceil( str_word_count( wp_strip_all_tags( $string ) ) / 200 ), 2 );
            if( $read_time == 0 ) {
            	return 1 . esc_html__( ' min', 'blog-postx' );
            } else {
            	return $read_time . esc_html__( ' mins', 'blog-postx' );
            }
        }
    }
endif;

if( ! function_exists( 'blog_postx_get_post_categories' ) ) :
    /**
     * Function contains post categories html
     * @return float
     */
    function blog_postx_get_post_categories( $post_id, $number = 1, $args = [] ) {
        $hide_on_mobile = '';
    	$n_categories = wp_get_post_categories($post_id, array( 'number' => absint( $number ) ));
        if( array_key_exists( 'hide_on_mobile', $args ) ) :
            $hide_on_mobile = ( ! $args['hide_on_mobile'] ) ? ' hide-on-mobile' : '';
        endif;
        $tag = ( is_single() ? 'div' : 'ul' );
		echo '<'. $tag .' class="post-categories'. esc_attr( $hide_on_mobile ) .'">';
			foreach( $n_categories as $n_category ) :
				echo '<li class="cat-item ' .esc_attr( 'cat-' . $n_category ). '"><a href="' .esc_url( get_category_link( $n_category ) ). '" rel="category tag">' .get_cat_name( $n_category ). '</a></li>';
			endforeach;
		echo '</'. $tag .'>';
    }
endif;

if( ! function_exists( 'blog_postx_loader_html' ) ) :
	/**
     * Preloader html
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
	function blog_postx_loader_html() {
        if( ! BPX\blog_postx_get_customizer_option( 'preloader_option' ) ) return;
	?>
        <div class="<?php echo esc_attr( 'blog_postx_loading_box preloader-style--two display-preloader--every-load' ); ?>">
			<div class="box">
				<div class="one"></div>
                <div class="two"></div>
                <div class="three"></div>
                <div class="four"></div>
                <div class="five"></div>
			</div>
		</div>
	<?php
	}
    add_action( 'blog_postx_page_prepend_hook', 'blog_postx_loader_html', 1 );
endif;

 if( ! function_exists( 'blog_postx_custom_header_html' ) ) :
    /**
     * Site custom header html
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    function blog_postx_custom_header_html() {
        /**
         * Get custom header markup
         * 
         * @since 1.0.0 
         */
        the_custom_header_markup();
    }
    add_action( 'blog_postx_page_prepend_hook', 'blog_postx_custom_header_html', 20 );
 endif;

 if( ! function_exists( 'blog_postx_pagination_fnc' ) ) :
    /**
     * Renders pagination html
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    function blog_postx_pagination_fnc() {
        if( is_null( paginate_links() ) ) {
            return;
        }
        $archive_pagination_type = BPX\blog_postx_get_customizer_option( 'archive_pagination_type' );
        // the_post_navigation
        switch($archive_pagination_type) {
            case 'default': 
                $next_link = get_previous_posts_link( 'Newer Posts' );
                $prev_link = get_next_posts_link( 'Older Posts' );
                ?>
                    <nav class="navigation posts-navigation">
                        <div class="nav-links">
                            <div class="nav-previous"><?php if( ! is_null( $prev_link ) ) echo wp_kses_post( $prev_link ); ?></div>
                            <div class="nav-next"><?php if( ! is_null( $next_link ) ) echo wp_kses_post( $next_link ); ?></div>
                        </div>
                    </nav>
                <?php
                    break;
            default: echo '<div class="pagination">' .wp_kses_post( paginate_links( array( 'prev_text' => '<i class="fa-solid fa-angles-left"></i>', 'next_text' => '<i class="fa-solid fa-angles-right"></i>', 'type' => 'list' ) ) ). '</div>';
        }
        
    }
    add_action( 'blog_postx_pagination_link_hook', 'blog_postx_pagination_fnc' );
 endif;

 if( ! function_exists( 'blog_postx_scroll_to_top_html' ) ) :
    /**
     * Scroll to top fnc
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    function blog_postx_scroll_to_top_html() {
        $stt_text = BPX\blog_postx_get_customizer_option( 'stt_text' );
        $classes = 'blog-postx-scroll-btn align--right display--fixed';
        $show_scroll_to_top_on_mobile = BPX\blog_postx_get_customizer_option( 'show_scroll_to_top_on_mobile' );
        if( ! $show_scroll_to_top_on_mobile ) $classes .= ' hide-on-mobile';
        ?>
            <div id="blog-postx-scroll-to-top" class="<?php echo esc_attr( $classes ); ?>">
                <div class="scroll-top-wrap">
                    <?php if( $stt_text ) echo '<span class="icon-text">'. esc_html( $stt_text ) .'</span>'; ?>
                    <div class="scroll-to-top-wrapper">
                        <span class="icon-holder"><i class="fas fa-angle-up"></i></span>
                    </div>
                </div>
            </div><!-- #blog-postx-scroll-to-top -->
        <?php
    }
    add_action( 'blog_postx_scroll_to_top_hook', 'blog_postx_scroll_to_top_html', 10 );
 endif;

 require get_template_directory() . '/inc/hooks/footer-hooks.php'; // footer hooks.
 require get_template_directory() . '/inc/hooks/archive-hooks.php'; // archive hooks.
 require get_template_directory() . '/inc/hooks/single-hooks.php'; // single hooks.

if ( ! function_exists( 'blog_postx_breadcrumb_trail' ) ) :
    /**
     * Theme default breadcrumb function.
     *
     * @since 1.0.0
     */
    function blog_postx_breadcrumb_trail() {
        if ( ! function_exists( 'breadcrumb_trail' ) ) {
            // load class file
            require_once get_template_directory() . '/inc/breadcrumb-trail/breadcrumb-trail.php';
        }

        // arguments variable
        $breadcrumb_args = array(
            'container' => 'div',
            'show_browse' => false
        );
        breadcrumb_trail( $breadcrumb_args );
    }
    add_action( 'blog_postx_breadcrumb_trail_hook', 'blog_postx_breadcrumb_trail' );
endif;

if( ! function_exists( 'blog_postx_breadcrumb_html' ) ) :
    /**
     * Theme breadcrumb
     *
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    function blog_postx_breadcrumb_html() {
        $site_breadcrumb_option = BPX\blog_postx_get_customizer_option( 'site_breadcrumb_option' );
        $single_post_layout = BPX\blog_postx_get_customizer_option( 'single_post_layout' );
        $show_breadcrumb_on_mobile = true;
	    $single_layout_post_meta = metadata_exists( 'post', get_the_ID(), 'single_layout' ) ? get_post_meta( get_the_ID(), 'single_layout', true ) : 'customizer-layout';
        if ( ! $site_breadcrumb_option ) return;
        if ( is_front_page() || is_home() ) return;
        $site_breadcrumb_type = BPX\blog_postx_get_customizer_option( 'site_breadcrumb_type' );
        $wrapperClass = 'blog-postx-breadcrumb-element';
        if( ( $single_post_layout !== 'layout-six' && $single_layout_post_meta !== 'customizer-layout' ) || $single_layout_post_meta !== 'layout-six' ) $wrapperClass .= ' row';
        if( ! $show_breadcrumb_on_mobile ) $wrapperClass .= ' hide-on-mobile';
            ?>
                <div class="<?php echo esc_attr( $wrapperClass ); ?>">
                    <div class="blog-postx-breadcrumb-wrap">
                        <?php
                            switch( $site_breadcrumb_type ) {
                                case 'yoast': if( blog_postx_compare_wand([blog_postx_function_exists( 'yoast_breadcrumb' )] ) ) yoast_breadcrumb();
                                        break;
                                case 'rankmath': if( blog_postx_compare_wand([blog_postx_function_exists( 'rank_math_the_breadcrumbs' )] ) ) rank_math_the_breadcrumbs();
                                        break;
                                case 'bcn': if( blog_postx_compare_wand([blog_postx_function_exists( 'bcn_display' )] ) ) bcn_display();
                                        break;
                                default: do_action( 'blog_postx_breadcrumb_trail_hook' );
                                        break;
                            }
                        ?>
                    </div>
                </div><!-- .row -->
        <?php
    }
endif;
add_action( 'blog_postx_before_main_content', 'blog_postx_breadcrumb_html', 10 );

if( ! function_exists( 'blog_postx_theme_mode_switch' ) ) :
    /**
     * Function to return either icon html or image html
     * 
     * @param type
     * @since 1.0.0
     */
    function blog_postx_theme_mode_switch( $mode, $theme_mode ) {
        $elementClass = ( $theme_mode == 'light' ) ? 'lightmode' : 'darkmode';
        switch( $mode['type'] ) :
            case 'icon' :
                echo '<i class="'. esc_attr( $mode['value'] . ' ' . $elementClass ) .'"></i>';
                break;
            case 'svg' :
                echo '<img class="'. esc_attr( $elementClass ) .'" src="'. esc_url( wp_get_attachment_image_url( $mode['value'], 'full' ) ) .'" loading="lazy">';
                break;
        endswitch;
    }
 endif;

if( ! function_exists( 'blog_postx_button_html' ) ) :
    /**
     * View all html
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    function blog_postx_button_html( $args ) {
        if( ! $args['show_button'] ) return;
        $is_after = true;
        $archive_read_more_text_on_mobile = true;
        $archive_read_more_button_on_mobile = true;

        $read_more_text_hide_on_mobile = ( ! $archive_read_more_text_on_mobile ) ? ' hide-on-mobile' : '';
        $read_more_button_hide_on_mobile = ( ! $archive_read_more_button_on_mobile ) ? ' hide-on-mobile' : '';
        
        $classes = isset( $args['classes'] ) ? 'blog-postx-button post-link-button' . ' ' .$args['classes'] : 'post-button blog-postx-button';
        $classes .= $read_more_button_hide_on_mobile;
        $classes .= ' after';
        $link = isset( $args['link'] ) ? $args['link'] : get_the_permalink();
        $text = isset( $args['text'] ) ? $args['text'] : apply_filters( 'blog_postx_global_button_label_fitler', esc_html__( 'continue Reading..', 'blog-postx' ) );
        $text_html = sprintf( '<span class="button-text%1$s">%2$s</span>', esc_attr( $read_more_text_hide_on_mobile ), esc_html( $text ) );
        $icon_html = '';
        echo apply_filters( 'blog_postx_button_html', sprintf( '<a class="%1$s" href="%2$s">%3$s%4$s</a>', esc_attr( $classes ), esc_url( $link ), wp_kses_post( $is_after ? $text_html : $icon_html ), wp_kses_post( $is_after ? $icon_html : $text_html ) ) );
    }
    add_action( 'blog_postx_section_block_view_all_hook', 'blog_postx_button_html', 10, 1 );
endif;