<?php
    /**
     * Base class for header and footer builder
     * 
     * @package Blog Postx Pro
     * @since 1.0.0
     */
    namespace Blog_Postx_Builder;
    use BlogPostx\CustomizerDefault as BPX;
    if( ! class_exists( 'Header_Builder_Render' ) ) :
        /**
         * Builder Base class
         * 
         * @since 1.0.0
         */
        class Header_Builder_Render extends Builder_Base {
            /**
             * Method that gets called when class is instantiated
             * 
             * @since 1.0.0
             */
            public function __construct() {
                $this->original_value = BPX\blog_postx_get_customizer_option( 'header_builder' );
                $this->builder_value = $this->original_value;
                $this->assign_values();
                $this->prepare_value_for_render();
                $this->render();
            }

            /**
             * Assign values
             * 
             * @since 1.0.0
             */
            public function assign_values() {
                /* Columns count */
                $this->columns_array = [ 
                    BPX\blog_postx_get_customizer_option( 'header_first_row_column' ),
                    BPX\blog_postx_get_customizer_option( 'header_second_row_column' ),
                    BPX\blog_postx_get_customizer_option( 'header_third_row_column' )
                ];
                /* Columns layout */
                $this->column_layouts_array = [
                    BPX\blog_postx_get_customizer_option( 'header_first_row_column_layout' ),
                    BPX\blog_postx_get_customizer_option( 'header_second_row_column_layout' ),
                    BPX\blog_postx_get_customizer_option( 'header_third_row_column_layout' )
                ];
                /* Column Alignments */
                $this->column_alignments_array = $this->organize_column_alignments();
            }

            /**
             * Column alignments
             * 
             * @since 1.0.0
             */
            public function organize_column_alignments() {
                $column_alignments = [
                    [
                        /* First Row */
                        BPX\blog_postx_get_customizer_option( 'header_first_row_column_one' ),
                        BPX\blog_postx_get_customizer_option( 'header_first_row_column_two' ),
                        BPX\blog_postx_get_customizer_option( 'header_first_row_column_three' )
                    ],
                    [
                        /* Second Row */
                        BPX\blog_postx_get_customizer_option( 'header_second_row_column_one' ),
                        BPX\blog_postx_get_customizer_option( 'header_second_row_column_two' ),
                        BPX\blog_postx_get_customizer_option( 'header_second_row_column_three' )
                    ],
                    [
                        /* Third Row */
                        BPX\blog_postx_get_customizer_option( 'header_third_row_column_one' ),
                        BPX\blog_postx_get_customizer_option( 'header_third_row_column_two' ),
                        BPX\blog_postx_get_customizer_option( 'header_third_row_column_three' )
                    ]
                ];

                $structured_alignements = [];
                if( count( $this->columns_array ) > 0 ) :
                    $columns_array_count = count( $this->columns_array );
                    for( $i = 0; $i < $columns_array_count; $i++ ) :
                        $structured_alignements[ $i ] = $column_alignments[ $i ];
                    endfor;
                endif;

                return $structured_alignements;
            }

            /**
             * Extra row classes
             * 
             * @since 1.0.0
             */
            public function get_extra_row_classes( $row_index ) {
                $row_widgets = $this->builder_value[ $row_index ];
                $only_widgets = array_reduce( $row_widgets, 'array_merge', [] );
                $header_sticky = $this->get_row_header_sticky_value( $row_index );
                $classes = '';
                if( $header_sticky ) $classes .= ' row-sticky';
                if( in_array( 'menu', $only_widgets ) ) $classes .= ' has-menu';
                return $classes;
            }

            /**
             * Header sticky controls in an array
             * 
             * @since 1.0.0
             */
            public function get_row_header_sticky_value( $row_index ) {
                $header_sticky_controls = [
                    BPX\blog_postx_get_customizer_option( 'header_first_row_header_sticky' ),
                    BPX\blog_postx_get_customizer_option( 'header_second_row_header_sticky' ),
                    BPX\blog_postx_get_customizer_option( 'header_third_row_header_sticky' )
                ];
                return $header_sticky_controls[ $row_index ];
            }

            /**
             * Get widget html
             * 
             * @since 1.0.0
             */
            public function get_widget_html( $widget ) {
                require get_template_directory() . '/inc/hooks/top-header-hooks.php';
                if( ! $widget ) return;
                switch( $widget ) :
                    case 'site-logo':
                        /**
                        * hook - blog_postx_header__site_branding_section_hook
                        * 
                        * @hooked - blog_postx_header_menu_part - 10
                        */
                        if( has_action( 'blog_postx_header__site_branding_section_hook' ) ) do_action( 'blog_postx_header__site_branding_section_hook' );
                        break;
                    case 'date-time':
                        /**
                        * hook - blog_postx_date_time_hook
                        * 
                        * @hooked - blog_postx_date_time_part - 10
                        */
                        if( has_action( 'blog_postx_date_time_hook' ) ) do_action( 'blog_postx_date_time_hook' );
                        break;
                    case 'social-icons':
                        /**
                        * hook - blog_postx_social_icons_hook
                        * 
                        * @hooked - blog_postx_social_part - 10
                        */
                        if( has_action( 'blog_postx_social_icons_hook' ) ) do_action( 'blog_postx_social_icons_hook' );
                        break;
                    case 'search':
                        /**
                         * hook - blog_postx_header_search_hook
                         * 
                         * @hooked - blog_postx_header_search_part - 10
                         */
                        if( has_action( 'blog_postx_header_search_hook' ) ) do_action( 'blog_postx_header_search_hook' );
                        break;
                    case 'menu':
                        /**
                         * hook - blog_postx_header__menu_section_hook
                         * 
                         * @hooked - blog_postx_header_menu_part - 10
                         */
                        if( has_action( 'blog_postx_header__menu_section_hook' ) ) do_action( 'blog_postx_header__menu_section_hook' );
                        break;
                    case 'button':
                        /**
                         * hook - blog_postx_header__custom_button_section_hook
                         * 
                         * @hooked - blog_postx_header_custom_button_part - 10
                         */
                        if( has_action( 'blog_postx_header__custom_button_section_hook' ) ) do_action( 'blog_postx_header__custom_button_section_hook' );
                        break;
                    case 'theme-mode':
                        /**
                         * hook - blog_postx_header_theme_mode_hook
                         * 
                         * @hooked - blog_postx_header_theme_mode_part - 10
                         */
                        if( has_action( 'blog_postx_header_theme_mode_hook' ) ) do_action( 'blog_postx_header_theme_mode_hook' );
                        break;
                    case 'off-canvas':
                        /**
                         * hook - blog_postx_header_off_canvas_hook
                         * 
                         * @hooked - blog_postx_header_canvas_menu_part - 10
                         */
                        if( has_action( 'blog_postx_header_off_canvas_hook' ) ) do_action( 'blog_postx_header_off_canvas_hook' );
                        break;
                    case 'secondary-menu':
                        /**
                         * hook - blog_postx_header__secondary_menu_section_hook
                         * 
                         * @hooked - blog_postx_header_secondary_menu - 10
                         */
                        if( has_action( 'blog_postx_header__secondary_menu_section_hook' ) ) do_action( 'blog_postx_header__secondary_menu_section_hook' );
                        break;
                endswitch;
            }
        }
    endif;