<?php

/**
 * Sanitize Functions.
 */
 
if ( ! function_exists( 'bizindustries_sanitize_logo' ) ) :
	/**
	 * Sanitize the logo title select option.
	 *
	 * @since 1.0.0	 *
	 * @param string $logo_option.
	 */
	 
	function bizindustries_sanitize_logo( $logo_option ) {
		if ( ! in_array( $logo_option, array(  'logo-only', 'title-only', 'title-desc', 'title-img' ) ) ) {
			$logo_option = 'logo-only';
		} 
		return $logo_option;
	}
endif;


if ( ! function_exists( 'bizindustries_sanitize_select' ) ) :

	/**
		* Sanitize select.
	 *
	 * @since 1.0.0
	 *	
	 */
	function bizindustries_sanitize_select( $input, $setting ) {

		// Ensure input is a slug.
		$input = sanitize_key( $input );

		// Get list of choices from the control associated with the setting.
		$choices = $setting->manager->get_control( $setting->id )->choices;

		// If the input is a valid key, return it; otherwise, return the default.
		return ( array_key_exists( $input, $choices ) ? $input : $setting->default );

	}

endif;



if ( ! function_exists( 'bizindustries_sanitize_choices' ) ) :
	/**
	 * Sanitize choices.
	 *
	 * @since 1.0.0
	 *	
	 */
	function bizindustries_sanitize_choices( $input, $setting ) {
		global $wp_customize; 
		$control = $wp_customize->get_control( $setting->id ); 
		if ( array_key_exists( $input, $control->choices ) ) {
			return $input;
		} else {
			return $setting->default;
		}
	}
	
endif;


if ( ! function_exists( 'bizindustries_sanitize_color_alpha' ) ) :
	/**
	 * Sanitize Color Alpha
	 *
	 * @since 1.0.0
	 *	
	 */
	function bizindustries_sanitize_color_alpha( $color ){
		$color = str_replace( '#', '', $color );
		if ( '' === $color ){
			return '';
		}
		// 3 or 6 hex digits, or the empty string.
		if ( preg_match('|^#([A-Fa-f0-9]{3}){1,2}$|', '#' . $color ) ) {
			// convert to rgb
			$colour = $color;
			if ( strlen( $colour ) == 6 ) {
				list( $r, $g, $b ) = array( $colour[0] . $colour[1], $colour[2] . $colour[3], $colour[4] . $colour[5] );
			} elseif ( strlen( $colour ) == 3 ) {
				list( $r, $g, $b ) = array( $colour[0] . $colour[0], $colour[1] . $colour[1], $colour[2] . $colour[2] );
			} else {
				return false;
			}
			$r = hexdec( $r );
			$g = hexdec( $g );
			$b = hexdec( $b );
			return 'rgba('.join( ',', array( 'r' => $r, 'g' => $g, 'b' => $b, 'a' => 1 ) ).')';

		}
		return strpos( trim( $color ), 'rgb' ) !== false ?  $color : false;
	}
endif;




if ( ! function_exists( 'bizindustries_sanitize_number_range' ) ) :
	/**
	 * Number Range sanitization callback example.
	 *
	 * @since 1.0.0	 
	 * - Sanitization: number_range
	 * - Control: number, tel
	 *
	 * Sanitization callback for 'number' or 'tel' type text inputs. This callback sanitizes
	 * `$number` as an absolute integer within a defined min-max range.
	 *
	 * @see absint() https://developer.wordpress.org/reference/functions/absint/
	 *
	 * @param int                  $number  Number to check within the numeric range defined by the setting.
	 * @param WP_Customize_Setting $setting Setting instance.
	 * @return int|string The number, if it is zero or greater and falls within the defined range; otherwise,
	 *                    the setting default.
	 */
	function bizindustries_sanitize_number_range( $number, $setting ) {
		// Ensure input is an absolute integer.
		$number = absint( $number );

		// Get the input attributes associated with the setting.
		$atts = $setting->manager->get_control( $setting->id )->input_attrs;

		// Get minimum number in the range.
		$min = ( isset( $atts['min'] ) ? $atts['min'] : $number );

		// Get maximum number in the range.
		$max = ( isset( $atts['max'] ) ? $atts['max'] : $number );

		// Get step.
		$step = ( isset( $atts['step'] ) ? $atts['step'] : 1 );

		// If the number is within the valid range, return it; otherwise, return the default
		return ( $min <= $number && $number <= $max && is_int( $number / $step ) ? $number : $setting->default );
	}
endif;


if ( ! function_exists( 'bizindustries_sanitize_checkbox' ) ) :
	/**
	* Sanitize Checkbox
	 *
	 * @since 1.0.0	 
	 *	
	*/
	function bizindustries_sanitize_checkbox( $input ) {
		if ( $input == 1 ) {
			return true;
		} else {
			return false;
		}
	}
endif; 

if ( ! function_exists( 'bizindustries_sanitize_html' ) ) :
	/**
	 * HTML sanitization 
	 *
	 * @since 1.0.0
	 * @see wp_filter_post_kses() https://developer.wordpress.org/reference/functions/wp_filter_post_kses/
	 *
	 * @param string $html HTML to sanitize.
	 * @return string Sanitized HTML.
	 */
	function bizindustries_sanitize_html( $html ) {
		return wp_kses_post( force_balance_tags( $html ) );
	}
endif;

if ( ! function_exists( 'bizindustries_sanitize_css' ) ) :
	/**
	 * Sanitize CSS code
	 *
	 * @since 1.0.0	 *
	 * @param $string
	 * @return string
	 */
	function bizindustries_sanitize_css($string) {
		$string = preg_replace( '@<(script|style)[^>]*?>.*?</\\1>@si', '', $string );
		$string = strip_tags($string);
		return trim( $string );
	}
endif;



function get_kses_extended_ruleset() {
    $kses_defaults = wp_kses_allowed_html( 'post' );

    $svg_args = array(
        'svg'   => array(
            'class'           => true,
            'aria-hidden'     => true,
            'aria-labelledby' => true,
            'role'            => true,
            'xmlns'           => true,
            'width'           => true,
            'height'          => true,
            'viewbox'         => true, // <= Must be lower case!
        ),
        'g'     => array( 'fill' => true ),
        'title' => array( 'title' => true ),
        'path'  => array(
            'd'    => true,
            'fill' => true,
        ),
    );
    return array_merge( $kses_defaults, $svg_args );
}


if ( ! function_exists( 'bizindustries_sanitize_sort' ) ) :
	/**
	 * Sanitize sort
	 *
	 * @since 1.0.0
	 *	
	 */
	function bizindustries_sanitize_sort($value) {
		if (is_array($value)) {
			foreach ($value as $key => $subvalue) {
				$value[$key] = esc_attr($subvalue);
			}
			return $value;
		}
		return esc_attr($value);
	}
endif;


if ( ! function_exists( 'bizindustries_dropdown_pages' ) ) :
	/**
	 * Sanitize dropdown pages.
	 *
	 * @since 1.0.0
	 *	
	 */
	function bizindustries_dropdown_pages( $page_id, $setting ) {
	  // Ensure $input is an absolute integer.
	  $page_id = absint( $page_id );
	  
	  // If $page_id is an ID of a published page, return it; otherwise, return the default.
	  return ( 'publish' == get_post_status( $page_id ) ? $page_id : $setting->default );
	}
endif;

if ( ! function_exists( 'bizindustries_sanitize_image' ) ) :
	/**
	 * Image sanitization callback example.
	 *
	 * Checks the image's file extension and mime type against a whitelist. If they're allowed,
	 * send back the filename, otherwise, return the setting default.
	 *
	 */
	function bizindustries_sanitize_image( $image, $setting ) {
		/*
		 * Array of valid image file types.
		 *
		 * The array includes image mime types that are included in wp_get_mime_types()
		 */
		$mimes = array(
			'jpg|jpeg|jpe' => 'image/jpeg',
			'gif'          => 'image/gif',
			'png'          => 'image/png',
			'bmp'          => 'image/bmp',
			'tif|tiff'     => 'image/tiff',
			'ico'          => 'image/x-icon'
		);
		// Return an array with file extension and mime_type.
		$file = wp_check_filetype( $image, $mimes );
		// If $image has a valid mime_type, return it; otherwise, return the default.
		return ( $file['ext'] ? $image : $setting->default );
	}
endif;



if ( ! function_exists( 'bizindustries_sanitize_email' ) ) :
	/**
	* Sanitizes email
	* @param  $input entered value
	* @return sanitized output
	*
	* @since 1.0.0
	*/
	function bizindustries_sanitize_email( $input ) {
		
		$links = explode( '|', $input );

		// Ensure $input is an url.
		$links = array_map( 'sanitize_email', $links );
		$links = implode('|', $links);

		return $links;
	}
endif; 


if ( ! function_exists( 'bizindustries_sanitize_map_iframe' ) ) :
	/**
	 * Google map iframe sanitization
	 *
	 *
	 */
	function bizindustries_sanitize_map_iframe( $input, $setting ) {
		$allowedtags = array(
			'iframe' => array(
				'src' => array(),
				'width' => array(),
				'height' => array(),
				'frameborder' => array(),
				'style'     => array(),
				'marginwidth' => array(),
				'marginheight' => array(),
			)
		);

		$input = wp_kses( $input, $allowedtags );

		return $input;
	}
endif;

if ( ! function_exists( 'bizindustries_content_width_selection' ) ) :
	/**
	 * Sanitize Content layout options
	 *
	 * @param string $input
	 * @return string
	 */
	function bizindustries_content_width_selection( $input ){
		$valid = array(
			'semi-width'   => esc_html__('1400px', 'bizindustries'),
			'defaul-width' => esc_html__('1170px', 'bizindustries'),
		 );

		 if ( array_key_exists( $input, $valid ) ) :
			return $input;
		 else :
			return '';
		 endif;
	}
endif;



