<?php
/**
 * ballyhoo functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package ballyhoo
 */

if ( ! defined( 'ballyhoo_VERSION' ) ) {
	// Replace the version number of the theme on each release.
	define( 'ballyhoo_VERSION', '1.0.16' );
}

if ( ! function_exists( 'ballyhoo_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function ballyhoo_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on ballyhoo, use a find and replace
		 * to change 'ballyhoo' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'ballyhoo', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus(
			array(
				'menu-1' => esc_html__( 'Primary', 'ballyhoo' ),
			)
		);

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support(
			'html5',
			array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'style',
				'script',
			)
		);

		// Add support for responsive embeds
		add_theme_support('responsive-embeds');

		// Add support for "align-wide" option
		add_theme_support('align-wide');

		// Set up the WordPress core custom background feature.
		add_theme_support(
			'custom-background',
			apply_filters(
				'ballyhoo_custom_background_args',
				array(
					'default-color' => 'ffffff',
					// 'default-image' => get_template_directory_uri(), '/imgs/book.jpg',
					'width' => 1400,
					'height' => 500,
					'flex-height' => false,
					'flex-width' => false,
					'uploads' => true,
				)
			)
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support(
			'custom-logo',
			array(
				'height'      => 100,
				'width'       => 100,
				'flex-width'  => true,
				'flex-height' => true,
			)
		);
		add_theme_support( 'wp-block-styles' );
		add_theme_support('custom-header');
	}
endif;
add_action( 'after_setup_theme', 'ballyhoo_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function ballyhoo_content_width() {
	$GLOBALS['ballyhoo_content_width'] = apply_filters( 'ballyhoo_content_width', 640 );
}
add_action( 'after_setup_theme', 'ballyhoo_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function ballyhoo_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'ballyhoo' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add blocks you want in the sidebar here.', 'ballyhoo' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
	register_sidebar(
		array(
			'name' => esc_html__('Footer','ballyhoo'),
			'id' => 'footer-widgets',
			'description' => esc_html__('Add footer blocks here', 'ballyhoo'),
			'before_widget' => '<section id="%1$s class="widget %2$s">',
			'after_widget' => '</section>',
			'before_title' => '<h2 class="widget-title">',
			'after_title' => '</h2>',
		)
	);
}
add_action( 'widgets_init', 'ballyhoo_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function ballyhoo_scripts() {
	wp_enqueue_style( 'ballyhoo-style', get_stylesheet_uri(), array(), ballyhoo_VERSION );
	wp_style_add_data( 'ballyhoo-style', 'rtl', 'replace' );

	wp_enqueue_script( 'ballyhoo-navigation', get_template_directory_uri() . '/js/navigation.js', array(), ballyhoo_VERSION, true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'ballyhoo_scripts' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}
function ballyhoo_slug_editor_styles() {
    wp_enqueue_style( 'ballyhoo-slug-editor-style', get_template_directory_uri() . '/editor-style.css' );
    wp_enqueue_style( 'ballyhoo-slug-editor-style-2', 'https://fonts.googleapis.com/css2?family=Open+Sans:ital,wght@0,400;0,700;1,400&display=swap');
}
add_action( 'enqueue_block_editor_assets', 'ballyhoo_slug_editor_styles' );
// Put everything from here into the block-patterns.php file later
if (function_exists('register_block_pattern_category')) {
	register_block_pattern_category(
		'ballyhoo',
		array('label' => esc_html__('Ballyhoo', 'ballyhoo'))
	);
}
if(function_exists('register_block_pattern')) {
	// Website header
	register_block_pattern(
		'ballyhoo/website-header',
		array(
			'title' => esc_html__('Website header', 'ballyhoo'),
			'categories' => array('ballyhoo'),
			'viewportWidth' => 1400,
			'content' => '<!-- wp:group {"backgroundColor":"#111111","className":"site-branding"} --><div class="wp-block-group site-branding has-111111-background-color has-background"><!-- wp:heading {"level":1,"textColor":"white","className":"site-title"} --><h1 class="site-title has-white-color has-text-color">Your Website Title Here</h1><!-- /wp:heading --><!-- wp:heading {"align":"center","textColor":"white","className":"site-description"} --><h2 class="site-description has-white-color has-text-color">Your Website Slogan Here</h2><!-- /wp:heading --></div><!-- /wp:group -->',
		)
	);
	// Promotional headings
	register_block_pattern(
		'ballyhoo/promotional-headings',
		array(
			'title' => esc_html__('Promotional headings', 'ballyhoo'),
			'categories' => array('ballyhoo'),
			'viewportWidth' => 1400,
			'content' => '<!-- wp:heading {"textAlign":"center","level":1,"className":"light-heading"} --><h1 class="light-heading">Your first light heading</h1><!-- /wp:heading --><!-- wp:heading {"textAlign:"center","level":2,"className":"light-heading"} --><h2 class="light-heading">Your second light heading</h2><!-- /wp:heading -->',
		)
	);
	// Footer
	register_block_pattern(
		'ballyhoo/footer-pattern',
		array(
			'title' => esc_html__('Footer', 'ballyhoo'),
			'categories' => array('ballyhoo'),
			'viewportWidth' => 1400,
			'content' => '<!-- wp:group {"tagName":"footer",backgroundColor":"#111111","className":"site-branding"} --><div class="wp-block-group site-branding has-111111-background-color has-background"><!-- wp:paragraph {"textColor":"white"} --><p class="has-white-color has-text-color">© [Current Year] <a href="http://example.com" data-type="URL" data-id="http://example.com">[Your Website Name]</a> All Rights Reserved</p><!-- /wp:paragraph --></div><!-- /wp:group -->',
		)
	);
	register_block_pattern(
		'ballyhoo/fact-file',
		array(
			'title' => esc_html__('Fact File', 'ballyhoo'),
			'categories' => array('ballyhoo'),
			'viewportWidth' => 1400,
			'content' => '<!-- wp:group --><div class="wp-block-group"><!-- wp:columns --><div class="wp-block-columns"><!-- wp:column {"verticalAlignment":"top","className":"fact-file"} --><div class="wp-block-column is-vertically-aligned-top fact-file"><!-- wp:heading {"textAlign":"center","fontSize":"large"} --><h2 class="has-text-align-center has-large-font-size">Fact File Title</h2><!-- /wp:heading --><!-- wp:group --><div class="wp-block-group"><!-- wp:columns --><div class="wp-block-columns"><!-- wp:column --><div class="wp-block-column"><!-- wp:image --><figure class="wp-block-image"><img src="" alt=""/></figure><!-- /wp:image --></div><!-- /wp:column --><!-- wp:column --><div class="wp-block-column"><!-- wp:paragraph --><p>Put a fact here.</p><!-- /wp:paragraph --><!-- wp:paragraph --><p>Put another fact here.</p><!-- /wp:paragraph --></div><!-- /wp:column --></div><!-- /wp:columns --></div><!-- /wp:group --></div><!-- /wp:column --></div><!-- /wp:columns --></div><!-- /wp:group -->',
		)
	);
}
function ballyhoo_add_font_awesome_icons() {
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/all.min.css');
}
add_action( 'wp_enqueue_scripts', 'ballyhoo_add_font_awesome_icons' );
function ballyhoo_add_google_fonts() {
	wp_enqueue_style( 'google_fonts', 'https://fonts.googleapis.com/css2?family=Open+Sans:ital,wght@0,400;0,700;1,400&display=swap');
}
add_action( 'wp_enqueue_scripts', 'ballyhoo_add_google_fonts' );
// Put everything from here into the block-styles.php file later
if (function_exists('register_block_style')) {
	// Register the individual stylesheets for each style
	function ballyhoo_register_block_styles() {
		wp_register_style( 'button-square', get_template_directory_uri() . '/block-stylesheets/button-square.css' );
		wp_register_style( 'button-square-outline', get_template_directory_uri() . '/block-stylesheets/button-square-outline.css' );
		wp_register_style( 'button-rounded-square', get_template_directory_uri() . '/block-stylesheets/button-rounded-square.css' );

		wp_register_style( 'code-dark', get_template_directory_uri() . '/block-stylesheets/code-dark.css' );
		wp_register_style( 'image-rounded-square', get_template_directory_uri() . '/block-stylesheets/image-rounded-square.css' );
		wp_register_style( 'pullquote-huge', get_template_directory_uri() . '/block-stylesheets/pullquote-huge.css' );

		wp_register_style( 'quote-huge', get_template_directory_uri() . '/block-stylesheets/quote-huge.css' );
		wp_register_style( 'quote-light', get_template_directory_uri() . '/block-stylesheets/quote-light.css' );
		wp_register_style( 'verse-dark', get_template_directory_uri() . '/block-stylesheets/verse-dark.css' );

		wp_register_style( 'verse-large', get_template_directory_uri() . '/block-stylesheets/verse-large.css' );
		wp_register_style( 'verse-large-dark', get_template_directory_uri() . '/block-stylesheets/verse-large-dark.css' );
		// Button: Square
		register_block_style(
			'core/button',
			array(
				'name' => 'ballyhoo-button-square',
				'label' => esc_html__('Square', 'ballyhoo'),
				'style_handle' => 'button-square',
			),
		);
		// Button: Square Outline
		register_block_style(
			'core/button',
			array(
				'name' => 'ballyhoo-button-square-outline',
				'label' => esc_html__('Square Outline', 'ballyhoo'),
				'style_handle' => 'button-square-outline',
			),
		);
		// Button: Rounded Square
		register_block_style(
			'core/button',
			array(
				'name' => 'ballyhoo-button-rounded-square',
				'label' => esc_html__('Rounded Square', 'ballyhoo'),
				'style_handle' => 'button-rounded-square',
			),
		);
		// Code: Dark
		register_block_style(
			'core/code',
			array(
				'name' => 'ballyhoo-code-dark',
				'label' => esc_html__('Dark', 'ballyhoo'),
				'style_handle' => 'code-dark',
			),
		);
		// Image: Rounded Square
		register_block_style(
			'core/image',
			array(
				'name' => 'ballyhoo-image-rounded-square',
				'label' => esc_html__('Rounded Square', 'ballyhoo'),
				'style_handle' => 'image-rounded-square',
			),
		);
		// Pullquote: Huge
		register_block_style(
			'core/pullquote',
			array(
				'name' => 'ballyhoo-pullquote-huge',
				'label' => esc_html__('Huge', 'ballyhoo'),
				'style_handle' => 'pullquote-huge',
			),
		);
		// Quote: Huge
		register_block_style(
			'core/quote',
			array(
				'name' => 'ballyhoo-quote-huge',
				'label' => esc_html__('Huge', 'ballyhoo'),
				'style_handle' => 'quote-huge',
			),
		);
		// Quote: Light
		register_block_style(
			'core/quote',
			array(
				'name' => 'ballyhoo-quote-light',
				'label' => esc_html__('Light', 'ballyhoo'),
				'style_handle' => 'quote-light',
			),
		);
		// Verse: Dark
		register_block_style(
			'core/verse',
			array(
				'name' => 'ballyhoo-verse-dark',
				'label' => esc_html__('Dark', 'ballyhoo'),
				'style_handle' => 'verse-dark',
			),
		);
		// Verse: Large
		register_block_style(
			'core/verse',
			array(
				'name' => 'ballyhoo-verse-large',
				'label' => esc_html__('Large', 'ballyhoo'),
				'style_handle' => 'verse-large',
			),
		);
		// Verse: Large Dark
		register_block_style(
			'core/verse',
			array(
				'name' => 'ballyhoo-verse-large-dark',
				'label' => esc_html__('Large Dark', 'ballyhoo'),
				'style_handle' => 'verse-large-dark',
			),
		);
	}
	add_action('wp_enqueue_scripts','ballyhoo_scripts');
	add_action( 'init', 'ballyhoo_register_block_styles' );
}