<?php
/**
 * Theme Options Dashboard Management
 *
 * Manage widgets, tools and staffs.
 *
 * @package Ayyash
 * @since   1.2.3
 * @version 1.0.0
 */

// phpcs:disable
namespace Ayyash\AdminOptions;
// phpcs:enable

if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	die();
}

/**
 * Ayyash Admin Settings
 */
class Ayyash_Admin_Options {

	/**
	 * Menu page title
	 *
	 * @since 1.2.3
	 * @var string $menu_page_title
	 */
	public static $page_title = 'Ayyash Options';

	/**
	 * Page title
	 *
	 * @since 1.2.3
	 * @var string $menu_title
	 */
	public static $menu_title = 'Ayyash Options';

	/**
	 * Plugin slug
	 *
	 * @since 1.2.3
	 * @var string $menu_slug
	 */
	public static $menu_slug = 'ayyash-options-page';

	/**
	 * @return Ayyash_Admin_Options
	 */
	protected static $instance;

	/**
	 * Holds arrays of plugin details.
	 *
	 * @since 1.0.0
	 * @since 2.5.0 the array has the plugin slug as an associative key.
	 *
	 * @var array
	 */
	public static $plugins = array();

	/**
	 * Constructor
	 */
	protected function __construct() {

		if ( ! is_admin() ) {
			return;
		}

		add_action( 'admin_footer', 'wp_print_request_filesystem_credentials_modal' );
		add_action( 'after_setup_theme', [ __CLASS__, 'init_admin_settings' ], 99 );
		add_action( 'wp_ajax_ayyash_activate_plugin', [ __CLASS__, 'ajax_activate_plugin' ] );
		add_action( 'wp_ajax_ayyash_deactivate_plugin', [ __CLASS__, 'ajax_deactivate_plugin' ] );
		add_action('admin_notices', [ __CLASS__, 'ayyash_author_admin_notice' ] );
		add_action( 'switch_theme', array( $this, 'update_dismiss' ) );
		add_action( 'admin_head', array( $this, 'dismiss' ) );
	}

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Admin settings init
	 */
	public static function init_admin_settings() {
		add_action( 'admin_menu', [ __CLASS__, 'add_admin_menu' ] );
	}

	/**
	 * Add main menu
	 *
	 * @since 1.0
	 */
	public static function add_admin_menu() {
		$page_title = self::$page_title;
		$menu_title = self::$menu_title;
		$capability = 'manage_options';
		$menu_slug  = self::$menu_slug;

		$hook = add_theme_page( $page_title, $menu_title, $capability, $menu_slug, [ __CLASS__, 'ayyash_content_callback' ] );
		add_action( 'load-' . $hook, [ __CLASS__, 'load' ] );
	}

	/**
	 * Load
	 */
	public static function load() {
		remove_all_actions( 'admin_notices' );
		add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue' ] );
	}

	/**
	 * Enqueue JS and Styles
	 */
	public static function enqueue() {

		// Admin Options.
		wp_enqueue_style( 'ayyash-admin-options', AYYASH_THEME_URI . 'assets/dist/css/admin-options.css', [], AYYASH_THEME_VERSION );

		// Plugin activate & deactivate ajax
		wp_enqueue_script( 'wp-util' );
		wp_enqueue_script( 'updates' );
		wp_enqueue_script( 'ayyash-ajax-admin-options', AYYASH_THEME_URI . 'assets/dist/js/adminOptions.js', [], AYYASH_THEME_VERSION, true );
		wp_localize_script( 'ayyash-ajax-admin-options', 'ayyashOptions',
			array(
				'admin_ajax' => admin_url( 'admin-ajax.php' ),
				'plugin_url' => admin_url( 'plugins.php' ),
				'nonce'      => wp_create_nonce( 'ayyash-admin' ),
			)
		);
	}

	/**
	 * Recommended Plugins
	 */
	public static function recommended_plugins() {
		$plugins = apply_filters(
			'ayyash_recommended_plugins',
			array(
				array(
					'slug' => 'ayyash-studio',
					'icon' => 'icon.svg',
					'name' => __( 'Ayyash Studio — The kick-start kit', 'ayyash' ),
					'file' => 'ayyash-studio/ayyash-studio.php',
				),
				array(
					'slug' => 'elementor',
					'icon' => 'icon.svg',
					'name' => __( 'Elementor', 'ayyash' ),
					'file' => 'elementor/elementor.php',
				),
				array(
					'slug' => 'absolute-addons',
					'icon' => 'icon-256x256.png',
					'name' => __( 'Absolute Addons', 'ayyash' ),
					'file' => 'absolute-addons/absolute-addons.php',
				),
				array(
					'slug' => 'absolute-thumbnail-column',
					'icon' => 'icon-256x256.png',
					'name' => __( 'Absolute Thumbnail Column', 'ayyash' ),
					'file' => 'absolute-thumbnail-column/absolute-thumbnail-column.php',
				),
				array(
					'slug' => 'regenerate-thumbnails',
					'icon' => 'icon-256x256..png',
					'name' => __( 'Regenerate Thumbnails', 'ayyash' ),
					'file' => 'regenerate-thumbnails/regenerate-thumbnails.php',
				),
			)
		);
		return $plugins;
	}

	/**
	 * Returns an Array of all the Available Google Fonts
	 *
	 * @return array
	 */
	public static function ayyash_docs_data() {
		static $docs;
		if ( null === $docs ) {
			/** @define "AYYASH_THEME_DIR" "./../../" */
			$docs = file_get_contents( AYYASH_THEME_DIR . 'assets/docs.json' );
			if ( $docs ) {
				$docs = json_decode( $docs, true );
			}
		}

		return $docs ? $docs : [];
	}

	/**
	 * Menu Callback
	 */
	public static function ayyash_content_callback() {
		self::ayyash_options_header();
		?>
		<div class="ayyash-options-wrap">
			<?php
			self::ayyash_options_content();
			self::ayyash_options_sidebar();
			?>
		</div><!-- end .ayyash-options-wrap -->
		<?php
	}

	/**
	 * Options Page header
	 */
	protected static function ayyash_options_header() {
		?>
		<div class="ayyash-options-header wrap">
			<div class="ayyash-options-header__content">
				<img
					src="data:image/svg+xml,%3Csvg%20width%3D%22177%22%20height%3D%2252%22%20viewBox%3D%220%200%20177%2052%22%20fill%3D%22none%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%0A%3Cpath%20d%3D%22M25.965%2050C39.2004%2050%2049.9299%2039.2549%2049.9299%2026C49.9299%2012.7452%2039.2004%202%2025.965%202C12.7295%202%202%2012.7452%202%2026C2%2039.2549%2012.7295%2050%2025.965%2050Z%22%20stroke%3D%22%23101318%22%20stroke-width%3D%222.23343%22%2F%3E%0A%3Cg%20clip-path%3D%22url%28%23clip0_58_2534%29%22%3E%0A%3Cpath%20d%3D%22M15.942%2026.2086C17.4401%2026.2086%2018.6546%2025.0304%2018.6546%2023.5769C18.6546%2022.1235%2017.4401%2020.9453%2015.942%2020.9453C14.4439%2020.9453%2013.2295%2022.1235%2013.2295%2023.5769C13.2295%2025.0304%2014.4439%2026.2086%2015.942%2026.2086Z%22%20fill%3D%22%23101318%22%2F%3E%0A%3Cpath%20d%3D%22M15.727%2018.0567C17.1064%2018.0567%2018.2246%2016.9719%2018.2246%2015.6337C18.2246%2014.2955%2017.1064%2013.2107%2015.727%2013.2107C14.3477%2013.2107%2013.2295%2014.2955%2013.2295%2015.6337C13.2295%2016.9719%2014.3477%2018.0567%2015.727%2018.0567Z%22%20fill%3D%22%23101318%22%2F%3E%0A%3Cpath%20d%3D%22M22.326%2016.9657C24.3274%2013.3713%2026.4776%2011.895%2029.2563%2015.8746C32.3162%2020.2553%2035.5745%2028.0699%2037.9067%2033.0764C39.1637%2035.7562%2039.2464%2039.5271%2034.946%2038.452C33.7882%2038.1632%2026.9076%2034.8095%2026.3783%2034.0071C25.7664%2033.1085%2026.246%2032.1297%2027.5361%2032.0174C28.3631%2031.9532%2034.4333%2034.4083%2035.343%2034.7613C33.3141%2031.9157%2030.2597%2027.6527%2026.1798%2021.9722C25.419%2024.5397%2024.2778%2028.3802%2022.7561%2033.4936C20.0931%2042.3995%2010.4007%2038.773%2014.3703%2031.1509C15.076%2029.8885%2015.7872%2028.6262%2016.504%2027.3639C18.4226%2027.1071%2019.8947%2025.5025%2019.8947%2023.5769C19.8947%2022.9992%2019.7623%2022.4536%2019.5308%2021.9722L20.2089%2020.7527C20.1317%2020.7634%2020.0601%2020.7688%2019.9939%2020.7688C19.018%2020.7688%2018.2241%2019.9985%2018.2241%2019.0678C18.2241%2018.1211%2019.018%2017.3508%2019.9939%2017.3508C20.9532%2017.3508%2021.7471%2018.1211%2021.7471%2019.0678C21.7471%2019.6455%2021.4329%2020.1911%2020.9367%2020.496C22.4914%2020.9292%2023.4177%2018.5062%2022.326%2016.9657Z%22%20fill%3D%22%23101318%22%2F%3E%0A%3C%2Fg%3E%0A%3Cpath%20d%3D%22M79.7789%2036.2791H70.4829V35.1311L73.2829%2033.1991L71.2949%2028.3831H62.1949L60.2069%2033.0591L64.8549%2034.8791L64.1549%2036.2791H55.4469V35.1311L58.2469%2032.9751L65.8629%2015.1671L69.5309%2013.7391L77.6229%2033.2831L80.4789%2034.8791L79.7789%2036.2791ZM66.8429%2017.5471L62.8949%2026.7871H70.6509L66.8429%2017.5471ZM95.6901%2022.5031L88.5501%2036.3071L82.4181%2043.6991L81.0741%2042.6631C82.0821%2041.5244%2082.8568%2040.5257%2083.3981%2039.6671C83.9581%2038.8084%2084.3501%2037.9404%2084.5741%2037.0631C84.8168%2036.2044%2084.9475%2035.2151%2084.9661%2034.0951L81.5781%2025.8631L78.9181%2024.2951V23.1471L84.2661%2020.7951L85.3301%2025.0231L88.6061%2033.0031L92.3301%2025.4431L88.4661%2022.7271L89.1661%2021.3271L95.6901%2021.3551V22.5031ZM112.542%2022.5031L105.402%2036.3071L99.2696%2043.6991L97.9256%2042.6631C98.9336%2041.5244%2099.7082%2040.5257%20100.25%2039.6671C100.81%2038.8084%20101.202%2037.9404%20101.426%2037.0631C101.668%2036.2044%20101.799%2035.2151%20101.818%2034.0951L98.4296%2025.8631L95.7696%2024.2951V23.1471L101.118%2020.7951L102.182%2025.0231L105.458%2033.0031L109.182%2025.4431L105.318%2022.7271L106.018%2021.3271L112.542%2021.3551V22.5031ZM128.541%2036.2791H122.801V33.8431L119.637%2036.3351C119.263%2036.4844%20118.797%2036.5591%20118.237%2036.5591C116.557%2036.5591%20115.334%2036.1484%20114.569%2035.3271C113.822%2034.4871%20113.449%2033.4791%20113.449%2032.3031C113.449%2031.6124%20113.682%2030.9404%20114.149%2030.2871C114.634%2029.6151%20115.315%2028.9804%20116.193%2028.3831H122.801V27.0951C122.801%2024.9671%20121.662%2023.9031%20119.385%2023.9031C118.19%2023.9031%20116.659%2024.5284%20114.793%2025.7791L113.953%2024.5191L117.929%2021.3551C118.713%2021.1497%20119.693%2021.0471%20120.869%2021.0471C122.717%2021.0471%20124.154%2021.4857%20125.181%2022.3631C126.207%2023.2404%20126.721%2024.5191%20126.721%2026.1991V33.8991L129.241%2034.8791L128.541%2036.2791ZM119.329%2033.6191C120.374%2033.6191%20121.531%2033.1431%20122.801%2032.1911V29.8111H118.013C117.639%2030.2591%20117.453%2030.8004%20117.453%2031.4351C117.453%2032.0511%20117.621%2032.5737%20117.957%2033.0031C118.311%2033.4137%20118.769%2033.6191%20119.329%2033.6191ZM139.032%2033.0311C139.032%2032.5831%20138.752%2032.1537%20138.192%2031.7431C137.65%2031.3324%20136.82%2030.8284%20135.7%2030.2311C134.244%2029.4471%20133.124%2028.7471%20132.34%2028.1311C131.556%2027.4964%20131.164%2026.7591%20131.164%2025.9191C131.164%2025.2471%20131.462%2024.5284%20132.06%2023.7631C132.657%2022.9977%20133.385%2022.3537%20134.244%2021.8311C135.121%2021.3084%20135.924%2021.0471%20136.652%2021.0471C137.548%2021.0471%20138.584%2021.1591%20139.76%2021.3831C140.954%2021.5884%20141.822%2021.8217%20142.364%2022.0831V26.5351H141.02C140.478%2025.4711%20139.825%2024.6404%20139.06%2024.0431C138.313%2023.4457%20137.548%2023.1471%20136.764%2023.1471C136.26%2023.1471%20135.858%2023.2871%20135.56%2023.5671C135.261%2023.8284%20135.112%2024.1551%20135.112%2024.5471C135.112%2025.0511%20135.392%2025.5177%20135.952%2025.9471C136.53%2026.3764%20137.398%2026.8991%20138.556%2027.5151C139.526%2028.0377%20140.292%2028.4857%20140.852%2028.8591C141.43%2029.2137%20141.925%2029.6337%20142.336%2030.1191C142.746%2030.5857%20142.952%2031.1084%20142.952%2031.6871C142.952%2032.3777%20142.634%2033.1057%20142%2033.8711C141.384%2034.6177%20140.628%2035.2524%20139.732%2035.7751C138.854%2036.2977%20138.042%2036.5591%20137.296%2036.5591C136.325%2036.5591%20135.177%2036.4564%20133.852%2036.2511C132.545%2036.0271%20131.612%2035.7844%20131.052%2035.5231V30.9871H132.396C133.012%2032.0884%20133.749%2032.9471%20134.608%2033.5631C135.485%2034.1604%20136.362%2034.4591%20137.24%2034.4591C137.762%2034.4591%20138.192%2034.3191%20138.528%2034.0391C138.864%2033.7591%20139.032%2033.4231%20139.032%2033.0311ZM162.344%2036.2791H156.604V27.0391C156.604%2026.0497%20156.455%2025.3684%20156.156%2024.9951C155.857%2024.6031%20155.335%2024.4071%20154.588%2024.4071C154.028%2024.4071%20153.468%2024.5004%20152.908%2024.6871C152.348%2024.8737%20151.713%2025.2284%20151.004%2025.7511V33.9551L153.524%2034.8791L152.824%2036.2791H144.592V35.1311L147.084%2033.1151V16.6511L144.592%2014.9431V13.7951L149.66%2011.8351H151.004V24.0711L154.56%2021.2711C154.952%2021.1217%20155.493%2021.0471%20156.184%2021.0471C157.509%2021.0471%20158.564%2021.4391%20159.348%2022.2231C160.132%2022.9884%20160.524%2024.0337%20160.524%2025.3591V33.9551L163.044%2034.8791L162.344%2036.2791Z%22%20fill%3D%22%23101318%22%2F%3E%0A%3Cdefs%3E%0A%3CclipPath%20id%3D%22clip0_58_2534%22%3E%0A%3Crect%20width%3D%2225.6369%22%20height%3D%2225.6744%22%20fill%3D%22white%22%20transform%3D%22translate%2813.1465%2013.1628%29%22%2F%3E%0A%3C%2FclipPath%3E%0A%3C%2Fdefs%3E%0A%3C%2Fsvg%3E%0A"
					alt="Ayyash">
				<div class="wp-heading-inline"><?php echo 'v' . esc_html( AYYASH_THEME_VERSION ); ?></div>
			</div><!-- end .ayyash-options-header__content -->
		</div><!-- end .ayyash-options-header -->
		<?php
	}

	/**
	 * Options Page Content
	 */
	public static function ayyash_options_content() {
		?>
		<div class="ayyash-options-content">
			<div class="ayyash-options-tabs" role="tablist">
				<?php
				self::get_tab_menu();
				self::get_tab_content();
				?>
			</div><!-- end .ayyash-options-tabs -->
		</div><!-- end .ayyash-options-content -->
		<?php
	}

	/**
	 * Content Tab Menu
	 */
	protected static function get_tab_menu() {
		$tabs = self::get_registered_tabs();
		?>
		<div class="ayyash-options-tabs__nav">
			<?php
			foreach ( $tabs as $slug => $tab ) {
				printf(
				'<a href="#%1$s" aria-controls="tab-content-%1$s" id="tab-nav-%1$s" class="ayyash-options-tabs__nav-item ayyash-options-tabs__nav-item--%1$s" role="tab"><i class="%3$s"></i> %2$s</a>', esc_attr( $slug ), esc_html( $tab['menu_title'] ), esc_html( $tab['icon'] )
				);
			}
			?>
		</div><!-- end .ayyash-options-tabs__nav -->
		<?php
	}

	/**
	 * Admin Tabs
	 *
	 * @return mixed|void
	 */
	protected static function get_registered_tabs() {
		$tabs = apply_filters(
			'ayyash_options_tabs',
			[
				'recommended-plugins' => [
					'menu_title' => esc_html__( 'Recommended Plugins', 'ayyash' ),
					'icon'       => '',
					'position'   => -1,
					'callback'   => [ __CLASS__, 'get_recommended_plugin' ],

				],
				'theme-documentation' => [
					'menu_title' => esc_html__( 'Documentation', 'ayyash' ),
					'icon'       => '',
					'position'   => 20,
					'callback'   => [ __CLASS__, 'get_documentation' ],
				],
			]
		);

		uasort( $tabs, [ __CLASS__, 'ayyash_options_uasort_position' ] );

		return $tabs;

	}

	public static function get_recommended_plugin() {
		get_template_part( 'inc/settings/content', 'recommended-plugins' );
	}

	public static function get_documentation() {
		get_template_part( 'inc/settings/content', 'theme-documentation' );

	}

	/**
	 * Tab Content
	 */
	protected static function get_tab_content() {
		$tabs = self::get_registered_tabs();
		?>
		<div class="ayyash-options-tabs__content">
			<?php foreach ( $tabs as $slug => $tab ) { ?>
			<div class="ayyash-options-tabs__content-item" id="<?php echo esc_attr( $slug ); ?>" role="tabpanel" aria-labelledby="tab-nav-<?php echo esc_attr( $slug ); ?>">
				<div class="ayyash-options-panel <?php echo esc_attr( $slug ); ?>">
					<div class="ayyash-options-panel__body">
						<div class="ayyash-options-panel__<?php echo esc_attr( $slug ); ?>">
						<?php
						if ( ! empty( $tab['callback'] ) && is_callable( $tab['callback'] ) ) {
							call_user_func( $tab['callback'], self::get_instance(), $tab );
						}
						?>
						</div>
					</div><!-- end .ayyash-options-panel__body -->
				</div><!-- end .ayyash-options-panel -->
			</div><!-- end .ayyash-options-tabs__content-item -->
			<?php } ?>
		</div><!-- end .ayyash-options-tabs__content -->
		<?php
	}

	/**
	 * Options Page Sidebar
	 */
	protected static function ayyash_options_sidebar() {
		?>
		<div class="ayyash-options-sidebar">
		</div><!-- end .ayyash-options-sidebar -->
		<?php
	}

	/**
	 * Sorting Tab Positions
	 *
	 * @param $a
	 * @param $b
	 *
	 * @return int
	 */
	public static function ayyash_options_uasort_position( $a, $b ) {

		if ( $a['position'] == $b['position'] ) {
			return 0;
		}

		return ( $a['position'] < $b['position'] ) ? - 1 : 1;
	}

	/**
	 * Ajax Install and Activating Plugin
	 *
	 * @return void
	 */
	public static function ajax_activate_plugin() {
		check_ajax_referer( 'ayyash-admin', 'nonce' );

		$mainfile = ( isset( $_POST['mainfile'] ) ) ? sanitize_text_field( wp_unslash( $_POST['mainfile'] ) ) : false;

		if ( $mainfile && current_user_can( 'install_plugins' ) ) {
			$activate = activate_plugin( $mainfile, '', false, true );
			if ( is_wp_error( $activate ) ) {
				wp_send_json_error( $activate->get_error_message() );
			}
			wp_send_json_success( __( 'Plugin Activated Successfully!', 'ayyash' ) );
		}
		wp_send_json_error();
	}

	/**
	 * Ajax Deactivating Plugin
	 *
	 * @return void
	 */
	public static function ajax_deactivate_plugin() {
		check_ajax_referer( 'ayyash-admin', 'nonce' );

		$mainfile = ( isset( $_POST['mainfile'] ) ) ? sanitize_text_field( wp_unslash( $_POST['mainfile'] ) ) : false;

		if ( $mainfile && current_user_can( 'install_plugins' ) ) {
			deactivate_plugins( $mainfile, false, false );
			if ( ! is_plugin_inactive( $mainfile ) ) {
				wp_send_json_error( __( 'Cannot deactivate this plugin!', 'ayyash' ) );
			}
			wp_send_json_success( __( 'Plugin Deactivated Successfully!', 'ayyash' ) );
		}
		wp_send_json_error();
	}

	/**
	 * Return plugin is inactive , not installed and update.
	 *
	 * @return array
	 */
	public static function plugin_status() {
		$recommended_plugins = self::recommended_plugins();
		$installed_plugins   = apply_filters( 'all_plugins', get_plugins() );

		$plugin_status = [];

		$plugin_status['msg']['not_installed'] = _n_noop(
			'This theme recommends the following plugin: ',
			'This theme recommends the following plugins: ',
			'ayyash'
		);
		$plugin_status['msg']['not_active']    = _n_noop(
			'The following recommended plugin is currently inactive: ',
			'The following recommended plugins are currently inactive: ',
			'ayyash'
		);
		$plugin_status['msg']['updates']       = _n_noop(
			'There is an update available for: ',
			'There are updates available for the following plugins: ',
			'ayyash'
		);

		foreach ( $recommended_plugins as $plugin ) {
			$is_installed = isset( $installed_plugins[ $plugin['file'] ] );
			$is_active    = is_plugin_active( $plugin['file'] );

			if ( ! $is_installed ) {
				$plugin_status['not_installed'][ $plugin['name'] ] = sprintf('<em><a href="%s" class="thickbox">%s</a></em>', esc_url( admin_url('plugin-install.php?tab=plugin-information&plugin=' . $plugin['slug'] . '&TB_iframe=true&width=772&height=239"') ), esc_html( $plugin['name'] ) );
			}

			if ( ! $is_active && $is_installed ) {
				$plugin_status['not_active'][ $plugin['name'] ] = sprintf('<em><a href="%s" class="thickbox">%s</a></em>', esc_url( admin_url('plugin-install.php?tab=plugin-information&plugin=' . $plugin['slug'] . '&TB_iframe=true&width=772&height=239"') ), esc_html( $plugin['name'] ) );
			}

			$repo_updates = get_site_transient( 'update_plugins' );
			if ( isset( $repo_updates->response[ $plugin['file'] ]->new_version ) ) {
				$plugin_status['updates'][ $plugin['name'] ] = sprintf('<em><a href="%s" class="thickbox">%s</a></em>', esc_url( admin_url('plugin-install.php?tab=plugin-information&plugin=' . $plugin['slug'] . '&TB_iframe=true&width=772&height=239"') ), esc_html( $plugin['name'] ) );
			}
		}

		return $plugin_status;
	}

	/**
	 * Delete dismissable nag option when theme is switched.
	 *
	 * This ensures that the user(s) is/are again reminded via nag of required
	 * and/or recommended plugins if they re-activate the theme.
	*/
	public function update_dismiss() {
		delete_metadata( 'user', null, 'ayyash_required_plugin_dismiss_notice', null, true );
	}

	/**
	 * Register dismissal of admin notices.
	 *
	 * Acts on the dismiss link in the admin nag messages.
	 * If clicked, the admin notice disappears and will no longer be visible to this user.
	 */
	public function dismiss() {
		if ( isset( $_GET['ayyash-required-dismiss'] ) && check_admin_referer( 'ayyash-required-dismiss-' . get_current_user_id() ) ) {
			update_user_meta( get_current_user_id(), 'ayyash_required_plugin_dismiss_notice', 1 );
		}
	}

	/**
	 * Comma Seperated String generator
	 *
	 * @param $array
	 *
	 * @return string
	 */
	public static function notice_str_generator( $array = [] ) {
		return join(' and ', array_filter( array_merge( array( join(', ', array_slice( $array, 0, -1) ) ), array_slice( $array, -1) ), 'strlen') );
	}

	/**
	 * Ayyash Admin Notice
	 *
	 * @return void
	 */
	public static function ayyash_author_admin_notice() {
		$plugin_status = self::plugin_status();

		//Check user compatibility for notices.
		if ( ! current_user_can( 'install_plugins' ) && ! current_user_can( 'activate_plugins' ) && ! current_user_can( 'update_plugins' ) ) {
			return;
		}

		//Check User is dismissed the notice.
		if ( 1 === intval( get_user_meta( get_current_user_id(), 'ayyash_required_plugin_dismiss_notice', true ) ) ) {
			return;
		}

		//Check there is any plugin status
		if ( 0 === count( $plugin_status ) ) {
			return;
		}

		$styles = 'display: block; margin: 0.5em 0.5em 0 0; clear: both;';
		?>
		<div class="notice notice-warning is-dismissible">
			<p>
				<strong>
					<?php
					if ( isset ( $plugin_status['not_installed'] ) ) { ?>
					<span style="<?php echo esc_attr( $styles ); ?>">
						<?php
						$msg = translate_nooped_plural( $plugin_status['msg']['not_installed'], count( $plugin_status['not_installed'] ), 'ayyash' );
						echo esc_html( $msg ) . wp_kses_post( self::notice_str_generator( $plugin_status['not_installed'] ) );
						?>
					</span>
					<?php }

					if ( isset ( $plugin_status['not_active'] ) ) { ?>
					<span style="<?php echo esc_attr( $styles ); ?>">
						<?php
						$msg = translate_nooped_plural( $plugin_status['msg']['not_active'], count( $plugin_status['not_active'] ), 'ayyash' );
						echo esc_html( $msg ) . wp_kses_post( self::notice_str_generator( $plugin_status['not_active'] ) );
						?>
					</span>
					<?php }

					if ( isset ( $plugin_status['updates'] ) ) { ?>
						<span style="<?php echo esc_attr( $styles ); ?>">
						<?php
						$msg = translate_nooped_plural( $plugin_status['msg']['updates'], count( $plugin_status['updates'] ), 'ayyash' );
						echo esc_html( $msg ) . wp_kses_post( self::notice_str_generator( $plugin_status['updates'] ) );
						?>
					</span>
					<?php } ?>
					<span style="<?php echo esc_attr( $styles ); ?>">
						<a href="<?php echo esc_url( wp_nonce_url( add_query_arg( 'ayyash-required-dismiss', 'ayyash_required_plugin_dismiss_notice' ), 'ayyash-required-dismiss-' . get_current_user_id() ) ); ?>" class="dismiss-notice" target="_parent"><?php esc_html_e( 'Dismiss this notice', 'ayyash' ); ?></a>
					</span>
				</strong>
			</p>

		</div>
		<?php
	}
}

Ayyash_Admin_Options::get_instance();
