<?php
/**
 * Customizer Helper Functions.
 *
 * @package Ayyash
 * @author ThemeRox
 */

if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	die();
}

if ( ! function_exists( 'ayyash_get_default' ) ) :

	/**
	 * Get default value for a theme option.
	 *
	 * @param string $name mod name.
	 *
	 * @return string String with default value.
	 */
	function ayyash_get_default( $name ) {
		global $ayyash_defaults;

		if ( $ayyash_defaults && $name && isset( $ayyash_defaults[ $name ] ) ) {
			return $ayyash_defaults[ $name ];
		}

		return '';
	}
endif;

if ( ! function_exists( 'ayyash_get_choices' ) ) :

	/**
	 * Get all Choices/Options for a dropdown.
	 *
	 * @param string $name mod name.
	 *
	 * @return array Array with all options.
	 */
	function ayyash_get_choices( $name ) {
		global $ayyash_defaults;

		if ( array_key_exists( $name, $ayyash_defaults['choices'] ) ) {
			return $ayyash_defaults['choices'][ $name ];
		}

		return [];
	}
endif;

if ( ! function_exists( 'ayyash_get_default_mod' ) ) :

	/**
	 * Get the mod value or default value if mod is not set.
	 *
	 * @param string $name mod name.
	 * @param array $mods mods to check before check default mods.
	 *
	 * @return string Mod value.
	 */
	function ayyash_get_default_mod( $name, $mods ) {
		global $ayyash_defaults;

		if ( array_key_exists( $name, $mods ) && '' !== $mods[ $name ] ) {
			return $mods[ $name ];
		} elseif ( array_key_exists( $name, $ayyash_defaults ) ) {
			return $ayyash_defaults[ $name ];
		}

		return '';
	}
endif;

if ( ! function_exists( 'ayyash_get_default_mods' ) ) :

	/**
	 * Get all Default Ayyash Mod Options and Values.
	 *
	 * @return array    Array with key as option names and value as option values.
	 */
	function ayyash_get_default_mods() {
		global $ayyash_defaults;

		return $ayyash_defaults;
	}
endif;

if ( ! function_exists( 'ayyash_get_mods' ) ) :

	/**
	 * Returns all mods set by the user, returns the default values if any mod is not set.
	 *
	 * @return array
	 */
	function ayyash_get_mods() {
		/**
		 * @var false|array $mods
		 */
		$mods = get_theme_mods();
		if ( ! $mods ) {
			$mods = [];
		}

		return array_merge( ayyash_get_default_mods(), $mods );
	}
endif;

if ( ! function_exists( 'ayyash_get_mod' ) ) :

	/**
	 * Wrapper for WordPress 'get_theme_mod' function.
	 *
	 * @param string $name mod name.
	 *
	 * @return mixed The string with the mod value.
	 */
	function ayyash_get_mod( $name ) {
		return get_theme_mod( $name, ayyash_get_default( $name ) );
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_float' ) ) :

	/**
	 * Sanitize function for WP_Customize setting to sanitize float values.
	 *
	 * @param mixed $value value to sanitize.
	 *
	 * @return float
	 */
	function ayyash_sanitize_float( $value ) {
		return floatval( $value );
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_choice' ) ) :

	/**
	 * Sanitize function for WP_Customize setting to sanitize select.
	 *
	 * @param string|int $value choice value.
	 * @param string|object $setting mod name.
	 *
	 * @return string|int
	 */
	function ayyash_sanitize_choice( $value, $setting ) {
		if ( is_object( $setting ) && isset( $setting->id ) ) {
			$setting = $setting->id;
		}

		$options = ayyash_get_choices( $setting );

		if ( ! in_array( $value, array_keys( $options ) ) ) { // phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			$value = ayyash_get_default( $setting );
		}

		return $value;
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_font_family' ) ) :

	/**
	 * Sanitize function for WP_Customize setting to sanitize Font Family.
	 *
	 * @param string $value font_family.
	 * @param string|object $setting font_family mod name.
	 *
	 * @return string
	 */
	function ayyash_sanitize_font_family( $value, $setting ) {

		if ( is_object( $setting ) && isset( $setting->id ) ) {
			$setting = $setting->id;
		}

		if ( ! is_string( $value ) || '' === $value ) {
			return '';
		} elseif ( ! in_array( $value, array_keys( ayyash_get_all_fonts( false ) ) ) ) { // phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			$value = ayyash_get_default( $setting );
		}

		return $value;
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_font_variant' ) ) :

	/**
	 * Sanitize function for WP_Customize setting to sanitize Font Family Variant.
	 *
	 * @param string $value Font Variant.
	 *
	 * @return string
	 */
	function ayyash_sanitize_font_variant( $value ) {

		$options = [
			'100',
			'100italic',
			'200',
			'200italic',
			'300',
			'300italic',
			'500',
			'500italic',
			'600',
			'600italic',
			'700',
			'700italic',
			'800',
			'800italic',
			'900',
			'900italic',
			'italic',
			'regular',
			'thin',
			'thin-italic',
			'bold',
			'bold-italic',
			'medium',
			'medium-italic',
			'extra-light',
			'extra-light-italic',
			'light',
			'light-italic',
			'serif',
			'serif-italic',
		];

		if ( ! is_string( $value ) || '' === $value ) {
			return 'regular';
		} elseif ( in_array( $value, $options, true ) ) {
			return $value;
		}

		return 'regular';
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_font_variant_array' ) ) {
	function ayyash_sanitize_font_variant_array( $variants = [] ) {

		if ( ! is_array( $variants ) ) {
			return ayyash_sanitize_font_variant( $variants );
		}

		foreach ( $variants as &$variant ) {
			$variant = ayyash_sanitize_font_variant( $variant );
		}

		return $variants;
	}
}

if ( ! function_exists( 'ayyash_sanitize_array_of_strings' ) ) {

	function ayyash_sanitize_array_of_strings( $array ) {
		return array_map( 'sanitize_text_field', $array );
	}
}


if ( ! function_exists( 'ayyash_sanitize_font_text_transform' ) ) :

	/**
	 * Sanitize function for WP_Customize setting to sanitize Font Text Transform.
	 *
	 * @param string $value Text transform.
	 *
	 * @return string
	 */
	function ayyash_sanitize_font_text_transform( $value ) {

		$options = [ 'none', 'uppercase', 'lowercase' ];

		if ( ! is_string( $value ) || '' === $value ) {
			return 'none';
		} elseif ( in_array( $value, $options, true ) ) {
			return $value;
		}

		return 'none';
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_font_subsets' ) ) :

	/**
	 * Sanitize function for WP_Customize setting to sanitize Font Subsets.
	 *
	 * @param string|string[] $values Font subsets.
	 *
	 * @return string[]
	 */
	function ayyash_sanitize_font_subsets( $values ) {

		$multi_values = ! is_array( $values ) ? explode( ',', $values ) : $values;

		return ! empty( $multi_values ) ? array_map( 'sanitize_text_field', $multi_values ) : [];
	}
endif;

if ( ! function_exists( 'ayyash_get_color_mods' ) ) :

	/**
	 * Determine if a mod is a color mod
	 *
	 * @param string $mod mod name.
	 *
	 * @return bool
	 */
	function ayyash_get_color_mods( $mod ) {
		return 0 === strpos( $mod, 'colors_' );
	}
endif;

if ( ! function_exists( 'ayyash_get_font_mods' ) ) :

	/**
	 * Determine if a mod is a typography mod
	 *
	 * @param string $mod mod name.
	 *
	 * @return bool
	 */
	function ayyash_get_font_mods( $mod ) {
		return 0 === strpos( $mod, 'typography_' );
	}
endif;

if ( ! function_exists( 'ayyash_is_font_family' ) ) :

	/**
	 * Checks if a given mod is Font Family
	 *
	 * @param string $mod mod name.
	 *
	 * @return bool
	 */
	function ayyash_is_font_family( $mod ) {
		return ayyash_string_ends_with( $mod, 'font_family' ) || ayyash_string_ends_with( $mod, 'font_variant' );
	}
endif;

if ( ! function_exists( 'ayyash_string_ends_with' ) ) :

	/**
	 * Determine if a string ends with a particulr value
	 *
	 * @param string $whole the full string.
	 * @param string $end part to check.
	 *
	 * @return bool
	 */
	function ayyash_string_ends_with( $whole, $end ) {
		return strpos( $whole, $end ) !== false && strpos( $whole, $end, strlen( $whole ) - strlen( $end ) ) !== false;
	}
endif;

if ( ! function_exists( 'ayyash_font_settings' ) ) :

	/**
	 * Prints the Font styles for a given section
	 *
	 * @param string $section mod section id.
	 *
	 * @return void CSS Font styles.
	 */
	function ayyash_font_settings( $section ) {
		$global_line_height = ayyash_get_mod( 'typography_global_line_height' );
		if ( ! $global_line_height ) {
			$global_line_height = 1.5;
		}
		$line_height  = ! ayyash_get_mod( $section . '_line_height' ) ? $global_line_height : ayyash_get_mod( $section . '_line_height' );
		$font_variant = ayyash_get_mod( $section . '_font_variant' );
		$font_weight  = 'regular' === $font_variant ? 'normal' : preg_replace( '/[^0-9]/', '', $font_variant );
		?>
		font:
		<?php
		printf(
			'%spx/%s "%s", -apple-system,BlinkMacSystemFont,"Segoe UI","Helvetica Neue",Arial,"Noto Sans",sans-serif,"Apple Color Emoji","Segoe UI Emoji","Segoe UI Symbol","Noto Color Emoji"',
			esc_attr( ayyash_get_mod( $section . '_font_size' ) ),
			esc_attr( $line_height ), // @phpstan-ignore-line
			esc_attr( ayyash_get_mod( $section . '_font_family' ) )
		);
		?>
		;
		font-weight: <?php echo esc_attr( $font_weight ); // @phpstan-ignore-line ?>;
		font-style: <?php echo strpos( $font_variant, 'italic' ) !== false ? 'italic' : 'normal'; ?>;
		text-transform: <?php echo esc_attr( ayyash_get_mod( $section . '_text_transform' ) ); ?>;
		letter-spacing: <?php echo esc_attr( ayyash_get_mod( $section . '_letter_spacing' ) ); ?>px;
		word-spacing: <?php echo esc_attr( ayyash_get_mod( $section . '_word_spacing' ) ); ?>px;
		<?php
	}
endif;

if ( ! function_exists( 'ayyash_sanitize_social_profiles' ) ) {
	/**
	 * Sanitize profiles control data.
	 *
	 * @param array|string $profiles JSON String or array of profile.
	 *
	 * @return array
	 */
	function ayyash_sanitize_social_profiles( $profiles ) {
		$profiles = is_array( $profiles ) ? $profiles : json_decode( $profiles, true );
		$temp     = [];
		foreach ( $profiles as $profile ) {
			$temp[] = ayyash_sanitize_social_profile( $profile );
		}

		return array_filter( $temp );
	}
}

if ( ! function_exists( 'ayyash_sanitize_social_profile' ) ) {
	/**
	 * Sanitize single profile data.
	 *
	 * @param array $args Profile data.
	 *
	 * @return array|bool
	 */
	function ayyash_sanitize_social_profile( $args ) {
		$args    = wp_parse_args(
			$args,
			[
				'label' => '',
				'url'   => '',
				'icon'  => '',
			]
		);
		$profile = [
			'label' => sanitize_text_field( $args['label'] ),
			'url'   => esc_url_raw( $args['url'] ),
			'icon'  => sanitize_text_field( $args['icon'] ),
		];

		if ( $profile['label'] && $profile['url'] && $profile['icon'] ) {
			return $profile;
		}

		return false;
	}
}

if ( ! function_exists( 'ayyash_theme_update_version_1_3_0' ) ) {

	/**
	 * Customizer upgrade for version greater than 1.2.5
	 *
	 * @return void
	 */
	function ayyash_theme_update_version_1_3_0() {
		$current_version = wp_get_theme()->get('Version');

		// Check if version is greater than 1.2.5
		if ( $current_version < '1.2.5' ) {
			return;
		}

		// Header Menu Style
		$menu_style = get_theme_mod( 'layout_header_menu_style' );
		if ( 'header-accent' === $menu_style ) {
			set_theme_mod('layout_header_menu_style', 'menu-primary');
			set_theme_mod('layout_header_menu_hover_style', 'menu-hover-secondary');
		} elseif ( 'header-white' === $menu_style ) {
			set_theme_mod('layout_header_menu_style', 'menu-white');
			set_theme_mod('layout_header_menu_hover_style', 'menu-hover-primary');
		} elseif ( 'default' === $menu_style ) {
			set_theme_mod('layout_header_menu_style', 'menu-text');
			set_theme_mod('layout_header_menu_hover_style', 'menu-hover-primary');
		}

		// Menu Left Right padding
		$menu_left_right_padding = get_theme_mod( 'layout_header_menu_lr_padding' );
		if ( $menu_left_right_padding ) {
			set_theme_mod('layout_header_menu_left_padding', $menu_left_right_padding );
			set_theme_mod('layout_header_menu_right_padding', $menu_left_right_padding );
		}

		// Submenu Top Bottom padding
		$submenu_top_bottom_padding = get_theme_mod( 'layout_header_submenu_tb_padding' );
		if ( $submenu_top_bottom_padding ) {
			set_theme_mod('layout_header_submenu_top_padding', $submenu_top_bottom_padding );
			set_theme_mod('layout_header_submenu_bottom_padding', $submenu_top_bottom_padding );
		}

		// Submenu Left Right padding
		$submenu_left_right_padding = get_theme_mod( 'layout_header_submenu_lr_padding' );
		if ( $submenu_left_right_padding ) {
			remove_theme_mod('layout_header_submenu_lr_padding');
			set_theme_mod('layout_header_submenu_left_padding', $submenu_left_right_padding );
			set_theme_mod('layout_header_submenu_right_padding', $submenu_left_right_padding );
		}

		// Header Button Top Bottom padding
		$header_btn_top_bottom_padding = get_theme_mod( 'layout_header_button_tb_padding' );
		if ( $header_btn_top_bottom_padding ) {
			remove_theme_mod('layout_header_button_tb_padding');
			set_theme_mod('layout_header_button_top_padding', $header_btn_top_bottom_padding );
			set_theme_mod('layout_header_button_bottom_padding', $header_btn_top_bottom_padding );
		}

		// Header Button Left Right padding
		$header_btn_left_right_padding = get_theme_mod( 'layout_header_button_lr_padding' );
		if ( $header_btn_left_right_padding ) {
			remove_theme_mod('layout_header_button_lr_padding');
			set_theme_mod('layout_header_button_left_padding', $header_btn_left_right_padding );
			set_theme_mod('layout_header_button_right_padding', $header_btn_left_right_padding );
		}

		// Footer Credits Top Bottom Margin
		$footer_credit_top_bottom_padding = get_theme_mod( 'layout_footer_credits_tb_margin' );
		if ( $footer_credit_top_bottom_padding ) {
			remove_theme_mod( 'layout_footer_credits_tb_margin' );
			set_theme_mod('layout_footer_credits_top_margin', $footer_credit_top_bottom_padding );
			set_theme_mod('layout_footer_credits_bottom_margin', $footer_credit_top_bottom_padding );
		}

	}
	add_action('after_setup_theme', 'ayyash_theme_update_version_1_3_0');
}

/**
 * Determines whether a plugin is active.
 *
 * @param string $plugin Path to the plugin file relative to the plugins directory.
 * @return bool True, if in the active plugins list. False, not in the list.
 */
function ayyash_is_plugin_active( $plugin ) {

	if ( in_array( $plugin, (array) get_option( 'active_plugins', array() ), true ) ) {
		return true;
	}

	if ( ! is_multisite() ) {
		return false;
	}

	$plugins = get_site_option( 'active_sitewide_plugins' );
	if ( isset( $plugins[ $plugin ] ) ) {
		return true;
	}

	return false;
}
