<?php
/**
 * WC Hook Helpers.
 *
 * @package Ayyash
 * @author ThemeRox
 */

if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	die();
}

if ( ! function_exists( 'ayyash_woocommerce_init' ) ) :
	/**
	 * Add theme support and remove default action hooks so we can replace them with our own.
	 *
	 * @return void
	 */
	function ayyash_woocommerce_init() {
		add_theme_support( 'woocommerce' );
		/**
		----------------------------------------------------------------------
		 * Enable Product Zoom
		 * ----------------------------------------------------------------------*/
		if ( ayyash_get_mod( 'woocommerce_general_enable_product_zoom' ) ) {
			add_theme_support( 'wc-product-gallery-zoom' );
		} else {
			remove_theme_support( 'wc-product-gallery-zoom' );
		}

		/**
		----------------------------------------------------------------------
		 * Enable Product Lightbox
		 *----------------------------------------------------------------------*/
		if ( ayyash_get_mod( 'woocommerce_general_enable_product_lightbox' ) ) {
			add_theme_support( 'wc-product-gallery-lightbox' );
		} else {
			remove_theme_support( 'wc-product-gallery-lightbox' );
		}

		/**
		----------------------------------------------------------------------
		 * Enable Product Gallery Slider
		 * ----------------------------------------------------------------------*/
		if ( ayyash_get_mod( 'woocommerce_general_enable_product_gallery_slider' ) ) {
			add_theme_support( 'wc-product-gallery-slider' );
		} else {
			remove_theme_support( 'wc-product-gallery-slider' );
		}

		// remove default content wrapper.
		remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper' );
		remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end' );

	}
endif;

if ( ! function_exists( 'ayyash_loop_columns' ) ) {
	/**
	 * Override theme default specification for product # per row
	 *
	 * @param int $columns
	 *
	 * @return int
	 */
	function ayyash_loop_columns( $columns ) {
		$loop_columns = absint( ayyash_get_mod( 'woocommerce_shop_archive_column' ) );

		if ( $loop_columns ) {
			return $loop_columns;
		}

		return $columns;
	}
}

if ( ! function_exists( 'ayyash_woocommerce_before_main_content' ) ) :
	/**
	 * Add markup layout before main WooCommerce content.
	 *
	 * @return void
	 */
	function ayyash_woocommerce_before_main_content() {
		$classes = apply_filters( 'ayyash_content_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );
		$view    = ayyash_get_current_screen();
		?>
		<div  id="content">
			<?php ayyash_title_bar( $view ); ?>
			<div <?php ayyash_get_content_class(); ?>>
				<div class="<?php echo esc_attr( $classes ); ?>">
					<div class="row">
						<?php ayyash_try_sidebar( $view, 'left' ); ?>
						<div id="primary" <?php ayyash_main_class(); ?>>
							<main id="main" class="site-main" role="main">
		<?php
	}
endif;

if ( ! function_exists( 'ayyash_woocommerce_after_main_content' ) ) :
	/**
	 * End of the main content wrapper
	 *
	 * @return void
	 */
	function ayyash_woocommerce_after_main_content() {
		// End content wrapper.
		?>
							</main><!-- #main -->
						</div><!-- #primary -->
						<?php ayyash_try_sidebar( ayyash_get_current_screen(), 'right' ); ?>
					</div><!-- .row -->
				</div><!-- .container -->
			</div>
		</div><!-- #content -->
		<?php

	}
endif;

if ( ! function_exists( 'ayyash_loop_shop_per_page' ) ) {
	/**
	 * Show number of products Per Page
	 *
	 * @param int $product_number
	 *
	 * @return int
	 */
	function ayyash_loop_shop_per_page( $product_number ) {
		$per_page = (int) ayyash_get_mod( 'woocommerce_shop_archive_per_page' );
		if ( $per_page < 0 || $per_page > 0 ) {
			return $per_page;
		}

		return $product_number;
	}
}

if ( ! function_exists( 'ayyash_wc_hide_page_title' ) ) {
	/**
	 * Show Title
	 *
	 * @return bool
	 */
	function ayyash_wc_hide_page_title() {
		return ayyash_get_mod( 'woocommerce_shop_archive_title' ) == 1;
	}
}

if ( ! function_exists( 'ayyash_remove_wc_breadcrumbs' ) ) {
	/**
	 * Remove Shop Page Breadcrumbs as we have default
	 *
	 * @return void
	 */
	function ayyash_remove_wc_breadcrumbs() {
		remove_action( 'woocommerce_before_main_content', 'woocommerce_breadcrumb', 20 );
	}
}

if ( ! function_exists( 'ayyash_wc_remove_result_count' ) ) {
	/**
	 * Remove Shop Filter
	 *
	 * @return void
	 */
	function ayyash_wc_remove_result_count() {
		if ( ayyash_get_mod( 'woocommerce_shop_archive_result_count' ) == 1 ) {
			woocommerce_result_count();
		}
	}
}

if ( ! function_exists( 'ayyash_wc_remove_archive_short' ) ) {
	/**
	 * Remove Shop Page Filter
	 *
	 * @return void
	 */
	function ayyash_wc_remove_archive_short() {
		if ( ayyash_get_mod( 'woocommerce_shop_archive_sort' ) == 1 ) {
			woocommerce_catalog_ordering();
		}
	}
}

if ( ! function_exists( 'ayyash_shop_header_open' ) ) {
	/**
	 * Ayyash Shop header open
	 *
	 * @return void
	 */
	function ayyash_shop_header_open() {
		echo '<div class="product-header">';
	}
}

if ( ! function_exists( 'ayyash_get_product_discount' ) ) {
	/**
	 * Display Product Discount Tag.
	 *
	 * @param int  $product_id  Product ID.
	 * @param int  $precision   Numeric round precision.
	 *
	 * @return void
	 */
	function ayyash_get_product_discount( $product_id = null, $precision = 0 ) {

		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_discount' ) ) {
			return;
		}

		$product_id = ! $product_id ? get_the_ID() : $product_id;
		$product    = wc_get_product( $product_id );
		$discount   = 0;

		if ( $product instanceof WC_Product ) {
			$sale_price    = $product->get_sale_price();
			$regular_price = $product->get_regular_price();

			if ( ! empty( $sale_price ) && $regular_price > $sale_price ) {
				$discount = ( ( (float) $regular_price - (float) $sale_price ) / (float) $regular_price ) * 100;
				$discount = round( $discount, $precision );
			}
		}

		if ( class_exists( 'Ayyash_Pro' ) ) {
			$product_label_type = ayyash_get_mod('woocommerce_shop_archive_label_type');
		} else {
			$product_label_type = '';
		}

		if ( $discount > 0 ) {
			printf(
				'<span class="product-badge product-discount %s">%2$s%% OFF</span>',
				esc_attr( $product_label_type ? $product_label_type : 'fill' ),
				(float) $discount
			);
		}
	}
}

if ( ! function_exists( 'ayyash_get_featured_badge' ) ) {
	/**
	 * Display Product Discount Tag.
	 *
	 * @param int  $product_id  Product ID.
	 *
	 * @return void
	 */
	function ayyash_get_featured_badge( $product_id = null ) {

		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_featured' ) ) {
			return;
		}

		$product_id  = ! $product_id ? get_the_ID() : $product_id;
		$product     = wc_get_product( $product_id );
		$is_featured = false;

		if ( $product instanceof WC_Product ) {
			$is_featured = $product->is_featured();
		}

		if ( class_exists( 'Ayyash_Pro' ) ) {
			$product_label_type = ayyash_get_mod('woocommerce_shop_archive_label_type');
		} else {
			$product_label_type = '';
		}

		if ( $is_featured ) {
			printf(
				'<span class="product-badge product-featured %s">Featured</span>',
				esc_attr( $product_label_type ? $product_label_type : 'fill' )
			);
		}
	}
}

if ( ! function_exists( 'ayyash_get_product_badge' ) ) {

	/**
	 * Get product badge.
	 *
	 * @param int   $product_id Product ID.
	 * @param array $check check.
	 *
	 * @return void
	 */
	function ayyash_get_product_badge( $product_id = null, $check = array() ) {

		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_product_badge' ) ) {
			return;
		}

		$product_id = ! $product_id ? get_the_ID() : $product_id;
		$product    = wc_get_product( $product_id );
		$check      = array( 'hot', 'sale' ); // @TODO check generating badge from product tag is necessary.

		if ( ! $product ) {
			return;
		}

		$product_tags = array_map(
			function ( $tag_id ) use ( $check ) {
				$term = get_term_by( 'id', $tag_id, 'product_tag' );

				// @phpstan-ignore-next-line
				if ( ! $term || is_wp_error( $term ) ) {
					return null;
				}

				/**
				 * @var WP_Term $term
				 */
				if ( ! empty( $check ) && ! in_array( $term->slug, $check ) ) { // phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
					return null;
				}

				if ( class_exists( 'Ayyash_Pro' ) ) {
					$product_label_type = ayyash_get_mod('woocommerce_shop_archive_label_type');
				} else {
					$product_label_type = '';
				}

				return sprintf(
					'<span class="product-badge badge-%s %s">%s</span>',
					esc_attr( $term->slug ),
					esc_attr( $product_label_type ? $product_label_type : 'fill' ),
					esc_html( $term->name )
				);
			},
			$product->get_tag_ids()
		);

		$product_tags = array_filter( $product_tags );

		if ( ! empty( $product_tags ) ) {
			echo implode( '', $product_tags ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}
}

if ( ! function_exists( 'ayyash_shop_labels' ) ) {
	/**
	 * Shop labels
	 *
	 * @return void
	 */
	function ayyash_shop_labels() {
		$product_label_style = ayyash_get_mod( 'woocommerce_shop_archive_label_style' );
		?>
		<div class="ayyash-product-labels <?php echo $product_label_style ? esc_attr( $product_label_style ) : 'block'; ?>">
			<?php
			/**
			 * Hook: ayyash_shop_labels
			 */
			do_action( 'ayyash_shop_labels' );
			?>
		</div><!-- end .ayyash-product-labels -->
		<?php
	}
}

if ( ! function_exists( 'ayyash_shop_thumbnails' ) ) {
	/**
	 * Shop thumbnails
	 *
	 * @return void
	 */
	function ayyash_shop_thumbnails() {
		?>
		<div class="ayyash-product-thumbnail">
			<?php
			ayyash_the_post_thumbnail(
				'shop-product-image',
				[
					'wrapper'   => false,
					'auto_size' => true,
				]
			);
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_shop_actions' ) ) {
	/**
	 * Shop Actions
	 *
	 * @return void
	 */
	function ayyash_shop_actions() {
		$action_pos  = ayyash_get_mod( 'woocommerce_shop_archive_action_pos' );
		$action_type = ayyash_get_mod( 'woocommerce_shop_archive_action_style' );

		if ( class_exists( 'Ayyash_Pro' ) ) {
			$classes = [];

			if ( $action_pos ) {
				$classes[] = $action_pos;
			} else {
				$classes[] = 'right';
			}

			if ( 'rounded' === $action_type ) {
				$classes[] = 'icon-rounded';
			}

			$classes = implode( ' ', $classes );
		} else {
			$classes = 'icon-rounded right';
		}

		?>
		<div class="ayyash-product-actions <?php echo esc_attr( $classes ); ?>">
			<?php

			/**
			 * Hook: ayyash_shop_actions
			 */
			do_action( 'ayyash_shop_actions' );
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_shop_header_close' ) ) {
	/**
	 * Shop header close
	 *
	 * @return void
	 */
	function ayyash_shop_header_close() {
		echo '</div>';
	}
}

if ( ! function_exists( 'ayyash_shop_content_open' ) ) {
	function ayyash_shop_content_open() {
		/**
		 * Shop content open
		 */
		echo '<div class="product-content">';
	}
}

if ( ! function_exists( 'ayyash_shop_category' ) ) {
	/**
	 * Shop category
	 *
	 * @return void
	 */
	function ayyash_shop_category() {
		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_product_category' ) ) {
			return;
		}

		global $product;
		$terms = get_the_terms( $product->get_id(), 'product_cat' );
		if ( ! empty( $terms ) ) {
			echo '<ul class="ayyash-product-category">';
			foreach ( $terms as $_term ) {
				$term_link = get_term_link( $_term );
				if ( ! $term_link || is_wp_error( $term_link ) ) {
					continue;
				}
				echo '<li><a href="' . esc_url( $term_link ) . '">' . esc_html( $_term->name ) . '</a></li>';
			}
			echo '</ul>';
		}
	}
}

if ( ! function_exists( 'ayyash_shop_title' ) ) {
	/**
	 * Shop title
	 *
	 * @return void
	 */
	function ayyash_shop_title() {
		if ( ! empty( get_the_title() ) ) {
			sprintf('<h3 class="ayyash-product-title"><a href="%s">%s</a></h3>', esc_url( get_the_permalink() ), esc_html( the_title() ) );
		}
	}
}

if ( ! function_exists( 'ayyash_shop_price' ) ) {
	/**
	 * Shop price
	 *
	 * @return void
	 */
	function ayyash_shop_price() {

		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_product_price' ) ) {
			return;
		}

		global $product;
		?>
		<div class="ayyash-product-price-wrapper inline">
			<?php echo wp_kses_post( $product->get_price_html() ); ?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_shop_rating' ) ) {
	/**
	 * Shop rating
	 *
	 * @return void
	 */
	function ayyash_shop_rating() {
		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_product_rating' ) ) {
			return;
		}
		global $product;
		if ( ! empty( $product->get_average_rating() ) ) { ?>
			<div class="ayyash-product-rating">
				<?php
				echo wc_get_rating_html( $product->get_average_rating() ); //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				if ( 1 === (int) get_theme_mod( 'woocommerce_shop_archive_show_product_rating_count' ) ) {
					echo sprintf('<span class="rating-count">(%d)</span>', esc_html( $product->get_rating_count() ) );
				}
				?>
			</div>
		<?php
		}
	}
}

if ( ! function_exists( 'ayyash_shop_content_close' ) ) {
	/**
	 * Shop content close
	 *
	 * @return void
	 */
	function ayyash_shop_content_close() {
		echo '</div>';
	}
}

if ( ! function_exists( 'ayyash_product_footer_classes' ) ) {
	/**
	 * Shop footer classes
	 *
	 * @return string
	 */
	function ayyash_product_footer_classes() {
		$classes[] = 'product-footer';
		$classes   = apply_filters( 'ayyash_product_footer_classes', $classes );
		$classes   = array_unique( array_filter( $classes ) );
		return implode(' ', $classes);
	}
}

if ( ! function_exists( 'ayyash_shop_footer_open' ) ) {
	/**
	 * Shop footer open
	 *
	 * @return void
	 */
	function ayyash_shop_footer_open() {
		echo '<div class="' . esc_attr( ayyash_product_footer_classes() ) . '">';
	}
}

if ( ! function_exists( 'ayyash_shop_footer_cart_wishlist_btn' ) ) {
	/**
	 * Check if cart or wishlist button exist in footer
	 *
	 * @return string
	 */
	function ayyash_shop_footer_cart_wishlist_btn( $classes ) {
		// Check if wishlist button is available in product footer
		if ( 'before_image' !== get_option( 'yith_wcwl_loop_position' ) && defined( 'YITH_WCWL' ) && 'yes' === get_option( 'yith_wcwl_show_on_loop' ) ) {
			$classes[] = 'footer-wishlist-btn';
		}

		// Check if cart button available in product footer @TODO
		//$cart_btn_position = get_theme_mod( 'woocommerce_shop_archive_btn_position' );
		//if ( 'in-footer' === $cart_btn_position ) {
			$classes[] = 'footer-cart-btn';
		//}

		return $classes;
	}
}

if ( ! function_exists( 'ayyash_shop_cart_btn' ) ) {
	function ayyash_shop_cart_btn() {
		?>
		<div class="<?php echo esc_attr( ayyash_product_btn_classes() ); ?>">
			<?php woocommerce_template_loop_add_to_cart(); ?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_shop_footer_close' ) ) {
	function ayyash_shop_footer_close() {
		echo '</div>';
	}
}

if ( ! function_exists( 'ayyash_mini_cart' ) ) :
	/**
	 * If theme option have enable search it will show search after menu item.
	 *
	 * @param string   $items The HTML list content for the menu items.
	 * @param stdClass $args  An object containing wp_nav_menu() arguments.
	 *
	 * @return string
	 */
	function ayyash_mini_cart() {
		if ( ayyash_get_mod( 'layout_header_show_mini_cart' ) ) {
			?>
			<div class="ayyash-header-cart-count d-none d-md-block"><a  href="<?php echo esc_url( wc_get_cart_url() ); ?>"><i class="ai ai-cart" aria-hidden="true"></i><span class="badge"><?php echo esc_html( WC()->cart->get_cart_contents_count() ); ?></span></a></div>
			<?php
		}
	}

endif;

if ( ! function_exists( 'ayyash_iconic_cart_count_fragments' ) ) {
	/**
	 * Add item count to cart freagments.
	 *
	 * @param array $fragments
	 *
	 * @return array
	 */
	function ayyash_iconic_cart_count_fragments( $fragments ) {
		$fragments['li.header-cart-count'] = '<li class="header-cart-count menu-item"><a class="header-cart" href="' . wc_get_cart_url() . '"><i class="ai ai-cart" aria-hidden="true"></i><span class="badge">' . WC()->cart->get_cart_contents_count() . '</span></a></li>';
		return $fragments;
	}
}

if ( ! function_exists( 'ayyash_mini_cart_fragments' ) ) {
	/**
	 * Ensure cart contents update when products are added to the cart via AJAX
	 *
	 * @param $fragments
	 *
	 * @return mixed
	 */
	function ayyash_mini_cart_fragments( $fragments ) {
		$fragments['.ayyash-header-cart-count .badge'] = '<span class="badge">' . esc_html( WC()->cart->get_cart_contents_count() ) . '</span>';
		return $fragments;
	}
}

if ( ! function_exists( 'ayyash_wc_change_number_related_products' ) ) {
	/**
	 * WC UpSell display args.
	 *
	 * @param array $args Args.
	 *
	 * @return array
	 */
	function ayyash_wc_upsell_display_args( $args ) {

		if ( is_product() ) {
			$args['posts_per_page'] = ayyash_get_mod( 'woocommerce_single_upsells_count' );
		} elseif ( is_cart() ) {
			$args['posts_per_page'] = ayyash_get_mod( 'woocommerce_cart_upsells_count' );
		} else {
			$args['posts_per_page'] = 2;
		}
		return $args;
	}
}

if ( ! function_exists( 'ayyash_wc_related_products_limit' ) ) {
	/**
	 * Related product limit.
	 *
	 * @return array
	 */
	function ayyash_wc_related_products_limit() {

		if ( is_product() ) {
			$args['posts_per_page'] = ayyash_get_mod( 'woocommerce_single_related_item_count' );
		} else {
			$args['posts_per_page'] = 4;
		}
		return $args;
	}
}

if ( ! function_exists( 'ayyash_cart_cross_sell_total' ) ) {
	/**
	 * Cross Sell Count
	 *
	 * @return int
	 */
	function ayyash_cart_cross_sell_total() {
		return absint( ayyash_get_mod( 'woocommerce_cart_cross_sell_count' ) );
	}
}

if ( ! function_exists( 'ayyash_stock_catalog' ) ) {
	/**
	 * Show product is in stock or out of stock.
	 *
	 * @return void
	 */
	function ayyash_stock_catalog() {
		global $product;
		if ( $product->is_in_stock() ) {
			/* translators: %s: Stock Quantity */
			echo ' <span class="stock-tag" data-bs-toggle="tooltip" data-bs-placement="bottom" title="' . sprintf( esc_attr__( 'Stock: %s', 'ayyash' ), esc_attr( $product->get_stock_quantity() ) ) . '">' . sprintf( esc_html__( '%s in stock', 'ayyash' ), esc_html( $product->get_stock_quantity() ) ) . '</span>';
		} else {
			/* translators: %s: Stock Quantity */
			echo ' <span class="stock-tag" data-bs-toggle="tooltip" data-bs-placement="bottom" title="' . sprintf( esc_attr__( 'Stock: %s', 'ayyash' ), esc_attr( $product->get_stock_quantity() ) ) . '">' . esc_html__( 'out of stock', 'ayyash' ) . '</span>';
		}
	}
}

if ( ! function_exists( 'ayyash_woocommerce_new_badge' ) ) {
	/**
	 * Show badge new if product is new.
	 *
	 * @return void
	 */
	function ayyash_woocommerce_new_badge() {
		global $product;
		$postdate = get_the_time( 'Y-m-d', $product->get_id() ); // Post date.
		if ( $postdate ) {
			// @phpstan-ignore-next-line
			$postdate = strtotime( $postdate ); // Timestamped post date.
			$newness  = 10;   // Newness in days as defined by option.
			if ( ( time() - ( 60 * 60 * 24 * $newness ) ) < $postdate ) { // If the product was published within the newness time frame display the new badge.
				echo '<div class="label-badge new"><span>' . esc_html__( 'New', 'ayyash' ) . '</span></div>';
			}
		}
	}
}

if ( ! function_exists( 'ayyash_sale_countdown_timer' ) ) {
	/**
	 * Sale Countdown timer
	 *
	 * @return void
	 */
	function ayyash_sale_countdown_timer() {
		global $product;

		$sale_date = $product->get_date_on_sale_to();

		if ( ! $sale_date ) {
			return;
		}

		echo '<div class="ayyash-countdown offer-time" data-date="' . esc_attr( $sale_date->date( 'Y/m/d' ) ) . '"></div>';
	}
}

if ( ! function_exists( 'ayyash_display_sold_out_loop_woocommerce' ) ) {
	/**
	 * Sold_out hook added
	 *
	 * @return void
	 */
	function ayyash_display_sold_out_loop_woocommerce() {

		if ( 1 !== (int) get_theme_mod( 'woocommerce_shop_archive_show_stock' ) && is_shop() ) {
			return;
		}
		global $product;

		if ( class_exists( 'Ayyash_Pro' ) ) {
			$product_label_type = ayyash_get_mod('woocommerce_shop_archive_label_type');
		} else {
			$product_label_type = '';
		}

		if ( ! $product->is_in_stock() ) {
			printf(
				'<span class="product-badge soldout %s">%s</span>',
				esc_attr( $product_label_type ? $product_label_type : 'fill' ),
				esc_html__( 'Out Of Stock', 'ayyash' )
			);
		}
	}
}

if ( ! function_exists( 'ayyash_filter_product_rating_html' ) ) {
	/**
	 * Filter Product rating
	 *
	 * @param string $rating_html rating html.
	 *
	 * @return string
	 */
	function ayyash_filter_product_rating_html( $rating_html ) {

		global $ayyash_products;

		if ( ! $ayyash_products ) {
			return $rating_html;
		}
		if ( isset( $ayyash_products['display_rating'] ) && 'yes' == $ayyash_products['display_rating'] ) {
			return $rating_html;
		}
		return '';
	}
}

if ( ! function_exists( 'ayyash_change_number_related_products' ) ) {
	/**
	 * Related Product Column Limit
	 *
	 * @param array $args
	 *
	 * @return array
	 */
	function ayyash_change_number_related_products( $args ) {

		$args['columns'] = absint( ayyash_get_mod( 'woocommerce_single_related_product_column' ) ); // # of columns per row
		return $args;
	}
}

if ( ! function_exists( 'ayyash_wc_gallery_thumb_size' ) ) {
	/**
	 * Product Gallery Thumb Image Size Resizing.
	 *
	 * @return array
	 */
	function ayyash_wc_gallery_thumb_size() {
		return array(
			'width'  => 100,
			'height' => 100,
			'crop'   => 0,
		);
	}
}

if ( ! function_exists( 'ayyash_variation_radio_buttons' ) ) {
	/**
	 * Product Attribute Radio buttons
	 *
	 * @param string $html Data.
	 * @param array  $args Arguments.
	 *
	 * @return string
	 */
	function ayyash_variation_radio_buttons( $html, $args ) {

		$args = wp_parse_args( $args, [
			'options'          => false,
			'attribute'        => false,
			'product'          => false,
			'selected'         => false,
			'name'             => '',
			'id'               => '',
			'class'            => '',
			'show_option_none' => __( 'Choose an option', 'ayyash' ),
		] );

		/**
		 * @var WC_Product_Variable $product
		 */
		$product   = $args['product'];
		$options   = $args['options'];
		$attribute = $args['attribute'];
		$name      = ! $args['name'] ? 'attribute_' . sanitize_title( $attribute ) : $args['name'];

		if ( false === $args['selected'] && $args['attribute'] && $args['product'] instanceof WC_Product ) {
			$selected_key     = 'attribute_' . sanitize_title( $args['attribute'] );
			$selected         = isset( $_REQUEST[ $selected_key ] ) ? wc_clean( wp_unslash( $_REQUEST[ $selected_key ] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$args['selected'] = ! $selected ? $product->get_variation_default_attribute( $args['attribute'] ) : $selected;
		}

		if ( empty( $options ) && ! empty( $product ) && ! empty( $attribute ) ) { //@phpstan-ignore-line
			$attributes = $product->get_variation_attributes();
			$options    = $attributes[ $attribute ];
		}

		$radios = '<div class="variation-radios ayyash-attributes ' . ayyash_get_mod('woocommerce_single_attribute', 'default') . '">';
		// Attribute label.
		$attrib_name = wc_attribute_label( $attribute, $product );

		if ( ! empty( $options ) ) {
			if ( $product && taxonomy_exists( $attribute ) ) { //@phpstan-ignore-line
				$terms = wc_get_product_terms( $product->get_id(), $attribute, [ 'fields' => 'all' ] );
				foreach ( $terms as $term ) {
					if ( in_array( $term->slug, $options, true ) ) {
						$id       = $name . '-' . $term->slug;
						$opt_name = apply_filters( 'woocommerce_variation_option_name', $term->name );
						$title    = sprintf(
						/* translators: 1. attribute label, 2. attribute value. */
							_x( '%1$s: %2$s', 'Product attribute option tooltip.', 'ayyash' ),
							$attrib_name,
							$opt_name
						);
						$output = '<span class="ayyash-attribute attribute-' . esc_attr( $name ) . '" data-bs-toggle="tooltip" data-bs-placement="bottom" title="' . esc_attr( $title ) . '" ';

						if ( 'attribute_pa_color' === $name ) {
							$color = get_term_meta( $term->term_id, 'color', true );
							$color = ! $color ? strtolower( $term->name ) : $color;

							$output .= 'style="background-color: ' . esc_attr( $color ) . '">';
							$output .= '<img alt="swatch" src="data:image/svg+xml,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22UTF-8%22%3F%3E%0A%3Csvg%20width%3D%2218px%22%20height%3D%2213px%22%20viewBox%3D%220%200%2018%2013%22%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%3E%0A%20%20%20%20%3C%21--%20Generator%3A%20Sketch%2041.2%20%2835397%29%20-%20http%3A%2F%2Fwww.bohemiancoding.com%2Fsketch%20--%3E%0A%20%20%20%20%3Ctitle%3EIcon%3C%2Ftitle%3E%0A%20%20%20%20%3Cdesc%3ECreated%20with%20Sketch.%3C%2Fdesc%3E%0A%20%20%20%20%3Cdefs%3E%3C%2Fdefs%3E%0A%20%20%20%20%3Cg%20id%3D%22Page-1%22%20stroke%3D%22none%22%20stroke-width%3D%221%22%20fill%3D%22none%22%20fill-rule%3D%22evenodd%22%3E%0A%20%20%20%20%20%20%20%20%3Cg%20id%3D%22Ecommerce%22%20transform%3D%22translate%28-844.000000%2C%20-3389.000000%29%22%20fill%3D%22%23FFFFFF%22%3E%0A%20%20%20%20%20%20%20%20%20%20%20%20%3Cg%20id%3D%22Product-Card-%231%22%20transform%3D%22translate%28150.000000%2C%202942.000000%29%22%3E%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%3Cg%20id%3D%22Color%22%20transform%3D%22translate%28683.000000%2C%20399.000000%29%22%3E%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%3Cpath%20d%3D%22M28.8443688%2C48.9224987%20L28.095269%2C48.1585481%20C27.8879809%2C47.9471506%2027.5523969%2C47.9471506%2027.3451088%2C48.1585481%20L17.6752%2C58.0207009%20C17.4673818%2C58.2320984%2017.1317978%2C58.2320984%2016.9255701%2C58.0207009%20L12.6546937%2C53.6656962%20C12.4474056%2C53.4542987%2012.1118216%2C53.4542987%2011.9055938%2C53.6656962%20L11.1554337%2C54.4301874%20C10.9481456%2C54.6415849%2010.9481456%2C54.9838218%2011.1554337%2C55.1946786%20L16.5369712%2C60.6834443%20C16.950487%2C61.105158%2017.6211249%2C61.1056986%2018.0351709%2C60.6839849%20L28.8438387%2C49.6875305%20C29.0516569%2C49.4766737%2029.052187%2C49.1338962%2028.8443688%2C48.9224987%22%20id%3D%22Icon%22%3E%3C%2Fpath%3E%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%3C%2Fg%3E%0A%20%20%20%20%20%20%20%20%20%20%20%20%3C%2Fg%3E%0A%20%20%20%20%20%20%20%20%3C%2Fg%3E%0A%20%20%20%20%3C%2Fg%3E%0A%3C%2Fsvg%3E" alt="' . esc_attr( $opt_name ) . '">';
						} else {
							$output .= '>' . esc_html( $term->name );
						}

						$output .= '</span>';

						$radios .= '<input type="radio" id="' . esc_attr( $id ) . '" name="' . esc_attr( $name ) . '" value="' . esc_attr( $term->slug ) . '"' . checked( sanitize_title( $args['selected'] ), $term->slug, false ) . '><label for="' . esc_attr( $id ) . '">' . $output . '</label>';
					}
				}
			} else {
				foreach ( $options as $option ) {
					$id     = $name . '-' . $option;
					$output = apply_filters( 'woocommerce_variation_option_name', $option );
					$title  = sprintf(
					/* translators: 1. attribute label, 2. attribute value. */
						_x( '%1$s: %2$s', 'Product attribute option tooltip.', 'ayyash' ),
						$attrib_name,
						$output
					);

					$output  = '<span class="ayyash-attribute attribute-' . esc_attr( $name ) . '" data-bs-toggle="tooltip" data-bs-placement="bottom" title="' . esc_attr( $title ) . '">' . esc_html( $output ) . '</span>';
					$checked = sanitize_title( $args['selected'] ) === $args['selected'] ? checked( $args['selected'], sanitize_title( $option ), false ) : checked( $args['selected'], $option, false );
					$radios .= '<input type="radio" id="' . esc_attr( $id ) . '" name="' . esc_attr( $name ) . '" value="' . esc_attr( $option ) . '"' . $checked . '><label for="' . esc_attr( $id ) . '">' . $output . '</label>';
				}
			}
		}

		$radios .= '</div>';

		return $html . $radios;
	}
}

if ( ! function_exists( 'ayyash_variation_check' ) ) {
	/**
	 * Check Product Variation.
	 *
	 * @param bool                 $active Activate.
	 * @param WC_Product_Variation $variation Variation.
	 *
	 * @return bool
	 */
	function ayyash_variation_check( $active, $variation ) {
		if ( ! $variation->is_in_stock() && ! $variation->backorders_allowed() ) {
			return false;
		}

		return $active;
	}
}

if ( ! function_exists( 'ayyash_product_loop_start' ) ) {
	/**
	 * WooCommerce loop start
	 *
	 * @return void
	 */
	function ayyash_product_loop_start() {
		echo sprintf( '<ul class="%s">', esc_attr( ayyash_product_loop_classes() ) );
	}
}

if ( ! function_exists( 'ayyash_product_loop_classes' ) ) {
	/**
	 * Ayyash Cart Button Classes
	 *
	 * @return string
	 */
	function ayyash_product_loop_classes() {
		$classes[] = 'products ayyash-products columns-' . esc_attr( wc_get_loop_prop( 'columns' ) );
		$classes   = apply_filters( 'ayyash_product_loop_classes', $classes );
		$classes   = array_unique( array_filter( $classes ) );
		return implode(' ', $classes);
	}
}

if ( ! function_exists( 'ayyash_product_classes' ) ) {
	/**
	 * Ayyash Cart Button Classes
	 *
	 * @return string
	 */
	function ayyash_product_classes() {
		$classes[] = 'ayyash-product-item';
		$classes[] = 'hover-grow-up';
		$classes[] = 'hover-actions';
		$classes[] = 'hover-button';
		$classes[] = 'ayyash-cart-button';
		$classes   = apply_filters( 'ayyash_product_classes', $classes );
		$classes   = array_unique( array_filter( $classes ) );
		return implode(' ', $classes);
	}
}

if ( ! function_exists( 'product_cat_item_class' ) ) {
	/**
	 * Ayyash Product category item class
	 *
	 * @return string
	 */
	function product_cat_item_class( $classes ) {
		$classes[] = 'ayyash-product-item';
		return $classes;
	}
}

if ( ! function_exists( 'ayyash_product_btn_classes' ) ) {
	/**
	 * Ayyash Cart Button Classes
	 *
	 * @return string
	 */
	function ayyash_product_btn_classes() {
		$classes[] = 'ayyash-cart-button';
		$classes   = apply_filters( 'ayyash_product_btn_classes', $classes );
		$classes   = array_unique( array_filter( $classes ) );
		return implode(' ', $classes);
	}
}

if ( ! function_exists( 'ayyash_single_product_actions' ) ) {
	/**
	 * Product Actions
	 *
	 * @return void
	 */
	function ayyash_single_product_actions() {
		?>
		<div class="ayyash-product-single-actions d-flex align-items-center">
			<?php
			/**
			 * Hook ayyash_product_actions.
			 */
			do_action('ayyash_product_actions');
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_single_product_share' ) ) {
	function ayyash_single_product_share() {
		?>
		<div class="share-product">
			<div class="share-title"><?php esc_html_e( 'Share:', 'ayyash' ); ?></div>
			<ul class="social-share-list d-flex justify-content-center align-items-center">
				<li class="share-item">
					<a target="_blank" class="facebook" data-social="facebook" title="<?php esc_attr_e( 'Share on Facebook', 'ayyash' ); ?>" href="https://www.facebook.com/sharer.php?u=<?php the_permalink(); ?>">
						<i class="ai ai-facebook" aria-hidden="true"></i>
					</a>
				</li>
				<li class="share-item">
					<a target="_blank" class="twitter" data-social="twitter" title="<?php esc_attr_e( 'Share on Twitter', 'ayyash' ); ?>" href="https://www.twitter.com/share?url=<?php the_permalink(); ?>&text=<?php the_title(); ?>">
						<i class="ai ai-twitter" aria-hidden="true"></i>
					</a>
				</li>
				<li class="share-item">
					<a target="_blank" class="pinterest" data-social="pinterest" title="<?php esc_attr_e( 'Share on Pinterest', 'ayyash' ); ?>" href="https://pinterest.com/pin/create/button/?url=<?php the_permalink(); ?>&amp;media=">
						<i class="ai ai-pinterest" aria-hidden="true"></i>
					</a>
				</li>
			</ul>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_my_account_content_title' ) ) {
	/**
	 * Function Name        : ayyash_my_account_content_title
	 * Function Hooked      : woocommerce_before_account_orders,
	 *                        woocommerce_before_account_orders,
	 *                        woocommerce_before_account_downloads,
	 *                        woocommerce_before_edit_account_form,
	 *                        woocommerce_before_edit_my_address
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_my_account_content_title() {
		if ( is_wc_endpoint_url( 'Dashboard' ) && in_the_loop() ) {
			?>
			<h4><?php esc_html_e( 'Dashboard', 'ayyash' ); ?></h4>
			<?php
		} elseif ( is_wc_endpoint_url( 'orders' ) && in_the_loop() ) {
			?>
			<h4><?php esc_html_e( 'Orders', 'ayyash' ); ?></h4>
			<?php
		} elseif ( is_wc_endpoint_url( 'downloads' ) && in_the_loop() ) {
			?>
			<h4><?php esc_html_e( 'Downloads', 'ayyash' ); ?></h4>
			<?php
		} elseif ( is_wc_endpoint_url( 'edit-account' ) && in_the_loop() ) {
			?>
			<h4><?php esc_html_e( 'Account Details', 'ayyash' ); ?></h4>
			<?php
		} elseif ( is_wc_endpoint_url( 'edit-address' ) && in_the_loop() ) {
			?>
			<h4><?php esc_html_e( 'Addresses', 'ayyash' ); ?></h4>
			<?php
		} else {
			?>
			<h4><?php esc_html_e( 'Dashboard', 'ayyash' ); ?></h4>
			<?php
		}
	}
}

if ( ! function_exists( 'ayyash_remove_my_account_menu_items' ) ) {
	function ayyash_remove_my_account_menu_items( $items ) {

		// Remove 'customer-logout' key / label pair from original $items array
		if ( array_key_exists( 'customer-logout', $items ) ) {
			unset( $items['customer-logout'] );
		}

		return $items;
	}
}

if ( ! function_exists( 'woocommerce_download_button_label' ) ) {
	/**
	 * Change Woocommerce Download Button label
	 */

	function woocommerce_download_button_label( $downloads ) {
		foreach ( $downloads as &$download ) {
			$download['download_name'] = __( 'Download', 'ayyash' );
		}
		return $downloads;
	}
}

if ( ! function_exists( 'ayyash_before_single_product' ) ) {
	function ayyash_before_single_product() {
		echo '<div class="row"><div class="col-md-6 col-sm-12">';
	}
}

if ( ! function_exists( 'ayyash_single_product_thumbnail_after' ) ) {
	function ayyash_single_product_thumbnail_after() {
		echo '</div><div class="col-md-6 col-sm-12">';
	}
}

if ( ! function_exists( 'ayyash_after_single_product' ) ) {
	function ayyash_after_single_product() {
		echo '</div></div>';
	}
}

if ( ! function_exists( 'ayyash_single_product_content_after' ) ) {
	function ayyash_single_product_content_after() {
		echo '</div><div class="col-md-12 col-sm-12">';
	}
}

if ( ! function_exists( 'ayyash_add_cart_link' ) ) {
	function ayyash_add_cart_link() {
		global $product;
		echo sprintf(
			'<a href="%s" data-quantity="%s" class="cart-btn %s" %s><i class="ai ai-cart" aria-hidden="true"></i> <span>%s</span></a>',
			esc_url( $product->add_to_cart_url() ),
			esc_attr( isset( $args['quantity'] ) ? $args['quantity'] : 1 ),
			esc_attr( isset( $args['class'] ) ? $args['class'] : 'button' ),
			isset( $args['attributes'] ) ? wc_implode_html_attributes( $args['attributes'] ) : '', //phpcs:ignore
			esc_html( $product->add_to_cart_text() )
		);
	}
}

if ( ! function_exists( 'ayyash_cart_cross_sell_display' ) ) {
	/**
	 * Remove woocommerce_cross_sell_display and display it after form
	 *
	 * @return void
	 */
	function ayyash_cart_cross_sell_display() {
		remove_action( 'woocommerce_cart_collaterals', 'woocommerce_cross_sell_display' );
		add_action( 'ayyash_after_cart_form', 'woocommerce_cross_sell_display' );
	}
}

if ( ! function_exists( 'ayyash_sidebar_archive_filter' ) ) {
	/**
	 * Display the shop page mobile filter.
	 *
	 * @return void
	 * @since  3.1.0
	 */
	function ayyash_sidebar_archive_filter() {
		if ( ! is_product() ) {
			?>
			<a href="#" id="ayyash-filter-toggle" class="d-block d-md-none shop-filter" role="button" aria-controls="filter-toggle">
				<svg width="16" height="12" viewBox="0 0 16 12" fill="none" xmlns="http://www.w3.org/2000/svg">
					<path fill-rule="evenodd" clip-rule="evenodd" d="M10.6875 0C9.33346 0 8.19475 0.970543 7.93213 2.25H0.5625C0.251842 2.25 0 2.50184 0 2.8125C0 3.12316 0.251842 3.375 0.5625 3.375H7.93213C8.19475 4.65446 9.33346 5.625 10.6875 5.625C12.0415 5.625 13.1802 4.65446 13.4429 3.375H15.1875C15.4982 3.375 15.75 3.12316 15.75 2.8125C15.75 2.50184 15.4982 2.25 15.1875 2.25H13.4429C13.1802 0.970543 12.0415 0 10.6875 0ZM10.6875 1.125C11.6261 1.125 12.375 1.87387 12.375 2.8125C12.375 3.75113 11.6261 4.5 10.6875 4.5C9.74887 4.5 9 3.75113 9 2.8125C9 1.87387 9.74887 1.125 10.6875 1.125Z" fill="var(--ayyash-color-heading)"></path>
					<path fill-rule="evenodd" clip-rule="evenodd" d="M5.0625 5.625C3.70846 5.625 2.56975 6.59554 2.30713 7.875H0.5625C0.251842 7.875 0 8.12684 0 8.4375C0 8.74816 0.251842 9 0.5625 9H2.30713C2.56975 10.2795 3.70846 11.25 5.0625 11.25C6.41654 11.25 7.55525 10.2795 7.81787 9H15.1875C15.4982 9 15.75 8.74816 15.75 8.4375C15.75 8.12684 15.4982 7.875 15.1875 7.875H7.81787C7.55525 6.59554 6.41654 5.625 5.0625 5.625ZM5.0625 6.75C6.00113 6.75 6.75 7.49887 6.75 8.4375C6.75 9.37613 6.00113 10.125 5.0625 10.125C4.12387 10.125 3.375 9.37613 3.375 8.4375C3.375 7.49887 4.12387 6.75 5.0625 6.75Z" fill="var(--ayyash-color-heading)"></path>
				</svg>
				<span><?php esc_html_e( 'Filter', 'ayyash' ); ?></span>
			</a>
			<?php
		}
	}
}

if ( ! function_exists( 'ayyash_filter_shop' ) ) {
	/**
	 * Function Name        : ayyash_filter_shop
	 * Function Hooked      : ayyash_filter_shop
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_filter_shop() {
		if ( is_shop() || is_product_category() ) {
			?>
			<div id="filter-toggle">
				<div class="container">
					<div class="d-flex align-items-center justify-content-between py-3">
						<a href="#" id="ayyash-filter-close" role="button" aria-controls="filter-toggle">
							<i class="ai ai-close" aria-hidden="true"></i>
							<span class="sr-only"><?php esc_html_e( 'Close Filter Toggle', 'ayyash' ); ?></span>
						</a>
					</div>
					<div class="widget-area main-sidebar">
						<?php dynamic_sidebar( 'shop-sidebar' ); ?>
					</div>
				</div>
			</div><!-- /filter-toggle -->
			<?php
		}
	}
}

if ( ! function_exists( 'ayyash_filter_backdrop_overlay' ) ) {
	/**
	 * Filter toggle overlay
	 *
	 * @return void
	 */
	function ayyash_filter_backdrop_overlay() {
		if ( is_shop() || is_product_category() && ayyash_get_mod( 'layout_header_mobile_backdrop_bg' ) ) {
			echo "<div class='ayyash-filter-overlay'></div>";
		}
	}
}

// End of file ayyash-woocommerce-hook-helper.php
