<?php
/**
 * Hook Helper.
 *
 * @author ThemeRox
 * @category HelperFunctions\HookHelper
 * @package Ayyash\Core
 */

if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	die();
}

if ( ! function_exists( 'ayyash_preloader' ) ) {
	function ayyash_preloader() {

		if ( 0 === (int) ayyash_get_mod( 'preload_enable' ) ) {
			return;
		}

		if ( class_exists('Elementor\Plugin' ) && ( Elementor\Plugin::$instance->preview->is_preview_mode() || Elementor\Plugin::$instance->editor->is_edit_mode() ) ) { //phpcs:ignore
			return;
		}

		global $revslider_is_preview_mode;
		if ( class_exists( 'RevSliderAdmin' ) && $revslider_is_preview_mode ) {
			return;
		}

		$animation_type  = ayyash_get_mod( 'preload_animation_type' );
		$animation_speed = ayyash_get_mod( 'preload_animation_speed' );
		$animation_cls   = 'animate__animated animate__' . $animation_type . ' animate__infinite';
		$preloader_img   = ayyash_get_mod( 'preload_img' );
		$text_size       = ayyash_get_mod( 'preload_text_size' );
		$img_size        = ayyash_get_mod( 'preload_img_size' );

		if ( $animation_speed ) {
			$animation_cls .= ' animate__' . $animation_speed;
		}

		?>
		<div id="ayyash-preloader" style="position: fixed;top: 0;bottom: 0;left: 0;right: 0;z-index: 9999999999;background: #ffffff;">
			<div class="ayyash-preloader__content" style="position: absolute;top: 50%;left: 50%;transform: translate(-50%, -50%);text-align: center;">
				<div class="<?php echo 'none' !== $animation_type ? esc_attr( $animation_cls ) : ''; ?>">
					<?php
					if ( $preloader_img && 1 === (int) ayyash_get_mod( 'preload_img_enable' ) ) {
						echo '<img src="' . esc_url( $preloader_img ) . '" alt="Ayyash preloader" width="' . esc_attr( $img_size ) . 'px" height="auto" />';
					}

					if ( 1 === (int) ayyash_get_mod( 'preload_text_enable' ) ) {
						echo '<p style="font-size:' . esc_attr( $text_size ) . 'px; margin: 0;">' . esc_html( ayyash_get_mod( 'preload_text' ) ) . '</p>';
					}
					?>
				</div>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_body_classes' ) ) {
	/**
	 * Body classes.
	 *
	 * @param string[] $classes body class list.
	 *
	 * @return array
	 */
	function ayyash_body_classes( $classes ) {

		// Apply single page/post layout.
		$site_layout = ayyash_get_post_layout_options( 'site_layout' );
		if ( 'default' === $site_layout ) {
			$site_layout = ayyash_get_mod( 'layout_global_site' );
		}
		//Add mobile menu style class
		if ( ayyash_get_mod( 'layout_header_mobile_menu_style' ) ) {
			$mobile_menu = ayyash_get_mod( 'layout_header_mobile_menu_style' );
			$classes[]   = $mobile_menu;
		}
		// Adds `singular` to singular pages, and `hfeed` to all other pages.
		$classes[] = is_singular() ? 'singular' : 'hfeed';
		$classes[] = 'theme-ayyash';
		$classes[] = 'boxed' === $site_layout ? 'boxed' : 'wide';

		// Adds a class of group-blog to blog with more than 1 published author.
		if ( is_multi_author() ) {
			$classes[] = 'group-blog';
		}

		return $classes;
	}
}

if ( ! function_exists( 'ayyash_post_classes' ) ) {
	/**
	 * Post class.
	 *
	 * @param string[] $classes post class list.
	 *
	 * @return array
	 */
	function ayyash_post_classes( $classes ) {
		if ( ! is_singular() ) {
			$classes[] = 'post-' . ayyash_get_content_layout( ayyash_get_current_screen() );
		} else {
			if ( 'post' === get_post_type() ) {
				$classes[] = 'post-normal post-single';
			}
		}

		return $classes;
	}
}

if ( ! function_exists( 'ayyash_allow_tags' ) ) {
	/**
	 * Ayyash Allow tags
	 *
	 * @param array $allowed_tags
	 *
	 * @return array
	 */
	function ayyash_allow_tags( $allowed_tags ) {
		$allowed_tags['noscript'] = [];

		return $allowed_tags;
	}
}

if ( ! function_exists( 'ayyash_custom_excerpt_length' ) ) {
	/**
	 * Override the_excerpt.
	 *
	 * @param int $length length.
	 *
	 * @return int
	 */
	function ayyash_custom_excerpt_length( $length ) {
		$excerpt_length = absint( ayyash_get_mod( 'layout_blog_excerpt_length' ) );
		if ( $excerpt_length ) {
			return $excerpt_length;
		}

		return $length;
	}
}

if ( ! function_exists( 'ayyash_category_transient_flusher' ) ) {
	/**
	 * Flush out the transients used in ayyash_categorized_blog.
	 *
	 * @return void
	 */
	function ayyash_category_transient_flusher() {
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		// Like, beat it. Dig?
		delete_transient( 'ayyash_category_count' );
	}
}

if ( ! function_exists( 'ayyash_skip_links' ) ) {
	/**
	 * Function Name        : ayyash_skip_links
	 * Function Hooked      : ayyash_header_skip_links
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_skip_links() {
		?>
		<a class="skip-link screen-reader-text" href="#content"><?php esc_html_e( 'Skip to content', 'ayyash' ); ?></a>
		<?php
	}
}

// secondary_head for top bar.
if ( ! function_exists( 'ayyash_secondary_head' ) ) {
	/**
	 * Function Name        : ayyash_action_secondary_head
	 * Function Hooked      : ayyash_header_secondary_header
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_secondary_head() {
		$header_container_cls = apply_filters( 'ayyash_header_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );

		// Removes secondary header from starter content file.
		if ( is_page_template( 'template-starter.php' ) ) {
			return;
		}

		if ( ayyash_get_mod( 'layout_header_secondary' ) ) {
			?>
			<header id="secondary-head" class="secondary-header" role="banner">
				<div class="<?php echo esc_attr( $header_container_cls ); ?>">
					<?php
					$mobile_hamburger_show = ayyash_get_mod( 'layout_header_menu_mobileshow' );

					switch ( $mobile_hamburger_show ) {
						case 'sm':
							$mobile_hamburger_show = 'item-d-md-show';
							break;
						case 'md':
							$mobile_hamburger_show = 'item-d-lg-show';
							break;
						case 'lg':
							$mobile_hamburger_show = 'item-d-xl-show';
							break;
						case 'xl':
							$mobile_hamburger_show = ' ';
							break;
						default:
							$mobile_hamburger_show = 'item-d-xl-show';
					}
					?>
					<div
						class="secondary-head-wrap d-flex align-items-center justify-content-between <?php echo esc_attr( $mobile_hamburger_show ); ?>">
						<?php ayyash_second_header(); ?>
						<?php
						if ( ayyash_get_mod( 'layout_header_secondary' ) ) {
							ayyash_mobile_icon();
						}
						?>
					</div>
				</div>
			</header>
			<?php
		}
	}
}

// master head for main menu.
if ( ! function_exists( 'ayyash_master_head' ) ) {
	/**
	 * Function Name        : ayyash_action_master_head
	 * Function Hooked      : ayyash_header_master_header
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_master_head() {
		$header_container_cls = apply_filters( 'ayyash_header_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );
		?>
		<header id="masthead"
				class="main-header <?php echo esc_attr( ayyash_transparent_background() ) . esc_attr( get_theme_mod( 'layout_header_bottom_border' ) === 1 ? ' header-border-bottom' : '' ); ?> "
				role="banner">
			<div class="<?php echo esc_attr( $header_container_cls ); ?>">
				<div class="masthead-wrap">
					<?php
					if ( is_page_template( 'template-starter.php' ) ) {
						?>
						<div class="site-branding">
							<a href="<?php echo esc_url( home_url() ); ?>">
								<img
									src="<?php echo esc_url( get_template_directory_uri() ) . '/assets/dist/images/starter-content/logo-white.svg'; ?>"
									alt="Ayyash">
							</a>
						</div>
						<?php
					} else {
						ayyash_site_branding();
					}

					ayyash_site_menu();
					ayyash_search_menu_icon();

					// Show mini-cart if WooCommerce Exist
					if ( class_exists( 'WooCommerce' ) ) {
						ayyash_mini_cart();
					}

					// Header button
					if ( is_page_template( 'template-starter.php' ) ) {
						ayyash_starter_header_button();
					} else {
						ayyash_header_button();
					}

					// Mobile icon
					if ( ! ayyash_get_mod( 'layout_header_secondary' ) ) {
						ayyash_mobile_icon();
					} elseif ( is_page_template( 'template-starter.php' ) ) {
						ayyash_mobile_icon();
					}
					?>
				</div>
			</div>
		</header>
		<?php
	}
}

// Mobile Menu.
if ( ! function_exists( 'ayyash_mobile_menu' ) ) {
	/**
	 * Function Name        : ayyash_mobile_menu
	 * Function Hooked      : ayyash_mobile_menu
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_mobile_menu() {

		$mobile_menu_container_cls = '';

		if ( ayyash_get_mod( 'layout_header_mobile_menu_style' ) === 'dropdown-mobile-menu' ) {
			$mobile_menu_container_cls = apply_filters( 'ayyash_mobile_menu_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );
		}

		$mobile_menu_view = ayyash_get_mod( 'layout_header_menu_mobileshow' );

		switch ( $mobile_menu_view ) {
			case 'sm':
				$mobile_menu_view = ' d-block d-md-none';
				break;
			case 'lg':
				$mobile_menu_view = ' d-block d-xl-none';
				break;
			case 'xl':
				$mobile_menu_view = ' d-block';
				break;
			case 'md':
			default:
				$mobile_menu_view = ' d-block d-lg-none';
				break;
		}
		$uid = wp_unique_id( 'search-' );
		?>
		<div id="navigation-mobile">
			<div class="<?php echo esc_attr( $mobile_menu_container_cls ) . esc_attr( $mobile_menu_view ); ?>">
				<?php
				if ( ayyash_get_mod( 'layout_header_mobile_menu_style' ) !== 'dropdown-mobile-menu' ) {
					?>
					<a href="#" id="ayyash-mobile-close" role="button" aria-controls="navigation-mobile">
						<i class="ai ai-close" aria-hidden="true"></i>
					</a>
				<?php } ?>

				<?php ayyash_get_mobile_menu(); ?>
			</div>
		</div><!-- /navigation-mobile -->
		<?php
	}
}

// Mobile menu backdrop
if ( ! function_exists( 'ayyash_mobile_backdrop_overlay' ) ) {
	/**
	 * Mobile menu toggle overlay
	 *
	 * @return void
	 */
	function ayyash_mobile_backdrop_overlay() {
		if ( ayyash_get_mod( 'layout_header_mobile_backdrop_bg' ) ) {
			echo "<div class='ayyash-menu-overlay'></div>";
		}
	}
}

// Comment

if ( ! function_exists( 'ayyash_commentfields_rowtag' ) ) {
	/**
	 * Adds the Proper opening markup for comment filed.
	 *
	 * @return void
	 */
	function ayyash_commentfields_rowtag() {
		echo '<div class="row">';
	}
}

if ( ! function_exists( 'ayyash_commentfields_rowtag_end' ) ) {
	/**
	 * Adds the Proper closing markup for comment filed.
	 *
	 * @return void
	 */
	function ayyash_commentfields_rowtag_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'ayyash_cancel_comment_reply_link' ) ) {
	/**
	 * Ayyash Comment Reply link
	 *
	 * @param string $link
	 *
	 * @return string
	 */
	function ayyash_cancel_comment_reply_link( $link ) {
		$tip = '<a data-bs-toggle="tooltip" data-bs-placement="bottom" title="' . esc_attr__( 'Cancel reply', 'ayyash' ) . '" ';

		return str_replace( '<a ', $tip, $link );
	}
}

/**
 * ----------------------------------------------------------------------
 * Page Option related Function
 *----------------------------------------------------------------------*/

// Secondary Header Enable Disable.

if ( ! function_exists( 'ayyash_single_secondary_header' ) ) {
	/**
	 * Function Name        : ayyash_single_secondary_header
	 * Description          : Remove secondary header for single page or post
	 * Function Hooked      : ayyash_single_option
	 * Function return Type : markup for secondary head or remove action
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_single_secondary_header() {
		$secondary_header = ayyash_get_post_layout_options( 'secondary_header' );
		if ( 'default' !== $secondary_header ) {
			if ( 'enable' === $secondary_header ) {
				add_action( 'ayyash_header_secondary_header', 'ayyash_secondary_head' );
			} else {
				remove_action( 'ayyash_header_secondary_header', 'ayyash_secondary_head' );
			}
		}
	}
}

if ( ! function_exists( 'ayyash_single_master_header' ) ) {
	/**
	 * Function Name        : ayyash_single_master_header
	 * Description          : Enable / Disable master head for single page
	 * Function Hooked      : ayyash_single_option
	 * Function return Type : html header markup
	 * Parameter            : post or page id
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_single_master_header() {
		$header_option = ayyash_get_post_layout_options( 'display_header' );
		if ( 'disable' === $header_option ) {
			remove_action( 'ayyash_header_master_header', 'ayyash_master_head' );
		}

	}
}

if ( ! function_exists( 'ayyash_grid_loop_layout_start' ) ) {
	/**
	 * Ayyash Grid Layout Loop Start
	 *
	 * @param string $ayyash_layout
	 *
	 * @return void
	 */
	function ayyash_grid_loop_layout_start( $ayyash_layout ) {
		if ( 'grid' == $ayyash_layout ) {
			?>
	<div class="post-grid-wrap"><?php
		}
	}
}

if ( ! function_exists( 'ayyash_grid_loop_layout_end' ) ) {
	/**
	 * Ayyash Grid Layout Loop End
	 *
	 * @param string $ayyash_layout
	 *
	 * @return void
	 */
	function ayyash_grid_loop_layout_end( $ayyash_layout ) {
		if ( 'grid' == $ayyash_layout ) {
			?></div><!-- .post-grid-wrap --><?php
		}
	}
}

if ( ! function_exists( 'ayyash_render_footer_main' ) ) {
	/**
	 * Function Name        : ayyash_render_footer_main
	 * Function Hooked      : ayyash_footer_bottom_widget
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_render_footer_main() {
		$footer_main_container_cls = apply_filters( 'ayyash_footer_main_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );
		$main_footer_option        = ayyash_get_post_layout_options( 'display_main_footer' );

		if ( is_page_template( 'template-starter.php' ) ) {
			return;
		}

		if ( ! is_active_sidebar( 'footer-widget' ) ) {
			return;
		}

		if ( 'disable' === $main_footer_option ) {
			return;
		}
		?>

		<div class="ayyash-footer footer-main" >
			<div class="ayyash-footer-widget footer-main">

			<?php
			/**
			 * Render Footer before Content.
			 *
			 * @hooked ayyash_before_footer_container
			 */
			do_action( 'ayyash_before_footer_container' );
			?>

			<div class="<?php echo esc_attr( $footer_main_container_cls ); ?>">
				<div class="row">
					<?php dynamic_sidebar( 'footer-widget' ); ?>
				</div>
			</div>

			<?php
			/**
			 * Render Footer after Content.
			 *
			 * @hooked ayyash_after_footer_container
			 */
			do_action( 'ayyash_after_footer_container' ); ?>

			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_render_footer_credits' ) ) {

	/**
	 * Function Name        : ayyash_render_footer_top
	 * Function Hooked      : ayyash_footer_top_widget
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_render_footer_credits() {
		$credit_option = ayyash_get_post_layout_options( 'display_credit' );
		if ( 'disable' === $credit_option ) {
			return;
		}
		?>
		<div class="ayyash-footer-credit-wrap">
			<div class="ayyash-footer-copyright">
			<div class="container-fluid">
				<div class="row">
					<div class="col-md-12">
						<div
							class="footer-credit">
							<div
								class="copyright text-center"><?php echo wp_kses_post( ayyash_get_mod( 'layout_footer_footer_text' ) ); ?></div>
							<!-- .copyright -->
							<div class="powered-by">
								<?php
								printf(
								/* translators: %1$s: WordPress, %2$s: heart icon, %3$s: Ayyash  */
									esc_html__( 'Proudly powered by %1$s. Made with %2$s by %3$s', 'ayyash' ),
									'<a href="' . esc_url( __( 'https://wordpress.org/', 'ayyash' ) ) . '" rel="noopener" target="_blank">WordPress</a>',
									'<i class="ai ai-heart"></i>',
									'<a href="' . esc_url( __( 'https://themerox.com/', 'ayyash' ) ) . '" rel="noopener" target="_blank">ThemeRox</a>'
								);
								?>
							</div><!-- .made-with -->
						</div>
					</div>
				</div>
			</div>
		</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_render_footer_social_icons' ) ) {
	/**
	 * Function Name        : ayyash_master_footer
	 * Function Hooked      : ayyash_footer_master_footer
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_render_footer_social_icons() {
		$footer_social_container_cls = apply_filters( 'ayyash_footer_social_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );
		$secondary_footer_option     = ayyash_get_post_layout_options( 'display_secondary_footer' );

		if ( is_page_template( 'template-starter.php' ) ) {
			return;
		}
		if ( ! ayyash_has_social_icons() ) {
			return;
		}
		if ( 'disable' === $secondary_footer_option ) {
			return;
		}
		?>
		<div class="ayyash-footer ayyash-footer-secondary">
			<div class="ayyash-row ayyash-footer-social-icon text-center">
				<div class="<?php echo esc_attr( $footer_social_container_cls ); ?>">
					<div class="row">
						<div class="col-md-12 ayyash-footer-social-share">
							<ul class="list-inline list-unstyled mb-0">
								<?php ayyash_footer_social_icons(); ?>
							</ul>
						</div>
					</div>
					<!-- end row -->
				</div>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_render_footer_menu' ) ) {
	/**
	 * Ayyash Footer Menu
	 *
	 * @return void
	 */
	function ayyash_render_footer_menu() {
		if ( ! has_nav_menu( 'footer_menu' ) ) {
			return;
		}
		?>
		<div class="ayyash-row ayyash-footer-menu text-center">
			<div class="container">
				<div class="row">
					<div class="col-md-12">
						<?php
						wp_nav_menu( [
							'menu_class'     => 'footer-nav',
							'menu_id'        => 'footer-nav',
							'items_wrap'     => '<ul id="%1$s" class="%2$s">%3$s</ul>',
							'theme_location' => 'footer_menu',
						] );
						?>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ayyash_render_go_to_top' ) ) {
	/**
	 * Function Name        : ayyash_go_to_top
	 * Function Hooked      : ayyash_footer_go_to_top
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_render_go_to_top() {
		if ( ayyash_get_mod( 'layout_footer_scroll_to_top' ) === 1 ) {
			?>
			<a href="#" class="cd-top"><i class="ti-angle-up"></i><span
					class="sr-only"><?php esc_html_e( 'Goto Top', 'ayyash' ); ?></span></a>
			<?php
		}
	}
}

if ( ! function_exists( 'ayyash_disable_mediaelement' ) ) {
	/**
	 * Disable WP MediaElement JS in frontend (shortcodes)
	 *
	 * @return string
	 */
	function ayyash_disable_mediaelement() {
		// returning anything other then mediaelement do the job.
		return 'plyr'; // use plyr
	}
}

if ( ! function_exists( 'ayyash_set_gallery_thumb_size' ) ) {
	/**
	 * Filter thumbnail size for gallery images (archive page)
	 *
	 * @param array $atts
	 *
	 * @return array
	 */
	function ayyash_set_gallery_thumb_size( $atts ) {
		if ( ! is_admin() && ! is_singular() ) {

			$ayyash_layout = ayyash_get_content_layout();
			if ( 'normal' === $ayyash_layout ) {
				$atts['size'] = 'post-thumbnails';
			} else {
				$atts['size'] = 'ayyash-gallery';
			}
		}

		return $atts;
	}
}

if ( ! function_exists( 'ayyash_pingback_header' ) ) {
	/**
	 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
	 *
	 * @return void
	 * @since Twenty Twenty-One 1.0
	 */
	function ayyash_pingback_header() {
		if ( is_singular() && pings_open() ) {
			?>
			<link rel="pingback" href="<?php echo esc_url( get_bloginfo( 'pingback_url' ) ); ?>">
			<?php
		}
	}
}

if ( ! function_exists( 'ayyash_supports_js' ) ) {
	/**
	 * Remove the `no-js` class from body if JS is supported.
	 *
	 * @return void
	 * @since Twenty Twenty-One 1.0
	 */
	function ayyash_supports_js() {
		?>
		<script>document.documentElement.className = document.documentElement.className.replace(/\bno-js\b/, '') + ' js';</script>
		<?php
	}
}

if ( ! function_exists( 'ayyash_home_section_category_and_date' ) ) {
	/**
	 * Function Name        : ayyash_home_section_category_and_date
	 * Function Hooked      : ayyash_home_page_sections
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */

	function ayyash_home_section_category_and_date() {
		do_action( 'before_ayyash_home_section_category_and_date' );
		?>
		<section class="ayyash-cat-menu-and-date-area">
			<div class="container">
				<div class="cat-menu-and-date-wrap d-flex flex-column flex-md-row  flex-wrap justify-content-between">
					<?php ayyash_cat_menu(); ?>
					<?php ayyash_date(); ?>
				</div>
			</div>
		</section>
		<?php
		do_action( 'after_ayyash_home_section_category_and_date' );
	}
}

if ( ! function_exists( 'ayyash_home_section_sticky_posts' ) ) {
	/**
	 * Function Name        : ayyash_home_section_sticky_posts
	 * Function Hooked      : ayyash_home_page_sections
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_home_section_sticky_posts() {

		//necessary variables
		$home_sticky_post_container_cls = apply_filters( 'ayyash_home_sticky_post_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );
		$slider                         = ayyash_get_mod( 'layout_home_enable_sticky_post_slider' );
		$post_per_page                  = ayyash_get_mod( 'layout_blog_home_sticky_posts_per_page' );
		$column                         = 3;

		// Default classes
		$wrapper_class = 'sticky-post-wrapper ayyash-slider-active';

		// Prepare Slider Settings
		if ( $slider ) {
			// variable come from customizer
			$slick_arrows   = ayyash_get_mod( 'layout_home_sticky_post_slider_enable_arrow' ) ? 'true' : 'false';
			$slick_dots     = ayyash_get_mod( 'layout_home_sticky_post_slider_enable_dots' ) ? 'true' : 'false';
			$slick_autoplay = ayyash_get_mod( 'layout_home_sticky_post_slider_enable_autoplay' ) ? 'true' : 'false';

			$slick_data = '{"slidesToShow":' . $column . ',"arrows":' . $slick_arrows . ',"dots":' . $slick_dots . ',"autoplay":' . $slick_autoplay . ',"responsive":[{"breakpoint":576,"settings":{"slidesToShow":1}},{"breakpoint":992,"settings":{"slidesToShow":2}}]}';
		} else {
			$wrapper_class = 'sticky-post-wrapper column-' . $column;
		}

		//prepare post query
		$post_in      = get_option( 'sticky_posts' );
		$args         = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $post_per_page,
			'post__in'            => $post_in,
			'ignore_sticky_posts' => true,
		);
		$sticky_posts = new WP_Query( $args );

		if ( ayyash_get_mod( 'layout_blog_home_sticky_post' ) ) :
			?>
			<section class="ayyash-sticky-post-area">
				<div class="<?php echo esc_attr( $home_sticky_post_container_cls ); ?>">
					<div class="<?php echo esc_attr( $wrapper_class ); ?>" <?php if ( $slider ) {
						echo 'data-slick = ' . esc_attr( $slick_data );
								} ?>>
						<?php
						if ( $sticky_posts->have_posts() ) {
							while ( $sticky_posts->have_posts() ) :
								$sticky_posts->the_post(); ?>
								<article class="single-post">
									<?php if ( has_post_thumbnail() ) { ?>
										<div class="thumbnail">
											<?php the_post_thumbnail(); ?>
										</div>
									<?php } ?>

									<div class="content">
										<div class="meta">
											<span class="date"><?php echo esc_html( get_the_date() ); ?> - </span>
											<?php
											$categories = get_the_category();
											if ( ! empty( $categories ) ) {
												echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
											}
											?>
										</div>
										<h3 class="title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
										</h3>
									</div>
								</article>
							<?php
							endwhile;
							wp_reset_postdata();
						}
						?>
					</div>
				</div>
			</section>
		<?php
		endif;
	}
}

if ( ! function_exists( 'ayyash_home_section_blog_popular_and_latest_post_tab' ) ) {
	/**
	 * Function Name        : ayyash_home_section_blog_popular_and_latest_post_tab
	 * Function Hooked      : ayyash_home_page_sections
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_home_section_blog_popular_and_latest_post_tab() {

		$post_number                                = ayyash_get_mod( 'layout_blog_home_post_tab_posts_per_page' ); // this  will come form customizer
		$home_popular_and_latest_post_container_cls = apply_filters( 'ayyash_home_popular_and_latest_post_container_class', ayyash_get_mod( 'layout_global_content_layout' ) );

		$latest_post = new WP_Query( array(
			'post_type'           => 'post',
			'posts_per_page'      => $post_number,
			'post_status'         => 'publish',
			'order'               => 'DESC',
			'orderby'             => 'ID',
			'ignore_sticky_posts' => true,
			'no_found_rows'       => true,
		) );

		$popular_post = new WP_Query( array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $post_number,
			'meta_key'            => 'ayyash_post_views_count',
			'orderby'             => 'meta_value_num',
			'order'               => 'DESC',
			'ignore_sticky_posts' => true,
		) );

		if ( ayyash_get_mod( 'layout_blog_home_post_tab' ) ) :
			?>
			<section class="ayyash-home-popular-and-latest-blog-section">
				<div class="<?php echo esc_attr( $home_popular_and_latest_post_container_cls ); ?>">

					<div class="row">
						<div class="col-md-12">
							<ul class="nav nav-tabs ayyash-tab-list" id="ayyashTab" role="tablist">
								<li class="nav-item" role="presentation">
									<button class="active" id="latest-tab" data-bs-toggle="tab" data-bs-target="#latest"
											type="button" role="tab" aria-controls="latest" aria-selected="true">
										<?php esc_html_e( 'Latest', 'ayyash' ); ?>
									</button>
								</li>
								<li class="nav-item" role="presentation">
									<button id="popular-tab" data-bs-toggle="tab" data-bs-target="#popular"
											type="button" role="tab" aria-controls="popular" aria-selected="false">
										<?php esc_html_e( 'Popular', 'ayyash' ); ?>
									</button>
								</li>
							</ul>
						</div>
					</div>

					<div class="row">
						<div class="col-md-12">
							<div class="tab-content" id="ayyashTabContent">
								<!-- latest post -->
								<?php if ( $latest_post->have_posts() ) :
									$latest_post->the_post(); ?>
									<div class="tab-pane fade show active" id="latest" role="tabpanel" aria-labelledby="latest-tab">
										<article class="blog-item first-item">
											<div class="thumbnail"><?php the_post_thumbnail(); ?></div>
											<div class="content">
												<div class="meta">
													<span class="date"><?php echo esc_html( get_the_date() ); ?></span>
													-
													<?php
													$categories = get_the_category();
													if ( ! empty( $categories ) ) {
														echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
													}
													?>
												</div>
												<h3 class="title"><a
														href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
											</div>
										</article>
										<div class="blog-list">
											<?php while ( $latest_post->have_posts() ) :
												$latest_post->the_post(); ?>
												<article class="blog-item">
													<?php if ( has_post_thumbnail() ) { ?>
														<div
															class="thumbnail"><?php the_post_thumbnail(); ?></div>
													<?php } ?>
													<div class="content">
														<div class="meta">
															<span
																class="date"><?php echo esc_html( get_the_date() ); ?></span>
															-
															<?php
															$categories = get_the_category();
															if ( ! empty( $categories ) ) {
																echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
															}
															?>
														</div>
														<h5 class="title"><a
																href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
														</h5>
													</div>
												</article>
											<?php endwhile; ?>
										</div>
									</div>
								<?php
								endif;
								wp_reset_postdata();
								?>

								<!-- Popular post -->
								<?php if ( $popular_post->have_posts() ) :
									$popular_post->the_post(); ?>
									<div class="tab-pane fade" id="popular" role="tabpanel" aria-labelledby="popular-tab">
										<article class="blog-item first-item">
											<div class="thumbnail"><?php the_post_thumbnail(); ?></div>
											<div class="content">
												<div class="meta">
													<span class="date"><?php echo esc_html( get_the_date() ); ?></span>
													-
													<?php
													$categories = get_the_category();
													if ( ! empty( $categories ) ) {
														echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
													}
													?>
												</div>
												<h3 class="title"><a
														href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
											</div>
										</article>
										<div class="blog-list">
											<?php while ( $popular_post->have_posts() ) :
												$popular_post->the_post(); ?>
												<article class="blog-item">
													<?php if ( has_post_thumbnail() ) { ?>
														<div class="thumbnail"><?php the_post_thumbnail(); ?></div>
													<?php } ?>
													<div class="content">
														<div class="meta">
															<span
																class="date"><?php echo esc_html( get_the_date() ); ?></span>
															-
															<?php
															$categories = get_the_category();
															if ( ! empty( $categories ) ) {
																echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
															}
															?>
														</div>
														<h5 class="title"><a
																href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
														</h5>
													</div>
												</article>
											<?php endwhile; ?>
										</div>
									</div>
								<?php
								endif;
								wp_reset_postdata();
								?>
							</div>
						</div>
					</div>
				</div>
			</section>
		<?php
		endif;
	}
}

if ( ! function_exists( 'ayyash_home_section_blog_important_posts' ) ) {
	/**
	 * Function Name        : ayyash_home_section_blog_important_posts
	 * Function Hooked      : ayyash_home_page_sections
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_home_section_blog_important_posts() {
		$ayyash_view = ayyash_get_current_screen();

		$section_title  = ayyash_get_mod( 'layout_blog_home_latest_posts_title' );
		$show_view_all  = ayyash_get_mod( 'layout_blog_home_latest_posts_view_btn' );
		$post_columns   = ayyash_get_mod( 'layout_blog_home_latest_posts_column' );
		$image_position = ayyash_get_mod( 'layout_blog_home_latest_posts_thumbnail_position' );
		$excerpt_length = ayyash_get_mod( 'layout_blog_home_latest_posts_excerpt_length' );
		$post_per_page  = ayyash_get_mod( 'layout_blog_home_latest_posts_per_page' );

		//prepare post query
		$args            = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'posts_per_page'      => $post_per_page,
			'order'               => 'DESC',
			'orderby'             => 'ID',
			'ignore_sticky_posts' => true,
		);
		$important_posts = new WP_Query( $args );

		?>
		<section class="ayyash-home-important-blog-section">
			<div
				class="<?php echo esc_attr( apply_filters( 'ayyash_home_important_blog_container_class', ayyash_get_mod( 'layout_global_content_layout' ) ) ); ?>">
				<div class="row">
					<!--sidebar will work in future  -->
					<?php ayyash_try_sidebar( $ayyash_view, 'left' ); ?>

					<div <?php ayyash_main_class(); ?>>
						<div class="ayyash-section-heading d-flex justify-content-between">
							<h2 class="title">
								<?php if ( $section_title ) {
									echo esc_html( $section_title );
								} ?>
							</h2>
							<?php if ( $show_view_all ) { ?>
								<div class="ayyash-read-more">
									<a href="<?php echo esc_url( get_post_type_archive_link( 'post' ) ); ?>">
										<span><?php esc_html_e( 'View All', 'ayyash' ); ?></span>
										<i class="ai ai-more"></i>
									</a>
								</div>
							<?php } ?>
						</div>
						<!-- .ayyash-home-section-heading -->
						<div class="ayyash-home-post-wrapper column-<?php echo esc_attr( $post_columns ); ?>">
							<?php
							if ( $important_posts->have_posts() ) {
								while ( $important_posts->have_posts() ) :
									$important_posts->the_post();
									?>
									<article class="post-item image-<?php echo esc_attr( $image_position ); ?>">
										<div class="thumbnail">
											<?php
											if ( has_post_thumbnail() ) {
												the_post_thumbnail();
											} else {
												?><img src="<?php echo esc_url( AYYASH_THEME_URI . 'assets/dist/images/post_demo.svg' ); ?>" alt="<?php the_title(); ?>"><?php
											}
											?>
										</div>

										<div class="content">
											<div class="meta">
												<span class="date"><?php echo esc_html( get_the_date() ); ?> - </span>
												<?php
												$categories = get_the_category();
												if ( ! empty( $categories ) ) {
													echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
												}
												?>
											</div>
											<h3 class="title"><a
													href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
											<p><?php echo esc_html( get_ayyash_excerpt( get_the_ID(), $excerpt_length ) ); ?></p>
										</div>
									</article>
								<?php
								endwhile;
								wp_reset_postdata();
							} else {
								esc_html_e( 'No post found', 'ayyash' );
							}
							?>
						</div>
					</div>
					<!-- .content area -->

					<!--sidebar will work in future -->
					<?php ayyash_try_sidebar( $ayyash_view, 'right' ); ?>

				</div>
				<!-- .row -->
			</div>
			<!-- .container -->
		</section>
		<?php
	}
}

if ( ! function_exists( 'ayyash_home_section_blog_post_by_category' ) ) {
	/**
	 * Function Name        : ayyash_home_section_blog_post_by_category
	 * Function Hooked      : ayyash_home_page_sections
	 * Function return Type : html
	 * Function Since       : 1.0.0
	 *
	 * @return void
	 */
	function ayyash_home_section_blog_post_by_category() {

		$selected_categories = ayyash_get_mod( 'layout_blog_home_categories_control' );
		if ( ! $selected_categories ) {
			return;
		}

		foreach ( $selected_categories as $category ) {

			//sidebar and main content grid
			$sidebar_grid     = ayyash_get_mod( 'layout_global_sidebar_grid' );
			$sidebar_classes  = 'widget-area main-sidebar col-md-' . $sidebar_grid;
			$content_grid     = ayyash_get_mod( 'layout_global_content_grid' );
			$sidebar_position = ayyash_get_mod( 'layout_blog_home_' . $category . '_sidebar_position' );

			//prepare content classes
			$content_classes = array( 'content-area' );
			if ( 'none' === $sidebar_position ) {
				$content_classes[] = 'col-md-12';
			} else {

				if ( $content_grid < 1 || $content_grid > 12 ) {
					$content_grid = ayyash_get_default( 'layout_global_content_grid' );
				}

				$content_classes[] = 'col-md-' . absint( $content_grid );
			}


			$term_details = get_term_by( 'slug', $category, 'category' );

			$section_title  = ayyash_get_mod( 'layout_blog_home_' . $category . '_title' );
			$show_view_all  = ayyash_get_mod( 'layout_blog_home_' . $category . '_view_btn' );
			$post_columns   = ayyash_get_mod( 'layout_blog_home_' . $category . '_column' );
			$image_position = ayyash_get_mod( 'layout_blog_home_' . $category . '_thumbnail_position' );
			$excerpt_length = ayyash_get_mod( 'layout_blog_home_' . $category . '_excerpt_length' );
			$post_per_page  = ayyash_get_mod( 'layout_blog_home_' . $category . '_post_per_page' );
			$sidebar        = ayyash_get_mod( 'layout_blog_home_' . $category . '_sidebar' );
			$category_id    = absint( $term_details->term_id );

			//prepare section title
			if ( empty( $section_title ) && ! empty( $category_id ) ) {
				$section_title = $term_details->name;
			}

			//prepare section category link
			$category_link = '#';
			if ( ! empty( $category_id ) ) {
				$category_link = get_category_link( $category_id );
			}

			//prepare post query
			$args           = array(
				'post_type'           => 'post',
				'post_status'         => 'publish',
				'posts_per_page'      => $post_per_page,
				'category__in'        => $category_id,
				'ignore_sticky_posts' => true,
			);
			$category_posts = new WP_Query( $args );
			?>
			<section class="ayyash-home-category-blog-section">
				<div
					class="<?php echo esc_attr( apply_filters( 'ayyash_home_category_blog_container_class', ayyash_get_mod( 'layout_global_content_layout' ) ) ); ?>">
					<div class="row">

						<?php
						if ( 'left' === $sidebar_position && is_active_sidebar( $sidebar ) ) {
							?>
							<div id="secondary" class="<?php echo esc_attr( $sidebar_classes ); ?>" role="complementary">
								<?php dynamic_sidebar( $sidebar ); ?>
							</div><!-- #secondary -->
							<?php
						}
						?>

						<div class="<?php echo esc_attr( implode( ' ', $content_classes ) ); ?>">
							<div class="ayyash-section-heading d-flex justify-content-between">
								<h2 class="title"><?php echo esc_html( $section_title ); ?></h2>
								<?php if ( $show_view_all ) { ?>
									<div class="ayyash-read-more">
										<a href="<?php echo esc_url( $category_link ); ?>">
											<span><?php esc_html_e( 'View All', 'ayyash' ); ?></span>
											<i class="ai ai-more"></i>
										</a>
									</div>
								<?php } ?>
							</div>
							<!-- .ayyash-home-section-heading -->
							<div class="ayyash-home-post-wrapper column-<?php echo esc_attr( $post_columns ); ?>">
								<?php
								if ( $category_posts->have_posts() ) {
									while ( $category_posts->have_posts() ) :
										$category_posts->the_post(); ?>
										<article class="post-item image-<?php echo esc_attr( $image_position ); ?>">
											<div class="thumbnail">
												<?php
												if ( has_post_thumbnail() ) {
													the_post_thumbnail();
												} else {
													?><img src="<?php echo esc_url( AYYASH_THEME_URI . 'assets/dist/images/post_demo.svg' ); ?>" alt="<?php the_title(); ?>"><?php
												}
												?>
											</div>

											<div class="content">
												<div class="meta">
													<span
														class="date"><?php echo esc_html( get_the_date() ); ?> - </span>
													<?php
													$categories = get_the_category();
													if ( ! empty( $categories ) ) {
														echo '<a class="category" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '">' . esc_html( $categories[0]->name ) . '</a>';
													}
													?>
												</div>
												<h3 class="title"><a
														href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
												<p><?php echo esc_html( get_ayyash_excerpt( get_the_ID(), $excerpt_length ) ); ?></p>
											</div>
										</article>
									<?php
									endwhile;
									wp_reset_postdata();
								}
								?>
							</div>
						</div>
						<!-- .content area -->

						<?php
						if ( 'right' === $sidebar_position && is_active_sidebar( $sidebar ) ) {
							?>
							<div id="secondary" class="<?php echo esc_attr( $sidebar_classes ); ?>" role="complementary">
								<?php dynamic_sidebar( $sidebar ); ?>
							</div><!-- #secondary -->
							<?php
						}
						?>
					</div>
					<!-- .row -->
				</div>
				<!-- .container -->
			</section>
			<?php
		}
	}
}

if ( ! function_exists( 'ayyash_add_no_js_attachment_image' ) ) {
	/**
	 *
	 * Filter the HTML img element representing an image attachment.
	 *
	 * @param string $html HTML img element or empty string on failure.
	 * @param int $attachment_id Image attachment ID.
	 * @param string|int[] $size Requested image size. Can be any registered image size name, or
	 *                                    an array of width and height values in pixels (in that order).
	 * @param bool $icon Whether the image should be treated as an icon.
	 * @param string[] $attr Array of attribute values for the image markup, keyed by attribute name.
	 *                                    See wp_get_attachment_image().
	 *
	 * @return string
	 */
	function ayyash_add_no_js_attachment_image( $html, $attachment_id, $size, $icon, $attr ) {

		if ( ! ayyash_maybe_lazyload_post_thumbnail() ) {
			return $html;
		}

		// Remove both filter cb
		remove_filter( 'wp_get_attachment_image', 'ayyash_add_no_js_attachment_image', 9999 );
		remove_filter( 'wp_get_attachment_image_attributes', 'ayyash_filter_attachment_image_attributes', 9999 );

		$html .= '<noscript>' . wp_get_attachment_image( $attachment_id, $size, $icon, $attr ) . '</noscript>';

		add_filter( 'wp_get_attachment_image_attributes', 'ayyash_filter_attachment_image_attributes', 9999 );
		add_filter( 'wp_get_attachment_image', 'ayyash_add_no_js_attachment_image', 9999, 5 );

		return $html;
	}
}

if ( ! function_exists( 'ayyash_extra_user_profile_fields' ) ) {
	/**
	 * Add Social Icon link in profile.php
	 */
	function ayyash_extra_user_profile_fields( $user ) {
		?>
		<h3><?php esc_html_e( 'Social Link', 'ayyash' ); ?></h3>

		<table class="form-table">
			<tr>
				<th><label for="ayyash_facebook_url"><?php esc_html_e( 'Facebook URL', 'ayyash' ); ?></label></th>
				<td>
					<input type="text" name="ayyash_facebook_url" id="ayyash_facebook_url" value="<?php echo esc_attr( get_the_author_meta( 'ayyash_facebook_url', $user->ID ) ); ?>" class="regular-text"/><br/>
					<span class="description"><?php esc_html_e( 'Enter Your Facebook Profile Link', 'ayyash' ); ?></span>
				</td>
			</tr>
			<tr>
				<th><label for="ayyash_instagram_url"><?php esc_html_e( 'Instagram URL', 'ayyash' ); ?></label></th>
				<td>
					<input type="text" name="ayyash_instagram_url" id="ayyash_instagram_url" value="<?php echo esc_attr( get_the_author_meta( 'ayyash_instagram_url', $user->ID ) ); ?>" class="regular-text"/><br/>
					<span class="description"><?php esc_html_e( 'Enter Your Instagram Profile Link', 'ayyash' ); ?></span>
				</td>
			</tr>
			<tr>
				<th><label for="ayyash_twitter_url"><?php esc_html_e( 'Twitter URL', 'ayyash' ); ?></label></th>
				<td>
					<input type="text" name="ayyash_twitter_url" id="ayyash_twitter_url" value="<?php echo esc_attr( get_the_author_meta( 'ayyash_twitter_url', $user->ID ) ); ?>" class="regular-text"/><br/>
					<span class="description"><?php esc_html_e( 'Enter Your Twitter Profile Link', 'ayyash' ); ?></span>
				</td>
			</tr>
			<tr>
				<th><label for="ayyash_pinterest_url"><?php esc_html_e( 'Pinterest URL', 'ayyash' ); ?></label></th>
				<td>
					<input type="text" name="ayyash_pinterest_url" id="ayyash_pinterest_url" value="<?php echo esc_attr( get_the_author_meta( 'ayyash_pinterest_url', $user->ID ) ); ?>" class="regular-text"/><br/>
					<span class="description"><?php esc_html_e( 'Enter Your Pinterest Profile Link', 'ayyash' ); ?></span>
				</td>
			</tr>

		</table>
	<?php }
}

if ( ! function_exists( 'ayyash_save_extra_user_profile_fields' ) ) {
	/**
	 * Save user profile fields
	 *
	 * @param $user_id
	 * @return false|void
	 */
	function ayyash_save_extra_user_profile_fields( $user_id ) {

		if ( ! current_user_can( 'edit_user', $user_id ) ) {
			return false;
		}

		check_admin_referer( 'update-user_' . $user_id );

		$facebook_url  = isset( $_POST['ayyash_facebook_url'] ) ? sanitize_text_field( wp_unslash( $_POST['ayyash_facebook_url'] ) ) : '';
		$instagram_url = isset( $_POST['ayyash_instagram_url'] ) ? sanitize_text_field( wp_unslash( $_POST['ayyash_instagram_url'] ) ) : '';
		$twitter_url   = isset( $_POST['ayyash_twitter_url'] ) ? sanitize_text_field( wp_unslash( $_POST['ayyash_twitter_url'] ) ) : '';
		$pinterest_url = isset( $_POST['ayyash_pinterest_url'] ) ? sanitize_text_field( wp_unslash( $_POST['ayyash_pinterest_url'] ) ) : '';

		update_user_meta( $user_id, 'ayyash_facebook_url', $facebook_url );
		update_user_meta( $user_id, 'ayyash_instagram_url', $instagram_url );
		update_user_meta( $user_id, 'ayyash_twitter_url', $twitter_url );
		update_user_meta( $user_id, 'ayyash_pinterest_url', $pinterest_url );

	}
}
