<?php
/**
 * Custom Functions
 * @package Architect Contractor
 * @since 1.0.0
 */

if( !function_exists('architect_contractor_site_logo') ):

    /**
     * Logo & Description
     */
    /**
     * Displays the site logo, either text or image.
     *
     * @param array $architect_contractor_args Arguments for displaying the site logo either as an image or text.
     * @param boolean $architect_contractor_echo Echo or return the HTML.
     *
     * @return string $architect_contractor_html Compiled HTML based on our arguments.
     */
    function architect_contractor_site_logo( $architect_contractor_args = array(), $architect_contractor_echo = true ){
        $architect_contractor_logo = get_custom_logo();
        $architect_contractor_site_title = get_bloginfo('name');
        $architect_contractor_contents = '';
        $architect_contractor_classname = '';
        $architect_contractor_defaults = array(
            'logo' => '%1$s<span class="screen-reader-text">%2$s</span>',
            'logo_class' => 'site-logo site-branding',
            'title' => '<a href="%1$s" class="custom-logo-name">%2$s</a>',
            'title_class' => 'site-title',
            'home_wrap' => '<h1 class="%1$s">%2$s</h1>',
            'single_wrap' => '<div class="%1$s">%2$s</div>',
            'condition' => (is_front_page() || is_home()) && !is_page(),
        );
        $architect_contractor_args = wp_parse_args($architect_contractor_args, $architect_contractor_defaults);
        /**
         * Filters the arguments for `architect_contractor_site_logo()`.
         *
         * @param array $architect_contractor_args Parsed arguments.
         * @param array $architect_contractor_defaults Function's default arguments.
         */
        $architect_contractor_args = apply_filters('architect_contractor_site_logo_args', $architect_contractor_args, $architect_contractor_defaults);
        
        $architect_contractor_show_logo  = get_theme_mod('architect_contractor_display_logo', false);
        $architect_contractor_show_title = get_theme_mod('architect_contractor_display_title', true);

        if ( has_custom_logo() && $architect_contractor_show_logo ) {
            $architect_contractor_contents .= sprintf($architect_contractor_args['logo'], $architect_contractor_logo, esc_html($architect_contractor_site_title));
            $architect_contractor_classname = $architect_contractor_args['logo_class'];
        }

        if ( $architect_contractor_show_title ) {
            $architect_contractor_contents .= sprintf($architect_contractor_args['title'], esc_url(get_home_url(null, '/')), esc_html($architect_contractor_site_title));
            // If logo isn't shown, fallback to title class for wrapper.
            if ( !$architect_contractor_show_logo ) {
                $architect_contractor_classname = $architect_contractor_args['title_class'];
            }
        }

        // If nothing is shown (logo or title both disabled), exit early
        if ( empty($architect_contractor_contents) ) {
            return;
        }

        $architect_contractor_wrap = $architect_contractor_args['condition'] ? 'home_wrap' : 'single_wrap';
        // $architect_contractor_wrap = 'home_wrap';
        $architect_contractor_html = sprintf($architect_contractor_args[$architect_contractor_wrap], $architect_contractor_classname, $architect_contractor_contents);
        /**
         * Filters the arguments for `architect_contractor_site_logo()`.
         *
         * @param string $architect_contractor_html Compiled html based on our arguments.
         * @param array $architect_contractor_args Parsed arguments.
         * @param string $architect_contractor_classname Class name based on current view, home or single.
         * @param string $architect_contractor_contents HTML for site title or logo.
         */
        $architect_contractor_html = apply_filters('architect_contractor_site_logo', $architect_contractor_html, $architect_contractor_args, $architect_contractor_classname, $architect_contractor_contents);
        if (!$architect_contractor_echo) {
            return $architect_contractor_html;
        }
        echo $architect_contractor_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    }

endif;

if( !function_exists('architect_contractor_site_description') ):

    /**
     * Displays the site description.
     *
     * @param boolean $architect_contractor_echo Echo or return the html.
     *
     * @return string $architect_contractor_html The HTML to display.
     */
    function architect_contractor_site_description($architect_contractor_echo = true){

        if ( get_theme_mod('architect_contractor_display_header_text', false) == true ) :
        $architect_contractor_description = get_bloginfo('description');
        if (!$architect_contractor_description) {
            return;
        }
        $architect_contractor_wrapper = '<div class="site-description"><span>%s</span></div><!-- .site-description -->';
        $architect_contractor_html = sprintf($architect_contractor_wrapper, esc_html($architect_contractor_description));
        /**
         * Filters the html for the site description.
         *
         * @param string $architect_contractor_html The HTML to display.
         * @param string $architect_contractor_description Site description via `bloginfo()`.
         * @param string $architect_contractor_wrapper The format used in case you want to reuse it in a `sprintf()`.
         * @since 1.0.0
         *
         */
        $architect_contractor_html = apply_filters('architect_contractor_site_description', $architect_contractor_html, $architect_contractor_description, $architect_contractor_wrapper);
        if (!$architect_contractor_echo) {
            return $architect_contractor_html;
        }
        echo $architect_contractor_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        endif;
    }

endif;

if( !function_exists('architect_contractor_posted_on') ):

    /**
     * Prints HTML with meta information for the current post-date/time.
     */
    function architect_contractor_posted_on( $architect_contractor_icon = true, $architect_contractor_animation_class = '' ){

        $architect_contractor_default = architect_contractor_get_default_theme_options();
        $architect_contractor_post_date = absint( get_theme_mod( 'architect_contractor_post_date',$architect_contractor_default['architect_contractor_post_date'] ) );

        if( $architect_contractor_post_date ){

            $architect_contractor_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
            if (get_the_time('U') !== get_the_modified_time('U')) {
                $architect_contractor_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
            }

            $architect_contractor_time_string = sprintf($architect_contractor_time_string,
                esc_attr(get_the_date(DATE_W3C)),
                esc_html(get_the_date()),
                esc_attr(get_the_modified_date(DATE_W3C)),
                esc_html(get_the_modified_date())
            );

            $architect_contractor_year = get_the_date('Y');
            $architect_contractor_month = get_the_date('m');
            $architect_contractor_day = get_the_date('d');
            $link = get_day_link($architect_contractor_year, $architect_contractor_month, $architect_contractor_day);

            $architect_contractor_posted_on = '<a href="' . esc_url($link) . '" rel="bookmark">' . $architect_contractor_time_string . '</a>';

            echo '<div class="entry-meta-item entry-meta-date">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $architect_contractor_animation_class ).'">';

            if( $architect_contractor_icon ){

                echo '<span class="entry-meta-icon calendar-icon"> ';
                architect_contractor_the_theme_svg('calendar');
                echo '</span>';

            }

            echo '<span class="posted-on">' . $architect_contractor_posted_on . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;

if( !function_exists('architect_contractor_posted_by') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function architect_contractor_posted_by( $architect_contractor_icon = true, $architect_contractor_animation_class = '' ){   

        $architect_contractor_default = architect_contractor_get_default_theme_options();
        $architect_contractor_post_author = absint( get_theme_mod( 'architect_contractor_post_author',$architect_contractor_default['architect_contractor_post_author'] ) );

        if( $architect_contractor_post_author ){

            echo '<div class="entry-meta-item entry-meta-author">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $architect_contractor_animation_class ).'">';

            if( $architect_contractor_icon ){
            
                echo '<span class="entry-meta-icon author-icon"> ';
                architect_contractor_the_theme_svg('user');
                echo '</span>';
                
            }

            $architect_contractor_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';
            echo '<span class="byline"> ' . $architect_contractor_byline . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;


if( !function_exists('architect_contractor_posted_by_avatar') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function architect_contractor_posted_by_avatar( $architect_contractor_date = false ){

        $architect_contractor_default = architect_contractor_get_default_theme_options();
        $architect_contractor_post_author = absint( get_theme_mod( 'architect_contractor_post_author',$architect_contractor_default['architect_contractor_post_author'] ) );

        if( $architect_contractor_post_author ){

            echo '<div class="entry-meta-left">';
            echo '<div class="entry-meta-item entry-meta-avatar">';
            echo wp_kses_post( get_avatar( get_the_author_meta( 'ID' ) ) );
            echo '</div>';
            echo '</div>';

            echo '<div class="entry-meta-right">';

            $architect_contractor_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';

            echo '<div class="entry-meta-item entry-meta-byline"> ' . $architect_contractor_byline . '</div>';

            if( $architect_contractor_date ){
                architect_contractor_posted_on($architect_contractor_icon = false);
            }
            echo '</div>';

        }

    }

endif;

if( !function_exists('architect_contractor_entry_footer') ):

    /**
     * Prints HTML with meta information for the categories, tags and comments.
     */
    function architect_contractor_entry_footer( $architect_contractor_cats = true, $architect_contractor_tags = true, $architect_contractor_edits = true){   

        $architect_contractor_default = architect_contractor_get_default_theme_options();
        $architect_contractor_post_category = absint( get_theme_mod( 'architect_contractor_post_category',$architect_contractor_default['architect_contractor_post_category'] ) );
        $architect_contractor_post_tags = absint( get_theme_mod( 'architect_contractor_post_tags',$architect_contractor_default['architect_contractor_post_tags'] ) );

        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            if( $architect_contractor_cats && $architect_contractor_post_category ){

                /* translators: used between list items, there is a space after the comma */
                $architect_contractor_categories = get_the_category();
                if ($architect_contractor_categories) {
                    echo '<div class="entry-meta-item entry-meta-categories">';
                    echo '<div class="entry-meta-wrapper">';
                
                    /* translators: 1: list of categories. */
                    echo '<span class="cat-links">';
                    foreach( $architect_contractor_categories as $architect_contractor_category ){

                        $architect_contractor_cat_name = $architect_contractor_category->name;
                        $architect_contractor_cat_slug = $architect_contractor_category->slug;
                        $architect_contractor_cat_url = get_category_link( $architect_contractor_category->term_id );
                        ?>

                        <a class="twp_cat_<?php echo esc_attr( $architect_contractor_cat_slug ); ?>" href="<?php echo esc_url( $architect_contractor_cat_url ); ?>" rel="category tag"><?php echo esc_html( $architect_contractor_cat_name ); ?></a>

                    <?php }
                    echo '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';
                }

            }

            if( $architect_contractor_tags && $architect_contractor_post_tags ){
                /* translators: used between list items, there is a space after the comma */
                $architect_contractor_tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'architect-contractor'));
                if( $architect_contractor_tags_list ){

                    echo '<div class="entry-meta-item entry-meta-tags">';
                    echo '<div class="entry-meta-wrapper">';

                    /* translators: 1: list of tags. */
                    echo '<span class="tags-links">';
                    echo wp_kses_post($architect_contractor_tags_list) . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';

                }

            }

            if( $architect_contractor_edits ){

                edit_post_link(
                    sprintf(
                        wp_kses(
                        /* translators: %s: Name of current post. Only visible to screen readers */
                            __('Edit <span class="screen-reader-text">%s</span>', 'architect-contractor'),
                            array(
                                'span' => array(
                                    'class' => array(),
                                ),
                            )
                        ),
                        get_the_title()
                    ),
                    '<span class="edit-link">',
                    '</span>'
                );
            }

        }
    }

endif;

if ( ! function_exists( 'architect_contractor_post_thumbnail' ) ) :

    /**
     * Displays an optional post thumbnail.
     *
     * Shows background style image with height class on archive/search/front,
     * and a normal inline image on single post/page views.
     */
    function architect_contractor_post_thumbnail( $architect_contractor_image_size = 'medium' ) {

        if ( post_password_required() || is_attachment() ) {
            return;
        }

        // Fallback image path
        $architect_contractor_default_image = get_template_directory_uri() . '/inc/homepage-setup/assets/homepage-setup-images/post-img1.png';

        // Image size → height class map
        $architect_contractor_size_class_map = array(
            'full'      => 'data-bg-large',
            'large'     => 'data-bg-big',
            'medium'    => 'data-bg-medium',
            'small'     => 'data-bg-small',
            'xsmall'    => 'data-bg-xsmall',
            'thumbnail' => 'data-bg-thumbnail',
        );

        $architect_contractor_class = isset( $architect_contractor_size_class_map[ $architect_contractor_image_size ] )
            ? $architect_contractor_size_class_map[ $architect_contractor_image_size ]
            : 'data-bg-medium';

        if ( is_singular() ) {
            the_post_thumbnail();
        } else {
            // 🔵 On archives → use background image style
            $architect_contractor_image = has_post_thumbnail()
                ? wp_get_attachment_image_src( get_post_thumbnail_id(), $architect_contractor_image_size )
                : array( $architect_contractor_default_image );

            $architect_contractor_bg_image = isset( $architect_contractor_image[0] ) ? $architect_contractor_image[0] : $architect_contractor_default_image;
            ?>
            <div class="post-thumbnail data-bg <?php echo esc_attr( $architect_contractor_class ); ?>"
                 data-background="<?php echo esc_url( $architect_contractor_bg_image ); ?>">
                <a href="<?php the_permalink(); ?>" class="theme-image-responsive" tabindex="0"></a>
            </div>
            <?php
        }
    }

endif;

if( !function_exists('architect_contractor_is_comment_by_post_author') ):

    /**
     * Comments
     */
    /**
     * Check if the specified comment is written by the author of the post commented on.
     *
     * @param object $architect_contractor_comment Comment data.
     *
     * @return bool
     */
    function architect_contractor_is_comment_by_post_author($architect_contractor_comment = null){

        if (is_object($architect_contractor_comment) && $architect_contractor_comment->user_id > 0) {
            $architect_contractor_user = get_userdata($architect_contractor_comment->user_id);
            $post = get_post($architect_contractor_comment->comment_post_ID);
            if (!empty($architect_contractor_user) && !empty($post)) {
                return $architect_contractor_comment->user_id === $post->post_author;
            }
        }
        return false;
    }

endif;

if( !function_exists('architect_contractor_breadcrumb') ) :

    /**
     * Architect Contractor Breadcrumb
     */
    function architect_contractor_breadcrumb($architect_contractor_comment = null){

        echo '<div class="entry-breadcrumb">';
        architect_contractor_breadcrumb_trail();
        echo '</div>';

    }

endif;