<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package ansupa
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */

function ansupa_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	// Header color scheme 
	$theme_color_scheme = get_theme_mod( 'ansupa_theme_color_scheme');
	$classes[] =  esc_attr( $theme_color_scheme );

	// When  color scheme is light or dark.
	$color_scheme = get_theme_mod( 'ansupa_color_scheme', 'default' );
	$classes[] = esc_attr( $color_scheme ) . '-version';
	
	// When global archive layout is checked.
	if ( is_archive() || ansupa_is_latest_posts() || is_404() || is_search() ) {
		$archive_sidebar = get_theme_mod( 'ansupa_archive_sidebar', 'right' ); 
		$classes[] = esc_attr( $archive_sidebar ) . '-sidebar';
	} else if ( is_single() ) { // When global post sidebar is checked.
    	$ansupa_post_sidebar_meta = get_post_meta( get_the_ID(), 'ansupa-select-sidebar', true );
    	if ( ! empty( $ansupa_post_sidebar_meta ) ) {
			$classes[] = esc_attr( $ansupa_post_sidebar_meta ) . '-sidebar';
    	} else {
			$global_post_sidebar = get_theme_mod( 'ansupa_global_post_layout', 'right' ); 
			$classes[] = esc_attr( $global_post_sidebar ) . '-sidebar';
    	}
	} elseif ( ansupa_is_frontpage_blog() || is_page() ) {
		if ( ansupa_is_frontpage_blog() ) {
			$page_id = get_option( 'page_for_posts' );
		} else {
			$page_id = get_the_ID();
		}

    	$ansupa_page_sidebar_meta = get_post_meta( $page_id, 'ansupa-select-sidebar', true );
		if ( ! empty( $ansupa_page_sidebar_meta ) ) {
			$classes[] = esc_attr( $ansupa_page_sidebar_meta ) . '-sidebar';
		} else {
			$global_page_sidebar = get_theme_mod( 'ansupa_global_page_layout', 'right' ); 
			$classes[] = esc_attr( $global_page_sidebar ) . '-sidebar';
		}
	}

    $classes[] = 'header-2 dark-version';

   if ( ! is_active_sidebar( 'front-page-sidebar' ) && ansupa_is_frontpage() ) { 
        $classes[] = 'no-home-sidebar';
   }

    
	// Site layout classes
	$site_layout = get_theme_mod( 'ansupa_site_layout', 'wide' );
	$classes[] = esc_attr( $site_layout ) . '-layout'; 

	return $classes;
}
add_filter( 'body_class', 'ansupa_body_classes' );

function ansupa_post_classes( $classes ) {
	if ( ansupa_is_page_displays_posts() ) {
		// Search 'has-post-thumbnail' returned by default and remove it.
		$key = array_search( 'has-post-thumbnail', $classes );
		unset( $classes[ $key ] );
		
		$archive_img_enable = get_theme_mod( 'ansupa_enable_archive_featured_img', true );

		if( has_post_thumbnail() && $archive_img_enable ) {
			$classes[] = 'has-post-thumbnail';
		} else {
			$classes[] = 'no-post-thumbnail';
		}
	}
  
	return $classes;
}
add_filter( 'post_class', 'ansupa_post_classes' );

/**
 * Excerpt length
 * 
 * @since ansupa 1.0.0
 * @return Excerpt length
 */
function ansupa_excerpt_length( $length ){
	if ( is_admin() ) {
		return $length;
	}

	$length = get_theme_mod( 'ansupa_archive_excerpt_length', 60 );
	return $length;
}
add_filter( 'excerpt_length', 'ansupa_excerpt_length', 999 );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function ansupa_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'ansupa_pingback_header' );

/**
 * Get an array of post id and title.
 * 
 */
function ansupa_get_post_choices() {
	$choices = array( '' => esc_html__( '--Select--', 'ansupa' ) );
	$args = array( 'numberposts' => -1, );
	$posts = get_posts( $args );

	foreach ( $posts as $post ) {
		$id = $post->ID;
		$title = $post->post_title;
		$choices[ $id ] = $title;
	}

	return $choices;
    wp_reset_postdata();
}

function ansupa_get_page_choices() {
    $pages = get_pages();
    $choices = array();
    $choices[0] = esc_html__( '--Select--', 'ansupa' );
    foreach ( $pages as $page ) {
        $choices[ $page->ID ] = $page->post_title;
    }
    return  $choices;
}

/**
 * Get an array of cat id and title.
 * 
 */
function ansupa_get_post_cat_choices() {
  $choices = array( '' => esc_html__( '--Select--', 'ansupa' ) );
	$cats = get_categories();

	foreach ( $cats as $cat ) {
		$id = $cat->term_id;
		$title = $cat->name;
		$choices[ $id ] = $title;
	}

	return $choices;
}

/**
 * Get an list of category sluf.
 * 
 */
function ansupa_get_category_slug_list($post_id) {
    $cat_list = wp_get_post_categories($post_id);
    $cat_slug = "";
    foreach($cat_list as $cat_id){
        $cat = get_category($cat_id);
        $cat_slug = $cat_slug . "".$cat->slug . " ";
    }

    return $cat_slug;
}


/**
 * Checks to see if we're on the homepage or not.
 */
function ansupa_is_frontpage() {
	return ( is_front_page() && ! is_home() );
}

/**
 * Checks to see if Static Front Page is set to "Your latest posts".
 */
function ansupa_is_latest_posts() {
	return ( is_front_page() && is_home() );
}

/**
 * Checks to see if Static Front Page is set to "Posts page".
 */
function ansupa_is_frontpage_blog() {
	return ( is_home() && ! is_front_page() );
}

/**
 * Checks to see if the current page displays any kind of post listing.
 */
function ansupa_is_page_displays_posts() {
	return ( ansupa_is_frontpage_blog() || is_search() || is_archive() || ansupa_is_latest_posts() );
}

/**
 * Shows a breadcrumb for all types of pages.  This is a wrapper function for the Breadcrumb_Trail class,
 * which should be used in theme templates.
 *
 * @since  1.0.0
 * @access public
 * @param  array $args Arguments to pass to Breadcrumb_Trail.
 * @return void
 */
function ansupa_breadcrumb( $args = array() ) {
	$breadcrumb = apply_filters( 'breadcrumb_trail_object', null, $args );

	if ( ! is_object( $breadcrumb ) )
		$breadcrumb = new Breadcrumb_Trail( $args );

	return $breadcrumb->trail();
}

/**
 * Pagination in archive/blog/search pages.
 */
function ansupa_posts_pagination() { 
	$archive_pagination = get_theme_mod( 'ansupa_archive_pagination_type', 'numeric' );
	if ( 'disable' === $archive_pagination ) {
		return;
	}
	if ( 'numeric' === $archive_pagination ) {
		the_posts_pagination( array(
            'prev_text'          => ansupa_get_svg( array( 'icon' => 'left-arrow' ) ),
            'next_text'          => ansupa_get_svg( array( 'icon' => 'left-arrow' ) ),
        ) );
	} elseif ( 'older_newer' === $archive_pagination ) {
        the_posts_navigation( array(
            'prev_text'          => ansupa_get_svg( array( 'icon' => 'left' ) ) . '<span>'. esc_html__( 'Older', 'ansupa' ) .'</span>',
            'next_text'          => '<span>'. esc_html__( 'Newer', 'ansupa' ) .'</span>' . ansupa_get_svg( array( 'icon' => 'right' ) ),
        )  );
	}
}

function ansupa_get_svg_by_url( $url = false ) {
	if ( ! $url ) {
		return false;
	}

	$social_icons = ansupa_social_links_icons();

	foreach ( $social_icons as $attr => $value ) {
		if ( false !== strpos( $url, $attr ) ) {
			return ansupa_get_svg( array( 'icon' => esc_attr( $value ) ) );
		}
	}
}

if ( ! function_exists( 'ansupa_the_excerpt' ) ) :

  /**
   * Generate excerpt.
   *
   * @since 1.0.0
   *
   * @param int     $length Excerpt length in words.
   * @param WP_Post $post_obj WP_Post instance (Optional).
   * @return string Excerpt.
   */
  function ansupa_the_excerpt( $length = 0, $post_obj = null ) {

    global $post;

    if ( is_null( $post_obj ) ) {
      $post_obj = $post;
    }

    $length = absint( $length );

    if ( 0 === $length ) {
      return;
    }

    $source_content = $post_obj->post_content;

    if ( ! empty( $post_obj->post_excerpt ) ) {
      $source_content = $post_obj->post_excerpt;
    }

    $source_content = preg_replace( '`\[[^\]]*\]`', '', $source_content );
    $trimmed_content = wp_trim_words( $source_content, $length, '&hellip;' );
    return $trimmed_content;

  }

endif;

function ansupa_get_section_content( $section_name, $content_type, $content_count ){

    $content = array();


    if (  in_array( $content_type, array( 'post', 'page' ) ) ) {
    $content_id = array();
    if ( 'post' === $content_type ) {
        for ( $i=1; $i <= $content_count; $i++ ) { 
            $content_id[] = get_theme_mod( "ansupa_{$section_name}_{$content_type}_" . $i );
            } 
    }else {
        for ( $i=1; $i <= $content_count; $i++ ) { 
            $content_id[] = get_theme_mod( "ansupa_{$section_name}_{$content_type}_" . $i );
        }
    }
    $args = array(
        'post_type' => $content_type,
        'post__in' => (array)$content_id,   
        'orderby'   => 'post__in',
        'posts_per_page' => absint( $content_count ),
        'ignore_sticky_posts' => true,
    );

    } else {
        $cat_content_id = get_theme_mod( "ansupa_{$section_name}_{$content_type}" );
        $args = array(
            'cat' => $cat_content_id,   
            'posts_per_page' =>  absint( $content_count ),
        );
    }

    $query = new WP_Query( $args );
    if ( $query->have_posts() ) {
        $i = 0;
        while ( $query->have_posts() ) {
            $query->the_post();

            $content[$i]['id'] = get_the_id();
            $content[$i]['title'] = get_the_title();
            $content[$i]['url'] = get_the_permalink();
            $content[$i]['content'] = get_the_content();
            $i++;
        }
        wp_reset_postdata();
    }

    return $content;
}

// Add auto p to the palces where get_the_excerpt is being called.
add_filter( 'get_the_excerpt', 'wpautop' );

if ( ! function_exists( 'ansupa_category_title' ) ) :

function ansupa_category_title( $title ) {
    if ( is_category() ) {
        $title = single_cat_title( '', false );
    }
    return $title;
}
add_filter( 'get_the_archive_title', 'ansupa_category_title' );

endif;

function ansupa_category_choices() {
    $tax_args = array(
        'hierarchical' => 0,
        'taxonomy'     => 'category',
    );
    $taxonomies = get_categories( $tax_args );
    $choices = array();
    $choices[0] = esc_html__( '--Select--', 'ansupa' );
    foreach ( $taxonomies as $tax ) {
        $choices[ $tax->term_id ] = $tax->name;
    }
    return  $choices;
}

function ansupa_return_social_icon( $social_link ) {
    // Get supported social icons.
    $social_icons = ansupa_social_links_icons();

    // Check in the URL for the url in the array.
    foreach ( $social_icons as $attr => $value ) {
        if ( false !== strpos( $social_link, $attr ) ) {
            return ansupa_get_svg( array( 'icon' => esc_attr( $value ) ) );
        }
    }
}


if ( ! function_exists( 'ansupa_ajax_enqueuer' ) ) :
    /**
     * ajax enquue
     *
     */
    function ansupa_ajax_enqueuer() {
        wp_register_script( "ansupa-ajax", get_template_directory_uri() . '/assets/js/blog-ajax.js', array( 'jquery' ), '', true );
        wp_localize_script( 'ansupa-ajax', 'ansupa', array( 'ajaxurl' => admin_url( 'admin-ajax.php' ) ) );        
        wp_enqueue_script( 'ansupa-ajax' );
    }
endif;
add_action( 'wp_enqueue_scripts', 'ansupa_ajax_enqueuer' );

if ( ! function_exists( 'ansupa_latest_posts_ajax_handler' ) ) :
    /**
     * ajax handler
     *
     */
    function ansupa_latest_posts_ajax_handler(){
        $blog = get_theme_mod( 'ansupa_blog_posts', 'cat' );
        $blog_num = get_theme_mod( 'ansupa_blog_posts_count', 4 ) ;

        $page = isset( $_POST['LBpageNumber'] ) ? absint( wp_unslash( $_POST['LBpageNumber'] ) ) : 1;
        header("Content-Type: text/html");

        $blog_content_type  = get_theme_mod( 'ansupa_blog_posts', 'cat' );
                
        $blog_posts_excerpt = get_theme_mod( 'ansupa_blog_posts_excerpt', 20 );

        // Query if the content type is either post or page.
        if (  in_array( $blog, array( 'post', 'page' ) ) ) {
            $content_id = array();
            if ( 'post' === $blog ) {
                for ( $i=1; $i <= $blog_num; $i++ ) {
                    $content_id[] = get_theme_mod( "ansupa_blog_posts{$blog}_" . $i );
                    }
            }else {
                for ( $i=1; $i <= $blog_num; $i++ ) {
                    $content_id[] = get_theme_mod( "ansupa_blog_posts{$blog}_" . $i );
                }
            }
            $args = array(
                'post_type' => $blog,
                'post__in' => (array)$content_id,
                'orderby'   => 'post__in',
                'posts_per_page' => absint( $blog_num ),
                'ignore_sticky_posts' => true,
                'orderby'           => 'post__in',
                'post_status'       => array( 'publish' ),
                'paged'             => $page,
            );
        } else {
            $cat_content_id = get_theme_mod( 'ansupa_blog_posts_cat' );
            $args = array(
                'post_type' => 'post',
                'cat' => $cat_content_id,
                'posts_per_page' =>  absint( $blog_num ),
                'ignore_sticky_posts' => true,
                'post_status'       => array( 'publish' ),
                'paged' => $page,

            );
        }
        $latest_posts = new WP_Query( $args );

        if ( $latest_posts -> have_posts() ) : while ( $latest_posts -> have_posts() ) : $latest_posts -> the_post(); 
             $banner_thumbnail = !empty( get_the_post_thumbnail_url( ) ) ? get_the_post_thumbnail_url( get_the_id(), 'medium-large' ) : get_template_directory_uri(). '/assets/img/no-featured-image.jpg';
            ?>
            <article class="hentry">
                <div class="featured-image" style="background-image:url('<?php echo esc_url( get_the_post_thumbnail_url( $content['id'] ) ) ; ?>')">
                    <a href="<?php echo esc_url( $content['url'] ); ?>" class="post-thumbnail-link"></a>
                </div>

                <div class="entry-container">
                    <div class="entry-meta">
                        <span class="cat-links">
                            <?php the_category( '', '', $content['id'] ); ?>
                        </span>
                    </div><!-- .entry-meta -->

                    <header class="entry-header">
                        <h2 class="entry-title"><a href="<?php the_permalink(); ?>" tabindex="0"><?php the_title(); ?></a></h2>
                    </header>

                    <div class="entry-content">
                        <p><?php echo esc_html( wp_trim_words( get_the_content(), $blog_posts_excerpt ) ); ?></p>
                    </div>

                    <div class="footer-meta">
                        <?php 

                        ansupa_post_author() ;

                        ansupa_posted_on( get_the_id() );

                        ?>

                        <?php if( $blog_posts !== 'page' ): ?>
                        <span class="comment"><a class="url fn n" href="<?php echo esc_url( $content['url'] ); ?>"><?php echo get_comments_number( $content['id'] ); ?></a></span>
                    <?php endif; ?>
                    </div><!-- .entry-meta -->
                </div><!-- .entry-container -->
            </article>

        <?php endwhile; endif;
        wp_reset_postdata();
        ?>
        <?php die();
    }
endif;
add_action("wp_ajax_ansupa_latest_posts_ajax_handler", "ansupa_latest_posts_ajax_handler");
add_action("wp_ajax_nopriv_ansupa_latest_posts_ajax_handler", "ansupa_latest_posts_ajax_handler");