<?php
/**
 * MeanThemes\Ampy\Sidebars\Component class
 *
 * @package ampy
 */

namespace MeanThemes\Ampy\Sidebars;

use MeanThemes\Ampy\Component_Interface;
use MeanThemes\Ampy\Templating_Component_Interface;
use function add_action;
use function add_filter;
use function register_sidebar;
use function esc_html__;
use function is_active_sidebar;
use function dynamic_sidebar;

/**
 * Class for managing sidebars.
 *
 * Exposes template tags:
 * * `ampy()->is_primary_sidebar_active()`
 * * `ampy()->display_primary_sidebar()`
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/
 */
class Component implements Component_Interface, Templating_Component_Interface {

	const PRIMARY_SIDEBAR_SLUG     = 'sidebar-1';
	const HEADER_HERO_WIDGETS_SLUG = 'header-hero-widgets';
	const FOOTER_HERO_WIDGETS_SLUG = 'footer-hero-widgets';
	const FOOTER_WIDGETS_SLUG      = 'footer-widgets';

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug() : string {
		return 'sidebars';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize() {
		add_action( 'widgets_init', array( $this, 'action_register_sidebars' ) );
		add_filter( 'body_class', array( $this, 'filter_body_classes' ) );
	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `ampy()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */
	public function template_tags() : array {
		return array(
			'is_primary_sidebar_active'     => array( $this, 'is_primary_sidebar_active' ),
			'display_primary_sidebar'       => array( $this, 'display_primary_sidebar' ),
			'is_header_hero_widgets_active' => array( $this, 'is_header_hero_widgets_active' ),
			'display_header_hero_widgets'   => array( $this, 'display_header_hero_widgets' ),
			'is_footer_hero_widgets_active' => array( $this, 'is_footer_hero_widgets_active' ),
			'display_footer_hero_widgets'   => array( $this, 'display_footer_hero_widgets' ),
			'display_footer_widgets'        => array( $this, 'display_footer_widgets' ),
			'is_footer_widgets_active'      => array( $this, 'is_footer_widgets_active' ),
		);
	}

	/**
	 * Registers the sidebars.
	 */
	public function action_register_sidebars() {
		register_sidebar(
			array(
				'name'          => esc_html__( 'Sidebar', 'ampy' ),
				'id'            => static::PRIMARY_SIDEBAR_SLUG,
				'description'   => esc_html__( 'Add widgets here.', 'ampy' ),
				'before_widget' => '<section id="%1$s" class="widget %2$s">',
				'after_widget'  => '</section>',
				'before_title'  => '<h3 class="widget-title">',
				'after_title'   => '</h3>',
			)
		);
		register_sidebar(
			array(
				'name'          => esc_html__( 'Header Hero', 'ampy' ),
				'id'            => static::HEADER_HERO_WIDGETS_SLUG,
				'description'   => esc_html__( 'Add widgets here.', 'ampy' ),
				'before_widget' => '<section id="%1$s" class="widget %2$s">',
				'after_widget'  => '</section>',
				'before_title'  => '<h3 class="widget-title">',
				'after_title'   => '</h3>',
			)
		);
		register_sidebar(
			array(
				'name'          => esc_html__( 'Footer Hero', 'ampy' ),
				'id'            => static::FOOTER_HERO_WIDGETS_SLUG,
				'description'   => esc_html__( 'Add widgets here.', 'ampy' ),
				'before_widget' => '<section id="%1$s" class="widget %2$s">',
				'after_widget'  => '</section>',
				'before_title'  => '<h3 class="widget-title">',
				'after_title'   => '</h3>',
			)
		);
		register_sidebar(
			array(
				'name'          => esc_html__( 'Footer widgets', 'ampy' ),
				'id'            => static::FOOTER_WIDGETS_SLUG,
				'description'   => esc_html__( 'Add widgets here.', 'ampy' ),
				'before_widget' => '<section id="%1$s" class="widget %2$s">',
				'after_widget'  => '</section>',
				'before_title'  => '<h3 class="widget-title">',
				'after_title'   => '</h3>',
			)
		);
	}

	/**
	 * Adds custom classes to indicate whether a sidebar is present to the array of body classes.
	 *
	 * @param array $classes Classes for the body element.
	 * @return array Filtered body classes.
	 */
	public function filter_body_classes( array $classes ) : array {
		if ( $this->is_primary_sidebar_active() ) {
			global $template;

			if ( ! in_array( basename( $template ), array( 'front-page.php', 'grid-template.php', '404.php', '500.php', 'offline.php' ) ) ) {
				$classes[] = 'has-sidebar';
			}
		}

		return $classes;
	}

	/**
	 * Checks whether the primary sidebar is active.
	 *
	 * @return bool True if the primary sidebar is active, false otherwise.
	 */
	public function is_primary_sidebar_active() : bool {
		return (bool) is_active_sidebar( static::PRIMARY_SIDEBAR_SLUG ) && ! is_page_template( 'no-sidebar-template.php' );
	}

	/**
	 * Displays the primary sidebar.
	 */
	public function display_primary_sidebar() {
		dynamic_sidebar( static::PRIMARY_SIDEBAR_SLUG );
	}

	/**
	 * Checks whether the header widgetized area is active.
	 *
	 * @return bool True if the header widgetized area is active, false otherwise.
	 */
	public function is_header_hero_widgets_active() : bool {
		return (bool) is_active_sidebar( static::HEADER_HERO_WIDGETS_SLUG );
	}

	/**
	 * Displays the header widgetized area.
	 */
	public function display_header_hero_widgets() {
		dynamic_sidebar( static::HEADER_HERO_WIDGETS_SLUG );
	}

	/**
	 * Checks whether the footer hero widgetized area is active.
	 *
	 * @return bool True if the footer hero  widgetized area is active, false otherwise.
	 */
	public function is_footer_hero_widgets_active() : bool {
		return (bool) is_active_sidebar( static::FOOTER_HERO_WIDGETS_SLUG );
	}

	/**
	 * Displays the footer hero  widgetized area.
	 */
	public function display_footer_hero_widgets() {
		dynamic_sidebar( static::FOOTER_HERO_WIDGETS_SLUG );
	}

	/**
	 * Checks whether the footer widgetized area is active.
	 *
	 * @return bool True if the footer widgetized area is active, false otherwise.
	 */
	public function is_footer_widgets_active() : bool {
		return (bool) is_active_sidebar( static::FOOTER_WIDGETS_SLUG );
	}

	/**
	 * Displays the hero widgetized area.
	 */
	public function display_footer_widgets() {
		dynamic_sidebar( static::FOOTER_WIDGETS_SLUG );
	}
}
