<?php
/**
 * Amalie functions and definitions
 *
 * Set up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme you can override certain functions (those wrapped
 * in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before
 * the parent theme's file, so the child theme functions would be used.
 *
 * @link https://codex.wordpress.org/Theme_Development
 * @link https://codex.wordpress.org/Child_Themes
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters,
 * {@link https://codex.wordpress.org/Plugin_API}
 *
 * @package Amalie
 * @since Amalie 1.0
 */

/**
 * Set the content width based on the theme's design and stylesheet.
 *
 * @since Amalie 1.0
 */
if ( ! isset( $content_width ) ) {
	$content_width = 792;
}
if ( ! function_exists( 'amalie_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * @since Amalie 1.0
 */
function amalie_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on amalie, use a find and replace
	 * to change 'amalie-lite' to the name of your theme in all the template files
	 */
	load_theme_textdomain( 'amalie-lite', get_template_directory() . '/languages' );
	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );
	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );
	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * See: https://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 990, 510, true );
	// This theme uses wp_nav_menu() in two locations.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu',      'amalie-lite' ),
		'social'  => esc_html__( 'Social Links Menu', 'amalie-lite' ),
	) );
	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form', 'comment-form', 'comment-list', 'gallery', 'caption'
	) );
	/*
	 * Enable support for Post Formats.
	 *
	 * See: https://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'post-formats', array(
		'aside', 'image', 'video', 'quote', 'link', 'gallery', 'status', 'audio', 'chat'
	) );
	// Setup the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'amalie_custom_background_args', array(
		'default-color'      => '#2b2a26',
		'default-attachment' => 'fixed',
	) ) );
	/*
	 * Enable support for custom logo.
	 *
	 *  @since Amalie Lite 1.1
	 */
	add_theme_support( 'custom-logo', array(
		'height'      => 9999,
		'width'       => 9999,
		'flex-height' => true,
	) );
	/*
	 * This theme styles the visual editor to resemble the theme style,
	 * specifically font, colors, icons, and column width.
	 */
	add_editor_style( array( 'css/editor-style.css', 'genericons/genericons.css', amalie_fonts_url() ) );
	
	// Adding support for core block visual styles.
	add_theme_support( 'wp-block-styles' );
	
	// Add support for full and wide align images.
	add_theme_support( 'align-wide' );
		
	// Add support for custom color scheme.
	add_theme_support( 'editor-color-palette', array(
			array(
				'name'  => esc_html__( 'Green', 'amalie-lite' ),
				'slug'  => 'strong-green',
				'color' => '#597D04',
			),
		) );
}
endif; // amalie_setup
add_action( 'after_setup_theme', 'amalie_setup' );
/**
 * Register widget area.
 *
 * @since Amalie 1.0
 *
 * @link https://codex.wordpress.org/Function_Reference/register_sidebar
 */
function amalie_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Widget Area', 'amalie-lite' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'amalie-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'amalie_widgets_init' );
if ( ! function_exists( 'amalie_fonts_url' ) ) :
/**
 * Register Google fonts for amalie.
 *
 * @since Amalie 1.0
 *
 * @return string Google fonts URL for the theme.
 */
function amalie_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';
	/* translators: If there are characters in your language that are not supported by Noto Sans, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== esc_html_x( 'on', 'Noto Sans font: on or off', 'amalie-lite' ) ) {
		$fonts[] = 'Noto Sans:400italic,700italic,400,700';
	}
	/* translators: To add an additional character subset specific to your language, translate this to 'greek', 'cyrillic', 'devanagari' or 'vietnamese'. Do not translate into your own language. */
	$subset = esc_html_x( 'no-subset', 'Add new subset (greek, cyrillic, devanagari, vietnamese)', 'amalie-lite' );
	if ( 'cyrillic' == $subset ) {
		$subsets .= ',cyrillic,cyrillic-ext';
	} elseif ( 'greek' == $subset ) {
		$subsets .= ',greek,greek-ext';
	} elseif ( 'devanagari' == $subset ) {
		$subsets .= ',devanagari';
	} elseif ( 'vietnamese' == $subset ) {
		$subsets .= ',vietnamese';
	}
	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'display' => 'swap',
			'subset' => urlencode( $subsets ),
		), '//fonts.googleapis.com/css' );
	}
	require_once get_theme_file_path( 'inc/wptt-webfont-loader.php' );

	return wptt_get_webfont_url( esc_url_raw( $fonts_url ) );
}
endif;
/**
 * Enqueue scripts and styles.
 *
 * @since Amalie 1.0
 */
function amalie_scripts() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'amalie-fonts', amalie_fonts_url(), array(), null );
	// Add Genericons, used in the main stylesheet.
	wp_enqueue_style( 'genericons', get_template_directory_uri() . '/genericons/genericons.css', array(), '3.2' );
	// Load our main stylesheet.
	wp_enqueue_style( 'amalie-style', get_stylesheet_uri() );
	wp_enqueue_script( 'amalie-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20141010', true );
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
	if ( is_singular() && wp_attachment_is_image() ) {
		wp_enqueue_script( 'amalie-keyboard-image-navigation', get_template_directory_uri() . '/js/keyboard-image-navigation.js', array( 'jquery' ), '20141010' );
	}
	wp_enqueue_script( 'amalie-script', get_template_directory_uri() . '/js/functions.js', array( 'jquery' ), '20141212', true );
	wp_localize_script( 'amalie-script', 'screenReaderText', array(
		'expand'   => '<span class="screen-reader-text">' . esc_html__( 'expand child menu', 'amalie-lite' ) . '</span>',
		'collapse' => '<span class="screen-reader-text">' . esc_html__( 'collapse child menu', 'amalie-lite' ) . '</span>',
	) );
}
add_action( 'wp_enqueue_scripts', 'amalie_scripts' );

/**
 * Enqueue theme styles within Gutenberg.
 */
function amalie_gutenberg_styles() {

	// Load the theme styles within Gutenberg.
	wp_enqueue_style( 'amalie-gutenberg', get_theme_file_uri( '/editor.css' ), false, '1.1.1', 'all' );

	// Add custom fonts to Gutenberg.
	wp_enqueue_style( 'amalie-fonts', amalie_fonts_url(), array(), null );
}
add_action( 'enqueue_block_editor_assets', 'amalie_gutenberg_styles' );

if (!function_exists('amalie_admin_scripts')) {
	function amalie_admin_scripts($hook) {
			wp_enqueue_style('amalie-admin', get_template_directory_uri() . '/admin/admin.css');
	}
}
add_action('admin_enqueue_scripts', 'amalie_admin_scripts');
/** 
* JavaScript Detection. 
* 
* Adds a `js` class to the root `<html>` element when JavaScript is detected. 
* 
* @since Amalie 1.0
 */ 
function amalie_javascript_detection() { 
 	echo "<script>(function(){document.documentElement.className='js'})();</script>\n"; 
} 
add_action( 'wp_head', 'amalie_javascript_detection', 0 );
/**
 * Add featured image as background image to post navigation elements.
 *
 * @since Amalie 1.0
 *
 * @see wp_add_inline_style()
 */
function amalie_post_nav_background() {
	if ( ! is_single() ) {
		return;
	}
	$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '', true );
	$next     = get_adjacent_post( false, '', false );
	$css      = '';
	if ( is_attachment() && 'attachment' == $previous->post_type ) {
		return;
	}
	if ( $previous &&  has_post_thumbnail( $previous->ID ) ) {
		$prevthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $previous->ID ), 'post-thumbnail' );
		$css .= '
			.post-navigation .nav-previous { background-image: url(' . esc_url( $prevthumb[0] ) . '); }
			.post-navigation .nav-previous .post-title, .post-navigation .nav-previous a:hover .post-title, .post-navigation .nav-previous .meta-nav { color: #fff; }
			.post-navigation .nav-previous a:before { background-color: rgba(0, 0, 0, 0.4); }
		';
	}
	if ( $next && has_post_thumbnail( $next->ID ) ) {
		$nextthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $next->ID ), 'post-thumbnail' );
		$css .= '
			.post-navigation .nav-next { background-image: url(' . esc_url( $nextthumb[0] ) . '); }
			.post-navigation .nav-next .post-title, .post-navigation .nav-next a:hover .post-title, .post-navigation .nav-next .meta-nav { color: #fff; }
			.post-navigation .nav-next a:before { background-color: rgba(0, 0, 0, 0.4); }
		';
	}
	wp_add_inline_style( 'amalie-style', $css );
}
add_action( 'wp_enqueue_scripts', 'amalie_post_nav_background' );
/**
 * Display descriptions in main navigation.
 *
 * @since Amalie 1.0
 *
 * @param string  $item_output The menu item output.
 * @param WP_Post $item        Menu item object.
 * @param int     $depth       Depth of the menu.
 * @param array   $args        wp_nav_menu() arguments.
 * @return string Menu item with possible description.
 */
function amalie_nav_description( $item_output, $item, $depth, $args ) {
	if ( 'primary' == $args->theme_location && $item->description ) {
		$item_output = str_replace( $args->link_after . '</a>', '<div class="menu-item-description">' . $item->description . '</div>' . $args->link_after . '</a>', $item_output );
	}
	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'amalie_nav_description', 10, 4 );
/**
 * Add a `screen-reader-text` class to the search form's submit button.
 *
 * @since Amalie 1.0
 *
 * @param string $html Search form HTML.
 * @return string Modified search form HTML.
 */
function amalie_search_form_modify( $html ) {
	return str_replace( 'class="search-submit"', 'class="search-submit screen-reader-text"', $html );
}
add_filter( 'get_search_form', 'amalie_search_form_modify' );
/***** Include Admin *****/

if (is_admin()) {
	require_once('admin/admin.php');
}
/**
 * Theme Update Script
 *
 * Runs if version number saved in theme_mod "version" doesn't match current theme version.
 */
function amalie_update_check() {
	
// Return if update has already been run
	if ( -1 == get_theme_mod( 'custom_logo', -1 ) ) {
		return;
	}
	
	// If we're not on 3.5 yet, exit now
	if ( ! function_exists( 'the_custom_logo' ) ) {
		return;
	}
	// If a logo has been set previously, update to use logo feature introduced in WordPress 4.5
	if ( function_exists( 'the_custom_logo' ) && get_theme_mod( 'amalie_logo', false ) ) {
		// Since previous logo was stored a URL, convert it to an attachment ID
		$logo = attachment_url_to_postid( get_theme_mod( 'amalie_logo' ) );
		if ( is_int( $logo ) ) {
			set_theme_mod( 'custom_logo', attachment_url_to_postid( get_theme_mod( 'amalie_logo' ) ) );
		}
		remove_theme_mod( 'amalie_logo' );
	}
}
add_action( 'after_setup_theme', 'amalie_update_check' );
/**
 * Implement the Custom Header feature.
 *
 * @since Amalie 1.0
 */
require get_template_directory() . '/inc/custom-header.php';
/**
 * Custom template tags for this theme.
 *
 * @since Amalie 1.0
 */
require get_template_directory() . '/inc/template-tags.php';
/**
 * Customizer additions.
 *
 * @since Amalie 1.0
 */
require get_template_directory() . '/inc/customizer.php';
/**
 * Load Jetpack compatibility file.
 *
 * @since Amalie 1.0
 */
require( get_template_directory() . '/inc/jetpack.php' );