<?php
/**
 * Function describe for All2 
 * 
 * @package all2
 */

/**
 * Set a constant that holds the theme's minimum supported PHP version.
 */
define( 'ALL2_MIN_PHP_VERSION', '7.0' );

/**
 * Immediately after theme switch is fired we we want to check php version and
 * revert to previously active theme if version is below our minimum.
 */
add_action( 'after_switch_theme', 'all2_test_for_min_php' );

/**
 * Switches back to the previous theme if the minimum PHP version is not met.
 */
function all2_test_for_min_php() {

	// Compare versions.
	if ( version_compare( PHP_VERSION, ALL2_MIN_PHP_VERSION, '<' ) ) {
		// Site doesn't meet themes min php requirements, add notice...
		add_action( 'admin_notices', 'all2_min_php_not_met_notice' );
		// ... and switch back to previous theme.
		switch_theme( get_option( 'theme_switched' ) );
		return false;

	};
}

if ( ! function_exists( 'wp_body_open' ) ) {
        function wp_body_open() {
                do_action( 'wp_body_open' );
        }
}

/**
 * An error notice that can be displayed if the Minimum PHP version is not met.
 */
function all2_min_php_not_met_notice() {
	?>
	<div class="notice notice-error is_dismissable">
		<p>
			<?php esc_html_e( 'You need to update your PHP version to run this theme.', 'all2' ); ?> <br />
			<?php
			printf(
				/* translators: 1 is the current PHP version string, 2 is the minmum supported php version string of the theme */
				esc_html__( 'Actual version is: %1$s, required version is: %2$s.', 'all2' ),
				PHP_VERSION,
				ALL2_MIN_PHP_VERSION
			); // phpcs: XSS ok.
			?>
		</p>
	</div>
	<?php
}


if ( ! function_exists( 'all2_setup' ) ) :
	/**
	 * all2 setup.
	 *
	 * Set up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support post thumbnails.
	 *
	 */
	function all2_setup() {

		// Define and register starter content to showcase the theme on new sites.
		$starter_content = array(

			'widgets' => array(

				'footer-column-1-widget-area' => array(
					'recent-comments'
				),

				'footer-column-2-widget-area' => array(
					'recent-posts'
				),

				'footer-column-3-widget-area' => array(
					'calendar'
				),
			),

			'posts' => array(
				'home',
				'blog',
				'about',
				'contact'
			),

			// Default to a static front page and assign the front and posts pages.
			'options' => array(
				'show_on_front' => 'page',
				'page_on_front' => '{{home}}',
				'page_for_posts' => '{{blog}}',
			),

			// Set the front page section theme mods to the IDs of the core-registered pages.
			'theme_mods' => array(
				'all2_slider_display' => 1,
				'all2_slide1_image' => esc_url( get_stylesheet_directory_uri() . '/images/slider/1.jpg' ),
				'all2_slide2_image' => esc_url( get_stylesheet_directory_uri() . '/images/slider/2.jpg' ),
				'all2_slide3_image' => esc_url( get_stylesheet_directory_uri() . '/images/slider/3.jpg' ),
			),

			'nav_menus' => array(
				// Assign a menu to the "primary" location.
				'primary' => array(
					'name' => __( 'Primary Menu', 'all2' ),
					'items' => array(
						'link_home',
						'page_blog',
						'page_contact',
						'page_about',
					),
				),

				// Assign a menu to the "footer" location.
				'footer' => array(
					'name' => __( 'Footer Menu', 'all2' ),
					'items' => array(
						'link_home',
						'page_about',
						'page_blog',
						'page_contact',
					),
				),
			),
		);

		$starter_content = apply_filters( 'all2_starter_content', $starter_content );
		add_theme_support( 'starter-content', $starter_content );
	}
endif; // all2_setup
add_action( 'after_setup_theme', 'all2_setup' );




add_action( 'wp_enqueue_scripts', 'all2_enqueue_styles' );
function all2_enqueue_styles() {

  wp_enqueue_style( 'allingrid-stylesheet', get_template_directory_uri() . '/style.css' );
	wp_enqueue_style( 'all2-child-style', get_stylesheet_uri(), array( 'all2-stylesheet' ) );
}

function all2_load_scripts() {

	wp_enqueue_script( 'pgwslideshow-js',
		get_stylesheet_directory_uri() . '/js/pgwslideshow.js',
		array('jquery') );

	wp_enqueue_script( 'all2-js', get_stylesheet_directory_uri() . '/js/all2.js',
			array( 'jquery', 'pgwslideshow-js' ) );
}
add_action( 'wp_enqueue_scripts', 'all2_load_scripts' );

if ( ! function_exists( 'all2_display_slider' ) ) :
	/**
	 * Displays the slider
	 */
	function all2_display_slider() { ?>
		<div class="container">
			<ul class="pgwSlideshow">
				<?php
					// display slides
					for ( $i = 1; $i <= 3; ++$i ) {
							
							$defaultSlideImage = get_stylesheet_directory_uri().'/images/slider/' . $i .'.jpg';

							$slideImage = get_theme_mod( 'all2_slide'.$i.'_image', $defaultSlideImage );
						?>
							<li>
								<img src="<?php echo esc_attr( $slideImage ); ?>" />
							</li>
		<?php		} ?>
			</ul>
		</div>
	<?php 
	}
endif; // all2_display_slider

if ( ! function_exists( 'all2_sanitize_checkbox' ) ) :
	/**
	 * Sanitization callback for 'checkbox' type controls. This callback sanitizes `$checked`
	 * as a boolean value, either TRUE or FALSE.
	 *
	 * @param bool $checked Whether the checkbox is checked.
	 * @return bool Whether the checkbox is checked.
	 */
	function all2_sanitize_checkbox( $checked ) {
		// Boolean check.
		return ( ( isset( $checked ) && true == $checked ) ? true : false );
	}
endif; // all2_sanitize_checkbox

if ( ! function_exists( 'all2_sanitize_html' ) ) :

	function all2_sanitize_html( $html ) {
		return wp_kses_post( $html );
	}

endif; // all2_sanitize_html

if ( ! function_exists( 'all2_sanitize_url' ) ) :

	function all2_sanitize_url( $url ) {
		return esc_url_raw( $url );
	}

endif; // all2_sanitize_url

if ( ! function_exists( 'all2_customize_register' ) ) :
	/**
	 * Register theme settings in the customizer
	 */
	function all2_customize_register( $wp_customize ) {

		/**
		 * Add Slider Section
		 */
		$wp_customize->add_section(
			'all2_slider_section',
			array(
				'title'       => __( 'Slider', 'all2' ),
				'capability'  => 'edit_theme_options',
			)
		);

		// Add display slider option
		$wp_customize->add_setting(
				'all2_slider_display',
				array(
						'default'           => 0,
						'sanitize_callback' => 'all2_sanitize_checkbox',
				)
		);

		$wp_customize->add_control( new WP_Customize_Control( $wp_customize, 'all2_slider_display',
								array(
									'label'          => __( 'Display Slider on a Static Front Page', 'all2' ),
									'section'        => 'all2_slider_section',
									'settings'       => 'all2_slider_display',
									'type'           => 'checkbox',
								)
							)
		);
		
		for ($i = 1; $i <= 3; ++$i) {

			$slideImageId = 'all2_slide'.$i.'_image';
			$defaultSliderImagePath = get_stylesheet_directory_uri().'/images/slider/'.$i.'.jpg';
			
			// Add Slide Image
			$wp_customize->add_setting( $slideImageId,
				array(
					'default' => $defaultSliderImagePath,
					'sanitize_callback' => 'all2_sanitize_url'
				)
			);

			$wp_customize->add_control( new WP_Customize_Image_Control( $wp_customize, $slideImageId,
					array(
						'label'   	 => sprintf( esc_html__( 'Slide #%s Image', 'all2' ), $i ),
						'section' 	 => 'all2_slider_section',
						'settings'   => $slideImageId,
					) 
				)
			);
		}
	}
endif; // all2_customize_register
add_action('customize_register', 'all2_customize_register');