<?php
/**
 * The functions file is used to initialize everything in the theme.  It controls how the theme is loaded and
 * sets up the supported features, default actions, and default filters.  If making customizations, users
 * should create a child theme and make changes to its functions.php file (not this one).  Friends don't let
 * friends modify parent theme files. ;)
 *
 * Child themes should do their setup on the 'after_setup_theme' hook with a priority of 11 if they want to
 * override parent theme features.  Use a priority of 9 if wanting to run before the parent theme.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * @package Adroa
 * @subpackage Template
 * @version 0.1.2
 * @since 0.1.0
 * @author Thiago Senna <thiago@thremes.com.br>
 * @copyright Copyright (c) 2012, Thiago Senna
 * @link http://thremes.com.br/themes/adroa
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

/* Load the core theme framework. */
require_once( trailingslashit( get_template_directory() ) . 'library/hybrid.php' );
new Hybrid();

/* Do theme setup on the 'after_setup_theme' hook. */
add_action( 'after_setup_theme', 'adroa_theme_setup' );

/**
 * Theme setup function.  This function adds support for theme features and defines the default theme
 * actions and filters.
 *
 * @since 0.1.0
 */
function adroa_theme_setup() {

    /* Get action/filter hook prefix. */
    $prefix = hybrid_get_prefix();

    /* Add theme support for core framework features. */
    add_theme_support( 'hybrid-core-menus', array( 'primary', 'subsidiary' ) );
    add_theme_support( 'hybrid-core-sidebars', array( 'primary' ) );
    add_theme_support( 'hybrid-core-widgets' );
    add_theme_support( 'hybrid-core-shortcodes' );
    add_theme_support( 'hybrid-core-theme-settings', array( 'about', 'footer' ) );
    add_theme_support( 'hybrid-core-drop-downs' );
    add_theme_support( 'hybrid-core-template-hierarchy' );

    /* Add theme support for framework extensions. */
    add_theme_support( 'theme-layouts', array( '1c', '2c-l', '2c-r' ) );
    add_theme_support( 'post-stylesheets' );
    add_theme_support( 'dev-stylesheet' );
    add_theme_support( 'loop-pagination' );
    add_theme_support( 'get-the-image' );
    add_theme_support( 'breadcrumb-trail' );
    add_theme_support( 'cleaner-gallery' );

    /* Add theme support for WordPress features. */
    add_theme_support( 'automatic-feed-links' );
    add_theme_support( 'post-formats', array( 'image', 'gallery' ) );

    /* Add support for WordPress custom background. */
    add_theme_support(
        'custom-background',
        array(
            'default-image' => trailingslashit( get_template_directory_uri() ) . 'images/graphy.png',
            'wp-head-callback' => 'adroa_custom_background_callback'
        )
    );

    /* Add support for WordPress custom header image. */
    add_theme_support(
        'custom-header',
        array(
            'wp-head-callback' => '__return_false',
            'admin-head-callback' => '__return_false',
            'header-text' => false,
            'default-image' => 'remove-header',
            'width' => 960,
            'height' => 210
        )
    );
	/* Add featured-header support. */
	require_once( trailingslashit( get_template_directory() ) . 'featured-header.php' );

    /* Embed width/height defaults. */
    add_filter( 'embed_defaults', 'adroa_embed_defaults' );

    /* Set content width. */
    hybrid_set_content_width( 600 );

    /* Filter the sidebar widgets. */
    add_filter( 'sidebars_widgets', 'adroa_disable_sidebars' );
    add_action( 'template_redirect', 'adroa_one_column' );

    /* Add classes to the comments pagination. */
    add_filter( 'previous_comments_link_attributes', 'adroa_previous_comments_link_attributes' );
    add_filter( 'next_comments_link_attributes', 'adroa_next_comments_link_attributes' );

    /* Wraps <blockquote> around quote posts. */
    add_filter( 'the_content', 'adroa_quote_content' );

    /* Adds the featured image to image posts if no content is found. */
    add_filter( 'the_content', 'adroa_image_content' );

    /* Filters the image/gallery post format archive galleries. */
    add_filter( "{$prefix}_archive_portfolio_item_columns", 'adroa_archive_portfolio_item_columns' );

    /* Load some required font styles */
    add_action( 'wp_enqueue_scripts', 'adroa_load_font_styles' );
}

/**
 * Sets the number of columns to show on portfolio archives pages based on the
 * layout that is currently being used.
 *
 * @since 0.1.0
 * @param int $columns Number of portfolio item columns to display.
 * @return int $columns
 */
function adroa_archive_portfolio_item_columns( $columns ) {

    /* Only run the code if the theme supports the 'theme-layouts' feature. */
    if ( current_theme_supports( 'theme-layouts' ) ) {

        /* Get the current theme layout. */
        $layout = theme_layouts_get_layout();

        if ( 'layout-1c' == $layout )
            $columns = 3;
    }

    return $columns;
}

/**
 * Wraps the output of the quote post format content in a <blockquote> element if the user hasn't added a
 * <blockquote> in the post editor.
 *
 * @since 0.1.0
 * @param string $content The post content.
 * @return string $content
 */
function adroa_quote_content( $content ) {

    if ( has_post_format( 'quote' ) ) {
        preg_match( '/<blockquote.*?>/', $content, $matches );

        if ( empty( $matches ) )
            $content = "<blockquote>{$content}</blockquote>";
    }

    return $content;
}

/**
 * Returns the featured image for the image post format if the user didn't add any content to the post.
 *
 * @since 0.1.0
 * @param string $content The post content.
 * @return string $content
 */
function adroa_image_content( $content ) {

    if ( has_post_format( 'image' ) && '' == $content ) {
        if ( is_singular() )
            $content = get_the_image( array( 'size' => 'full', 'meta_key' => false, 'link_to_post' => false ) );
        else
            $content = get_the_image( array( 'size' => 'full', 'meta_key' => false ) );
    }

    return $content;
}

/**
 * Function for deciding which pages should have a one-column layout.
 *
 * @since 0.1.0
 */
function adroa_one_column() {

    if ( !is_active_sidebar( 'primary' ) && !is_active_sidebar( 'secondary' ) )
        add_filter( 'get_theme_layout', 'adroa_theme_layout_one_column' );

    elseif ( is_attachment() && 'layout-default' == theme_layouts_get_layout() )
        add_filter( 'get_theme_layout', 'adroa_theme_layout_one_column' );
}

/**
 * Filters 'get_theme_layout' by returning 'layout-1c'.
 *
 * @since 0.1.0
 * @param string $layout The layout of the current page.
 * @return string
 */
function adroa_theme_layout_one_column( $layout ) {
    return 'layout-1c';
}

/**
 * Disables sidebars if viewing a one-column page.
 *
 * @since 0.1.0
 * @param array $sidebars_widgets A multidimensional array of sidebars and widgets.
 * @return array $sidebars_widgets
 */
function adroa_disable_sidebars( $sidebars_widgets ) {

    if ( current_theme_supports( 'theme-layouts' ) && !is_admin() ) {

        if ( 'layout-1c' == theme_layouts_get_layout() ) {
            $sidebars_widgets['primary'] = false;
            $sidebars_widgets['secondary'] = false;
        }
    }

    return $sidebars_widgets;
}

/**
 * Overwrites the default widths for embeds.  This is especially useful for making sure videos properly
 * expand the full width on video pages.  This function overwrites what the $content_width variable handles
 * with context-based widths.
 *
 * @since 0.1.0
 */
function adroa_embed_defaults( $args ) {

    $args['width'] = hybrid_get_content_width();

    if ( current_theme_supports( 'theme-layouts' ) ) {

        $layout = theme_layouts_get_layout();

        if ( 'layout-1c' == $layout )
            $args['width'] = 940;
    }

    return $args;
}

/**
 * Adds 'class="prev" to the previous comments link.
 *
 * @since 0.1.0
 * @param string $attributes The previous comments link attributes.
 * @return string
 */
function adroa_previous_comments_link_attributes( $attributes ) {
    return $attributes . ' class="prev"';
}

/**
 * Adds 'class="next" to the next comments link.
 *
 * @since 0.1.0
 * @param string $attributes The next comments link attributes.
 * @return string
 */
function adroa_next_comments_link_attributes( $attributes ) {
    return $attributes . ' class="next"';
}

/**
 * Returns the number of images attached to the current post in the loop.
 *
 * @since 0.1.0
 * @return int
 */
function adroa_get_image_attachment_count() {
    $images = get_children( array( 'post_parent' => get_the_ID(), 'post_type' => 'attachment', 'post_mime_type' => 'image', 'numberposts' => -1 ) );
    return count( $images );
}

/**
 * Returns a set of image attachment links based on size.
 *
 * @author Justin Tadlock
 * @link http://justintadlock.com
 * @since 0.1.0
 * @return string Links to various image sizes for the image attachment.
 */
function adroa_get_image_size_links() {

    /* If not viewing an image attachment page, return. */
    if ( !wp_attachment_is_image( get_the_ID() ) )
        return;

    /* Set up an empty array for the links. */
    $links = array();

    /* Get the intermediate image sizes and add the full size to the array. */
    $sizes = get_intermediate_image_sizes();
    $sizes[] = 'full';

    /* Loop through each of the image sizes. */
    foreach ( $sizes as $size ) {

        /* Get the image source, width, height, and whether it's intermediate. */
        $image = wp_get_attachment_image_src( get_the_ID(), $size );

        /* Add the link to the array if there's an image and if $is_intermediate (4th array value) is true or full size. */
        if ( !empty( $image ) && ( true === $image[3] || 'full' == $size ) )
            $links[] = "<a class='image-size-link' href='" . esc_url( $image[0] ) . "'>{$image[1]} &times; {$image[2]}</a>";
    }

    /* Join the links in a string and return. */
    return join( ' <span class="sep">/</span> ', $links );
}

/**
 * This is a fix for when a user sets a custom background color with no custom background image.  What
 * happens is the theme's background image hides the user-selected background color.  If a user selects a
 * background image, we'll just use the WordPress custom background callback.
 *
 * @since 0.1.0
 * @link http://core.trac.wordpress.org/ticket/16919
 */
function adroa_custom_background_callback() {

    /* Get the background image. */
    $image = get_background_image();

    /* If there's an image, just call the normal WordPress callback. We won't do anything here. */
    if ( !empty( $image ) ) {
        _custom_background_cb();
        return;
    }

    /* Get the background color. */
    $color = get_background_color();

    /* If no background color, return. */
    if ( empty( $color ) )
        return;

    /* Use 'background' instead of 'background-color'. */
    $style = "background: #{$color};";

    ?>
<style type="text/css">body.custom-background { <?php echo trim( $style ); ?> }</style>
<?php

}

/**
 * Displays an attachment image's metadata and exif data while viewing a singular attachment page.
 *
 * Note: This function will most likely be restructured completely in the future.  The eventual plan is to
 * separate each of the elements into an attachment API that can be used across multiple themes.  Keep
 * this in mind if you plan on using the current filter hooks in this function.
 *
 * @author Justin Tadlock
 * @link http://justintadlock.com
 * @since 0.1.0
 */
function adroa_image_info() {

    /* Set up some default variables and get the image metadata. */
    $meta = wp_get_attachment_metadata( get_the_ID() );
    $items = array();
    $list = '';

    /* Add the width/height to the $items array. */
    $items['dimensions'] = sprintf( __( '<span class="prep">Dimensions:</span> %s', 'adroa' ), '<span class="image-data"><a href="' . esc_url( wp_get_attachment_url() ) . '">' . sprintf( __( '%1$s &#215; %2$s pixels', 'adroa' ), $meta['width'], $meta['height'] ) . '</a></span>' );

    /* If a timestamp exists, add it to the $items array. */
    if ( !empty( $meta['image_meta']['created_timestamp'] ) )
        $items['created_timestamp'] = sprintf( __( '<span class="prep">Date:</span> %s', 'adroa' ), '<span class="image-data">' . date( get_option( 'date_format' ), $meta['image_meta']['created_timestamp'] ) . '</span>' );

    /* If a camera exists, add it to the $items array. */
    if ( !empty( $meta['image_meta']['camera'] ) )
        $items['camera'] = sprintf( __( '<span class="prep">Camera:</span> %s', 'adroa' ), '<span class="image-data">' . $meta['image_meta']['camera'] . '</span>' );

    /* If an aperture exists, add it to the $items array. */
    if ( !empty( $meta['image_meta']['aperture'] ) )
        $items['aperture'] = sprintf( __( '<span class="prep">Aperture:</span> %s', 'adroa' ), '<span class="image-data">' . sprintf( __( 'f/%s', 'adroa' ), $meta['image_meta']['aperture'] ) . '</span>' );

    /* If a focal length is set, add it to the $items array. */
    if ( !empty( $meta['image_meta']['focal_length'] ) )
        $items['focal_length'] = sprintf( __( '<span class="prep">Focal Length:</span> %s', 'adroa' ), '<span class="image-data">' . sprintf( __( '%s mm', 'adroa' ), $meta['image_meta']['focal_length'] ) . '</span>' );

    /* If an ISO is set, add it to the $items array. */
    if ( !empty( $meta['image_meta']['iso'] ) )
        $items['iso'] = sprintf( __( '<span class="prep">ISO:</span> %s', 'adroa' ), '<span class="image-data">' . $meta['image_meta']['iso'] . '</span>' );

    /* If a shutter speed is given, format the float into a fraction and add it to the $items array. */
    if ( !empty( $meta['image_meta']['shutter_speed'] ) ) {

        if ( ( 1 / $meta['image_meta']['shutter_speed'] ) > 1 ) {
            $shutter_speed = '1/';

            if ( number_format( ( 1 / $meta['image_meta']['shutter_speed'] ), 1 ) ==  number_format( ( 1 / $meta['image_meta']['shutter_speed'] ), 0 ) )
                $shutter_speed .= number_format( ( 1 / $meta['image_meta']['shutter_speed'] ), 0, '.', '' );
            else
                $shutter_speed .= number_format( ( 1 / $meta['image_meta']['shutter_speed'] ), 1, '.', '' );
        } else {
            $shutter_speed = $meta['image_meta']['shutter_speed'];
        }

        $items['shutter_speed'] = sprintf( __( '<span class="prep">Shutter Speed:</span> %s', 'adroa' ), '<span class="image-data">' . sprintf( __( '%s sec', 'adroa' ), $shutter_speed ) . '</span>' );
    }

    /* Allow devs to overwrite the array of items. */
    $items = apply_atomic( 'image_info_items', $items );

    /* Loop through the items, wrapping each in an <li> element. */
    foreach ( $items as $item )
        $list .= "<li>{$item}</li>";

    /* Format the HTML output of the function. */
    $output = '<div class="image-info"><h3>' . __( 'Image Info', 'adroa' ) . '</h3><ul>' . $list . '</ul></div>';

    /* Display the image info and allow devs to overwrite the final output. */
    echo apply_atomic( 'image_info', $output );
}

/**
 * Load specific font stylesheets needed by the theme.
 *
 * @since 0.1.0
 */
function adroa_load_font_styles() {

    wp_register_style( 'openSans', 'http://fonts.googleapis.com/css?family=Open+Sans:400,600,400italic' );
    wp_enqueue_style( 'openSans' );

    wp_register_style( 'ptSerif', 'http://fonts.googleapis.com/css?family=PT+Serif' );
    wp_enqueue_style( 'ptSerif' );
}

?>
