<?php
/**
 * SVG icons related functions and filters
 *
 * @package Adonis
 */

/**
 * Add SVG definitions to the footer.
 */
function adonis_include_svg_icons() {
	// Define SVG sprite file.
	$svg_icons = get_parent_theme_file_path( '/assets/images/svg-icons.svg' );

	// If it exists, include it.
	if ( file_exists( $svg_icons ) ) {
		require_once( $svg_icons );
	}
}
add_action( 'wp_footer', 'adonis_include_svg_icons', 9999 );


/**
 * Return SVG markup.
 *
 * @param array $args {
 *     Parameters needed to display an SVG.
 *
 *     @type string $icon  Required SVG icon filename.
 *     @type string $title Optional SVG title.
 *     @type string $desc  Optional SVG description.
 * }
 * @return string SVG markup.
 */
function adonis_get_svg( $args = array() ) {
	// Make sure $args are an array.
	if ( empty( $args ) ) {
		return __( 'Please define default parameters in the form of an array.', 'adonis' );
	}

	// Define an icon.
	if ( false === array_key_exists( 'icon', $args ) ) {
		return __( 'Please define an SVG icon filename.', 'adonis' );
	}

	// Set defaults.
	$defaults = array(
		'icon'        => '',
		'title'       => '',
		'desc'        => '',
		'fallback'    => false,
	);

	// Parse args.
	$args = wp_parse_args( $args, $defaults );

	// Set aria hidden.
	$aria_hidden = ' aria-hidden="true"';

	// Set ARIA.
	$aria_labelledby = '';

	/*
	 * adonis doesn't use the SVG title or description attributes; non-decorative icons are described with .screen-reader-text.
	 *
	 * However, child themes can use the title and description to add information to non-decorative SVG icons to improve accessibility.
	 *
	 * Example 1 with title: <?php echo adonis_get_svg( array( 'icon' => 'arrow-right', 'title' => __( 'This is the title', 'textdomain' ) ) ); ?>
	 *
	 * Example 2 with title and description: <?php echo adonis_get_svg( array( 'icon' => 'arrow-right', 'title' => __( 'This is the title', 'textdomain' ), 'desc' => __( 'This is the description', 'textdomain' ) ) ); ?>
	 *
	 * See https://www.paciellogroup.com/blog/2013/12/using-aria-enhance-svg-accessibility/.
	 */
	if ( $args['title'] ) {
		$aria_hidden     = '';
		$unique_id       = uniqid();
		$aria_labelledby = ' aria-labelledby="title-' . $unique_id . '"';

		if ( $args['desc'] ) {
			$aria_labelledby = ' aria-labelledby="title-' . $unique_id . ' desc-' . $unique_id . '"';
		}
	}

	// Begin SVG markup.
	$svg = '<svg class="icon icon-' . esc_attr( $args['icon'] ) . '"' . $aria_hidden . $aria_labelledby . ' role="img">';

	// Display the title.
	if ( $args['title'] ) {
		$svg .= '<title id="title-' . $unique_id . '">' . esc_html( $args['title'] ) . '</title>';

		// Display the desc only if the title is already set.
		if ( $args['desc'] ) {
			$svg .= '<desc id="desc-' . $unique_id . '">' . esc_html( $args['desc'] ) . '</desc>';
		}
	}

	/*
	 * Display the icon.
	 *
	 * The whitespace around `<use>` is intentional - it is a work around to a keyboard navigation bug in Safari 10.
	 *
	 * See https://core.trac.wordpress.org/ticket/38387.
	 */
	$svg .= ' <use href="#icon-' . esc_attr( $args['icon'] ) . '" xlink:href="#icon-' . esc_attr( $args['icon'] ) . '"></use> ';

	// Add some markup to use as a fallback for browsers that do not support SVGs.
	if ( $args['fallback'] ) {
		$svg .= '<span class="svg-fallback icon-' . esc_attr( $args['icon'] ) . '"></span>';
	}

	$svg .= '</svg>';

	return $svg;
}

/**
 * Display SVG icons in social links menu.
 *
 * @param  string  $item_output The menu item output.
 * @param  WP_Post $item        Menu item object.
 * @param  int     $depth       Depth of the menu.
 * @param  array   $args        wp_nav_menu() arguments.
 * @return string  $item_output The menu item output with social icon.
 */
function adonis_nav_menu_social_icons( $item_output, $item, $depth, $args ) {
	// Get supported social icons.
	$social_icons = adonis_social_links_icons();

	// Change SVG icon inside social links menu if there is supported URL.
	if ( 'social-links-menu' === $args->menu_class ) {
		foreach ( $social_icons as $attr => $value ) {
			if ( false !== strpos( $item_output, $attr ) ) {
				$item_output = str_replace( $args->link_after, '</span>' . adonis_get_svg( array( 'icon' => esc_attr( $value ) ) ), $item_output );
			}
		}
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'adonis_nav_menu_social_icons', 10, 4 );

/**
 * Add dropdown icon if menu item has children.
 *
 * @param  string $title The menu item's title.
 * @param  object $item  The current menu item.
 * @param  array  $args  An array of wp_nav_menu() arguments.
 * @param  int    $depth Depth of menu item. Used for padding.
 * @return string $title The menu item's title with dropdown icon.
 */
function adonis_dropdown_icon_to_menu_link( $title, $item, $args, $depth ) {
	if ( 'menu-1' === $args->theme_location ) {
		foreach ( $item->classes as $value ) {
			if ( 'menu-item-has-children' === $value || 'page_item_has_children' === $value ) {
				$title = $title . adonis_get_svg( array( 'icon' => 'caret-down' ) );
			}
		}
	}

	return $title;
}
add_filter( 'nav_menu_item_title', 'adonis_dropdown_icon_to_menu_link', 10, 4 );

/**
 * Returns an array of supported social links (URL and icon name).
 *
 * @return array $social_links_icons
 */
function adonis_social_links_icons() {
	// Supported social links icons.
	$social_links_icons = array(
		'apple.com'			=> 'apple',
		'amazon.com'		=> 'amazon',
		'behance.net'     	=> 'behance',
		'behance.com'     	=> 'behance',
		'codepen.io'      	=> 'codepen',
		'deviantart.com'  	=> 'deviantart',
		'digg.com'        	=> 'digg',
		'discord.com'		=> 'discord',
		'discord.gg'		=> 'discord',
		'dribbble.com'    	=> 'dribbble',
		'dropbox.com'     	=> 'dropbox',
		'facebook.com'    	=> 'facebook',
		'flickr.com'      	=> 'flickr',
		'foursquare.com'  	=> 'foursquare',
		'plus.google.com' 	=> 'google-plus',
		'github.com'      	=> 'github',
		'instagram.com'   	=> 'instagram',
		'komoot.com'	  	=> 'komoot',
		'line.me'			=> 'line',
		'linkedin.com'    	=> 'linkedin',
		'mailto:'         	=> 'envelope-o',
		'medium.com'      	=> 'medium',
		'pinterest.com'   	=> 'pinterest-p',
		'getpocket.com'   	=> 'get-pocket',
		'reddit.com'      	=> 'reddit-alien',
		'sketchfab.com'	  	=> 'sketchfab',
		'skype.com'       	=> 'skype',
		'skype:'          	=> 'skype',
		'slideshare.net'  	=> 'slideshare',
		'snapchat.com'    	=> 'snapchat-ghost',
		'soundcloud.com'  	=> 'soundcloud',
		'spotify.com'     	=> 'spotify',
		'stumbleupon.com' 	=> 'stumbleupon',
		'tumblr.com'      	=> 'tumblr',
		'twitch.tv'       	=> 'twitch',
		'twitter.com'     	=> 'twitter',
		'x.com'     		=> 'x-twitter',
		'vimeo.com'       	=> 'vimeo',
		'vine.co'         	=> 'vine',
		'vk.com'          	=> 'vk',
		'wordpress.org'   	=> 'wordpress',
		'wordpress.com'   	=> 'wordpress',
		'yelp.com'        	=> 'yelp',
		'youtube.com'     	=> 'youtube',
		'tiktok.com'     	=> 'tiktok',
		'/feed'				=> 'rss',
		'tel:'				=> 'phone',
		'bsky.app'		  	=> 'bluesky',
		'threads.com'	 	=> 'threads',
		'xing.com'	  	  	=> 'xing',
		't.me'	  		  	=> 'telegram',
		'fetlife.com'	  	=> 'fetlife',
		'bandcamp.com'     	=> 'bandcamp',
		'etsy.com'     		=> 'etsy',
	);

	/**
	 * Filter adonis social links icons.
	 *
	 * @param array $social_links_icons
	 */
	return apply_filters( 'adonis_social_links_icons', $social_links_icons );
}
