<?php
/**
 * Auxilary class for REST.
 *
 * @link       http://sproutient.com
 * @since      1.0.0
 *
 * @package    zypento
 * @subpackage zypento/includes
 */

namespace Zypento_Blocks\Misc;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Auxilary class for REST.
 *
 * @since      1.0.0
 * @package    zypento
 * @subpackage zypento/includes
 * @author     Sproutient <dev@sproutient.com>
 */
class Utilities {

	/**
	 * Retrieves WooCommerce terms (categories or tags).
	 *
	 * @param string $type Type of terms to retrieve ('categories' or 'tags').
	 * @return array Array of term IDs and names.
	 */
	public static function get_woo_terms( $type = 'categories' ) {

		$taxonomy = 'product_cat';
		if ( 'tags' === $type ) {
			$taxonomy = 'product_tag';
		}

		$terms = get_terms(
			array(
				'taxonomy'   => $taxonomy,
				'hide_empty' => false,
			)
		);

		$terms_array = array();
		foreach ( $terms as $term ) {
			if ( is_object( $term ) && isset( $term->term_id ) && isset( $term->name ) ) {
				$terms_array[ esc_html( $term->term_id ) ] = esc_html( $term->name );
			} elseif ( is_array( $term ) && array_key_exists( 'term_id', $term ) && array_key_exists( 'name', $term ) ) {
				$terms_array[ esc_html( $term['term_id'] ) ] = esc_html( $term['name'] );
			}
		}

		return $terms_array;

	}

	/**
	 * Retrieves data for a specific WooCommerce term.
	 *
	 * @param int $term_id The ID of the term to retrieve.
	 * @return array Array containing term data (id, name, link, desc).
	 */
	public static function get_woo_term_data( $term_id ) {

		$term = get_term( $term_id );
		if ( is_wp_error( $term ) || ! $term ) {
			return array();
		}

		return array(
			'id'   => esc_html( $term->term_id ),
			'name' => esc_html( $term->name ),
			'link' => esc_url( get_term_link( $term ) ),
			'desc' => esc_html( $term->description ),
		);

	}

	/**
	 * Retrieves product IDs by term and type.
	 *
	 * @param int    $term_id   The term ID to filter products.
	 * @param string $type      The type of products to retrieve ('featured', 'on_sale', 'recent', 'sold', 'rated').
	 * @param string $taxonomy  The taxonomy to filter products (default 'product_cat').
	 * @param int    $limit     Number of products to retrieve (default 8).
	 * @return array            Array of product IDs.
	 */
	public static function get_products_by_type( $term_id, $type = 'featured', $taxonomy = 'product_cat', $limit = 8 ) {
		$args = array(
			'post_type'      => 'product',
			'posts_per_page' => $limit,
			'fields'         => 'ids',
			'tax_query'      => array(
				array(
					'taxonomy' => $taxonomy,
					'field'    => 'term_id',
					'terms'    => $term_id,
				),
			),
			'post_status'    => 'publish',
		);

		switch ( $type ) {
			case 'featured':
				$args['tax_query'][] = array(
					'taxonomy' => 'product_visibility',
					'field'    => 'name',
					'terms'    => 'featured',
					'operator' => 'IN',
				);
				break;
			case 'on_sale':
				$product_ids_on_sale = wc_get_product_ids_on_sale();
				$args['post__in']    = ! empty( $product_ids_on_sale ) ? $product_ids_on_sale : array( 0 );
				break;
			case 'recent':
				$args['orderby'] = 'date';
				$args['order']   = 'DESC';
				break;
			case 'sold':
				$args['meta_key'] = 'total_sales';
				$args['orderby']  = 'meta_value_num';
				$args['order']    = 'DESC';
				break;
			case 'rated':
				$args['meta_key'] = '_wc_average_rating';
				$args['orderby']  = 'meta_value_num';
				$args['order']    = 'DESC';
				break;
		}

		$query = new \WP_Query( $args );
		return $query->posts;
	}

	/**
	 * Retrieves detailed information for a WooCommerce product.
	 *
	 * @param int $product_id The ID of the product to retrieve details for.
	 * @return array Array containing product details (id, name, rating, prices, cart info, images, etc.).
	 */
	public static function get_product_details( $product_id ) {
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return array();
		}

		// Default values.
		$in_cart        = false;
		$cart_qty       = 0;
		$variation_data = array();

		// Check if WooCommerce cart is available and product is in cart.
		if ( function_exists( 'WC' ) && WC()->cart ) {
			foreach ( WC()->cart->get_cart() as $cart_item ) {
				if ( $cart_item['product_id'] === $product_id || $cart_item['variation_id'] === $product_id ) {
					$in_cart  = true;
					$cart_qty = $cart_item['quantity'];
					// If this is a variation, get variation attributes.
					if ( $cart_item['variation_id'] ) {
						$variation_data = $cart_item['variation'];
					}
					break;
				}
			}
		}

		$is_variation         = $product->is_type( 'variation' );
		$parent_id            = $is_variation ? $product->get_parent_id() : null;
		$variation_attributes = $is_variation ? $product->get_attributes() : array();

		// Get featured image URL.
		$featured_image = '';
		if ( $product->get_image_id() ) {
			$featured_image = wp_get_attachment_url( $product->get_image_id() );
		}

		// Get gallery image URLs.
		$gallery           = array();
		$gallery_image_ids = $product->get_gallery_image_ids();
		foreach ( $gallery_image_ids as $img_id ) {
			$gallery[] = wp_get_attachment_url( $img_id );
		}

		return array(
			'id'                   => $product->get_id(),
			'name'                 => $product->get_name(),
			'rating'               => $product->get_average_rating(),
			'on_sale'              => $product->is_on_sale(),
			'sale_price'           => $product->get_sale_price(),
			'regular_price'        => $product->get_regular_price(),
			'sku'                  => $product->get_sku(),
			'in_cart'              => $in_cart,
			'cart_qty'             => $cart_qty,
			'is_variation'         => $is_variation,
			'parent_id'            => $parent_id,
			'variation_data'       => $variation_data,
			'variation_attributes' => $variation_attributes,
			'featured_image'       => $featured_image,
			'gallery'              => $gallery,
			'price_html'           => $product->get_price_html(),
			'sale_price_html'      => wc_price( $product->get_sale_price() ),
			'regular_price_html'   => wc_price( $product->get_regular_price() ),
			'type'                 => $product->get_type(),
		);
	}

}
