<?php
/**
 * The file that defines the stock notifier class
 *
 * @link       http://sproutient.com
 * @since      1.0.0
 *
 * @package    zypento
 * @subpackage zypento/includes
 */

namespace Zypento_Blocks\Features;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use \Zypento_Blocks\Misc\Utilities;

/**
 * The blocks class.
 *
 * @since      1.0.0
 * @package    zypento
 * @subpackage zypento/includes
 * @author     Sproutient <dev@sproutient.com>
 */
class Woo {

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		add_filter( 'zypento_blocks_js_variables', array( $this, 'public_variables' ) );
		add_action( 'init', array( $this, 'register_blocks' ) );
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );

	}

	/**
	 * Add Public JS variables.
	 *
	 * @since    1.0.0
	 * @access   public
	 * @param string $variables Current gutenberg status.
	 */
	public function public_variables( $variables ) {

		if ( ! array_key_exists( 'api', $variables ) ) {

			$variables['api'] = array();

		}

		$variables['api']['addToCart'] = get_rest_url( null, ZYPENTO_BLOCKS_NAMESPACE . '/add-to-cart' );

		if ( ! array_key_exists( 'labels', $variables ) ) {

			$variables['labels'] = array();

		}

		return $variables;

	}

	/**
	 * Register REST Routes.
	 */
	public function register_routes() {

		global $zypento_blocks_objects;

		register_rest_route(
			ZYPENTO_BLOCKS_NAMESPACE,
			'/add-to-cart',
			array(

				// Here we register the readable endpoint for collections.
				array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'process_add_to_cart' ),
					'args'                => array(
						'product' => array(
							'description'       => esc_html__( 'Name.', 'zypento-blocks' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_blocks_objects['rest_aux'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_blocks_objects['rest_aux'], 'sanitize_string' ),
							'required'          => true,
							'default'           => '',
						),
						'qty'     => array(
							'description'       => esc_html__( 'Email.', 'zypento-blocks' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_blocks_objects['rest_aux'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_blocks_objects['rest_aux'], 'sanitize_string' ),
							'required'          => true,
							'default'           => '',
						),
						'nonce'   => array(
							'description'       => esc_html__( 'Nonce.', 'zypento-blocks' ),
							'type'              => 'bool',
							'sanitize_callback' => function( $value ) {
								return (bool) $value;
							},
							'validate_callback' => function( $value ) {
								return wp_verify_nonce( $value, 'zypento-blocks' );
							},
							'required'          => true,
							'default'           => false,
						),

					),
					'permission_callback' => '__return_true',
				),
				// Register our schema callback.
				'schema' => array( $zypento_blocks_objects['rest_aux'], 'get_schema' ),

			)
		);

	}

	/**
	 * Add product to cart.
	 *
	 * @param WP_REST_Request $request Current request.
	 */
	public function process_add_to_cart( $request ) {

		global $zypento_blocks_objects;

		$data['result']         = false;
		$data['data']           = array();
		$data['data']['error']  = array();
		$data['data']['reason'] = esc_html__( 'Something went wrong.', 'zypento-blocks' );

		$nonce = '';
		if ( isset( $request['nonce'] ) ) {
			$nonce = $request['nonce'];
		}

		$product = '';
		if ( isset( $request['product'] ) ) {
			$product = $request['product'];
		}

		$quantity = '';
		if ( isset( $request['qty'] ) ) {
			$quantity = intval( $request['qty'] );
		}

		if ( $nonce && $product ) {

			if ( function_exists( 'WC' ) ) {
				if ( null === WC()->session ) {
					WC()->session = new \WC_Session_Handler();
					WC()->session->init();
				}
				if ( null === WC()->cart ) {
					WC()->cart = new \WC_Cart();
					WC()->cart->get_cart();
				}
			}

			if ( function_exists( 'WC' ) && WC()->cart ) {
				$cart          = WC()->cart;
				$cart_item_key = $cart->find_product_in_cart( $cart->generate_cart_id( $product ) );
				if ( $cart_item_key ) {
					if ( $quantity > 0 ) {
						$cart->set_quantity( $cart_item_key, $quantity );
						$data['result']          = true;
						$data['data']['message'] = esc_html__( 'Cart updated.', 'zypento-blocks' );
					} else {
						$cart->remove_cart_item( $cart_item_key );
						$data['result']          = true;
						$data['data']['message'] = esc_html__( 'Product removed from cart.', 'zypento-blocks' );
					}
				} else {
					if ( $quantity > 0 ) {
						$cart->add_to_cart( $product, $quantity );
						$data['result']          = true;
						$data['data']['message'] = esc_html__( 'Product added to cart.', 'zypento-blocks' );
					} else {
						$data['result']         = false;
						$data['data']['reason'] = esc_html__( 'Quantity must be greater than zero.', 'zypento-blocks' );
					}
				}
			} else {
				$data['data']['reason'] = esc_html__( 'Cart not available.', 'zypento-blocks' );
			}
		}

		return $data;
	}

	/**
	 * Register testimonial blocks.
	 *
	 * @since  1.0.0
	 * @access public
	 */
	public function register_blocks() {

		register_block_type(
			ZYPENTO_BLOCKS_PLUGIN_PATH . 'assets/js/blocks/woocommerce/collection',
			array(
				'render_callback' => array( $this, 'render_collection' ),
			)
		);

	}

	/**
	 * Render the WooCommerce collection block.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param array $atts Block attributes.
	 * @return string
	 */
	public function render_collection( $atts ) {

		ob_start();
		$this->render_collection_content( $atts );
		$content = ob_get_contents();
		ob_end_clean();
		return $content;

	}

	/**
	 * Render the content for the WooCommerce collection block.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param array $atts Block attributes.
	 */
	public function render_collection_content( $atts ) {

		if ( ! is_array( $atts ) || ! array_key_exists( 'type', $atts ) || ! in_array( $atts['type'], array( 'category', 'tag' ), true ) || ! array_key_exists( 'term', $atts ) || '' === $atts['term'] ) {
			echo esc_html__( 'Please select a category or tag', 'zypento-blocks' );
			return;
		}

		$term_data = Utilities::get_woo_term_data( $atts['term'] );

		if ( ! is_array( $term_data ) || ! array_key_exists( 'name', $term_data ) || '' === $term_data['name'] || ! array_key_exists( 'link', $term_data ) || '' === $term_data['link'] ) {
			echo esc_html__( 'Something went wrong.', 'zypento-blocks' );
			return;
		}

		$tax = 'product_cat';
		if ( 'tag' === $atts['type'] ) {
			$tax = 'product_tag';
		}

		$zypento_blocks_featured_products  = Utilities::get_products_by_type( $atts['term'], 'featured', $tax, 8 );
		$on_sale_products                  = Utilities::get_products_by_type( $atts['term'], 'on_sale', $tax, 8 );
		$zypento_blocks_recent_products    = Utilities::get_products_by_type( $atts['term'], 'recent', $tax, 8 );
		$zypento_blocks_most_sold_products = Utilities::get_products_by_type( $atts['term'], 'sold', $tax, 8 );
		$zypento_blocks_top_rated_products = Utilities::get_products_by_type( $atts['term'], 'rated', $tax, 8 );
		$currency_symbol                   = get_woocommerce_currency_symbol();

		include ZYPENTO_BLOCKS_PLUGIN_PATH . 'partials/woocommerce/collection.php';

	}

}

