<?php
/**
 * The file that defines the core plugin class
 *
 * @link       http://zypento.com
 * @since      1.0.0
 *
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 * @param string          $type The type of request.
 * @param string $type The type of request.
 */

namespace Zypento_Agilecrm\Misc;

/**
 * The core plugin class.
 *
 * @since      1.0.0
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 * @author     Zypento <dev@zypento.com>
 */
class Utilities {

	/**
	 * Makes a request to the Agile CRM API.
	 *
	 * @param array $creds The credentials for the API request.
	 * @param array $data The data to send in the request.
	 * @return bool True if the request was successful, false otherwise.
	 */
	public function agile_request( $creds, $data ) {

		$auth_scheme = 'Basic';

		// Encoding credentials for HTTP Basic Authentication.
		$api_key = base64_encode( $creds['username'] . ':' . $creds['api'] );

		$url = $creds['domain'] . '/dev/api/contacts';

		$response = wp_remote_request(
			$url,
			array(
				'method'      => 'POST',
				'timeout'     => 45,
				'redirection' => 5,
				'httpversion' => '1.0',
				'blocking'    => true,
				'headers'     => array(
					'Authorization' => $auth_scheme . ' ' . $api_key,
					'Content-Type'  => 'application/json',
				),
				'body'        => wp_json_encode( $data ),
			)
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		if ( ! is_wp_error( $response ) && ( in_array( $response['response']['code'], array( 200, 201 ), true ) ) ) {
			return true;
		}
	}

	/**
	 * Get user details.
	 *
	 * @param WP_REST_Request $request Current request.
	 * @param string          $type The type of request.
	 */
	public function process_settings( $request, $type ) {

		global $zypento_agilecrm_objects;

		$proceed = true;

		$data = $zypento_agilecrm_objects['rest']->setup_response_data();

		$data = $this->process_data( $data, $proceed, $request, $type );

		$response = $zypento_agilecrm_objects['rest']->prepare( $data, $request );

		// Return all of our post response data.
		return $response;

	}

	/**
	 * Process data.
	 *
	 * @param Array           $data Value.
	 * @param Boolean         $proceed Value.
	 * @param WP_REST_Request $request Current request.
	 * @param string          $type The type of request.
	 */
	public function process_data( $data, &$proceed, $request, $type ) {

		global $zypento_agilecrm_objects;

		$id = '';

		$nonce = '';
		if ( isset( $request['nonce'] ) ) {
			$nonce = $request['nonce'];
		}

		$value = '';
		if ( isset( $request['value'] ) && '' !== $request['value'] ) {
			$value = $request['value'];
			$value = json_decode( $value, true );
		}

		if ( ! ( $nonce && $zypento_agilecrm_objects['rest']->role() && is_array( $value ) ) ) {
			return $data;
		}

		$settings = $this->parse_settings( $id, $value, $data, $proceed );

		if ( ! $proceed ) {
			return $data;
		}

		$update = false;
		if ( '' !== $id && 'cf7' === $type ) {
			delete_post_meta( $id, 'zyp_agile' );
			$update = update_post_meta( $id, 'zyp_agile', $settings );
		}
		if ( '' !== $id && 'ninja' === $type ) {
			delete_option( 'zypento_ninja_' . $id );
			$update = update_option( 'zypento_ninja_' . $id, $settings );
		}

		if ( ! $update ) {
			$data['data']['reason'] = esc_html__( 'Something went wrong. Check Agilecrm username/password.', 'zypento-agilecrm' );
			return $data;
		}

		$data['result']         = true;
		$data['data']['reason'] = '';

		return $data;

	}

	/**
	 * Parse settings.
	 *
	 * @param Array   $id Value.
	 * @param Array   $value Value.
	 * @param Array   $data Value.
	 * @param Boolean $proceed Value.
	 */
	public function parse_settings( &$id, $value, &$data, &$proceed ) {

		$settings = array(
			'system' => '',
			'custom' => '',
		);

		if ( ! array_key_exists( 'id', $value ) || ! array_key_exists( 'system', $value ) || ! array_key_exists( 'custom', $value ) ) {
			$data['data']['reason'][ $key1 ] = esc_html__( 'Something went wrong.', 'zypento-agilecrm' );
			$proceed                         = false;
		}

		if ( array_key_exists( 'id', $value ) ) {
			$id = $value['id'];
		}

		if ( array_key_exists( 'system', $value ) ) {
			$settings['system'] = $value['system'];
		}

		if ( array_key_exists( 'custom', $value ) ) {
			$settings['custom'] = $value['custom'];
		}

		return $settings;

	}

}
