<?php
/**
 * The file that defines the core plugin class
 *
 * @link       http://zypento.com
 * @since      1.0.0
 *
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 */

namespace Zypento_Agilecrm\Core;

/**
 * The core plugin class.
 *
 * @since      1.0.0
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 * @author     Zypento <dev@zypento.com>
 */
class Admin {

	/**
	 * Settings fields.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $settings;

	/**
	 * Agile CRM fields.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $crm_fields;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		global $zypento_agilecrm_objects;

		$this->set_settings();
		$this->set_crm_fields();

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_assets' ) );
		add_action( 'admin_menu', array( $this, 'create_admin_pages' ) );
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );

	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public function set_crm_fields() {

		$this->crm_fields = array(
			'first_name' => esc_html__( 'First Name', 'zypento-agilecrm' ),
			'last_name'  => esc_html__( 'Last Name', 'zypento-agilecrm' ),
			'email'      => esc_html__( 'Email', 'zypento-agilecrm' ),
			'phone'      => esc_html__( 'Phone', 'zypento-agilecrm' ),
			'YOUTUBE'    => esc_html__( 'YouTube', 'zypento-agilecrm' ),
			'LINKEDIN'   => esc_html__( 'LinkedIn', 'zypento-agilecrm' ),
			'URL'        => esc_html__( 'Website URL', 'zypento-agilecrm' ),
			'address'    => esc_html__( 'Address', 'zypento-agilecrm' ),
			'city'       => esc_html__( 'City', 'zypento-agilecrm' ),
			'state'      => esc_html__( 'State', 'zypento-agilecrm' ),
			'zip'        => esc_html__( 'Zip', 'zypento-agilecrm' ),
			'country'    => esc_html__( 'Country', 'zypento-agilecrm' ),
		);

	}

	/**
	 * Get CRM fields.
	 *
	 * @since    1.0.0
	 * @access   public
	 * @return   array    The CRM fields.
	 */
	public function get_crm_fields() {
		return $this->crm_fields;
	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public function set_settings() {

		$this->settings = array(
			'domain'   => array(
				'label'   => esc_html__( 'Domain', 'zypento-agilecrm' ),
				'error'   => esc_html__( 'Please enter the Agilecrm domain.', 'zypento-agilecrm' ),
				'message' => esc_html__( 'Enter full domain ex: http://example.agilecrm.com', 'zypento-agilecrm' ),
			),
			'username' => array(
				'label'   => esc_html__( 'Username', 'zypento-agilecrm' ),
				'error'   => esc_html__( 'Please enter the Agilecrm account email.', 'zypento-agilecrm' ),
				'message' => esc_html__( 'Enter your username.', 'zypento-agilecrm' ),
			),
			'api'      => array(
				'label'   => esc_html__( 'API Key', 'zypento-agilecrm' ),
				'error'   => esc_html__( 'Please enter the Agilecrm API key.', 'zypento-agilecrm' ),
				'message' => esc_html__( 'Enter your API Key', 'zypento-agilecrm' ),
			),
		);

	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public function get_settings() {

		return $this->settings;

	}

	/**
	 * Register all of the hooks related to the admin
	 *
	 * @since    1.0.0
	 * @access   public
	 * @param    String $hook The hook.
	 */
	public function admin_enqueue_assets( $hook ) {

		$admin_pages = array( 'toplevel_page_zypento-agilecrm' );

		$admin_pages = apply_filters( 'zypento_agilecrm_admin_pages', $admin_pages );

		if ( ! in_array( $hook, $admin_pages, true ) ) {
			return;
		}

		$variables['userLoggedIn'] = false;
		$variables['nonce']        = '';
		$variables['wpRestNonce']  = '';
		$variables['api']          = array();
		$variables['api']['admin'] = array();

		$variables['labels'] = array();

		$variables['labels']['settings']            = array();
		$variables['labels']['settings']['custom']  = esc_html__( 'Please enter only spaces and Alphanumeric characters.', 'zypento-agilecrm' );
		$variables['labels']['settings']['success'] = esc_html__( 'Settings saved.', 'zypento-agilecrm' );
		$variables['labels']['settings']['errors']  = $this->settings;
		$variables['labels']['settings']['wait']    = esc_html__( 'Please wait...', 'zypento-agilecrm' );
		$variables['labels']['settings']['error']   = esc_html__( 'Something went wrong, Please try again...', 'zypento-agilecrm' );
        $variables['labels']['settings']['all']    = esc_html__( 'Please fix all errors and try again...', 'zypento-agilecrm' );

		if ( is_user_logged_in() && current_user_can( 'administrator' ) ) {

			$variables['nonce']       = esc_html( wp_create_nonce( 'zypento-agilecrm' ) );
			$variables['wpRestNonce'] = esc_html( wp_create_nonce( 'wp_rest' ) );

			$variables['api']['admin']['settings'] = get_rest_url( null, ZYPENTO_AGILECRM_NAMESPACE . '/admin-settings' );

		}

		$variables = apply_filters( 'zypento_agilecrm_admin_variables', $variables );

		wp_enqueue_style( ZYPENTO_AGILECRM_NAME, ZYPENTO_AGILECRM_PLUGIN_URL . '/assets/css/admin.css', array(), ZYPENTO_AGILECRM_VERSION, 'all' );
		wp_register_script( ZYPENTO_AGILECRM_NAME, ZYPENTO_AGILECRM_PLUGIN_URL . '/assets/js/admin.js', array( 'jquery' ), ZYPENTO_AGILECRM_VERSION, true );
		wp_localize_script( ZYPENTO_AGILECRM_NAME, 'zypentoAgileAdminVariables', $variables );
		wp_enqueue_script( ZYPENTO_AGILECRM_NAME );

	}

	/**
	 * Create admin pages.
	 *
	 * @since    1.0.0
	 */
	public function create_admin_pages() {

		if ( ! array_key_exists( 'zypento-agilecrm', $GLOBALS['admin_page_hooks'] ) ) {

			add_menu_page(
				__( 'Zypento Agilecrm', 'zypento-agilecrm' ),
				__( 'Zypento Agilecrm', 'zypento-agilecrm' ),
				'manage_options',
				'zypento-agilecrm',
				array( $this, 'admin_page_display' ),
				'dashicons-media-code',
				20
			);

			do_action( 'zypento_admin_pages' );

		}

	}

	/**
	 * Display admin pagecontent.
	 *
	 * @since    1.0.0
	 */
	public function admin_page_display() {

		global $zypento_agilecrm_objects;
		$zypento_agilecrm_settings        = $this->settings;
		$settings                         = get_option( 'zyp_agilecrm_settings', array() );
		$zypento_agilecrm_settings_values = array();
		foreach ( $zypento_agilecrm_settings as $key => $value ) {
			if ( array_key_exists( $key, $settings ) ) {
				$zypento_agilecrm_settings_values[ $key ] = $settings[ $key ];
				continue;
			}
			$zypento_agilecrm_settings_values[ $key ] = '';
		}
		include ZYPENTO_AGILECRM_PLUGIN_PATH . 'partials/admin-page.php';

	}

	/**
	 * Register REST Routes.
	 */
	public function register_routes() {

		global $zypento_agilecrm_objects;

		register_rest_route(
			ZYPENTO_AGILECRM_NAMESPACE,
			'/admin-settings',
			array(

				// Here we register the readable endpoint for collections.
				array(
					'methods'             => 'GET, POST',
					'callback'            => array( $this, 'process_settings' ),
					'args'                => array(
						'action' => array(
							'description'       => esc_html__( 'Action.', 'zypento-agilecrm' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_agilecrm_objects['rest'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_agilecrm_objects['rest'], 'sanitize_string' ),
							'required'          => true,
							'default'           => '',
						),
						'value'  => array(
							'description'       => esc_html__( 'Value.', 'zypento-agilecrm' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_agilecrm_objects['rest'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_agilecrm_objects['rest'], 'sanitize_string' ),
							'required'          => false,
							'default'           => '',
						),
						'nonce'  => array(
							'description'       => esc_html__( 'Nonce.', 'zypento-agilecrm' ),
							'type'              => 'bool',
							'sanitize_callback' => function( $value ) {
								return (bool) $value;
							},
							'validate_callback' => function( $value ) {
								return wp_verify_nonce( $value, 'zypento-agilecrm' );
							},
							'required'          => true,
							'default'           => false,
						),
					),
					'permission_callback' => function() {
						return current_user_can( 'manage_options' );
					},
				),
				// Register our schema callback.
				'schema' => array( $zypento_agilecrm_objects['rest'], 'get_schema' ),

			)
		);

	}

	/**
	 * Get user details.
	 *
	 * @param WP_REST_Request $request Current request.
	 */
	public function process_settings( $request ) {

		global $zypento_agilecrm_objects;

		$proceed = true;

		$data = $zypento_agilecrm_objects['rest']->setup_response_data();

		$data = $this->process_data( $data, $proceed, $request );

		$response = $zypento_agilecrm_objects['rest']->prepare( $data, $request );

		// Return all of our post response data.
		return $response;

	}

	/**
	 * Process data.
	 *
	 * @param Array           $data Value.
	 * @param Boolean         $proceed Value.
	 * @param WP_REST_Request $request Current request.
	 */
	public function process_data( $data, &$proceed, $request ) {

		global $zypento_agilecrm_objects;

		$nonce = '';
		if ( isset( $request['nonce'] ) ) {
			$nonce = $request['nonce'];
		}

		$value = '';
		if ( isset( $request['value'] ) && '' !== $request['value'] ) {
			$value = $request['value'];
			$value = json_decode( $value, true );
		}

		if ( ! ( $nonce && $zypento_agilecrm_objects['rest']->role() && is_array( $value ) ) ) {
			return $data;
		}

		$settings = $this->parse_settings( $value, $data, $proceed );

		if ( ! $proceed ) {
			return $data;
		}

		delete_option( 'zyp_agilecrm_settings' );
		$update = update_option( 'zyp_agilecrm_settings', $settings );

		if ( ! $update ) {
			$data['data']['reason'] = esc_html__( 'Something went wrong. Check Agilecrm username/password.', 'zypento-agilecrm' );
			return $data;
		}

		$data['result']         = true;
		$data['data']['reason'] = '';

		return $data;

	}

	/**
	 * Parse settings.
	 *
	 * @param Array   $value Value.
	 * @param Array   $data Value.
	 * @param Boolean $proceed Value.
	 */
	public function parse_settings( $value, &$data, &$proceed ) {

		$zypento_agilecrm_settings = $this->settings;

		$settings = array();

		foreach ( $zypento_agilecrm_settings as $key1 => $value1 ) {

			if ( ! is_array( $data['data']['reason'] ) ) {
				$data['data']['reason'] = array();
			}

			if ( array_key_exists( $key1, $value ) && ! empty( $value[ $key1 ] ) ) {

				if ( 'domain' === $key1 && ! filter_var( $value[ $key1 ], FILTER_VALIDATE_URL ) ) {
					$data['data']['reason'][ $key1 ] = sprintf( '%1$s %2$s', esc_html__( 'Invalid domain', 'zypento-agilecrm' ), $key1 );
					$proceed                         = false;
					continue;
				}

				$settings[ $key1 ] = wp_strip_all_tags( $value[ $key1 ] );
				continue;
			}

			$data['data']['reason'][ $key1 ] = sprintf( '%1$s %2$s', esc_html__( 'Empty', 'zypento-agilecrm' ), $key1 );
			$proceed                         = false;

		}

		return $settings;

	}

	/**
	 * Make an API call to Agile CRM.
	 *
	 * @since    1.0.0
	 * @param    array $data Data to be sent in the API call.
	 */
	public function api_call( $data ) {

		global $zypento_agilecrm_objects;

		$creds = get_option( 'zyp_agilecrm_settings', array() );

		if ( ! is_array( $creds ) || empty( $creds ) ) {
			return;
		}

		foreach ( $this->settings as $key => $value ) {

			if ( ! array_key_exists( $key, $creds ) ) {
				return;
			}
		}

		$domain_url = $creds['domain'];
		$parsed_url = wp_parse_url( $domain_url );
		if ( false === $parsed_url || ! isset( $parsed_url['host'] ) ) {
			return;
		}

		$result = $zypento_agilecrm_objects['utilities']->agile_request( $creds, $data );

	}

}
