<?php
/**
 * The file that defines the core plugin class
 *
 * @link       http://zypento.com
 * @since      1.0.0
 *
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 */

namespace Zypento_Agilecrm\Features;

/**
 * The core plugin class.
 *
 * @since      1.0.0
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 * @author     Zypento <dev@zypento.com>
 */
class Ninja {

	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		add_action( 'zypento_admin_pages', array( $this, 'create_admin_pages' ) );
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
		add_action( 'ninja_forms_after_submission', array( $this, 'process_submission' ) );
		add_filter( 'zypento_agilecrm_admin_pages', array( $this, 'add_assets' ) );
		add_filter( 'zypento_agilecrm_admin_variables', array( $this, 'admin_variables' ) );

	}

	/**
	 * Create admin pages.
	 *
	 * @since    1.0.0
	 */
	public function create_admin_pages() {

		add_submenu_page(
			'zypento-agilecrm',
			__( 'Ninja Forms', 'zypento-agilecrm' ),
			__( 'Ninja Forms', 'zypento-agilecrm' ),
			'manage_options',
			'zypento-agilecrm-ninja',
			array( $this, 'admin_page_display' ),
		);

	}

	/**
	 * Create admin pages.
	 *
	 * @since    1.0.0
	 * @param array $admin_pages Admin pages.
	 */
	public function add_assets( $admin_pages ) {

		$admin_pages[] = 'zypento-agilecrm_page_zypento-agilecrm-ninja';

		return $admin_pages;

	}

	/**
	 * Register REST Routes.
	 */
	public function register_routes() {

		global $zypento_agilecrm_objects;

		register_rest_route(
			ZYPENTO_AGILECRM_NAMESPACE,
			'/ninja',
			array(

				// Here we register the readable endpoint for collections.
				array(
					'methods'             => 'GET, POST',
					'callback'            => array( $this, 'process_settings' ),
					'args'                => array(
						'action' => array(
							'description'       => esc_html__( 'Action.', 'zypento-agilecrm' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_agilecrm_objects['rest'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_agilecrm_objects['rest'], 'sanitize_string' ),
							'required'          => true,
							'default'           => '',
						),
						'value'  => array(
							'description'       => esc_html__( 'Value.', 'zypento-agilecrm' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_agilecrm_objects['rest'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_agilecrm_objects['rest'], 'sanitize_string' ),
							'required'          => false,
							'default'           => '',
						),
						'nonce'  => array(
							'description'       => esc_html__( 'Nonce.', 'zypento-agilecrm' ),
							'type'              => 'bool',
							'sanitize_callback' => function( $value ) {
								return (bool) $value;
							},
							'validate_callback' => function( $value ) {
								return wp_verify_nonce( $value, 'zypento-agilecrm' );
							},
							'required'          => true,
							'default'           => false,
						),
					),
					'permission_callback' => function() {
						return current_user_can( 'manage_options' );
					},
				),
				// Register our schema callback.
				'schema' => array( $zypento_agilecrm_objects['rest'], 'get_schema' ),

			)
		);

	}

	/**
	 * Display admin pagecontent.
	 *
	 * @since    1.0.0
	 */
	public function admin_page_display() {

		global $zypento_agilecrm_objects;

		$forms = array();

		if ( function_exists( 'Ninja_Forms' ) ) {
			$forms = Ninja_Forms()->form()->get_forms();
		}

		include ZYPENTO_AGILECRM_PLUGIN_PATH . 'partials/ninja-admin-page.php';

	}

	/**
	 * Add admin variables.
	 *
	 * @since    1.0.0
	 * @param    array $variables Admin variables.
	 * @return   array Modified admin variables.
	 */
	public function admin_variables( $variables ) {

		if ( ! array_key_exists( 'api', $variables ) || ! is_array( $variables['api'] ) ) {
			$variables['api'] = array();
		}
		if ( ! array_key_exists( 'admin', $variables['api'] ) || ! is_array( $variables['api']['admin'] ) ) {
			$variables['api']['admin'] = array();
		}
		$variables['api']['admin']['ninja'] = get_rest_url( null, ZYPENTO_AGILECRM_NAMESPACE . '/ninja' );
		return $variables;

	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 * @param    String $id Id of form.
	 */
	public function display_crm_fields( $id ) {

		global $zypento_agilecrm_objects;

		if ( '' === $id ) {
			return;
		}

		$ninja_fields = array();

		$ninja_fields_model = Ninja_Forms()->form( $id )->get_fields();

		foreach ( $ninja_fields_model as $ninja_field_model ) {
			$n_model                         = $ninja_field_model->get_settings();
			$ninja_fields[ $n_model['key'] ] = $n_model['label'];
		}

		$ninja_fields_output           = array();
		$ninja_fields_output['select'] = esc_attr__( 'Select', 'zypento-agilecrm' );

		$current_data = get_option( 'zypento_ninja_' . $id, array() );

		?>
		<div class="zypento-agilecrm-field-map-items">
			<div class="zypento-agilecrm-field-map-system-items">
			<?php

			foreach ( $zypento_agilecrm_objects['admin']->get_crm_fields() as $key => $value ) {

				$current_value = 'select';
				if ( is_array( $current_data ) && array_key_exists( 'system', $current_data ) && array_key_exists( $key, $current_data['system'] ) ) {
					$current_value = $current_data['system'][ $key ];
				}

				?>

				<div data-key="<?php echo esc_attr( $key ); ?>" data-type="system" class="zypento-agilecrm-field-map-item">

					<div class="zypento-agilecrm-field-map-item-content">

						<div>
							<p><?php echo esc_html( $value ); ?></p>
						</div>

						<div>
							<select>
							<option value="select"><?php esc_attr_e( 'Select', 'zypento-agilecrm' ); ?></option>
							<?php
							foreach ( $ninja_fields as $ninja_field_key => $ninja_field_value ) :
								if ( 'submit' !== $ninja_field_key ) :
									$ninja_fields_output[ $ninja_field_key ] = $ninja_field_value;
									?>
								<option <?php selected( $current_value, esc_attr( $ninja_field_key ), true ); ?> value="<?php echo esc_attr( $ninja_field_key ); ?>"><?php echo esc_attr( $ninja_field_value ); ?></option>
									<?php
									endif;
								endforeach;
							?>
							</select>
						</div>

					</div>

				</div>

				<?php

			}

			?>
			</div>
			<div data-crm="<?php echo esc_attr( wp_json_encode( $ninja_fields_output, JSON_FORCE_OBJECT ) ); ?>" class="zypento-agilecrm-field-map-custom-items">
				<?php if ( is_array( $current_data ) && array_key_exists( 'custom', $current_data ) ) : ?>
					<?php
					foreach ( $current_data['custom'] as $ninja_key1 => $ninja_value1 ) :
						?>

						<div data-type="custom" class="zypento-agilecrm-field-map-item">

						<p class="zypento-agilecrm-field-map-item-message">
						<?php esc_html_e( 'Please enter only spaces and Alphanumeric characters.', 'zypento-agilecrm' ); ?>
							<span class="zypento-agilecrm-field-map-item-delete"></span>
						</p>
						<div class="zypento-agilecrm-field-map-item-content">
							<div>
								<p><input type="text" value="<?php echo esc_attr( $ninja_key1 ); ?>" /></p>
							</div>

							<div>
								<select>
								<option value="select"><?php esc_attr_e( 'Select', 'zypento-agilecrm' ); ?></option>
							<?php
							foreach ( $ninja_fields as $ninja_field_key => $ninja_field_value ) :

								?>
									<option <?php selected( esc_attr( $ninja_value1 ), esc_attr( $ninja_field_key ), true ); ?> value="<?php echo esc_attr( $ninja_field_key ); ?>"><?php echo esc_attr( $ninja_field_value ); ?></option>
								<?php
								endforeach;
							?>
								</select>
							</div>
						</div>

						</div>

					<?php endforeach; ?>
				<?php endif; ?>
			</div>
			<p class="zypento-agilecrm-field-map-message"></p>
			<div class="zypento-agilecrm-field-map-actions">
				<span data-type="add"><?php esc_html_e( 'Add', 'zypento-agilecrm' ); ?></span>
				<span data-type="submit"><?php esc_html_e( 'Submit', 'zypento-agilecrm' ); ?></span>
			</div>
		</div>

		<?php

	}

	/**
	 * Process settings.
	 *
	 * @since    1.0.0
	 * @param    WP_REST_Request $request The request object.
	 */
	public function process_settings( $request ) {

		global $zypento_agilecrm_objects;

		return $zypento_agilecrm_objects['utilities']->process_settings( $request, 'ninja' );

	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 * @param    String $form_data Id of form.
	 */
	public function process_submission( $form_data ) {

		global $zypento_agilecrm_objects;

		if ( ! is_array( $form_data ) || ! array_key_exists( 'form_id', $form_data ) ) {
			return;
		}

		// Get the form ID.
		$form_id = $form_data['form_id'];

		$settings = get_option( 'zypento_ninja_' . $form_id, array() );

		if ( ! is_array( $settings ) || ! array_key_exists( 'system', $settings ) || ! array_key_exists( 'custom', $settings ) ) {
			return;
		}

		// Get the submitted fields and their values.
		$fields = $form_data['fields'];

		$posted_data = array();

		// Process the submitted data as needed.
		foreach ( $fields as $field ) {
			$field_id    = $field['id'];
			$field_value = $field['value'];

			// Get the field model.
			$field_model    = Ninja_Forms()->form( $form_id )->get_field( $field_id );
			$field_settings = $field_model->get_settings();
			$field_key      = $field_settings['key'];

			$posted_data[ $field_key ] = $field_value;
		}

		$properties = array();

		foreach ( $settings['system'] as $key => $value ) {
			if ( ! array_key_exists( $value, $posted_data ) ) {
				continue;
			}
			$property = array(
				'type'  => 'SYSTEM',
				'name'  => esc_html( $key ),
				'value' => esc_html( $posted_data[ $value ] ),
			);
			if ( in_array( $key, array( 'email', 'phone' ), true ) ) {
				$property['subtype'] = 'work';
			}
			if ( in_array( $key, array( 'YOUTUBE', 'LINKEDIN', 'URL' ), true ) ) {
				$property['name']    = 'website';
				$property['subtype'] = $key;
			}
			$properties[] = $property;
		}

		foreach ( $settings['custom'] as $key => $value ) {
			if ( ! array_key_exists( $value, $posted_data ) ) {
				continue;
			}
			$properties[] = array(
				'type'  => 'CUSTOM',
				'name'  => esc_html( $key ),
				'value' => esc_html( $posted_data[ $value ] ),
			);
		}

		$data = array(
			'star_value' => '',
			'lead_score' => '',
			'tags'       => array(
				'Lead',
			),
			'properties' => $properties,
		);

		$zypento_agilecrm_objects['admin']->api_call( $data );

	}

}
