<?php
/**
 * The file that defines the core plugin class
 *
 * @link       http://zypento.com
 * @since      1.0.0
 *
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 */

namespace Zypento_Agilecrm\Features;

/**
 * The core plugin class.
 *
 * @since      1.0.0
 * @package    zypento-agilecrm
 * @subpackage zypento-agilecrm/includes
 * @author     Zypento <dev@zypento.com>
 */
class Cf7 {

	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		add_action( 'zypento_admin_pages', array( $this, 'create_admin_pages' ) );
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
		add_action( 'wpcf7_before_send_mail', array( $this, 'process_submission' ) );
		add_filter( 'zypento_agilecrm_admin_pages', array( $this, 'add_assets' ) );
		add_filter( 'zypento_agilecrm_admin_variables', array( $this, 'admin_variables' ) );

	}

	/**
	 * Create admin pages.
	 *
	 * @since    1.0.0
	 */
	public function create_admin_pages() {

		add_submenu_page(
			'zypento-agilecrm',
			__( 'Contact Form 7', 'zypento-agilecrm' ),
			__( 'Contact Form 7', 'zypento-agilecrm' ),
			'manage_options',
			'zypento-agilecrm-cf7',
			array( $this, 'admin_page_display' ),
		);

	}

	/**
	 * Create admin pages.
	 *
	 * @since    1.0.0
	 * @param    array $admin_pages Admin pages.
	 */
	public function add_assets( $admin_pages ) {

		$admin_pages[] = 'zypento-agilecrm_page_zypento-agilecrm-cf7';

		return $admin_pages;

	}

	/**
	 * Add admin variables.
	 *
	 * @since    1.0.0
	 * @param    array $variables Admin variables.
	 * @return   array Modified admin variables.
	 */
	public function admin_variables( $variables ) {

		if ( ! array_key_exists( 'api', $variables ) || ! is_array( $variables['api'] ) ) {
			$variables['api'] = array();
		}
		if ( ! array_key_exists( 'admin', $variables['api'] ) || ! is_array( $variables['api']['admin'] ) ) {
			$variables['api']['admin'] = array();
		}
		$variables['api']['admin']['cf7'] = get_rest_url( null, ZYPENTO_AGILECRM_NAMESPACE . '/cf7' );
		return $variables;

	}

	/**
	 * Register REST Routes.
	 */
	public function register_routes() {

		global $zypento_agilecrm_objects;

		register_rest_route(
			ZYPENTO_AGILECRM_NAMESPACE,
			'/cf7',
			array(

				// Here we register the readable endpoint for collections.
				array(
					'methods'             => 'GET, POST',
					'callback'            => array( $this, 'process_settings' ),
					'args'                => array(
						'action' => array(
							'description'       => esc_html__( 'Action.', 'zypento-agilecrm' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_agilecrm_objects['rest'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_agilecrm_objects['rest'], 'sanitize_string' ),
							'required'          => true,
							'default'           => '',
						),
						'value'  => array(
							'description'       => esc_html__( 'Value.', 'zypento-agilecrm' ),
							'type'              => 'string',
							'validate_callback' => array( $zypento_agilecrm_objects['rest'], 'validate_string' ),
							'sanitize_callback' => array( $zypento_agilecrm_objects['rest'], 'sanitize_string' ),
							'required'          => false,
							'default'           => '',
						),
						'nonce'  => array(
							'description'       => esc_html__( 'Nonce.', 'zypento-agilecrm' ),
							'type'              => 'bool',
							'sanitize_callback' => function( $value ) {
								return (bool) $value;
							},
							'validate_callback' => function( $value ) {
								return wp_verify_nonce( $value, 'zypento-agilecrm' );
							},
							'required'          => true,
							'default'           => false,
						),
					),
					'permission_callback' => function() {
						return current_user_can( 'manage_options' );
					},
				),
				// Register our schema callback.
				'schema' => array( $zypento_agilecrm_objects['rest'], 'get_schema' ),

			)
		);

	}

	/**
	 * Process settings.
	 *
	 * @since    1.0.0
	 * @param    WP_REST_Request $request The request object.
	 */
	public function process_settings( $request ) {

		global $zypento_agilecrm_objects;

		return $zypento_agilecrm_objects['utilities']->process_settings( $request, 'cf7' );

	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 * @param    String $id Id of form.
	 */
	public function display_crm_fields( $id ) {

		global $zypento_agilecrm_objects;

		if ( '' === $id ) {
			return;
		}

		$cf7        = \WPCF7_ContactForm::get_instance( $id );
		$cf7_fields = $cf7->scan_form_tags();

		$cf7_fields_output           = array();
		$cf7_fields_output['select'] = esc_attr__( 'Select', 'zypento-agilecrm' );

		$current_data = get_post_meta( $id, 'zyp_agile', true );

		?>
		<div class="zypento-agilecrm-field-map-items">
			<div class="zypento-agilecrm-field-map-system-items">
			<?php

			foreach ( $zypento_agilecrm_objects['admin']->get_crm_fields() as $key => $value ) {

				$current_value = 'select';
				if ( is_array( $current_data ) && array_key_exists( 'system', $current_data ) && array_key_exists( $key, $current_data['system'] ) ) {
					$current_value = $current_data['system'][ $key ];
				}

				?>

				<div data-key="<?php echo esc_attr( $key ); ?>" data-type="system" class="zypento-agilecrm-field-map-item">

					<div class="zypento-agilecrm-field-map-item-content">

						<div>
							<p><?php echo esc_html( $value ); ?></p>
						</div>

						<div>
							<select>
							<option value="select"><?php esc_attr_e( 'Select', 'zypento-agilecrm' ); ?></option>
							<?php
							foreach ( $cf7_fields as $cf7_field ) :
								if ( 'submit' !== $cf7_field->type ) :
									$cf7_fields_output[ $cf7_field->name ] = $cf7_field->name;
									?>
								<option <?php selected( $current_value, esc_attr( $cf7_field->name ), true ); ?> value="<?php echo esc_attr( $cf7_field->name ); ?>"><?php echo esc_attr( $cf7_field->name ); ?></option>
									<?php
									endif;
								endforeach;
							?>
							</select>
						</div>

					</div>

				</div>

				<?php

			}

			?>
			</div>
			<div data-crm="<?php echo esc_attr( wp_json_encode( $cf7_fields_output, JSON_FORCE_OBJECT ) ); ?>" class="zypento-agilecrm-field-map-custom-items">
				<?php if ( is_array( $current_data ) && array_key_exists( 'custom', $current_data ) ) : ?>
					<?php
					foreach ( $current_data['custom'] as $cf7_key1 => $cf7_value1 ) :
						?>

						<div data-type="custom" class="zypento-agilecrm-field-map-item">

						<p class="zypento-agilecrm-field-map-item-message">
						<?php esc_html_e( 'Please enter only spaces and Alphanumeric characters.', 'zypento-agilecrm' ); ?>
							<span class="zypento-agilecrm-field-map-item-delete"></span>
						</p>
						<div class="zypento-agilecrm-field-map-item-content">
							<div>
								<p><input type="text" value="<?php echo esc_attr( $cf7_key1 ); ?>" /></p>
							</div>

							<div>
								<select>
								<option value="select"><?php esc_attr_e( 'Select', 'zypento-agilecrm' ); ?></option>
							<?php
							foreach ( $cf7_fields as $cf7_field ) :

								?>
									<option <?php selected( esc_attr( $cf7_value1 ), esc_attr( $cf7_field->name ), true ); ?> value="<?php echo esc_attr( $cf7_field->name ); ?>"><?php echo esc_attr( $cf7_field->name ); ?></option>
								<?php
								endforeach;
							?>
								</select>
							</div>
						</div>

						</div>

					<?php endforeach; ?>
				<?php endif; ?>
			</div>
			<p class="zypento-agilecrm-field-map-message"></p>
			<div class="zypento-agilecrm-field-map-actions">
				<span data-type="add"><?php esc_html_e( 'Add', 'zypento-agilecrm' ); ?></span>
				<span data-type="submit"><?php esc_html_e( 'Submit', 'zypento-agilecrm' ); ?></span>
			</div>
		</div>

		<?php

	}

	/**
	 * Display admin pagecontent.
	 *
	 * @since    1.0.0
	 */
	public function admin_page_display() {

		global $zypento_agilecrm_objects;

		$forms = array();

		if ( function_exists( 'wpcf7_add_shortcode' ) ) :
			$args = array(
				'post_type'      => 'wpcf7_contact_form',
				'posts_per_page' => -1,
				'fields'         => 'ids',
			);

			$forms = get_posts( $args );
		endif;

		include ZYPENTO_AGILECRM_PLUGIN_PATH . 'partials/cf7-admin-page.php';

	}

	/**
	 * Register all of the hooks related to the admin.
	 *
	 * @since    1.0.0
	 * @access   public
	 * @param    String $contact_form Id of form.
	 */
	public function process_submission( $contact_form ) {

		global $zypento_agilecrm_objects;

		$form_id    = $contact_form->id(); // Get the form ID.
		$submission = \WPCF7_Submission::get_instance();
		$settings   = get_post_meta( $form_id, 'zyp_agile', true );

		if ( ! is_array( $settings ) || ! array_key_exists( 'system', $settings ) || ! array_key_exists( 'custom', $settings ) || ! $submission ) {
			return $contact_form;
		}

		if ( $submission ) {
			$posted_data = $submission->get_posted_data();
		}

		if ( ! is_array( $posted_data ) ) {
			return $contact_form;
		}

		$properties = array();

		foreach ( $settings['system'] as $key => $value ) {
			if ( ! array_key_exists( $value, $posted_data ) ) {
				continue;
			}
			$property = array(
				'type'  => 'SYSTEM',
				'name'  => esc_html( $key ),
				'value' => esc_html( $posted_data[ $value ] ),
			);
			if ( in_array( $key, array( 'email', 'phone' ), true ) ) {
				$property['subtype'] = 'work';
			}
			if ( in_array( $key, array( 'YOUTUBE', 'LINKEDIN', 'URL' ), true ) ) {
				$property['name']    = 'website';
				$property['subtype'] = $key;
			}
			$properties[] = $property;
		}

		foreach ( $settings['custom'] as $key => $value ) {
			if ( ! array_key_exists( $value, $posted_data ) ) {
				continue;
			}
			$properties[] = array(
				'type'  => 'CUSTOM',
				'name'  => esc_html( $key ),
				'value' => esc_html( $posted_data[ $value ] ),
			);
		}

		$data = array(
			'star_value' => '',
			'lead_score' => '',
			'tags'       => array(
				'Lead',
			),
			'properties' => $properties,
		);

		$zypento_agilecrm_objects['admin']->api_call( $data );

		return $contact_form;

	}

}
