<?php
if (!defined('ABSPATH')) exit;

class ZUBBIN_UN_Settings {
  static function opt() { return 'zubbin_un_settings'; }

  /**
   * Apply remote state returned by Central (heartbeat/sync/plans).
   * Keeps Node backwards-compatible with older Central versions.
   *
   * Supported shapes:
   * - body.entitlement.{billing_status,plan_key,limits,features,block_reason,enabled,plan{limits,features}}
   * - body.node.{billing_status,plan_key,limits,plan_limits}
   */
  static function apply_remote_state($body) {
    if (!is_array($body)) return;

    $ent = is_array($body['entitlement'] ?? null) ? $body['entitlement'] : null;
    $node = is_array($body['node'] ?? null) ? $body['node'] : null;

    $plan_key = '';
    $billing_status = '';
    $block_reason = '';
    $enabled = null;
    $limits = null;
    $features = null;

    if (is_array($ent)) {
      $plan_key = sanitize_key((string)($ent['plan_key'] ?? ($ent['plan']['key'] ?? '')));
      $billing_status = (string)($ent['billing_status'] ?? ($ent['status'] ?? ''));
      $block_reason = (string)($ent['block_reason'] ?? '');
      if (array_key_exists('enabled', $ent)) {
        $enabled = (int)$ent['enabled'];
      }

      if (is_array($ent['limits'] ?? null)) {
        $limits = $ent['limits'];
      } elseif (is_array($ent['plan']['limits'] ?? null)) {
        $limits = $ent['plan']['limits'];
      }

      if (is_array($ent['features'] ?? null)) {
        $features = $ent['features'];
      } elseif (is_array($ent['plan']['features'] ?? null)) {
        $features = $ent['plan']['features'];
      }
    }

    if ($plan_key === '' && is_array($node)) {
      $plan_key = sanitize_key((string)($node['plan_key'] ?? ''));
    }
    if ($billing_status === '' && is_array($node)) {
      $billing_status = (string)($node['billing_status'] ?? '');
    }
    if ($limits === null && is_array($node)) {
      if (is_array($node['plan_limits'] ?? null)) {
        $limits = $node['plan_limits'];
      } elseif (is_array($node['limits'] ?? null)) {
        $limits = $node['limits'];
      }
    }

    $updates = [
      'entitlement_checked_at' => current_time('mysql'),
    ];

    if ($plan_key !== '') {
      $updates['plan_key'] = $plan_key;
    }
    if ($billing_status !== '') {
      $updates['billing_status'] = sanitize_text_field($billing_status);
    }
    if ($block_reason !== '') {
      $updates['block_reason'] = sanitize_text_field($block_reason);
    }
    if ($enabled !== null) {
      $updates['entitlement_enabled'] = (int)$enabled;
    }
    if (is_array($limits)) {
      $updates['plan_limits'] = $limits;
    }
    if (is_array($features)) {
      $updates['plan_features'] = $features;
    }

    self::save($updates);
  }

  static function defaults() {
    return [
      'central_url' => defined('ZUBBIN_UN_DEFAULT_CENTRAL_URL') ? (string)ZUBBIN_UN_DEFAULT_CENTRAL_URL : '',
      'central_api_base' => '',
      'node_key' => '',
      'node_secret' => '',

      'notify_email' => get_option('admin_email'),
      'contact_name' => '',
      'contact_email' => get_option('admin_email'),
      'contact_phone' => '',
      'contact_company' => '',

      // support details (cached from Central)
      'support_email' => '',
      'support_url' => '',
      'support_phone' => '',
      'support_whatsapp' => '',

      // billing/plan (cached from Central)
      'plan_key' => 'free',
      'billing_status' => '',
      'plan_limits' => [],
      'plan_features' => [],
      'block_reason' => '',
      'entitlement_enabled' => 1,
      'entitlement_checked_at' => '',

      // stored on central, used by central alerts engine
      'webhook_url' => '',
      // default ON (if webhook_url is set, Central will send webhooks automatically)
      'webhook_enabled' => 1,

      // monitoring
      'check_url' => home_url('/'),
      'check_timeout' => 10,

      // state
      'last_status' => 'unknown',
      'last_http' => 0,
      'last_response_ms' => 0,
      'last_message' => '',
      'last_ok_at' => '',
      'last_error' => '',

      // health snapshot (last run)
      'last_health' => [],
    ];
  }

  static function get() {
    $cur = get_option(self::opt(), []);
    if (!is_array($cur)) $cur = [];
    return array_merge(self::defaults(), $cur);
  }

  static function save($data) {
    $cur = self::get();
    $new = array_merge($cur, is_array($data) ? $data : []);
    update_option(self::opt(), $new);
  }

  static function paired($s=null) {
    $s = $s ?: self::get();
    return !empty($s['central_url']) && !empty($s['node_key']) && !empty($s['node_secret']);
  }

  static function api_bases($central_url) {
    $u = rtrim((string)$central_url, '/');
    if ($u === '') return [];
    $c = [];

    if (strpos($u, 'rest_route=') !== false) {
      $c[] = $u;
    } elseif (strpos($u, '/wp-json/') !== false) {
      if (preg_match('#/wp-json/wsum/v\d+#', $u)) {
        $c[] = $u;
      } else {
        $c[] = rtrim($u, '/').'/wsum/v2';
      }
    } else {
      $c[] = $u.'/wp-json/wsum/v2';
      $c[] = $u.'/index.php?rest_route=/wsum/v2';
      $c[] = $u.'/?rest_route=/wsum/v2';
    }

    $out = [];
    foreach ($c as $x) {
      if (!in_array($x, $out, true)) $out[] = $x;
    }
    return $out;
  }

  static function cached_api_base() {
    $s = self::get();
    $b = (string)($s['central_api_base'] ?? '');
    return $b === '' ? '' : $b;
  }

  static function set_cached_api_base($base) {
    $base = trim((string)$base);
    if ($base !== '') self::save(['central_api_base' => $base]);
  }

  static function endpoint_urls($central_url, $route) {
    $route = '/'.ltrim((string)$route, '/');

    $bases = [];
    $cached = self::cached_api_base();
    if ($cached !== '') $bases[] = rtrim($cached, '/');
    foreach (self::api_bases($central_url) as $b) $bases[] = rtrim($b, '/');

    $uniq = [];
    foreach ($bases as $b) {
      if ($b === '') continue;
      if (!in_array($b, $uniq, true)) $uniq[] = $b;
    }

    $urls = [];
    foreach ($uniq as $b) {
      if (strpos($b, 'rest_route=') !== false) {
        if (preg_match('/rest_route=([^&]*)/', $b)) {
          $urls[] = preg_replace('/rest_route=([^&]*)/', 'rest_route='.rawurlencode('/wsum/v2'.$route), $b);
        } else {
          $glue = (strpos($b, '?') !== false) ? '&' : '?';
          $urls[] = $b.$glue.'rest_route='.rawurlencode('/wsum/v2'.$route);
        }
      } else {
        $urls[] = $b.$route;
      }
    }
    return $urls;
  }

  static function base($central_url) {
    $bases = self::api_bases($central_url);
    return isset($bases[0]) ? $bases[0] : rtrim((string)$central_url, '/').'/wp-json/wsum/v2';
  }

  static function sanitize_url($url) {
    $url = trim((string)$url);
    return $url === '' ? '' : esc_url_raw($url);
  }
}
