<?php
if (!defined('ABSPATH')) exit;

class ZUBBIN_UN_Onboard {

  static function activate() {
    // Ensure local log table exists.
    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::install();
      ZUBBIN_UN_Logger::info('activate', 'Plugin activated');
    }

    // Migrate settings from older Node builds (older option keys) if present.
    $cur_raw = get_option(ZUBBIN_UN_Settings::opt(), null);
    $cur_is_empty = empty($cur_raw) || !is_array($cur_raw);
    if ($cur_is_empty) {
      $legacy_keys = [
        'wsum_settings',
        'wsum_node_settings',
        'wsum_node_agent_settings',
        'wsum_node_agent',
        'wp_site_uptime_node_agent_settings',
      ];
      foreach ($legacy_keys as $lk) {
        $old = get_option($lk, null);
        if (is_array($old) && !empty($old)) {
          $mapped = [];
          $map = [
            'central_url' => ['central_url','central','centralUrl'],
            'node_key' => ['node_key','nodeKey','key'],
            'node_secret' => ['node_secret','nodeSecret','secret'],
            'notify_email' => ['notify_email','email'],
            'contact_name' => ['contact_name','name'],
            'contact_email' => ['contact_email','contactEmail'],
            'contact_phone' => ['contact_phone','phone'],
            'contact_company' => ['contact_company','company'],
            'webhook_url' => ['webhook_url','webhook'],
            'webhook_enabled' => ['webhook_enabled','webhookOn'],
            'check_url' => ['check_url','checkUrl'],
            'check_timeout' => ['check_timeout','timeout'],
          ];
          foreach ($map as $new_key => $cands) {
            foreach ($cands as $cand) {
              if (isset($old[$cand]) && $old[$cand] !== '') {
                $mapped[$new_key] = $old[$cand];
                break;
              }
            }
          }
          if (!empty($mapped)) {
            // Normalize a few important fields.
            if (isset($mapped['central_url'])) $mapped['central_url'] = esc_url_raw((string)$mapped['central_url']);
            if (isset($mapped['webhook_url'])) $mapped['webhook_url'] = esc_url_raw((string)$mapped['webhook_url']);
            if (isset($mapped['check_url'])) $mapped['check_url'] = esc_url_raw((string)$mapped['check_url']);
            if (isset($mapped['check_timeout'])) $mapped['check_timeout'] = absint($mapped['check_timeout']);
            if (isset($mapped['webhook_enabled'])) $mapped['webhook_enabled'] = !empty($mapped['webhook_enabled']) ? 1 : 0;

            ZUBBIN_UN_Settings::save($mapped);
            if (class_exists('ZUBBIN_UN_Logger')) {
              ZUBBIN_UN_Logger::info('migrate', 'Migrated legacy settings', ['from_option' => $lk]);
            }
            break;
          }
        }
      }
    }

    // Ensure cron schedule registered + scheduled.
    if (class_exists('ZUBBIN_UN_Cron')) {
      ZUBBIN_UN_Cron::activate();
    }

    $s = ZUBBIN_UN_Settings::get();
    $updates = [];

    // Default Central URL for client installs.
    if (empty($s['central_url']) && defined('ZUBBIN_UN_DEFAULT_CENTRAL_URL')) {
      $updates['central_url'] = (string)ZUBBIN_UN_DEFAULT_CENTRAL_URL;
      $s['central_url'] = $updates['central_url'];
    }

    // Attempt tokenless auto-bootstrap if not already paired.
    if (!ZUBBIN_UN_Settings::paired($s) && !empty($s['central_url'])) {
      $r = ZUBBIN_UN_Client::auto_bootstrap($s['central_url']);
      if ((int)$r['http'] === 200 && !empty($r['body']['ok']) && !empty($r['body']['node_key']) && !empty($r['body']['node_secret'])) {
        $updates['node_key'] = (string)$r['body']['node_key'];
        $updates['node_secret'] = (string)$r['body']['node_secret'];
        $updates['last_error'] = '';
        if (class_exists('ZUBBIN_UN_Logger')) {
          ZUBBIN_UN_Logger::info('auto_bootstrap', 'Auto-registration succeeded', ['http' => (int)$r['http']]);
        }
      } else {
        $err = (string)($r['body']['error'] ?? 'auto_bootstrap_failed');
        $msg = (string)($r['body']['message'] ?? '');
        $updates['last_error'] = $msg !== '' ? ($err . ': ' . $msg) : $err;
        if (class_exists('ZUBBIN_UN_Logger')) {
          ZUBBIN_UN_Logger::warn('auto_bootstrap', 'Auto-registration failed', ['http' => (int)$r['http'], 'error' => $err, 'message' => $msg]);
        }
      }
    }

    if (!empty($updates)) {
      ZUBBIN_UN_Settings::save($updates);
    }

    // If paired, push initial profile settings to Central.
    $s2 = ZUBBIN_UN_Settings::get();
    if (ZUBBIN_UN_Settings::paired($s2)) {
      $sr = ZUBBIN_UN_Client::sync($s2);
      if (class_exists('ZUBBIN_UN_Logger')) {
        ZUBBIN_UN_Logger::info('sync', 'Initial sync sent', ['http' => (int)$sr['http']]);
      }
    }
  }
}
