<?php
if (!defined('ABSPATH')) exit;

class ZUBBIN_UN_Logger {

  const TABLE_SUFFIX = 'zubbin_un_node_logs';
  const MAX_ROWS = 500;
  const CACHE_GROUP = 'zubbin_un_logger';

  static function table() {
    global $wpdb;
    // Defensive: ensure table name is derived only from the DB prefix.
    $prefix = isset($wpdb->prefix) ? (string) $wpdb->prefix : 'wp_';
    $prefix = preg_replace('/[^A-Za-z0-9_]/', '', $prefix);
    return $prefix . self::TABLE_SUFFIX;
  }

  static function cache_key($suffix) {
    // Cache key varies with DB prefix to avoid collisions in multisite/multi-install.
    return 'zubbin_unlog_' . md5(self::table() . '|' . (string) $suffix);
  }

  static function cache_invalidate() {
    wp_cache_delete(self::cache_key('count'), self::CACHE_GROUP);
    // We cache recent() by limit; delete a small set of common limits.
    foreach ([20, 50, 100, 200, 500] as $lim) {
      wp_cache_delete(self::cache_key('recent_' . $lim), self::CACHE_GROUP);
    }
  }

  static function install() {
    global $wpdb;
    $table = self::table();
    $charset = $wpdb->get_charset_collate();

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';

    $sql = "CREATE TABLE {$table} (
      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
      ts DATETIME NOT NULL,
      level VARCHAR(12) NOT NULL,
      event VARCHAR(64) NOT NULL,
      message TEXT NOT NULL,
      context LONGTEXT NULL,
      PRIMARY KEY  (id),
      KEY ts (ts)
    ) {$charset};";

    dbDelta($sql);
  }

  static function redact($context) {
    if (!is_array($context)) return $context;
    $deny = ['node_secret','bootstrap_token','secret','token','X-WSUM-NODE-SECRET'];
    $out = $context;
    foreach ($deny as $k) {
      if (isset($out[$k])) $out[$k] = '***';
    }
    // Deep redact common nested payloads
    foreach (['headers','body','payload'] as $sub) {
      if (isset($out[$sub]) && is_array($out[$sub])) {
        foreach ($deny as $k) {
          if (isset($out[$sub][$k])) $out[$sub][$k] = '***';
        }
      }
    }
    return $out;
  }

  static function log($level, $event, $message, $context = []) {
    global $wpdb;
    $table = self::table();

    // Avoid repeated SHOW TABLES calls.
    static $table_ok = null;
    if ($table_ok === null) {
      // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
      $exists = $wpdb->get_var( $wpdb->prepare('SHOW TABLES LIKE %s', $table) );
      $table_ok = ($exists === $table);
    }
    if (!$table_ok) return;

    $level = substr((string)$level, 0, 12);
    $event = substr(sanitize_key((string)$event), 0, 64);
    $message = (string)$message;

    $ctx = self::redact($context);
    $ctx_json = !empty($ctx) ? wp_json_encode($ctx) : null;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->insert($table, [
      'ts' => current_time('mysql'),
      'level' => $level,
      'event' => $event,
      'message' => $message,
      'context' => $ctx_json,
    ], ['%s','%s','%s','%s','%s']);

    // Invalidate caches after any write.
    wp_cache_delete('count', self::CACHE_GROUP);
    wp_cache_delete('recent_100', self::CACHE_GROUP);
    // Also clear common recent cache variants.
    for ($i = 10; $i <= 500; $i += 10) {
      wp_cache_delete('recent_' . $i, self::CACHE_GROUP);
    }

    self::trim();
  }

  static function trim() {
    global $wpdb;
    $table = self::table();
    // Keep last MAX_ROWS rows.
    $count = wp_cache_get('count', self::CACHE_GROUP);
    if ($count === false) {
      // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
      $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
      // Cache for a short period; this is used only for trimming.
      wp_cache_set('count', $count, self::CACHE_GROUP, 60);
    }
    if ($count <= self::MAX_ROWS) return;

    $to_delete = $count - self::MAX_ROWS;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
    $wpdb->query( $wpdb->prepare("DELETE FROM {$table} ORDER BY id ASC LIMIT %d", $to_delete) );

    // Invalidate caches after trim.
    wp_cache_delete('count', self::CACHE_GROUP);
    for ($i = 10; $i <= 500; $i += 10) {
      wp_cache_delete('recent_' . $i, self::CACHE_GROUP);
    }
  }

  static function recent($limit = 100) {
    global $wpdb;
    $table = self::table();
    $limit = max(1, min(500, (int)$limit));
    $cache_key = 'recent_' . $limit;
    $cached = wp_cache_get($cache_key, self::CACHE_GROUP);
    if ($cached !== false) {
      return $cached;
    }
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
    $rows = $wpdb->get_results( $wpdb->prepare("SELECT * FROM {$table} ORDER BY id DESC LIMIT %d", $limit), ARRAY_A );
    wp_cache_set($cache_key, $rows, self::CACHE_GROUP, 30);
    return $rows;
  }

  static function clear() {
    global $wpdb;
    $table = self::table();
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
    $wpdb->query("TRUNCATE TABLE {$table}");

    // Invalidate caches after clear.
    wp_cache_delete('count', self::CACHE_GROUP);
    for ($i = 10; $i <= 500; $i += 10) {
      wp_cache_delete('recent_' . $i, self::CACHE_GROUP);
    }
  }

  static function info($event, $message, $context = []) { self::log('info', $event, $message, $context); }
  static function warn($event, $message, $context = []) { self::log('warning', $event, $message, $context); }
  static function error($event, $message, $context = []) { self::log('error', $event, $message, $context); }
}
