<?php
if (!defined('ABSPATH')) exit;

/**
 * Tier-1 + Tier-2 health checks for the Node.
 *
 * Tier-1:
 * - Fatal/white-screen capture (shutdown handler stores last fatal)
 * - WP-Cron health (last tick timestamp)
 *
 * Tier-2:
 * - Database connectivity (SELECT 1)
 * - REST /wp-json/ loopback check (best-effort, may fail on some hosts)
 */
class ZUBBIN_UN_Health {

  static function init() {
    // Capture fatal errors (white screen) on shutdown.
    register_shutdown_function([__CLASS__, 'capture_fatal']);
  }

  static function opt_fatal() { return 'zubbin_un_last_fatal'; }
  static function opt_cron_last() { return 'zubbin_un_last_cron_tick'; }

  static function mark_cron_tick() {
    update_option(self::opt_cron_last(), time(), false);
  }

  static function capture_fatal() {
    $e = error_get_last();
    if (!is_array($e) || empty($e['type'])) return;

    // Fatal types.
    $fatal_types = [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR, E_RECOVERABLE_ERROR];
    if (!in_array((int)$e['type'], $fatal_types, true)) return;

    $data = [
      'ts' => time(),
      'type' => (int)$e['type'],
      'message' => (string)($e['message'] ?? ''),
      'file' => (string)($e['file'] ?? ''),
      'line' => (int)($e['line'] ?? 0),
    ];
    update_option(self::opt_fatal(), $data, false);
  }

  static function pull_fatal_and_clear() {
    $f = get_option(self::opt_fatal(), null);
    if (is_array($f) && !empty($f['ts'])) {
      delete_option(self::opt_fatal());
      return $f;
    }
    return null;
  }

  static function db_check() {
    global $wpdb;
    try {
      $v = $wpdb->get_var( 'SELECT 1' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
      return ((string)$v === '1');
    } catch (Throwable $t) {
      return false;
    }
  }

  static function rest_check() {
    // Best-effort loopback to REST index.
    $url = home_url('/wp-json/');
    $res = wp_remote_get($url, [
      'timeout' => 4,
      'redirection' => 1,
      'headers' => ['User-Agent' => 'WebSentinelNode/'.ZUBBIN_UN_VERSION],
    ]);
    if (is_wp_error($res)) return false;
    $http = (int)wp_remote_retrieve_response_code($res);
    if ($http < 200 || $http >= 400) return false;
    $raw = (string)wp_remote_retrieve_body($res);
    $json = json_decode($raw, true);
    return is_array($json) && !empty($json['namespaces']);
  }

  static function cron_last_ts() {
    $ts = get_option(self::opt_cron_last(), 0);
    return (int)$ts;
  }

  static function snapshot($s, $check_ms) {
    $cron_ts = self::cron_last_ts();
    $db_ok = self::db_check();
    $rest_ok = self::rest_check();
    $fatal = self::pull_fatal_and_clear();

    return [
      'wp' => get_bloginfo('version'),
      'php' => PHP_VERSION,
      'ms' => (int)$check_ms,
      'cron_last' => $cron_ts ? gmdate('Y-m-d H:i:s', $cron_ts) : '',
      'db_ok' => $db_ok ? 1 : 0,
      'rest_ok' => $rest_ok ? 1 : 0,
      'fatal' => $fatal,
    ];
  }
}
