<?php
if (!defined('ABSPATH')) exit;

class ZUBBIN_UN_Cron {

  static function interval_seconds() {
    // Default: 5 minutes. Can be overridden by Central via heartbeat response.
    $s = class_exists('ZUBBIN_UN_Settings') ? ZUBBIN_UN_Settings::get() : [];
    $mins = isset($s['heartbeat_minutes']) ? absint($s['heartbeat_minutes']) : 0;
    if ($mins < 1) $mins = 5;
    if ($mins > 60) $mins = 60;
    return $mins * 60;
  }

  static function schedules($s) {
    // Dynamic interval based on saved settings (updated from Central).
    $s['zubbin_un_dynamic'] = ['interval'=>self::interval_seconds(),'display'=>'Web Sentinel dynamic'];
    return $s;
  }

  static function activate() {
    // Ensure schedule exists during activation *and* at runtime.
    add_filter('cron_schedules', [__CLASS__, 'schedules']);
    // Schedule ASAP (WordPress needs the custom schedule registered to schedule it).
    if (!wp_next_scheduled('zubbin_un_monitor_tick')) {
      wp_schedule_event(time()+120, 'zubbin_un_dynamic', 'zubbin_un_monitor_tick');
    }
  }

  static function init() {
    add_filter('cron_schedules', [__CLASS__, 'schedules']);
    add_action('zubbin_un_monitor_tick', [__CLASS__, 'tick']);
    if (!wp_next_scheduled('zubbin_un_monitor_tick')) {
      wp_schedule_event(time()+120, 'zubbin_un_dynamic', 'zubbin_un_monitor_tick');
    }
  }

  static function maybe_reschedule($new_minutes) {
    $new_minutes = absint($new_minutes);
    if ($new_minutes < 1) return;
    if ($new_minutes > 60) $new_minutes = 60;

    $s = ZUBBIN_UN_Settings::get();
    $cur = isset($s['heartbeat_minutes']) ? absint($s['heartbeat_minutes']) : 0;
    if ($cur < 1) $cur = 5;
    if ($cur === $new_minutes) return;

    ZUBBIN_UN_Settings::save(['heartbeat_minutes' => $new_minutes]);

    // Reschedule with the updated dynamic interval.
    wp_clear_scheduled_hook('zubbin_un_monitor_tick');
    if (!wp_next_scheduled('zubbin_un_monitor_tick')) {
      wp_schedule_event(time()+120, 'zubbin_un_dynamic', 'zubbin_un_monitor_tick');
    }

    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::info('cron_reschedule', 'Updated monitor interval from Central', ['minutes'=>$new_minutes]);
    }
  }

  static function deactivate() {
    wp_clear_scheduled_hook('zubbin_un_monitor_tick');
  }

  static function tick() {
    $s = ZUBBIN_UN_Settings::get();
    if (!ZUBBIN_UN_Settings::paired($s)) return;

    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::info('cron_tick', 'Monitor tick running');
    }

    // record that WP-Cron successfully ran the monitor tick
    if (class_exists('ZUBBIN_UN_Health')) {
      ZUBBIN_UN_Health::mark_cron_tick();
    }

    $check = ZUBBIN_UN_Monitor::check($s);

    // Tier-1 + Tier-2 health snapshot
    $health = [];
    if (class_exists('ZUBBIN_UN_Health')) {
      $health = ZUBBIN_UN_Health::snapshot($s, (int)$check['ms']);
      // store a local copy for quick admin visibility
      ZUBBIN_UN_Settings::save(['last_health' => $health]);
    }

    $hb = ZUBBIN_UN_Client::heartbeat($s, $check['status'], $check['http'], $check['ms'], $check['msg'], $health);

    // Best-effort: apply entitlement/plan state returned by Central.
    if (class_exists('ZUBBIN_UN_Settings') && isset($hb['body']) && is_array($hb['body'])) {
      ZUBBIN_UN_Settings::apply_remote_state($hb['body']);
    }

    if (class_exists('ZUBBIN_UN_Logger')) {
      $ok = ((int)$hb['http'] === 200 && !empty($hb['body']['ok']));
      ZUBBIN_UN_Logger::info('heartbeat', $ok ? 'Heartbeat delivered to Central' : 'Heartbeat failed', [
        'status' => (string)$check['status'],
        'http' => (int)$hb['http'],
        'error' => (string)($hb['body']['error'] ?? ''),
      ]);
    }

    $upd = [
      'last_status' => $check['status'],
      'last_response_ms' => (int)$check['ms'],
      'last_message' => (string)$check['msg'],
      'last_http' => (int)$hb['http'],
      'last_error' => (string)($hb['body']['error'] ?? ''),
    ];

    if ((int)$hb['http'] === 200 && !empty($hb['body']['ok'])) {
      $upd['last_ok_at'] = current_time('mysql');
      $upd['last_error'] = '';

      // Central may return effective monitoring defaults.
      if (!empty($hb['body']['node']['heartbeat_minutes'])) {
        self::maybe_reschedule((int)$hb['body']['node']['heartbeat_minutes']);
      } elseif (!empty($hb['body']['central']['heartbeat_minutes'])) {
        self::maybe_reschedule((int)$hb['body']['central']['heartbeat_minutes']);
      }
    }

    ZUBBIN_UN_Settings::save($upd);
  }
}
