<?php
if (!defined('ABSPATH')) exit;

class ZUBBIN_UN_Client {

  static function post_json($url, $body=[], $headers=[], $timeout=20) {
    $res = wp_remote_post($url, [
      'timeout'     => $timeout,
      'redirection' => 5,
      'blocking'    => true,
      'headers'     => array_merge([
        'Content-Type' => 'application/json',
        'Accept'       => 'application/json',
      ], $headers),
      'body'        => wp_json_encode($body),
    ]);

    if (is_wp_error($res)) {
      return ['http'=>0,'body'=>['ok'=>false,'error'=>'wp_error','message'=>$res->get_error_message()]];
    }

    $http = (int) wp_remote_retrieve_response_code($res);
    $raw  = (string) wp_remote_retrieve_body($res);

    // Some hosts/security layers prepend HTML/notices before JSON.
    // Try to recover the JSON payload if possible.
    $raw_trim = trim($raw);
    $json = json_decode($raw_trim, true);

    if (!is_array($json)) {
      $first_obj = strpos($raw_trim, '{');
      $last_obj  = strrpos($raw_trim, '}');
      $first_arr = strpos($raw_trim, '[');
      $last_arr  = strrpos($raw_trim, ']');

      $cand = '';
      if ($first_obj !== false && $last_obj !== false && $last_obj > $first_obj) {
        $cand = substr($raw_trim, $first_obj, $last_obj - $first_obj + 1);
      } elseif ($first_arr !== false && $last_arr !== false && $last_arr > $first_arr) {
        $cand = substr($raw_trim, $first_arr, $last_arr - $first_arr + 1);
      }

      if ($cand !== '') {
        $json = json_decode($cand, true);
      }
    }

    if (!is_array($json)) {
      $snippet = $raw_trim;
      if (strlen($snippet) > 600) $snippet = substr($snippet, 0, 600).'…';
      $json = ['ok'=>false,'error'=>'bad_json','http'=>$http,'raw'=>$snippet];
    }

    return ['http'=>$http,'body'=>$json];
  }


  static function post_json_fallback($central_url, $route, $body=[], $headers=[], $timeout=20) {
    $urls = ZUBBIN_UN_Settings::endpoint_urls($central_url, $route);
    $last = ['http'=>0,'body'=>['ok'=>false,'error'=>'no_endpoints','message'=>'No Central endpoint URLs could be built']];
    foreach ($urls as $u) {
      $res = self::post_json($u, $body, $headers, $timeout);
      $last = $res;

      if ((int)($res['http'] ?? 0) !== 404) {
        $base = $u;
        $pos = strpos($base, '/wsum/v2');
        if ($pos !== false) $base = substr($base, 0, $pos + strlen('/wsum/v2'));
        ZUBBIN_UN_Settings::set_cached_api_base($base);
        return $res;
      }
    }

    $attempts = $urls;
    $b = $last['body'];
    if (!is_array($b)) $b = ['ok'=>false,'error'=>'bad_json','message'=>''];
    if (!isset($b['attempted_urls'])) $b['attempted_urls'] = $attempts;
    $last['body'] = $b;
    return $last;
  }

  static function hdrs($s) {
    return ['X-WSUM-NODE-SECRET' => (string)$s['node_secret']];
  }

  static function bootstrap($central_url, $bootstrap_token) {
    return self::post_json_fallback($central_url, '/bootstrap', [
      'bootstrap_token' => (string)$bootstrap_token,
      'site_url' => home_url('/'),
      'site_name' => get_bloginfo('name'),
    ], [], 25);
  }

  // Tokenless onboarding (Central must expose /auto-bootstrap)
  static function auto_bootstrap($central_url) {
    return self::post_json_fallback($central_url, '/auto-bootstrap', [
      'site_url'  => home_url('/'),
      'site_name' => get_bloginfo('name'),
    ], [], 25);
  }

  static function heartbeat($s, $status, $http_code, $ms, $message, $health = []) {
    $payload = [
      'node_key' => (string)$s['node_key'],
      'node_secret' => (string)$s['node_secret'],
      'status' => (string)$status,
      'http_code' => (int)$http_code,
      'response_ms' => (int)$ms,
      'message' => (string)$message,
    ];

    // Optional health payload (Tier-1 + Tier-2 checks)
    if (is_array($health) && !empty($health)) {
      $payload['health'] = $health;
    }

    return self::post_json_fallback($s['central_url'], '/heartbeat', $payload, self::hdrs($s), 25);
  }

  static function sync($s) {
    return self::post_json_fallback($s['central_url'], '/sync', [
      'node_key' => (string)$s['node_key'],
      'node_secret' => (string)$s['node_secret'],
      'notify_email' => (string)$s['notify_email'],
      'contact_name' => (string)$s['contact_name'],
      'contact_email' => (string)$s['contact_email'],
      'contact_phone' => (string)$s['contact_phone'],
      'contact_company' => (string)$s['contact_company'],
      'webhook_url' => (string)$s['webhook_url'],
      'webhook_enabled' => (int)$s['webhook_enabled'],
    ], self::hdrs($s), 25);
  }




  static function plans($s) {
    return self::post_json_fallback($s['central_url'], '/plans', [
      'node_key' => (string)$s['node_key'],
      'node_secret' => (string)$s['node_secret'],
    ], self::hdrs($s), 20);
  }

  static function checkout($s, $plan_key, $cadence) {
    return self::post_json_fallback($s['central_url'], '/checkout', [
      'node_key' => (string)$s['node_key'],
      'node_secret' => (string)$s['node_secret'],
      'plan_key' => sanitize_key((string)$plan_key),
      'cadence'  => ($cadence==='yearly') ? 'yearly' : 'monthly',
    ], self::hdrs($s), 25);
  }

  static function portal($s) {
    return self::post_json_fallback($s['central_url'], '/portal', [
      'node_key' => (string)$s['node_key'],
      'node_secret' => (string)$s['node_secret'],
    ], self::hdrs($s), 25);
  }

  static function test_auth($s) {
    // lightweight call: sync with no changes (still validates headers/secret)
    return self::post_json_fallback($s['central_url'], '/sync', [
      'node_key' => (string)$s['node_key'],
      'node_secret' => (string)$s['node_secret'],
    ], self::hdrs($s), 20);
  }
}