<?php
if (!defined('ABSPATH')) exit;

class ZUBBIN_UN_Admin {

  static function init() {
    add_action('admin_menu', [__CLASS__, 'menu']);
    add_action('admin_enqueue_scripts', [__CLASS__, 'assets']);
    add_action('admin_footer', [__CLASS__, 'footer_branding']);

    add_action('admin_post_zubbin_un_save_settings', [__CLASS__, 'save_settings']);
    add_action('admin_post_zubbin_un_bootstrap', [__CLASS__, 'bootstrap']);
    add_action('admin_post_zubbin_un_auto_register', [__CLASS__, 'auto_register']);
    add_action('admin_post_zubbin_un_reset_registration', [__CLASS__, 'reset_registration']);
    add_action('admin_post_zubbin_un_force_sync', [__CLASS__, 'force_sync']);
    add_action('admin_post_zubbin_un_test_auth', [__CLASS__, 'test_auth']);
    add_action('admin_post_zubbin_un_clear_logs', [__CLASS__, 'clear_logs']);
    add_action('admin_post_zubbin_un_start_checkout', [__CLASS__, 'start_checkout']);
    add_action('admin_post_zubbin_un_open_portal', [__CLASS__, 'open_portal']);
  }

  

  public static function register_post_handlers() {
    // admin-post handlers must be registered even when UI is not loaded
    add_action("admin_post_zubbin_un_save_settings", [__CLASS__, "save_settings"]);
    add_action("admin_post_zubbin_un_bootstrap", [__CLASS__, "bootstrap"]);
    add_action("admin_post_zubbin_un_auto_register", [__CLASS__, "auto_register"]);
    add_action("admin_post_zubbin_un_force_sync", [__CLASS__, "force_sync"]);
    add_action("admin_post_zubbin_un_test_auth", [__CLASS__, "test_auth"]);
    add_action("admin_post_zubbin_un_clear_logs", [__CLASS__, "clear_logs"]);
    add_action("admin_post_zubbin_un_start_checkout", [__CLASS__, "start_checkout"]);
    add_action("admin_post_zubbin_un_open_portal", [__CLASS__, "open_portal"]);
  }
  /**
   * Safely read a POST key intended to act as an internal action flag.
   *
   * @param string $key
   * @return string
   */
  private static function post_key($key) {
    // This helper only reads an internal action flag. The nonce is verified
    // immediately in the calling flow before any state-changing action occurs.
    if (!isset($_POST[$key])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
      return "";
    }
    return sanitize_key(wp_unslash($_POST[$key])); // phpcs:ignore WordPress.Security.NonceVerification.Missing
  }

static function menu() {
    add_menu_page('Zubbin Uptime Node','Zubbin Uptime Node','manage_options','zubbin_un',[__CLASS__,'page'],'dashicons-shield',56);
  }

  static function assets($hook) {
    if (strpos((string)$hook, 'zubbin_un') === false) return;
    wp_enqueue_style('zubbin-un-admin', ZUBBIN_UN_URL.'assets/css/admin.css', [], ZUBBIN_UN_VERSION);
    wp_enqueue_script('zubbin-un-admin', ZUBBIN_UN_URL.'assets/js/admin.js', ['jquery'], ZUBBIN_UN_VERSION, true);
  }

  static function page() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');

    $s = ZUBBIN_UN_Settings::get();
    // Tab selection is a read-only UI concern (no state change). Nonce is not required.
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    $tab = isset($_GET['tab']) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'dashboard';

    echo '<div class="wrap"><div class="wsum-header"><h1>Zubbin Uptime Node</h1><div class="wsum-mini-logo"><img src="'.esc_url(zubbin_un_brand_asset_url('zuptime-mark-32.png')).'" alt="Zubbin" /></div></div>';
    echo '<h2 class="nav-tab-wrapper">';
    $tabs = [
      'dashboard'=>'Dashboard',
      'onboarding'=>'Onboarding',
      'settings'=>'Settings',
      'sync'=>'Sync',
      'logs'=>'Activity',
      'privacy' => 'Privacy',
      'upgrade'=>'Upgrade',
      'help'=>'Help',
      'contact'=>'Contact',
    ];
    // Whitelist tab to avoid processing arbitrary user input.
    if (!isset($tabs[$tab])) {
      $tab = 'dashboard';
    }
    foreach ($tabs as $k=>$label) {
      $cls = ($k===$tab) ? 'nav-tab nav-tab-active' : 'nav-tab';
      echo '<a class="'.esc_attr($cls).'" href="'.esc_url(admin_url('admin.php?page=zubbin_un&tab='.$k)).'">'.esc_html($label).'</a>';
    }
    echo '</h2>';

    if ($tab==='onboarding') self::tab_onboarding($s);
    elseif ($tab==='settings') self::tab_settings($s);
    elseif ($tab==='sync') self::tab_sync($s);
    elseif ($tab==='logs') self::tab_logs();
    elseif ($tab === 'privacy') self::tab_privacy($s);
    elseif ($tab==='upgrade') self::tab_upgrade($s);
    elseif ($tab==='help') self::tab_help();
    elseif ($tab==='contact') self::tab_contact($s);
    else self::tab_dashboard($s);

    echo '</div>';
  }

  /**
   * Lightweight branding header (admin only).
   * Purely presentational.
   */
  static function branding_bar() {
    $img = zubbin_un_brand_asset_url('zuptime-admin-42.png');
    $out  = '<div class="ws-brandbar">';
    $out .= '<img class="ws-brand-logo" src="' . esc_url($img) . '" alt="Zubbin" />';
    $out .= '<div class="ws-brand-links">';
    $out .= '<a href="' . esc_url('https://www.zubbin.com') . '" target="_blank" rel="noopener">' . esc_html('www.zubbin.com') . '</a>';
    $out .= '<span class="ws-brand-sep">|</span>';
    $out .= '<a href="' . esc_url('https://websentinel.biz') . '" target="_blank" rel="noopener">' . esc_html('websentinel.biz') . '</a>';
    $out .= '</div></div>';
    return $out;
  }

  static function footer_branding() {
    if (!isset($_GET['page']) || sanitize_key(wp_unslash($_GET['page'])) !== 'zubbin_un') { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
      return;
    }
    $img = zubbin_un_brand_asset_url('zuptime-admin-42.png');
    $zubbin = 'https://www.zubbin.com';
    $biz = 'https://websentinel.biz';
    echo '<div class="wsum-footer-branding">';
    printf(
      '<img class="wsum-footer-img" src="%s" alt="%s" />',
      esc_url($img),
      esc_attr__('Zubbin', 'zubbin-uptime-node')
    );
    echo '<div class="wsum-footer-links">';
    printf(
      '<a href="%s" target="_blank" rel="noopener">%s</a>',
      esc_url($zubbin),
      esc_html('www.zubbin.com')
    );
    echo ' <span class="wsum-footer-sep">|</span> ';
    printf(
      '<a href="%s" target="_blank" rel="noopener">%s</a>',
      esc_url($biz),
      esc_html('websentinel.biz')
    );
    echo '</div>';
    echo '</div>';
  }


  static function notice($type, $msg) {
    echo '<div class="notice notice-'.esc_attr($type).'"><p>'.wp_kses_post($msg).'</p></div>';
  }


  /**
   * Soft status badge indicating whether this node is connected to Central.
   * Uses only locally stored sync/monitor fields.
   *
   * @param array $s settings/state array
   * @return string safe HTML
   */
  static function central_badge($s) {
    $paired = class_exists('ZUBBIN_UN_Settings') ? ZUBBIN_UN_Settings::paired($s) : false;

    $last_http = isset($s['last_http']) ? (int)$s['last_http'] : 0;
    $last_status = isset($s['last_status']) ? (string)$s['last_status'] : '';
    $last_ok_at = isset($s['last_ok_at']) ? (string)$s['last_ok_at'] : '';

    $ok_recent = false;
    if ($last_ok_at !== '') {
      $ts = strtotime($last_ok_at);
      if ($ts) {
        // Consider "recent" if within the last 30 minutes.
        $ok_recent = (time() - $ts) <= (30 * 60);
      }
    }

    $connected = $paired && ($last_http === 200) && ($last_status === 'up' || $ok_recent);

    $icon = zubbin_un_brand_asset_url('zuptime-mark-32.png');

    if ($connected) {
      $label = __('Connected to Central', 'zubbin-uptime-node');
      $cls = 'zubbin-un-badge zubbin-un-badge-ok';
    } elseif ($paired) {
      $label = __('Central connection pending', 'zubbin-uptime-node');
      $cls = 'zubbin-un-badge zubbin-un-badge-warn';
    } else {
      $label = __('Not paired with Central', 'zubbin-uptime-node');
      $cls = 'zubbin-un-badge zubbin-un-badge-muted';
    }

    return '<div class="zubbin-un-badge-row"><span class="' . esc_attr($cls) . '"><img class="zubbin-un-badge-icon" src="' . esc_url($icon) . '" alt="" />' . esc_html($label) . '</span></div>';
  }

  static function tab_dashboard($s) {
    echo '<div class="ws-card"><h2>Status</h2>';
    echo wp_kses(
      self::central_badge($s),
      [
        'div'  => [ 'class' => true ],
        'span' => [ 'class' => true ],
        'img'  => [
          'class' => true,
          'src'   => true,
          'alt'   => true,
        ],
      ]
    );

    echo '<p><strong>Paired:</strong> ' . esc_html( ZUBBIN_UN_Settings::paired($s) ? 'Yes' : 'No' ) . '</p>';
    echo '<p><strong>Last Status:</strong> '.esc_html((string)$s['last_status']).'</p>';
    echo '<p><strong>Response:</strong> '.esc_html((string)$s['last_response_ms']).' ms</p>';
    echo '<p><strong>Message:</strong> '.esc_html((string)$s['last_message']).'</p>';
    echo '<p><strong>Last Central HTTP:</strong> '.esc_html((string)$s['last_http']).'</p>';
    echo '<p><strong>Last OK At:</strong> '.esc_html((string)$s['last_ok_at']).'</p>';
    if (!empty($s['last_error'])) self::notice('error', '<strong>Last Error:</strong> '.esc_html((string)$s['last_error']));
    echo '<p class="description">Alerts are sent by Central (email/webhook/SMS). This node only reports status.</p>';
    echo '</div>';

    if (ZUBBIN_UN_Settings::paired($s)) {
      echo '<div class="ws-card"><h2>Central Auth</h2>';
      echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
      wp_nonce_field('zubbin_un_test_auth');
      echo '<input type="hidden" name="action" value="zubbin_un_test_auth">';
      submit_button('Test Central Auth','secondary');
      echo '</form></div>';
    }

    // Recent activity
    if (class_exists('ZUBBIN_UN_Logger')) {
      $rows = ZUBBIN_UN_Logger::recent(5);
      if (!empty($rows)) {
        echo '<div class="ws-card"><h2>Recent Activity</h2>';
        echo '<ul style="margin:0;padding-left:18px;">';
        foreach ($rows as $r) {
          $ts = isset($r['ts']) ? (string)$r['ts'] : '';
          $lvl = isset($r['level']) ? strtoupper((string)$r['level']) : '';
          $msg = isset($r['message']) ? (string)$r['message'] : '';
          echo '<li><span style="opacity:.75">'.esc_html($ts).' • '.esc_html($lvl).'</span> — '.esc_html($msg).'</li>';
        }
        echo '</ul>';
        echo '<p class="description"><a href="'.esc_url(admin_url('admin.php?page=zubbin_un&tab=logs')).'">View full activity log →</a></p>';
        echo '</div>';
      }
    }
  }

  static function tab_logs() {
    echo '<div class="ws-card"><h2>Activity Log</h2>';
    echo '<p class="description">This log shows what the node has been doing (heartbeats, syncs, onboarding, health checks). Secrets are redacted.</p>';

    echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'" style="margin:8px 0 16px;">';
    wp_nonce_field('zubbin_un_clear_logs');
    echo '<input type="hidden" name="action" value="zubbin_un_clear_logs">';
    submit_button('Clear Log','secondary', 'submit', false);
    echo '</form>';

    if (!class_exists('ZUBBIN_UN_Logger')) {
      self::notice('warning', 'Logger not available.');
      echo '</div>';
      return;
    }

    $rows = ZUBBIN_UN_Logger::recent(200);
    if (empty($rows)) {
      self::notice('info', 'No activity recorded yet. Heartbeats run every 5 minutes once paired.');
      echo '</div>';
      return;
    }

    echo '<div style="overflow:auto;">';
    echo '<table class="widefat striped">';
    echo '<thead><tr><th style="width:160px;">Time</th><th style="width:90px;">Level</th><th style="width:140px;">Event</th><th>Message</th><th style="width:35%;">Context</th></tr></thead><tbody>';
    foreach ($rows as $r) {
      $ctx = (string)($r['context'] ?? '');
      $ctx_pretty = '';
      if ($ctx !== '') {
        $decoded = json_decode($ctx, true);
        $ctx_pretty = is_array($decoded) ? wp_json_encode($decoded, JSON_PRETTY_PRINT) : $ctx;
      }
      echo '<tr>';
      echo '<td>' . esc_html((string)($r['ts'] ?? '')) . '</td>';
      echo '<td>' . esc_html(strtoupper((string)($r['level'] ?? ''))) . '</td>';
      echo '<td>' . esc_html((string)($r['event'] ?? '')) . '</td>';
      echo '<td>' . esc_html((string)($r['message'] ?? '')) . '</td>';
      echo '<td><pre style="white-space:pre-wrap;max-height:160px;overflow:auto;margin:0;">'.esc_html($ctx_pretty).'</pre></td>';
      echo '</tr>';
    }
    echo '</tbody></table>';
    echo '</div>';
    echo '</div>';
  }

  static function clear_logs() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_clear_logs');
    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::clear();
      ZUBBIN_UN_Logger::info('logs', 'Activity log cleared');
    }
    wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=logs'));
    exit;
  }

  static function tab_onboarding($s) {
    echo '<div class="ws-card"><h2>Onboarding</h2>';
    echo '<p>This node can <strong>auto-register</strong> with your Central server (recommended). Manual token registration is still available below.</p>';

    // Auto-register
    if (!ZUBBIN_UN_Settings::paired($s)) {
      echo '<h3 style="margin-top:16px;">Auto-register (recommended)</h3>';
      echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
      wp_nonce_field('zubbin_un_auto_register');
      echo '<input type="hidden" name="action" value="zubbin_un_auto_register">';
      echo '<p><label><strong>Central URL</strong><br><input class="regular-text" name="central_url" value="'.esc_attr((string)$s['central_url']).'" placeholder="https://zubbin.com" required></label></p>';
      echo '<p class="description">Auto-registration requests a node key/secret from Central via a secure endpoint. No bootstrap token needed.</p>';
      submit_button('Auto-register Now','primary');
      echo '</form>';
      echo '<hr style="margin:18px 0;">';
    } else {
      self::notice('success','This node is already registered with Central.');

      echo '<h3 style="margin-top:16px;color:#b32d2e;">Reset registration</h3>';
      echo '<p class="description">If this site was already registered on Central (or you changed domains), you can reset the local node key/secret and register again.</p>';
      echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'" onsubmit="return confirm(\'Reset registration on this site? You will need to auto-register again.\');">';
      wp_nonce_field('zubbin_un_reset_registration');
      echo '<input type="hidden" name="action" value="zubbin_un_reset_registration">';
      submit_button('Reset Registration','delete');
      echo '</form>';

    }

    // Manual register
    echo '<h3>Manual registration (bootstrap token)</h3>';
    echo '<p>Use this only if auto-registration is disabled on Central.</p>';
    echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
    wp_nonce_field('zubbin_un_bootstrap');
    echo '<input type="hidden" name="action" value="zubbin_un_bootstrap">';
    echo '<p><label><strong>Central URL</strong><br><input class="regular-text" name="central_url" value="'.esc_attr((string)$s['central_url']).'" required></label></p>';
    echo '<p><label><strong>Bootstrap Token</strong><br><input class="regular-text" name="bootstrap_token" value="" required></label></p>';
    submit_button('Register (Token)','secondary');
    echo '</form>';
    echo '</div>';

    echo '<div class="ws-card"><h2>After Registration</h2><ol>';
    echo '<li>Wait for the first heartbeat (runs every 5 minutes)</li>';
    echo '<li>Go to <strong>Sync</strong> tab and click <strong>Force Sync</strong></li>';
    echo '<li>Go to <strong>Payment</strong> tab to start subscription (if required)</li>';
    echo '</ol></div>';
  }

  static function auto_register() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_auto_register');

    $central_url = isset($_POST['central_url']) ? esc_url_raw( wp_unslash( $_POST['central_url'] ) ) : '';
    if (empty($central_url)) {
      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=onboarding&err=missing_central'));
      exit;
    }

    // Save central URL first.
    ZUBBIN_UN_Settings::save(['central_url' => $central_url]);
    $s = ZUBBIN_UN_Settings::get();

    if (ZUBBIN_UN_Settings::paired($s)) {
      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=dashboard&msg=already_paired'));
      exit;
    }

    $r = ZUBBIN_UN_Client::auto_bootstrap($central_url);
    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::info('auto_bootstrap', 'Auto-register requested from admin', ['http' => (int)$r['http']]);
    }
    if ((int)$r['http'] === 200 && !empty($r['body']['ok']) && !empty($r['body']['node_key']) && !empty($r['body']['node_secret'])) {
      ZUBBIN_UN_Settings::save([
        'node_key' => (string)$r['body']['node_key'],
        'node_secret' => (string)$r['body']['node_secret'],
        'last_error' => '',
      ]);
      $s2 = ZUBBIN_UN_Settings::get();
      if (ZUBBIN_UN_Settings::paired($s2)) {
        $sr = ZUBBIN_UN_Client::sync($s2);
        if (class_exists('ZUBBIN_UN_Logger')) {
          ZUBBIN_UN_Logger::info('sync', 'Sync after auto-register', ['http' => (int)$sr['http']]);
        }
      }
      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=dashboard&msg=auto_registered'));
      exit;
    }

    $err = (string)($r['body']['error'] ?? 'auto_bootstrap_failed');
    $msg = (string)($r['body']['message'] ?? '');
    $raw = isset($r['body']['raw']) ? (string)$r['body']['raw'] : '';
    if ($msg === '' && $raw !== '') $msg = $raw;
    ZUBBIN_UN_Settings::save(['last_error' => $msg !== '' ? ($err . ': ' . $msg) : $err]);
    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::warn('auto_bootstrap', 'Auto-register failed (admin)', ['http' => (int)$r['http'], 'error' => $err, 'message' => $msg]);
    }
    wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=onboarding&err=auto_register_failed'));
    exit;
  }


  static function reset_registration() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_reset_registration');

    // Keep Central URL & contact info, but clear all pairing/identity + cached API base and entitlement.
    ZUBBIN_UN_Settings::save([
      'node_key' => '',
      'node_secret' => '',
      'central_api_base' => '',
      'last_error' => '',
      'billing_status' => '',
      'block_reason' => '',
      'plan_key' => '',
      'limits' => null,
      'features' => null,
    ]);

    if (class_exists('ZUBBIN_UN_Logger')) {
      ZUBBIN_UN_Logger::info('reset', 'Registration reset from admin', []);
    }

    wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=onboarding&msg=reset_done'));
    exit;
  }

  static function tab_settings($s) {
    echo '<div class="ws-card"><h2>Settings</h2>';
    echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
    wp_nonce_field('zubbin_un_save_settings');
    echo '<input type="hidden" name="action" value="zubbin_un_save_settings">';

    echo '<table class="form-table">';
    echo '<tr><th>Notify Email</th><td><input class="regular-text" name="notify_email" value="'.esc_attr((string)$s['notify_email']).'"><p class="description">Central sends alerts to this email.</p></td></tr>';
    echo '<tr><th>Contact Name</th><td><input class="regular-text" name="contact_name" value="'.esc_attr((string)$s['contact_name']).'"></td></tr>';
    echo '<tr><th>Contact Email</th><td><input class="regular-text" name="contact_email" value="'.esc_attr((string)$s['contact_email']).'"></td></tr>';
    echo '<tr><th>Contact Phone</th><td><input class="regular-text" name="contact_phone" value="'.esc_attr((string)$s['contact_phone']).'"><p class="description">Used for Central SMS if enabled.</p></td></tr>';
    echo '<tr><th>Company</th><td><input class="regular-text" name="contact_company" value="'.esc_attr((string)$s['contact_company']).'"></td></tr>';

    echo '<tr><th>Webhook URL</th><td><input class="regular-text" name="webhook_url" value="'.esc_attr((string)$s['webhook_url']).'"><p class="description">Central will call this when alerts fire (if enabled globally + per-node).</p></td></tr>';
    echo '<tr><th>Webhook Enabled</th><td><label><input type="checkbox" name="webhook_enabled" value="1" '.checked(!empty($s['webhook_enabled']), true, false).'> Enable for this node</label></td></tr>';

    echo '<tr><th>Check URL</th><td><input class="regular-text" name="check_url" value="'.esc_attr((string)$s['check_url']).'"></td></tr>';
    echo '<tr><th>Timeout</th><td><input type="number" name="check_timeout" value="'.esc_attr((string)$s['check_timeout']).'" min="3" max="60"></td></tr>';
    echo '</table>';

    submit_button('Save Settings','primary');
    echo '</form>';
    echo '</div>';
  }

  static function tab_sync($s) {
    echo '<div class="ws-card"><h2>Sync</h2>';

    if (!ZUBBIN_UN_Settings::paired($s)) {
      self::notice('warning', 'Pair the node first in <strong>Onboarding</strong>.');
      echo '</div>';
      return;
    }

    echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
    wp_nonce_field('zubbin_un_force_sync');
    echo '<input type="hidden" name="action" value="zubbin_un_force_sync">';
    submit_button('Force Sync Now','secondary');
    echo '</form>';

    echo '<p class="description">Sync pushes notify_email + contact + webhook settings to Central.</p>';
    echo '</div>';
  }
  
  static function tab_privacy($s) {
  echo '<div class="ws-card"><h2>Privacy</h2>';

  echo '<p>This plugin sends uptime and configuration data to the Central endpoint you configure.</p>';

  echo '<h3>Data sent</h3><ul>';
  echo '<li><strong>Site URL</strong> and <strong>Site Name</strong></li>';
  echo '<li><strong>Node identifier</strong> assigned by Central (node_key)</li>';
  echo '<li><strong>Uptime status</strong> (up/down/unknown), <strong>HTTP status code</strong>, and <strong>response time</strong></li>';
  echo '<li><strong>Short diagnostic message</strong> (e.g., http_500)</li>';
  echo '<li><strong>Notification email</strong> (notify_email) and optional contact details (name/email/phone/company)</li>';
  echo '<li>Optional <strong>webhook settings</strong> (URL + enabled)</li>';
  echo '</ul>';

  echo '<h3>Purpose</h3>';
  echo '<p>This data is used solely for uptime monitoring and alert routing via Central.</p>';

  echo '<h3>Control</h3>';
  echo '<p>The Central endpoint is explicitly configured by the site administrator. This plugin does not connect to any external service until you provide a Central URL and register the node.</p>';

  echo '</div>';
}

  static function tab_help() {
    echo '<div class="ws-card"><h2>Help</h2><ol>';
    echo '<li>Central → Bootstrap tab → Generate token</li>';
    echo '<li>Node → Onboarding tab → Register Now</li>';
    echo '<li>Wait 5 minutes for heartbeat</li>';
    echo '<li>Node → Sync → Force Sync</li>';
    echo '<li>Central sends alerts on UP/DOWN transitions</li>';
    echo '</ol></div>';
  }

  static function tab_contact($s) {
    echo '<div class="wsum-contact-brand"><img src="' . esc_url( zubbin_un_brand_asset_url( 'zuptime-mark-32.png' ) ) . '" alt="' . esc_attr__( 'Z Uptime', 'zubbin-uptime-node' ) . '" /></div>';
    echo '<div class="ws-card"><h2>Contact</h2>';

    // Support (pulled from Central and cached locally)
    $support_email = (string)($s['support_email'] ?? '');
    $support_url = (string)($s['support_url'] ?? '');
    $support_phone = (string)($s['support_phone'] ?? '');
    $support_wa = (string)($s['support_whatsapp'] ?? '');
    $node_key = (string)($s['node_key'] ?? '');

    echo '<h3 style="margin-top:0;">Support</h3>';
    if ($support_email === '' && $support_url === '' && $support_phone === '' && $support_wa === '') {
      self::notice('info', 'Support details will appear here after you run <strong>Sync</strong> or visit the <strong>Payment</strong> tab.');
    } else {
      echo '<ul>';
      if ($support_email !== '') {
        echo '<li><strong>Email:</strong> <a href="mailto:'.esc_attr($support_email).'">'.esc_html($support_email).'</a></li>';
      }
      if ($support_url !== '') {
        echo '<li><strong>Portal:</strong> <a href="'.esc_url($support_url).'" target="_blank" rel="noopener noreferrer">'.esc_html($support_url).'</a></li>';
      }
      if ($support_phone !== '') {
        echo '<li><strong>Phone:</strong> '.esc_html($support_phone).'</li>';
      }
      if ($support_wa !== '') {
        // Accept either a full URL or a phone number. If it looks like a number, build wa.me link.
        $wa_link = $support_wa;
        if (preg_match('/^\+?[0-9][0-9\s\-\(\)]{6,}$/', $support_wa)) {
          $digits = preg_replace('/\D+/', '', $support_wa);
          $wa_link = 'https://wa.me/'.$digits;
        }
        echo '<li><strong>WhatsApp:</strong> <a href="'.esc_url($wa_link).'" target="_blank" rel="noopener noreferrer">'.esc_html($support_wa).'</a></li>';
      }
      echo '</ul>';
    }

    if ($node_key !== '') {
      echo '<p class="description">Include your Node Key in support requests: <code>'.esc_html($node_key).'</code></p>';
    }

    echo '<hr style="margin:18px 0;">';
    echo '<h3>Site Owner (this WordPress site)</h3>';
    echo '<ul>';
    echo '<li><strong>Email:</strong> '.esc_html((string)($s['contact_email'] ?? '')).'</li>';
    echo '<li><strong>Phone:</strong> '.esc_html((string)($s['contact_phone'] ?? '')).'</li>';
    echo '<li><strong>Company:</strong> '.esc_html((string)($s['contact_company'] ?? '')).'</li>';
    echo '</ul>';

    echo '</div>';
  }

  static function save_settings() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_save_settings');

    $data = [
      'notify_email' => sanitize_email( wp_unslash( $_POST['notify_email'] ?? '' ) ),
      'contact_name' => sanitize_text_field( wp_unslash( $_POST['contact_name'] ?? '' ) ),
      'contact_email' => sanitize_email( wp_unslash( $_POST['contact_email'] ?? '' ) ),
      'contact_phone' => sanitize_text_field( wp_unslash( $_POST['contact_phone'] ?? '' ) ),
      'contact_company' => sanitize_text_field( wp_unslash( $_POST['contact_company'] ?? '' ) ),
      'webhook_url' => isset($_POST['webhook_url']) ? esc_url_raw( wp_unslash( $_POST['webhook_url'] ) ) : '',
      'webhook_enabled' => !empty($_POST['webhook_enabled']) ? 1 : 0,
      'check_url' => isset($_POST['check_url']) ? esc_url_raw( wp_unslash( $_POST['check_url'] ) ) : home_url('/'),
      'check_timeout' => max(3, min(60, (int) (isset($_POST['check_timeout']) ? absint( wp_unslash( $_POST['check_timeout'] ) ) : 10))),
    ];

    if ($data['notify_email'] === '') $data['notify_email'] = get_option('admin_email');
    if ($data['contact_email'] === '') $data['contact_email'] = $data['notify_email'];
    if ($data['check_url'] === '') $data['check_url'] = home_url('/');

    ZUBBIN_UN_Settings::save($data);

    // optional auto-sync when paired
    $s = ZUBBIN_UN_Settings::get();
    if (ZUBBIN_UN_Settings::paired($s)) {
      $r = ZUBBIN_UN_Client::sync($s);

      if ((int)$r['http'] === 200 && !empty($r['body']['central']) && is_array($r['body']['central'])) {
        $c = $r['body']['central'];
        ZUBBIN_UN_Settings::save([
          'support_email' => isset($c['support_email']) ? sanitize_email((string)$c['support_email']) : '',
          'support_url' => isset($c['support_url']) ? esc_url_raw((string)$c['support_url']) : '',
          'support_phone' => isset($c['support_phone']) ? sanitize_text_field((string)$c['support_phone']) : '',
          'support_whatsapp' => isset($c['support_whatsapp']) ? sanitize_text_field((string)$c['support_whatsapp']) : '',
        ]);
      }

    }

    wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=settings'));
    exit;
  }

  static function bootstrap() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_bootstrap');

    $central = isset($_POST['central_url']) ? trim( (string) esc_url_raw( wp_unslash( $_POST['central_url'] ) ) ) : '';
    $token   = isset($_POST['bootstrap_token']) ? trim( sanitize_text_field( wp_unslash( $_POST['bootstrap_token'] ) ) ) : '';

    if ($central==='' || $token==='') {
      wp_die('Missing Central URL or Bootstrap Token.');
    }

    $r = ZUBBIN_UN_Client::bootstrap($central, $token);

    if ((int)$r['http'] === 200 && !empty($r['body']['ok']) && !empty($r['body']['node_key']) && !empty($r['body']['node_secret'])) {
      ZUBBIN_UN_Settings::save([
        'central_url' => esc_url_raw($central),
        'node_key' => (string)$r['body']['node_key'],
        'node_secret' => (string)$r['body']['node_secret'],
        'last_error' => '',
        'last_http' => 200,
        'last_status' => 'unknown',
        'last_ok_at' => '',
      ]);

      // push initial sync immediately
      $s = ZUBBIN_UN_Settings::get();
      ZUBBIN_UN_Client::sync($s);

      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=dashboard'));
      exit;
    }

  }

  static function force_sync() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_force_sync');

    $s = ZUBBIN_UN_Settings::get();
    if (!ZUBBIN_UN_Settings::paired($s)) wp_die('Not paired');

    $r = ZUBBIN_UN_Client::sync($s);
    if ((int)($r['http'] ?? 0) === 200 && !empty($r['body']) && is_array($r['body'])) {
      ZUBBIN_UN_Settings::apply_remote_state($r['body']);
    }
    if ((int)$r['http'] !== 200) {
    }

    wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=sync'));
    exit;
  }

  static function test_auth() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_test_auth');

    $s = ZUBBIN_UN_Settings::get();
    if (!ZUBBIN_UN_Settings::paired($s)) wp_die('Not paired');

    $r = ZUBBIN_UN_Client::test_auth($s);
    if ((int)$r['http'] === 200) {
      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=dashboard'));
      exit;
    }

  }


  static function tab_upgrade($s) {
    echo '<div class="ws-card"><h2>' . esc_html__('Upgrade', 'zubbin-uptime-node') . '</h2>';
    echo '<p class="description">' . esc_html__('Upgrade is handled securely by your Central dashboard. This site does not process payments.', 'zubbin-uptime-node') . '</p>';

    if (!ZUBBIN_UN_Settings::paired($s)) {
      self::notice('warning', esc_html__('Connect to your Central dashboard first (Onboarding tab).', 'zubbin-uptime-node'));
      echo '</div>';
      return;
    }

    // Flash notice from prior actions (checkout/portal).
    $flash = get_transient('zubbin_un_flash_notice');
    if (is_array($flash) && !empty($flash['type']) && !empty($flash['message'])) {
      delete_transient('zubbin_un_flash_notice');
      self::notice((string)$flash['type'], (string)$flash['message']);
    }

    // Fetch plan info from Central (best-effort).
    $plan_key = isset($s['plan_key']) ? (string)$s['plan_key'] : 'free';
    $limits = isset($s['plan_limits']) && is_array($s['plan_limits']) ? $s['plan_limits'] : [];
    $plans = [];

    $r = ZUBBIN_UN_Client::plans($s);
    if ((int)$r['http'] === 200 && !empty($r['body']['ok'])) {
      // Cache entitlement/plan state if provided.
      if (is_array($r['body'])) {
        ZUBBIN_UN_Settings::apply_remote_state($r['body']);
      }
      $node = is_array($r['body']['node'] ?? null) ? $r['body']['node'] : [];
      $plan_key = (string)($node['plan_key'] ?? $plan_key);

      $plans = is_array($r['body']['plans'] ?? null) ? $r['body']['plans'] : [];

      // Cache support + plan info for display elsewhere.
      $central = is_array($r['body']['central'] ?? null) ? $r['body']['central'] : [];
      ZUBBIN_UN_Settings::save([
        'support_email' => (string)($central['support_email'] ?? ''),
        'support_url' => (string)($central['support_url'] ?? ''),
        'support_phone' => (string)($central['support_phone'] ?? ''),
        'support_whatsapp' => (string)($central['support_whatsapp'] ?? ''),
        'plan_key' => $plan_key,
      ]);
    } else {
      // If Central can't be reached or doesn't support the plans endpoint yet,
      // still allow users to click Upgrade (Central will validate plans).
      $http = (int)($r['http'] ?? 0);
      self::notice('warning', sprintf(
        /* translators: %s: HTTP status code returned by Central when loading plans. */
        esc_html__('Could not load plans from Central (HTTP %s). Upgrade buttons will still work, but prices may not display. Check your Central URL and that this site is paired.', 'zubbin-uptime-node'),
        $http ? (string)$http : '0'
      ));
    }

    // Refresh local cache after any remote updates.
    $s = ZUBBIN_UN_Settings::get();
    $billing_status = isset($s['billing_status']) ? (string)$s['billing_status'] : '';
    $block_reason = isset($s['block_reason']) ? (string)$s['block_reason'] : '';

    echo '<p><strong>' . esc_html__('Current plan:', 'zubbin-uptime-node') . '</strong> ' . esc_html($plan_key) . '</p>';

    if ($billing_status !== '') {
      echo '<p><strong>' . esc_html__('Billing status:', 'zubbin-uptime-node') . '</strong> ' . esc_html($billing_status) . '</p>';
    }

    if ($billing_status === 'blocked' || $billing_status === 'inactive' || $billing_status === 'past_due' || $billing_status === 'unpaid') {
      $msg = esc_html__('This site is not currently entitled to paid features. If you believe this is a mistake, open the billing portal or contact support.', 'zubbin-uptime-node');
      if ($block_reason !== '') {
        $msg .= ' ' . esc_html($block_reason);
      }
      self::notice('warning', $msg);
    }

    // Helpful for support troubleshooting (do not show full key).
    $nk = isset($s['node_key']) ? (string)$s['node_key'] : '';
    if ($nk !== '') {
      $mask = (strlen($nk) > 10) ? substr($nk, 0, 4) . '…' . substr($nk, -4) : $nk;
      echo '<p><small>' . esc_html__('Node key:', 'zubbin-uptime-node') . ' ' . esc_html($mask) . '</small></p>';
    }

    // Render upgrade options (Starter/Pro) with monthly/yearly.
    echo '<div class="ws-grid">';
    foreach (['starter','pro'] as $k) {
      $row = [];
      foreach ($plans as $p) {
        if (is_array($p) && (string)($p['key'] ?? '') === $k) { $row = $p; break; }
      }

      $name = (string)($row['name'] ?? ucfirst($k));
      // Central implementations may use different field names; accept several.
      // Newer Central returns nested arrays: price.monthly/yearly and limits.*
      $price_obj = is_array($row['price'] ?? null) ? $row['price'] : [];
      $limits_obj = is_array($row['limits'] ?? null) ? $row['limits'] : [];

      $m_price = (string)($price_obj['monthly'] ?? ($row['monthly'] ?? ($row['monthly_price'] ?? ($row['price_monthly'] ?? ($row['amount_monthly'] ?? '')))));
      $y_price = (string)($price_obj['yearly'] ?? ($row['yearly'] ?? ($row['yearly_price'] ?? ($row['price_yearly'] ?? ($row['amount_yearly'] ?? '')))));

      echo '<div class="ws-card">';
      echo '<h3>' . esc_html($name) . '</h3>';

      // Limits summary if provided
      $sites = (string)($limits_obj['sites'] ?? ($row['sites'] ?? ''));
      $min_interval = (string)($limits_obj['min_interval'] ?? ($limits_obj['min_check_interval'] ?? ($row['min_interval'] ?? ($row['min_check_interval'] ?? ''))));
      if ($sites !== '' || $min_interval !== '') {
        echo '<ul class="ul-disc">';
	      // translators: %s: number of sites allowed on this plan.
	      if ($sites !== '') echo '<li>' . sprintf(esc_html__('%s sites', 'zubbin-uptime-node'), esc_html($sites)) . '</li>';
	      // translators: %s: minimum allowed check interval in seconds.
	      if ($min_interval !== '') echo '<li>' . sprintf(esc_html__('Minimum check interval: %s seconds', 'zubbin-uptime-node'), esc_html($min_interval)) . '</li>';
        echo '</ul>';
      }

      echo '<div class="ws-row">';
      // Monthly button (always show; price label is optional)
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;margin-right:8px;">';
        wp_nonce_field('zubbin_un_start_checkout');
        echo '<input type="hidden" name="action" value="zubbin_un_start_checkout" />';
        echo '<input type="hidden" name="plan_key" value="' . esc_attr($k) . '" />';
        echo '<input type="hidden" name="cadence" value="monthly" />';
      if ($m_price !== '') {
        // translators: %s: formatted price (example: $19).
        submit_button(sprintf(esc_html__('Monthly %s', 'zubbin-uptime-node'), '$'.esc_html($m_price)), 'primary', 'submit', false);
      } else {
        submit_button(esc_html__('Monthly', 'zubbin-uptime-node'), 'primary', 'submit', false);
      }
        echo '</form>';

      // Yearly button (always show; price label is optional)
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;">';
        wp_nonce_field('zubbin_un_start_checkout');
        echo '<input type="hidden" name="action" value="zubbin_un_start_checkout" />';
        echo '<input type="hidden" name="plan_key" value="' . esc_attr($k) . '" />';
        echo '<input type="hidden" name="cadence" value="yearly" />';
      if ($y_price !== '') {
        // translators: %s: formatted price (example: $190).
        submit_button(sprintf(esc_html__('Yearly %s', 'zubbin-uptime-node'), '$'.esc_html($y_price)), 'secondary', 'submit', false);
      } else {
        submit_button(esc_html__('Yearly', 'zubbin-uptime-node'), 'secondary', 'submit', false);
      }
        echo '</form>';

      echo '</div>'; // row
      echo '</div>'; // card
    }
    echo '</div>'; // grid

    // Portal button if available
    echo '<hr />';
    echo '<p>' . esc_html__('Already subscribed? Manage billing in the secure customer portal.', 'zubbin-uptime-node') . '</p>';
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
    wp_nonce_field('zubbin_un_open_portal');
    echo '<input type="hidden" name="action" value="zubbin_un_open_portal" />';
    submit_button(esc_html__('Open Billing Portal', 'zubbin-uptime-node'), 'secondary');
    echo '</form>';

    echo '</div>';
  }

  static function start_checkout() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_start_checkout');

    $s = ZUBBIN_UN_Settings::get();
    if (!ZUBBIN_UN_Settings::paired($s)) wp_die('Not paired');

    $plan_key = isset($_POST['plan_key']) ? sanitize_key(wp_unslash($_POST['plan_key'])) : 'starter'; // phpcs:ignore WordPress.Security.NonceVerification.Missing
    $cadence = isset($_POST['cadence']) ? sanitize_key(wp_unslash($_POST['cadence'])) : 'monthly'; // phpcs:ignore WordPress.Security.NonceVerification.Missing
    $cadence = ($cadence === 'yearly') ? 'yearly' : 'monthly';

    $r = ZUBBIN_UN_Client::checkout($s, $plan_key, $cadence);
    $url = '';
    if ((int)$r['http'] === 200 && !empty($r['body']['ok'])) {
      $url = (string)($r['body']['checkout_url'] ?? '');
    }

    if ($url === '') {
      $http = (int)($r['http'] ?? 0);
      $err  = (string)($r['body']['error'] ?? 'checkout_failed');
      $msg  = (string)($r['body']['message'] ?? '');
      $detail = (string)($r['body']['detail']['error']['message'] ?? '');
      if ($detail !== '') {
        $msg = $detail;
      }
      if ($msg === '') {
        $msg = sprintf(
          /* translators: %1$s: error code; %2$s: http status */
          esc_html__('Upgrade failed (%1$s, HTTP %2$s). Please check your Central plan price IDs and Stripe configuration.', 'zubbin-uptime-node'),
          esc_html($err),
          esc_html($http ? (string)$http : '0')
        );
      }
      set_transient('zubbin_un_flash_notice', ['type'=>'error','message'=>$msg], 60);
      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=upgrade'));
      exit;
    }

	    $target = esc_url_raw($url);
	    $host   = wp_parse_url($target, PHP_URL_HOST);
	    if ($host) {
	      add_filter('allowed_redirect_hosts', function($hosts) use ($host) {
	        if (!in_array($host, $hosts, true)) $hosts[] = $host;
	        return $hosts;
	      });
	    }
	    wp_safe_redirect($target);
	    exit;
  }

  static function open_portal() {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('zubbin_un_open_portal');

    $s = ZUBBIN_UN_Settings::get();
    if (!ZUBBIN_UN_Settings::paired($s)) wp_die('Not paired');

    $r = ZUBBIN_UN_Client::portal($s);
    $url = '';
    if ((int)$r['http'] === 200 && !empty($r['body']['ok'])) {
      $url = (string)($r['body']['portal_url'] ?? '');
    }

    if ($url === '') {
      $http = (int)($r['http'] ?? 0);
      $err  = (string)($r['body']['error'] ?? 'portal_failed');
      $msg  = (string)($r['body']['message'] ?? '');
      if ($msg === '') {
        $msg = sprintf(
          /* translators: %1$s: error code; %2$s: http status */
          esc_html__('Billing portal unavailable (%1$s, HTTP %2$s). If you just subscribed, wait a minute and try again. Otherwise, confirm Stripe webhooks are configured on Central.', 'zubbin-uptime-node'),
          esc_html($err),
          esc_html($http ? (string)$http : '0')
        );
      }
      set_transient('zubbin_un_flash_notice', ['type'=>'warning','message'=>$msg], 60);
      wp_safe_redirect(admin_url('admin.php?page=zubbin_un&tab=upgrade'));
      exit;
    }

	    $target = esc_url_raw($url);
	    $host   = wp_parse_url($target, PHP_URL_HOST);
	    if ($host) {
	      add_filter('allowed_redirect_hosts', function($hosts) use ($host) {
	        if (!in_array($host, $hosts, true)) $hosts[] = $host;
	        return $hosts;
	      });
	    }
	    wp_safe_redirect($target);
	    exit;
  }

}
