<?php //phpcs:ignore
/**
 * Main Plugin File
 *
 * The primary entry point for the ZooMeet plugin.
 *
 * @package SOVLIX\ZMEET
 */

namespace SOVLIX\ZMEET;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check if the main class already exists.
 *
 * @since 1.0.0
 */
if ( ! class_exists( 'Zmeet_Boot' ) ) {
	/**
	 * ZooMeet Plugin Main Class
	 *
	 * Sets up and initializes the ZooMeet plugin.
	 *
	 * @since 1.0.0
	 */
	class Zmeet_Boot {
		/**
		 * Class constructor.
		 *
		 * Hooks into the 'plugins_loaded' action to initiate the plugin.
		 */
		private function __construct() {
			$this->init_promo_sdk();
			add_action( 'plugins_loaded', array( $this, 'init_plugin' ) );
		}

		/**
		 * Initialize a singleton instance of the plugin.
		 *
		 * @return Zmeet An instance of the Zmeet class.
		 * @since  1.0.0
		 */
		public static function init() {
			static $instance = false;

			if ( ! $instance ) {
				$instance = new self();
			}

			return $instance;
		}

		/**
		 * Initialize the ZooMeet plugin.
		 *
		 * Initiates the necessary components for the plugin, such as Assets,
		 * and either the Admin or Frontend components based on the current context.
		 *
		 * @return void
		 * @since  1.0.0
		 */
		public function init_plugin() {

			// Initialize Ajax component for asynchronous actions.
			if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
				new Zmeet_Ajax();
			}

			new Zmeet_Assets();

			if ( is_admin() ) {
				new Zmeet_Admin();
			} else {
				new Zmeet_Frontend();
			}

			new Zmeet_Post_Types();
			new Zmeet_API();
			new Zmeet_Shortcodes();
			new Zmeet_Gutenberg();
			\SOVLIX\ZMEET\Zoom\Zmeet_Zoom_Api::instance();

			// Get the list of active plugins.
			$active_plugins = get_option( 'active_plugins' );

			// Check if the Elementor plugin is active.
			if ( in_array( 'elementor/elementor.php', $active_plugins, true ) ) {
				new Elementor\Zmeet_Elementor_Integrator();
			}
		}

		/**
		 * Initialize Promo SDK
		 *
		 * @return void
		 * @since  1.23.10
		 */
		private function init_promo_sdk() {
			// Load SDK main class.
			$sdk_path = ZMEET_INCLUDES . '/lib/promo-sdk/class-promo-sdk.php';

			if ( ! file_exists( $sdk_path ) ) {
				return;
			}

			require_once $sdk_path;

			// Initialize SDK with configuration.
			\Promo_SDK::init_sdk(
				array(
					'product_slug'         => 'zoomeet',
					'plugin_page_slug'     => 'zmeet',
					'hide_modal_demo_link' => true,
					'api_endpoint'         => 'https://sovlix.com/wp-json/leetcampaign/v1/campaigns/active/zoomeet',
					'cache_duration'       => 3600,
					'text_domain'          => 'zoomeet',
					'prefix'               => 'zoomeet',
					'is_pro'               => zoo_fs()->is_premium(),
				)
			);
		}
	}

	// Kick-off the ZooMeet plugin.
	Zmeet_Boot::init();
}
