<?php
/**
 * Notice Renderer Class
 *
 * @package PromoSDK
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


if ( ! class_exists( 'Notice_Renderer' ) ) {
	/**
	 * Notice Renderer Class
	 *
	 * Handles rendering of admin notice banner component.
	 *
	 * @since 1.0.0
	 */
	class Notice_Renderer {
		/**
		 * Configuration array
		 *
		 * @var array
		 */
		private $config = array();

		/**
		 * Countdown Handler instance
		 *
		 * @var Countdown_Handler
		 */
		private $countdown_handler = null;

		/**
		 * Constructor
		 *
		 * @param array             $config Configuration array.
		 * @param Countdown_Handler $countdown_handler Countdown handler instance.
		 */
		public function __construct( $config, $countdown_handler ) {
			$this->config            = $config;
			$this->countdown_handler = $countdown_handler;
		}

		/**
		 * Render notice banner
		 *
		 * @param array $promo_data Promotional data.
		 * @return void
		 */
		public function render( $promo_data ) {
			$product_slug = isset( $promo_data['product_slug'] ) ? $promo_data['product_slug'] : 'unknown';

			if ( ! $this->should_show_notice( $promo_data ) ) {
				return;
			}

			// Get countdown end time.
			$countdown_end_time = $this->countdown_handler->get_countdown_end_time( $promo_data );

			if ( null === $countdown_end_time ) {
				return; // Don't show if not in active period.
			}

			// Fire action before notice display.
			do_action( 'promo_sdk_before_notice', $promo_data );

			// Include template.
			$template_path = __DIR__ . '/templates/notice-template.php';

			if ( file_exists( $template_path ) ) {
				include $template_path;
			}

			// Fire action after notice display.
			do_action( 'promo_sdk_after_notice', $promo_data );
		}

		/**
		 * Check if notice should be shown
		 *
		 * @param array $promo_data Promotional data.
		 * @return bool True if should show, false otherwise.
		 */
		private function should_show_notice( $promo_data ) {
			$product_slug = isset( $promo_data['product_slug'] ) ? $promo_data['product_slug'] : 'unknown';

			if ( ! is_array( $promo_data ) ) {
				return false;
			}

			// Check if this is a pro version - don't show notices for pro users.
			if ( isset( $this->config['is_pro'] ) && $this->config['is_pro'] ) {
				return false;
			}

			// Check if notice is enabled.
			if ( ! isset( $promo_data['notice_enabled'] ) || ! $promo_data['notice_enabled'] ) {
				return false;
			}

			// Check if on plugin admin page.
			if ( ! $this->is_plugin_page() ) {
				return false;
			}

			// Check if promo is in active cycle.
			if ( ! $this->countdown_handler->is_in_active_cycle( $promo_data ) ) {
				return false;
			}

			// Check if user dismissed notice.
			if ( $this->is_dismissed( $promo_data ) ) {
				return false;
			}

			// Check if required fields exist.
			$required_fields = array( 'id', 'notice_title', 'notice_button_text', 'notice_button_url' );

			foreach ( $required_fields as $field ) {
				if ( ! isset( $promo_data[ $field ] ) || empty( $promo_data[ $field ] ) ) {
					return false;
				}
			}

			return true;
		}

		/**
		 * Check if on plugin admin page
		 *
		 * @return bool True if on plugin page, false otherwise.
		 */
		private function is_plugin_page() {
			// Only check if we're in admin area.
			// JavaScript will handle specific page filtering.
			return is_admin();
		}

		/**
		 * Check if user dismissed notice
		 *
		 * @param array $promo_data Promotional data.
		 * @return bool True if dismissed, false otherwise.
		 */
		private function is_dismissed( $promo_data ) {
			$product_slug = isset( $promo_data['product_slug'] ) ? $promo_data['product_slug'] : '';

			if ( empty( $product_slug ) ) {
				return false;
			}

			$user_id        = get_current_user_id();
			$meta_key       = "promo_sdk_dismissed_notice_{$product_slug}";
			$dismissed_time = get_user_meta( $user_id, $meta_key, true );

			if ( empty( $dismissed_time ) ) {
				return false;
			}

			$current_time = time();

			// Check if dismiss_duration is set (not 0 or null).
			$dismiss_duration = apply_filters( 'promo_sdk_dismiss_duration', $this->config['dismiss_duration'] );

			if ( empty( $dismiss_duration ) ) {
				// No dismiss_duration set, use campaign end date.
				$campaign_end_time = $this->get_campaign_end_timestamp( $promo_data );

				if ( null !== $campaign_end_time ) {
					// Check if campaign has ended.
					if ( $current_time >= $campaign_end_time ) {
						// Campaign ended, delete dismiss meta and allow showing again.
						delete_user_meta( $user_id, $meta_key );
						return false;
					}
					// Campaign still active, keep dismissed.
					return true;
				}

				// If no campaign end time, fall back to default behavior (stay dismissed).
				return true;
			}

			// Use dismiss_duration if set.
			if ( ( $current_time - $dismissed_time ) > $dismiss_duration ) {
				// Dismiss duration expired, delete meta and show again.
				delete_user_meta( $user_id, $meta_key );
				return false;
			}

			return true;
		}

		/**
		 * Get campaign end timestamp
		 *
		 * @param array $promo_data Promotional data.
		 * @return int|null Campaign end timestamp or null.
		 */
		private function get_campaign_end_timestamp( $promo_data ) {
			// Check for recurring campaign.
			if ( isset( $promo_data['is_recurring'] ) && $promo_data['is_recurring'] ) {
				// For recurring campaigns, use recurring_end_time if available.
				if ( isset( $promo_data['recurring_end_time'] ) && ! empty( $promo_data['recurring_end_time'] ) ) {
					return strtotime( $promo_data['recurring_end_time'] );
				}
			}

			// For non-recurring campaigns, use end_date.
			if ( isset( $promo_data['end_date'] ) && ! empty( $promo_data['end_date'] ) ) {
				return strtotime( $promo_data['end_date'] );
			}

			return null;
		}
	}
}
