<?php // phpcs:ignore
/**
 * Zmeet_Shortcodes Class
 *
 * This class manages the frontend functionality for the ZooMeet plugin.
 *
 * @package SOVLIX\ZMEET
 */

namespace SOVLIX\ZMEET;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Zmeet_Shortcodes' ) ) {
	/**
	 * Zmeet_Shortcodes Class
	 *
	 * Handles frontend-related tasks and functionality for the ZooMeet plugin.
	 *
	 * @since 1.0.0
	 */
	class Zmeet_Shortcodes {
		/**
		 * Zmeet_Shortcodes constructor.
		 *
		 * Initializes the frontend class.
		 */
		public function __construct() {
			add_shortcode( 'zmeet-zoom-meeting', array( $this, 'zmeet_zoom_shortcode_render' ) );
			add_shortcode( 'zmeet-zoom-webinar', array( $this, 'zmeet_zoom_webinar_shortcode_render' ) );
			add_shortcode( 'zmeet-zoom-meeting-list', array( $this, 'zmeet_zoom_meeting_list_shortcode_render' ) );
			add_shortcode( 'zmeet-zoom-webinar-list', array( $this, 'zmeet_zoom_webinar_list_shortcode_render' ) );
		}

		/**
		 * Renders the ZooMeet Zoom shortcode.
		 *
		 * This function handles the rendering of the ZooMeet Zoom shortcode, which is used to display
		 * Zoom meeting information on a WordPress site. It processes the shortcode attributes, retrieves
		 * meeting details from the post meta, and generates the necessary HTML and CSS to display the
		 * meeting information.
		 *
		 * @param array $atts The shortcode attributes. Includes:
		 *   - 'id' (int): The ID of the post containing the Zoom meeting details.
		 *
		 * @return string The rendered HTML content for the shortcode.
		 */
		public function zmeet_zoom_shortcode_render( $atts ) {
			if ( ! empty( $atts ) ) {
				// Extract shortcode attributes, including the 'id' attribute if provided.
				$atts = shortcode_atts(
					array(
						'id'            => $atts['id'],
						'hide_details'  => isset( $atts['hide_details'] ) ? $atts['hide_details'] : '',
						'hide_timer'    => isset( $atts['hide_timer'] ) ? $atts['hide_timer'] : '',
						'hide_calendar' => zoo_fs()->can_use_premium_code__premium_only() && isset( $atts['hide_calendar'] ) ? $atts['hide_calendar'] : '',
					),
					$atts,
					'zmeet-zoom-meeting'
				);

				// Check if the post exists.
				if ( ! get_post_status( $atts['id'] ) ) {
					return '<h1>' . esc_html__( 'The meeting has ended or does not exist.', 'zoomeet' ) . '</h1>';
				}

				ob_start();

				$serialize_options = get_post_meta( $atts['id'], 'zmeet__meeting_settings', true );
				$options           = maybe_unserialize( $serialize_options );

				$user_id           = get_current_user_id();
				$current_user_data = get_userdata( $user_id );
				$current_user_name = $current_user_data ? $current_user_data->display_name : '';

				$meeting_status       = get_post_meta( $atts['id'], 'zmeet_meeting_status', true );
				$meeting_start_status = get_post_meta( $atts['id'], 'zmeet_meeting_start_status', true );
				$hide_sidebar         = $options['hide_sidebar'];
				$time_zone            = $options['meeting_timezone'];

				$gmt_array       = zmeet_get_gmt_offset( $options['meeting_timezone'] );
				$gmt_offset_val  = $gmt_array['gmt_offset_val'];
				$gmt_offset      = $gmt_array['gmt_offset'];
				$should_register = false;
				$attendee_login  = false;
				$login_status;
				$meeting_as_product;
				$product_id;
				$po_meeting_id = $atts['id'];
				$hide_details  = $atts['hide_details'];
				$hide_timer    = $atts['hide_timer'];
				$hide_calendar = $atts['hide_calendar'];

				if ( zoo_fs()->can_use_premium_code__premium_only() ) {
					$cookie_key   = "zmeet_login_status_{$po_meeting_id}";
					$login_status = isset( $_COOKIE[ $cookie_key ] ) ? sanitize_text_field( wp_unslash( $_COOKIE[ $cookie_key ] ) ) : '';

					$meeting_as_product = get_post_meta( get_the_ID(), 'zmeet_connect_as_product', true );
					$product_id         = get_post_meta( get_the_ID(), 'zmeet_connect_product_id', true );
					$should_register    = isset( $options['enable_should_register'] ) ? $options['enable_should_register'] : false;
				}

				// Use the specified or default post ID to get the post meta.
				$serialize_options = get_post_meta( $atts['id'], 'zmeet__meeting_settings', true );
				$options           = maybe_unserialize( $serialize_options );

				$start_url           = get_post_meta( $atts['id'], 'zmeet_zoom_start_url', true );
				$meeting_id          = get_post_meta( $atts['id'], 'zmeet_zoom_meeting_id', true );
				$join_url            = get_post_meta( $atts['id'], 'zmeet_zoom_join_url', true );
				$meeting_duration    = get_post_meta( $atts['id'], 'zmeet_zoom_meeting_duration', true );
				$meeting_description = get_post_meta( $atts['id'], 'meeting_description', true );
				$thumbnail_html      = get_the_post_thumbnail( $atts['id'] );
				$meeting_title       = get_the_title( $atts['id'] );
				$meeting_type        = intval( $options['meeting_type'] );
				$enable_user_zone    = isset( $options['display_time_user_zone'] ) ? $options['display_time_user_zone'] : '';

				$zoom_api      = \SOVLIX\ZMEET\Zoom\Zmeet_Zoom_Api::instance();
				$zoom_response = json_decode( $zoom_api->get_meeting_info( $meeting_id ) );
				$start_time    = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $options['startDateTime'] ) + $gmt_offset_val );

				$meeting_id;
				$password = '';

				$next_meeting_time   = '';
				$meeting_deleted     = false;
				$meeting_not_created = false;
				$is_schedule         = false;

				if ( isset( $zoom_response->code ) ) {
					if ( 3001 === $zoom_response->code ) {
						$meeting_deleted = true;
					}

					if ( 2300 === $zoom_response->code ) {
						$meeting_not_created = true;
					}
				}

				if ( zoo_fs()->can_use_premium_code__premium_only() ) {
					$calculated_time   = zmeet_pro_calculate_next_meeting_time( $options, $gmt_offset, $gmt_offset_val );
					$next_meeting_time = $calculated_time['start_time'];
					$attendee_login    = 'login_successful' === $login_status ? true : false;
				} else {
					$next_meeting_time = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $options['startDateTime'] ) + $gmt_offset_val );
				}

				if ( ( strtotime( gmdate( 'Y-m-d H:i:s' ) ) + $gmt_offset_val ) - strtotime( $next_meeting_time ) <= 0 ) {
					$is_schedule = true;
				}

				if ( $options['enable_recurring_meeting'] && ! empty( $zoom_response->occurrences ) ) {
					$recurrence                = $zoom_response->recurrence;
					$recurring_repeat_interval = $recurrence->repeat_interval;

					if ( 1 === $recurrence->type ) {
						$is_schedule           = false;
						$recurring_repeat_name = 1 < $recurring_repeat_interval ? 'days' : 'day';
					} elseif ( 2 === $recurrence->type ) {
						$is_schedule = true;

						// Convert weekly_days numbers to actual day names (Sunday as day 1).
						if ( ! empty( $recurrence->weekly_days ) ) {
							$weekly_days = explode( ',', $recurrence->weekly_days );
							$day_names   = array(
								1 => 'Sunday',
								2 => 'Monday',
								3 => 'Tuesday',
								4 => 'Wednesday',
								5 => 'Thursday',
								6 => 'Friday',
								7 => 'Saturday',
							);

							// Convert the numeric days into day names.
							$recurring_repeat_name = ( 1 < $recurring_repeat_interval ) ? 'weeks' : 'week';

							$recurring_repeat_name .= ' on ' . implode(
								', ',
								array_map(
									function ( $day ) use ( $day_names ) {
										return $day_names[ (int) $day ]; // Get day name from map.
									},
									$weekly_days
								)
							);

						} else {
							$recurring_repeat_name = 'week';
						}
					} elseif ( 3 === $recurrence->type ) {
						$is_schedule           = true;
						$recurring_repeat_name = 1 < $recurring_repeat_interval ? 'months' : 'month';
					}

					// Process occurrences.
					$recurring_next_occurrence = $zoom_response->occurrences[0];
					$next_meeting_time         = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $recurring_next_occurrence->start_time ) );
					$recurring_duration        = $recurring_next_occurrence->duration;
					$recurring_end_date        = gmdate( 'd M Y', strtotime( end( $zoom_response->occurrences )->start_time ) );
				}

				if ( ! $options['enable_recurring_meeting'] ) {

					if ( ( strtotime( gmdate( 'Y-m-d H:i:s' ) ) + $gmt_offset_val ) - strtotime( $start_time ) <= 0 ) {
						$next_meeting_time = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $start_time ) - $gmt_offset_val );
						$is_schedule       = true;
					}

					if ( ( strtotime( gmdate( 'Y-m-d H:i:s' ) ) + $gmt_offset_val ) - strtotime( $start_time ) >= 0 ) {
						$next_meeting_time = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $start_time ) );
					}
				}

				if ( zoo_fs()->can_use_premium_code__premium_only() ) {
					$attendee_login = 'login_successful' === $login_status ? true : false;
				}

				?>
				<div class="zmeet-wrapper" data-zmeet-starttime="<?php echo esc_attr( $next_meeting_time ); ?>" data-zmeet-timezone="<?php echo esc_attr( $options['meeting_timezone'] ); ?>" data-zmeet-title="<?php echo esc_attr( $meeting_title ); ?>" data-zmeet-permalilk="<?php echo esc_url( get_permalink( $po_meeting_id ) ); ?>" data-zmeet-enable-user-zone="<?php echo esc_attr( $enable_user_zone ); ?>">
					<div class="zmeet-col-12" style="float: none;">
						<?php
						if ( ! empty( $thumbnail_html ) ) {
							?>
								<div class="meeting-thumbnail">
									<?php echo wp_kses_post( $thumbnail_html ); ?>
								</div>
							<?php
						}
						?>

						<?php
						if ( ! empty( $meeting_description ) ) {
							?>
								<div class="meeting-details">
									<?php echo wp_kses_post( $meeting_description ); ?>
								</div>
							<?php
						}
						?>
					</div>

					<?php
					if ( 'yes' !== $hide_timer ) {
						if ( ( $should_register && $attendee_login ) || current_user_can( 'manage_options' ) || ! $should_register ) {
							?>
						<div class="zmeet-col-12" style="float: none;">
							<div class="zmeet-sidebar">
								<div class="zmeet-single-widget zmeet-single-widget-countdown zmeet-shortcode-widget">
									<h4 class="zmeet-widget-title" style="margin:0; color:#fff;"><?php esc_html_e( 'Time to go', 'zoomeet' ); ?></h4>
									<div class="zmeet-widget-inner">
									<?php if ( ! $meeting_not_created ) { ?>
										<?php if ( ! $meeting_deleted ) { ?>
											<?php if ( 'end' !== $meeting_status ) { ?>
												<?php if ( $is_schedule ) { ?>
													<div class="zmeet_start_time" data-meeting-start-time="<?php echo esc_attr( $next_meeting_time ); ?>" data-meeting-time-offset="<?php echo esc_attr( $gmt_offset_val ); ?>" > </div>
													<?php } else { ?>
														<span class="zmeet-countdown-wrapper">
															<span class="zmeet-countdown-value"><?php esc_html_e( 'Meeting is running', 'zoomeet' ); ?></span>
															<span class="zmeet-countdown-label"><?php esc_html_e( 'The meeting is started and running', 'zoomeet' ); ?></span>
														</span>
													<?php } ?>
											<?php } else { ?>
												<span class="zmeet-countdown-wrapper">
													<span class="zmeet-countdown-value"><?php esc_html_e( 'Meeting is finished', 'zoomeet' ); ?></span>
													<span class="zmeet-countdown-label"><?php esc_html_e( 'This meeting has been ended by the host.', 'zoomeet' ); ?></span>
												</span>
											<?php } ?>

										<?php } else { ?>
											<span class="zmeet-countdown-wrapper">
												<span class="zmeet-countdown-value"><?php esc_html_e( 'Meeting does not exist', 'zoomeet' ); ?></span>
												<span class="zmeet-countdown-label"><?php esc_html_e( 'The meeting is not exists in zoom account.', 'zoomeet' ); ?></span>
											</span>
										<?php } ?>
									<?php } else { ?>
										<?php if ( current_user_can( 'manage_options' ) ) { ?>
											<span class="zmeet-countdown-wrapper">
												<span class="zmeet-countdown-value"><?php esc_html_e( 'Meeting not created', 'zoomeet' ); ?></span>
												<span class="zmeet-countdown-label"><?php esc_html_e( 'You have to setup zoom credentils first from settings.', 'zoomeet' ); ?></span>
											</span>
										<?php } else { ?>
											<span class="zmeet-countdown-wrapper">
												<span class="zmeet-countdown-value"><?php esc_html_e( 'Meeting does not exist', 'zoomeet' ); ?></span>
												<span class="zmeet-countdown-label"><?php esc_html_e( 'The meeting is not exists in zoom account.', 'zoomeet' ); ?></span>
											</span>
										<?php } ?>
									<?php } ?>
									</div>
								</div>
							</div>
						</div>
							<?php
						}
					}
					?>

					<div class="zmeet-col-12" style="float: none;">
						<?php if ( ! $meeting_not_created ) { ?>
							<?php if ( ! $meeting_deleted ) { ?>
								<?php if ( 'end' !== $meeting_status ) { ?>
									<?php if ( ( $should_register && $attendee_login ) || current_user_can( 'manage_options' ) || ! $should_register ) { ?>
										<div class="zmeet-shortcode-zoom-join-links">
											<a target="_blank" href="<?php echo esc_url( $join_url ); ?>" rel="nofollow" class="zmeet-button zmeet-join-app"><?php esc_html_e( 'JOIN IN ZOOM APP', 'zoomeet' ); ?></a>

											<a class="zmeet-button zmeet-zoom-join-web" target="_blank" href="<?php echo esc_attr( add_query_arg( array( 'display_meeting' => '1' ), get_permalink( $atts['id'] ) ) ); ?>" title="JOIN IN BROWSER"><?php esc_html_e( 'JOIN IN BROWSER', 'zoomeet' ); ?></a>

											<?php if ( current_user_can( 'manage_options' ) ) { ?>
											<a target="_blank" href="<?php echo esc_url( $start_url ); ?>" rel="nofollow" class="zmeet-button zmeet-join-app"><?php esc_html_e( 'START MEETING', 'zoomeet' ); ?></a>
											<?php } ?>
										</div>
										<?php
									} else {
										require_once ZMEET_PRO_INCLUDES . '/Templates/zmeet-register.php';
									}
									?>

								<?php } else { ?>
										<div class="meeting-not-started">
											<?php esc_html_e( 'The meeting has ended by the host.', 'zoomeet' ); ?>
										</div>
								<?php } ?>

								<?php if ( current_user_can( 'manage_options' ) ) { ?>
									<div class="zmeet-single-widget zmeet-single-widget-host-actions" style="margin-top: 20px;">
										<div class="zmeet-widget-inner">
											<?php if ( 'end' !== $meeting_status ) { ?>
												<?php if ( 'start' !== $meeting_start_status ) { ?>
													<button class="zmeet-meeting-status zmeet-button" data-meeting-status="start" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'Start Meeting ?', 'zoomeet' ); ?></button>
												<?php } else { ?>
													<button class="zmeet-meeting-status zmeet-button" data-meeting-status="stop" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'Stop Meeting ?', 'zoomeet' ); ?></button>
												<?php } ?>
												<button class="zmeet-meeting-status zmeet-button" style="margin-top: 10px; background: #ea4335;" data-meeting-status="end" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'End Meeting ?', 'zoomeet' ); ?></button>
											<?php } else { ?>
												<button class="zmeet-meeting-status zmeet-button" data-meeting-status="resume" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'Enable Meeting Join ?', 'zoomeet' ); ?></button>
											<?php } ?>
											<p style="margin-top: 10px; font-size: 12px; opacity: 0.8;"><?php esc_html_e( 'You are seeing this because you are the author of this meeting', 'zoomeet' ); ?></p>
										</div>
									</div>
									<?php } ?>
							
							<?php } else { ?>
								<div class="meeting-not-started">
									<?php esc_html_e( 'Meeting does not exist.', 'zoomeet' ); ?>
								</div>
							<?php } ?>
						<?php } else { ?>
							<?php if ( current_user_can( 'manage_options' ) ) { ?>
								<div class="meeting-not-started">
									<?php esc_html_e( 'Meeting not created yet. Please setup api in settings first.', 'zoomeet' ); ?>
								</div>
							<?php } else { ?>
								<div class="meeting-not-started">
									<?php esc_html_e( 'Meeting does not exist.', 'zoomeet' ); ?>
								</div>
							<?php } ?>
						<?php } ?>
					</div>
								
					<?php
					if ( 'yes' !== $hide_timer ) {
						if ( ( $should_register && $attendee_login ) || current_user_can( 'manage_options' ) || ! $should_register ) {
							?>
							<div class="zmeet-col-12" style="float: none;">
								<div class="zmeet-sidebar">
									<div class="zmeet-single-widget zmeet-single-widget-detail">
											<h4 class="zmeet-widget-title" style="margin:0; color:#fff;"><?php esc_html_e( 'Details', 'zoomeet' ); ?></h4>
											<div class="zmeet-widget-inner">
												<dl>
													<dt><?php esc_html_e( 'Topic:', 'zoomeet' ); ?></dt>
													<dd><?php echo esc_html( the_title() ); ?></dd>
													<dt><?php esc_html_e( 'Hosted By:', 'zoomeet' ); ?></dt>
													<dd>
													<?php
														echo esc_html( $current_user_name );
													?>
													</dd>

													<?php if ( ! $options['enable_recurring_meeting'] ) { ?>
														<dt><?php esc_html_e( 'Start Time:', 'zoomeet' ); ?></dt>
														<dd class="mhbu-tm">
														<?php
														echo esc_html( zmeet_format_meeting_datetime( strtotime( $start_time ) ) );
														?>
														</dd>

													<?php } ?>

													<?php if ( $options['enable_recurring_meeting'] && ! empty( $zoom_response->occurrences ) ) { ?>
														<dt><?php 2 === $meeting_type ? esc_html_e( 'Recurring Meeting schedule:', 'zoomeet' ) : esc_html_e( 'Recurring Webinar schedule:', 'zoomeet' ); ?></dt>
														<dd style="margin-bottom:0;"> 
															<?php esc_html_e( 'From:', 'zoomeet' ); ?> 
															<span class="muhb-st"> <?php echo esc_html( zmeet_format_meeting_datetime( strtotime( $next_meeting_time ), 'd M Y' ) ); ?></span>
														</dd>
														<dd> 
															<?php esc_html_e( 'To:', 'zoomeet' ); ?> 
															<span class="muhb-et"> <?php echo esc_html( zmeet_format_meeting_datetime( strtotime( end( $zoom_response->occurrences )->start_time ), 'd M Y' ) ); ?></span>
														</dd>

														<dt><?php 2 === $meeting_type ? esc_html_e( 'Meeting recurrence:', 'zoomeet' ) : esc_html_e( 'Webinar recurrence:', 'zoomeet' ); ?></dt>
														<dd> <?php echo esc_html( zmeet_get_recurrence_description( $options ) ); ?></dd>

														<dt><?php esc_html_e( 'Next Start Time:', 'zoomeet' ); ?></dt>
														<dd class="mhbu-tm">
															<?php
															echo esc_html( zmeet_format_meeting_datetime( strtotime( $next_meeting_time ) ) );
															?>
														</dd>


														<dt><?php esc_html_e( 'Duration:', 'zoomeet' ); ?></dt>
														<dd class="zmeet-duration">
														<?php
														echo esc_html( zmeet_format_duration_display( $recurring_duration ) );
														?>

														</dd>

													<?php } ?>

													<?php if ( ! $options['enable_recurring_meeting'] ) { ?>
														<dt><?php esc_html_e( 'Duration:', 'zoomeet' ); ?></dt>
														<dd class="zmeet-duration">
														<?php
														echo esc_html( zmeet_format_duration_display( $meeting_duration ) );
														?>
														</dd>
													<?php } ?>

													<dt><?php esc_html_e( 'Current Timezone:', 'zoomeet' ); ?></dt>
													<dd class="zmeet-tz">
													<?php
														echo esc_html( $time_zone );
													?>
													</dd>
													
												</dl>
											</div>
										</div>

										<div class="zmeet-buy-btn-wpapper">
											<?php
											if ( ! empty( $product_id ) && 'yes' === $meeting_as_product && ! current_user_can( 'manage_options' ) ) {
												$product_link = get_permalink( $product_id );
												?>
												<a href="<?php echo esc_url( $product_link ); ?>" class="zmeet-buy-button" target="__blank"> <?php esc_attr_e( 'Buy Now !', 'zoomeet' ); ?></a>
											
											<?php } ?>
										</div>
									</div>
							</div>
							<?php
						}
					}

					if ( 'yes' !== $hide_calendar ) {
						if ( zoo_fs()->can_use_premium_code__premium_only() ) {
							?>
							<div class="zmeet-col-12" style="float: none; margin-top:20px;">
								<div class="zmeet-sidebar">
									<?php
									require_once ZMEET_PRO_INCLUDES . '/Templates/zmeet-calander.php';
									?>
								</div>
							</div>
							<?php
						}
					}
					?>

				</div>
				<?php
			} else {
				?>
				<h1> <?php echo esc_html__( 'Please use correct shortcode', 'zoomeet' ); ?></h1>
				<?php
			}

			return ob_get_clean();
		}

		/**
		 * Renders the ZooMeet Zoom Webinar shortcode.
		 *
		 * This function handles the rendering of the ZooMeet Zoom shortcode, which is used to display
		 * Zoom meeting information on a WordPress site. It processes the shortcode attributes, retrieves
		 * meeting details from the post meta, and generates the necessary HTML and CSS to display the
		 * meeting information.
		 *
		 * @param array $atts The shortcode attributes. Includes:
		 *   - 'id' (int): The ID of the post containing the Zoom meeting details.
		 *
		 * @return string The rendered HTML content for the shortcode.
		 */
		public function zmeet_zoom_webinar_shortcode_render( $atts ) {
			if ( ! empty( $atts ) ) {
				// Extract shortcode attributes, including the 'id' attribute if provided.
				$atts = shortcode_atts(
					array(
						'id'            => $atts['id'],
						'hide_details'  => isset( $atts['hide_details'] ) ? $atts['hide_details'] : '',
						'hide_timer'    => isset( $atts['hide_timer'] ) ? $atts['hide_timer'] : '',
						'hide_calendar' => zoo_fs()->can_use_premium_code__premium_only() && isset( $atts['hide_calendar'] ) ? $atts['hide_calendar'] : '',
					),
					$atts,
					'zmeet-zoom-webinar'
				);

				// Check if the post exists.
				if ( ! get_post_status( $atts['id'] ) ) {
					return '<h1>' . esc_html__( 'The webinar has ended or does not exist.', 'zoomeet' ) . '</h1>';
				}

				ob_start();

				$serialize_options = get_post_meta( $atts['id'], 'zmeet__meeting_settings', true );
				$options           = maybe_unserialize( $serialize_options );

				$user_id           = get_current_user_id();
				$current_user_data = get_userdata( $user_id );
				$current_user_name = $current_user_data ? $current_user_data->display_name : '';

				$meeting_status       = get_post_meta( $atts['id'], 'zmeet_meeting_status', true );
				$meeting_start_status = get_post_meta( $atts['id'], 'zmeet_meeting_start_status', true );
				$hide_sidebar         = $options['hide_sidebar'];
				$time_zone            = $options['meeting_timezone'];
				$enable_user_zone     = isset( $options['display_time_user_zone'] ) ? $options['display_time_user_zone'] : '';

				$gmt_array       = zmeet_get_gmt_offset( $options['meeting_timezone'] );
				$gmt_offset_val  = $gmt_array['gmt_offset_val'];
				$gmt_offset      = $gmt_array['gmt_offset'];
				$should_register = false;
				$attendee_login  = false;
				$login_status;
				$meeting_as_product;
				$product_id;
				$po_meeting_id = $atts['id'];
				$hide_details  = $atts['hide_details'];
				$hide_timer    = $atts['hide_timer'];
				$hide_calendar = $atts['hide_calendar'];

				if ( zoo_fs()->can_use_premium_code__premium_only() ) {
					$cookie_key   = "zmeet_login_status_{$po_meeting_id}";
					$login_status = isset( $_COOKIE[ $cookie_key ] ) ? sanitize_text_field( wp_unslash( $_COOKIE[ $cookie_key ] ) ) : '';

					$meeting_as_product = get_post_meta( get_the_ID(), 'zmeet_connect_as_product', true );
					$product_id         = get_post_meta( get_the_ID(), 'zmeet_connect_product_id', true );
					$should_register    = isset( $options['enable_should_register'] ) ? $options['enable_should_register'] : false;
				}

				// Use the specified or default post ID to get the post meta.

				$start_url           = get_post_meta( $atts['id'], 'zmeet_zoom_start_url', true );
				$meeting_id          = get_post_meta( $atts['id'], 'zmeet_zoom_webinar_id', true );
				$join_url            = get_post_meta( $atts['id'], 'zmeet_zoom_join_url', true );
				$meeting_duration    = get_post_meta( $atts['id'], 'zmeet_zoom_meeting_duration', true );
				$meeting_description = get_post_meta( $atts['id'], 'meeting_description', true );
				$thumbnail_html      = get_the_post_thumbnail( $atts['id'] );
				$meeting_title       = get_the_title( $atts['id'] );
				$meeting_type        = intval( $options['meeting_type'] );

				$zoom_api      = \SOVLIX\ZMEET\Zoom\Zmeet_Zoom_Api::instance();
				$zoom_response = json_decode( $zoom_api->get_webinar_info( $meeting_id ) );
				$start_time    = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $options['startDateTime'] ) + $gmt_offset_val );

				$meeting_id;
				$password = '';

				$next_meeting_time   = '';
				$meeting_deleted     = false;
				$meeting_not_created = false;
				$is_schedule         = false;

				if ( isset( $zoom_response->code ) ) {
					if ( 3001 === $zoom_response->code ) {
						$meeting_deleted = true;
					}

					if ( 2300 === $zoom_response->code ) {
						$meeting_not_created = true;
					}
				}

				if ( $options['enable_recurring_meeting'] && ! empty( $zoom_response->occurrences ) ) {
					$recurrence                = $zoom_response->recurrence;
					$recurring_repeat_interval = $recurrence->repeat_interval;

					if ( 1 === $recurrence->type ) {
						$is_schedule           = false;
						$recurring_repeat_name = 1 < $recurring_repeat_interval ? 'days' : 'day';
					} elseif ( 2 === $recurrence->type ) {
						$is_schedule = true;

						// Convert weekly_days numbers to actual day names (Sunday as day 1).
						if ( ! empty( $recurrence->weekly_days ) ) {
							$weekly_days = explode( ',', $recurrence->weekly_days );
							$day_names   = array(
								1 => 'Sunday',
								2 => 'Monday',
								3 => 'Tuesday',
								4 => 'Wednesday',
								5 => 'Thursday',
								6 => 'Friday',
								7 => 'Saturday',
							);

							// Convert the numeric days into day names.
							$recurring_repeat_name = ( 1 < $recurring_repeat_interval ) ? 'weeks' : 'week';

							$recurring_repeat_name .= ' on ' . implode(
								', ',
								array_map(
									function ( $day ) use ( $day_names ) {
										return $day_names[ (int) $day ]; // Get day name from map.
									},
									$weekly_days
								)
							);

						} else {
							$recurring_repeat_name = 'week';
						}
					} elseif ( 3 === $recurrence->type ) {
						$is_schedule           = true;
						$recurring_repeat_name = 1 < $recurring_repeat_interval ? 'months' : 'month';
					}

					// Process occurrences.
					$recurring_next_occurrence = $zoom_response->occurrences[0];
					$next_meeting_time         = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $recurring_next_occurrence->start_time ) );
					$recurring_duration        = $recurring_next_occurrence->duration;
					$recurring_end_date        = gmdate( 'd M Y', strtotime( end( $zoom_response->occurrences )->start_time ) );
				}

				if ( ! $options['enable_recurring_meeting'] ) {

					if ( ( strtotime( gmdate( 'Y-m-d H:i:s' ) ) + $gmt_offset_val ) - strtotime( $start_time ) <= 0 ) {
						$next_meeting_time = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $start_time ) - $gmt_offset_val );
						$is_schedule       = true;
					}

					if ( ( strtotime( gmdate( 'Y-m-d H:i:s' ) ) + $gmt_offset_val ) - strtotime( $start_time ) >= 0 ) {
						$next_meeting_time = gmdate( 'Y-m-d\TH:i:s.v\Z', strtotime( $start_time ) );
					}
				}
				?>
				<div class="zmeet-wrapper" data-zmeet-starttime="<?php echo esc_attr( $next_meeting_time ); ?>" data-zmeet-timezone="<?php echo esc_attr( $options['meeting_timezone'] ); ?>" data-zmeet-title="<?php echo esc_attr( $meeting_title ); ?>" data-zmeet-permalilk="<?php echo esc_url( get_permalink( $po_meeting_id ) ); ?>" data-zmeet-enable-user-zone="<?php echo esc_attr( $enable_user_zone ); ?>">
					<div class="zmeet-col-12" style="float: none;">
						<?php
						if ( ! empty( $thumbnail_html ) ) {
							?>
								<div class="meeting-thumbnail">
									<?php echo wp_kses_post( $thumbnail_html ); ?>
								</div>
							<?php
						}
						?>

						<?php
						if ( ! empty( $meeting_description ) ) {
							?>
								<div class="meeting-details">
									<?php echo wp_kses_post( $meeting_description ); ?>
								</div>
							<?php
						}
						?>
					</div>

					<?php if ( 'yes' !== $hide_timer ) { ?>
					<div class="zmeet-col-12" style="float: none;">
						<div class="zmeet-sidebar">
							<div class="zmeet-single-widget zmeet-single-widget-countdown zmeet-shortcode-widget">
								<h4 class="zmeet-widget-title" style="margin:0; color:#fff;"><?php esc_html_e( 'Time to go', 'zoomeet' ); ?></h4>
								<div class="zmeet-widget-inner">
								<?php if ( ! $meeting_not_created ) { ?>
									<?php if ( ! $meeting_deleted ) { ?>
										<?php if ( 'end' !== $meeting_status ) { ?>
											<?php if ( $is_schedule ) { ?>
												<div class="zmeet_start_time" data-meeting-start-time="<?php echo esc_attr( $next_meeting_time ); ?>" data-meeting-time-offset="<?php echo esc_attr( $gmt_offset_val ); ?>" > </div>
												<?php } else { ?>
													<span class="zmeet-countdown-wrapper">
														<span class="zmeet-countdown-value"><?php esc_html_e( 'Webinar is running', 'zoomeet' ); ?></span>
														<span class="zmeet-countdown-label"><?php esc_html_e( 'The webinar is started and running', 'zoomeet' ); ?></span>
													</span>
												<?php } ?>
										<?php } else { ?>
											<span class="zmeet-countdown-wrapper">
												<span class="zmeet-countdown-value"><?php esc_html_e( 'Webinar is finished', 'zoomeet' ); ?></span>
												<span class="zmeet-countdown-label"><?php esc_html_e( 'This webinar has been ended by the host.', 'zoomeet' ); ?></span>
											</span>
										<?php } ?>

									<?php } else { ?>
										<span class="zmeet-countdown-wrapper">
											<span class="zmeet-countdown-value"><?php esc_html_e( 'Webinar does not exist', 'zoomeet' ); ?></span>
											<span class="zmeet-countdown-label"><?php esc_html_e( 'The webinar is not exists in zoom account.', 'zoomeet' ); ?></span>
										</span>
									<?php } ?>
								<?php } else { ?>
									<?php if ( current_user_can( 'manage_options' ) ) { ?>
										<span class="zmeet-countdown-wrapper">
											<span class="zmeet-countdown-value"><?php esc_html_e( 'Webinar not created', 'zoomeet' ); ?></span>
											<span class="zmeet-countdown-label"><?php esc_html_e( 'You have to setup zoom credentils first from settings.', 'zoomeet' ); ?></span>
										</span>
									<?php } else { ?>
										<span class="zmeet-countdown-wrapper">
											<span class="zmeet-countdown-value"><?php esc_html_e( 'Webinar does not exist', 'zoomeet' ); ?></span>
											<span class="zmeet-countdown-label"><?php esc_html_e( 'The webinar is not exists in zoom account.', 'zoomeet' ); ?></span>
										</span>
									<?php } ?>
								<?php } ?>
								</div>
							</div>
						</div>
					</div>
					<?php } ?>

					<div class="zmeet-col-12" style="float: none;">
						<?php if ( ! $meeting_not_created ) { ?>
							<?php if ( ! $meeting_deleted ) { ?>
								<?php if ( 'end' !== $meeting_status ) { ?>
									<?php if ( ( $should_register && $attendee_login ) || current_user_can( 'manage_options' ) || ! $should_register ) { ?>
										<div class="zmeet-shortcode-zoom-join-links">
											<a target="_blank" href="<?php echo esc_url( $join_url ); ?>" rel="nofollow" class="zmeet-button zmeet-join-app"><?php esc_html_e( 'JOIN IN ZOOM APP', 'zoomeet' ); ?></a>

											<a class="zmeet-button zmeet-zoom-join-web" target="_blank" href="<?php echo esc_attr( add_query_arg( array( 'display_meeting' => '1' ), get_permalink( $atts['id'] ) ) ); ?>" title="JOIN IN BROWSER"><?php esc_html_e( 'JOIN IN BROWSER', 'zoomeet' ); ?></a>

											<?php if ( current_user_can( 'manage_options' ) ) { ?>
											<a target="_blank" href="<?php echo esc_url( $start_url ); ?>" rel="nofollow" class="zmeet-button zmeet-join-app"><?php esc_html_e( 'START WEBINAR', 'zoomeet' ); ?></a>
											<?php } ?>
										</div>
										<?php
									} else {
										require_once ZMEET_PRO_INCLUDES . '/Templates/zmeet-register.php';
									}
									?>

								<?php } else { ?>
										<div class="meeting-not-started">
											<?php esc_html_e( 'The meeting has ended by the host.', 'zoomeet' ); ?>
										</div>
								<?php } ?>

								<?php if ( current_user_can( 'manage_options' ) ) { ?>
									<div class="zmeet-single-widget zmeet-single-widget-host-actions" style="margin-top: 20px;">
										<div class="zmeet-widget-inner">
											<?php if ( 'end' !== $meeting_status ) { ?>
												<?php if ( 'start' !== $meeting_start_status ) { ?>
													<button class="zmeet-meeting-status zmeet-button" data-meeting-status="start" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'Start Webinar ?', 'zoomeet' ); ?></button>
												<?php } else { ?>
													<button class="zmeet-meeting-status zmeet-button" data-meeting-status="stop" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'Stop Webinar ?', 'zoomeet' ); ?></button>
												<?php } ?>
												<button class="zmeet-meeting-status zmeet-button" style="margin-top: 10px; background: #ea4335;" data-meeting-status="end" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'End Webinar ?', 'zoomeet' ); ?></button>
											<?php } else { ?>
												<button class="zmeet-meeting-status zmeet-button" data-meeting-status="resume" data-post-id="<?php echo esc_attr( $atts['id'] ); ?>" data-meeting-id="<?php echo esc_attr( $atts['id'] ); ?>"><?php esc_html_e( 'Enable Webinar Join ?', 'zoomeet' ); ?></button>
											<?php } ?>
											<p style="margin-top: 10px; font-size: 12px; opacity: 0.8;"><?php esc_html_e( 'You are seeing this because you are the author of this webinar', 'zoomeet' ); ?></p>
										</div>
									</div>
									<?php } ?>
							
							<?php } else { ?>
								<div class="meeting-not-started">
									<?php esc_html_e( 'Meeting does not exist.', 'zoomeet' ); ?>
								</div>
							<?php } ?>
						<?php } else { ?>
							<?php if ( current_user_can( 'manage_options' ) ) { ?>
								<div class="meeting-not-started">
									<?php esc_html_e( 'Meeting not created yet. Please setup api in settings first.', 'zoomeet' ); ?>
								</div>
							<?php } else { ?>
								<div class="meeting-not-started">
									<?php esc_html_e( 'Meeting does not exist.', 'zoomeet' ); ?>
								</div>
							<?php } ?>
						<?php } ?>
					</div>
								
					<?php if ( 'yes' !== $hide_details ) { ?>
					<div class="zmeet-col-12" style="float: none;">
						<div class="zmeet-sidebar">
							<div class="zmeet-single-widget zmeet-single-widget-detail">
									<h4 class="zmeet-widget-title" style="margin:0; color:#fff;"><?php esc_html_e( 'Details', 'zoomeet' ); ?></h4>
									<div class="zmeet-widget-inner">
										<dl>
											<dt><?php esc_html_e( 'Topic:', 'zoomeet' ); ?></dt>
											<dd><?php echo esc_html( the_title() ); ?></dd>
											<dt><?php esc_html_e( 'Hosted By:', 'zoomeet' ); ?></dt>
											<dd>
											<?php
												echo esc_html( $current_user_name );
											?>
											</dd>

											<?php if ( ! $options['enable_recurring_meeting'] ) { ?>
												<dt><?php esc_html_e( 'Start Time:', 'zoomeet' ); ?></dt>
												<dd class="mhbu-tm">
												<?php
												echo esc_html( zmeet_format_meeting_datetime( strtotime( $start_time ) ) );
												?>
												</dd>

											<?php } ?>

											<?php if ( $options['enable_recurring_meeting'] && ! empty( $zoom_response->occurrences ) ) { ?>
												<dt><?php 2 === $meeting_type ? esc_html_e( 'Recurring Meeting schedule:', 'zoomeet' ) : esc_html_e( 'Recurring Webinar schedule:', 'zoomeet' ); ?></dt>
												<dd style="margin-bottom:0;"> 
													<?php esc_html_e( 'From:', 'zoomeet' ); ?> 
													<span class="muhb-st"> <?php echo esc_html( zmeet_format_meeting_datetime( strtotime( $next_meeting_time ), 'd M Y' ) ); ?></span>
												</dd>
												<dd> 
													<?php esc_html_e( 'To:', 'zoomeet' ); ?> 
													<span class="muhb-et"> <?php echo esc_html( zmeet_format_meeting_datetime( strtotime( end( $zoom_response->occurrences )->start_time ), 'd M Y' ) ); ?></span>
												</dd>

												<dt><?php 2 === $meeting_type ? esc_html_e( 'Meeting recurrence:', 'zoomeet' ) : esc_html_e( 'Webinar recurrence:', 'zoomeet' ); ?></dt>
												<dd> <?php echo esc_html( zmeet_get_recurrence_description( $options ) ); ?></dd>

												<dt><?php esc_html_e( 'Next Start Time:', 'zoomeet' ); ?></dt>
												<dd class="mhbu-tm">
													<?php
													echo esc_html( zmeet_format_meeting_datetime( strtotime( $next_meeting_time ) ) );
													?>
												</dd>


												<dt><?php esc_html_e( 'Duration:', 'zoomeet' ); ?></dt>
												<dd class="zmeet-duration">
												<?php
												echo esc_html( zmeet_format_duration_display( $recurring_duration ) );
												?>

												</dd>

											<?php } ?>

											<?php if ( ! $options['enable_recurring_meeting'] ) { ?>
												<dt><?php esc_html_e( 'Duration:', 'zoomeet' ); ?></dt>
												<dd class="zmeet-duration">
												<?php
												echo esc_html( zmeet_format_duration_display( $meeting_duration ) );
												?>
												</dd>
											<?php } ?>

											<dt><?php esc_html_e( 'Current Timezone:', 'zoomeet' ); ?></dt>
											<dd class="zmeet-tz">
											<?php
												echo esc_html( $time_zone );
											?>
											</dd>
											
										</dl>
									</div>
								</div>

								<div class="zmeet-buy-btn-wpapper">
									<?php
									if ( ! empty( $product_id ) && 'yes' === $meeting_as_product && ! current_user_can( 'manage_options' ) ) {
										$product_link = get_permalink( $product_id );
										?>
										<a href="<?php echo esc_url( $product_link ); ?>" class="zmeet-buy-button" target="__blank"> <?php esc_attr_e( 'Buy Now !', 'zoomeet' ); ?></a>
									
									<?php } ?>
								</div>
							</div>
					</div>
					<?php } ?>

				</div>
				<?php
			} else {
				?>
				<h1> <?php echo esc_html__( 'Please use correct shortcode', 'zoomeet' ); ?></h1>
				<?php
			}

			return ob_get_clean();
		}

		/**
		 * Renders the Zoom meeting list shortcode.
		 *
		 * This function starts output buffering, prints the container for the Zoom meeting list,
		 * and then returns the buffered content.
		 *
		 * @return string The buffered output of the Zoom meeting list container.
		 */
		public function zmeet_zoom_meeting_list_shortcode_render() {
			ob_start();

			?>
			<div id="zmeet_zoom_meeting_list">
			</div>
			
			<?php

			return ob_get_clean();
		}

		/**
		 * Renders the Zoom webinar list shortcode.
		 *
		 * This function starts output buffering, prints the container for the Zoom webinar list,
		 * and then returns the buffered content.
		 *
		 * @return string The buffered output of the Zoom webinar list container.
		 */
		public function zmeet_zoom_webinar_list_shortcode_render() {
			ob_start();

			?>
			<div id="zmeet_zoom_webinar_list">
			</div>
			
			<?php

			return ob_get_clean();
		}
	}
}
