<?php // phpcs:ignore
/**
 * Zmeet_Assets Class
 *
 * This class handles the loading of assets for the ZooMeet plugin.
 *
 * @package SOVLIX\ZMEET
 */

namespace SOVLIX\ZMEET;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Zmeet_Assets' ) ) {
	/**
	 * Zmeet_Assets Class
	 *
	 * Manages the loading of assets, such as styles and scripts, for the ZooMeet plugin.
	 *
	 * @since 1.0.0
	 */
	class Zmeet_Assets {

		/**
		 * Zmeet_Assets constructor.
		 *
		 * Initializes the Zmeet_Assets class and sets up actions for enqueueing assets.
		 */
		public function __construct() {
			add_action( 'admin_enqueue_scripts', array( $this, 'zmeet_admin_enqueue' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'zmeet_frontend_enqueue' ), 9999 );
		}

		/**
		 * ZooMeet enqueue for admin.
		 *
		 * @return void
		 *
		 * @since 1.0.0
		 */
		public function zmeet_admin_enqueue() {
			$create_meeting_url         = admin_url( 'admin.php?page=zmeet#/meeting/create' );
			$oauth_data                 = get_option( 'zmeet_zoom_global_oauth_data' );
			$server_auth_prepare        = zmeet_is_server_auth_prepare();
			$zmeet_zoom_settings        = get_option( 'zmeet_zoom_settings' );
			$current_user_id            = get_current_user_id();

			wp_enqueue_media();
			wp_enqueue_style( 'dashicons' );
			$screen = get_current_screen();

			wp_enqueue_style( 'zmeet-admin-style', ZMEET_ASSETS_PATH . '/css/admin.css', array(), ZMEET_VERSION, 'all' );
			wp_enqueue_style( 'zmeet-admin-fonts', ZMEET_ASSETS_PATH . '/css/fonts.css', array(), ZMEET_VERSION, 'all' );

			if ( $screen && ( strpos( $screen->id, 'zmeet' ) !== false ) ) {

				wp_enqueue_script( 'zmeet-admin-build', ZMEET_ASSETS_PATH . '/dist/main.build.js', array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-i18n' ), ZMEET_VERSION, true );
				wp_enqueue_script( 'zmeet-admin-js', ZMEET_ASSETS_PATH . '/js/admin.js', array( 'jquery' ), ZMEET_VERSION, true );

				// Set script translations.
				wp_set_script_translations( 'zmeet-admin-build', 'zoomeet', plugin_dir_path( ZMEET_FILE ) . 'languages/' );

				// Localize the script and pass the create meeting URL.
				wp_localize_script(
					'zmeet-admin-build',
					'zmeetMeetingsData',
					array(
						'createMeetingUrl'         => $create_meeting_url,
						'oauthData'                => ! empty( $oauth_data ) && $server_auth_prepare ? 'connected' : '',
						'zmeet_zoom_settings'      => $zmeet_zoom_settings,
						'zmeet_timezone'           => zmeet_timezone_offset(),
						'zmeet_password'           => wp_generate_password( 8, false ),
						'zoom_users'               => get_option( 'zmeet_zoom_users' ),
						'installed'                => zmeet_pro_plugin_exists(),
						'active'                   => zoo_fs()->can_use_premium_code__premium_only(),
						'license'                  => zoo_fs()->can_use_premium_code__premium_only(),
						'zoom_redirect_url'        => home_url( '/zmeet-zoom-oauth/' ),
						'assets_path'              => ZMEET_ASSETS_PATH,
						'checkout_url'             => zoo_fs()->get_upgrade_url(),
						'is_paying'                => zoo_fs()->is_paying(),
						'ajax_url'                 => admin_url( 'admin-ajax.php' ),
						'nonce'                    => wp_create_nonce( 'zmeet_admin_nonce' ),
						'siteurl'                  => get_site_url(),
						'mce_btn_name'             => __( 'Meeting', 'zoomeet' ),
						'hide_floating_create_btn' => false,
						'hide_floating_update_btn' => false,
						'is_admin'                 => is_admin(),
						'strings'                  => Zmeet_Translate::strings(),
						'settingsUrl'              => admin_url( 'admin.php?page=zmeet-settings' ),
						'noticeData'               => ( new \SOVLIX\ZMEET\API\Zmeet_Settings() )->get_admin_notices_data(),
					)
				);

				wp_enqueue_style( 'zmeet-admin-build-style', ZMEET_ASSETS_PATH . '/dist/main.build.css', array(), ZMEET_VERSION, 'all' );
			}
		}

		/**
		 * ZooMeet enqueue for frontend.
		 *
		 * @return void
		 *
		 * @since 1.0.0
		 */
		public function zmeet_frontend_enqueue() {

			$oauth_data          = get_option( 'zmeet_zoom_global_oauth_data' );
			$server_auth_prepare = zmeet_is_server_auth_prepare();
			$zmeet_zoom_settings = get_option( 'zmeet_zoom_settings' );
			$current_user_id     = get_current_user_id();
			wp_enqueue_style( 'dashicons' );

			if ( is_user_logged_in() ) {

				wp_enqueue_script( 'zmeet-frontend-main-build', ZMEET_ASSETS_PATH . '/dist/main.build.js', array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-i18n' ), ZMEET_VERSION, true );

				wp_enqueue_media();
				wp_enqueue_style( 'dashicons' );

				wp_localize_script(
					'zmeet-frontend-main-build',
					'zmeetMeetingsData',
					array(
						'oauthData'                => ! empty( $oauth_data ) && $server_auth_prepare ? 'connected' : '',
						'zmeet_zoom_settings'      => $zmeet_zoom_settings,
						'zmeet_timezone'           => zmeet_timezone_offset(),
						'zmeet_password'           => wp_generate_password( 8, false ),
						'zoom_users'               => get_option( 'zmeet_zoom_users' ),
						'installed'                => true,
						'active'                   => true,
						'siteurl'                  => get_site_url(),
						'hide_floating_create_btn' => true,
						'hide_floating_update_btn' => true,
						'mce_btn_name'             => __( 'Meeting', 'zoomeet' ),
						'is_admin'                 => is_admin(),
						'ajax_url'                 => admin_url( 'admin-ajax.php' ),
						'nonce'                    => wp_create_nonce( 'zmeet_admin_nonce' ),
						'strings'                  => Zmeet_Translate::strings(),
					)
				);
			}

			wp_enqueue_style( 'zmeet-admin-fonts', ZMEET_ASSETS_PATH . '/css/fonts.css', array(), ZMEET_VERSION, 'all' );

			wp_enqueue_script( 'zmeet-frontend-js', ZMEET_ASSETS_PATH . '/js/frontend.js', array( 'jquery', 'moment' ), ZMEET_VERSION, true );

			wp_enqueue_script( 'zmeet-frontend-build', ZMEET_ASSETS_PATH . '/dist/frontend.build.js', array( 'wp-blocks', 'wp-element', 'wp-i18n' ), ZMEET_VERSION, true );

			wp_enqueue_style( 'zmeet-frontend-build-style', ZMEET_ASSETS_PATH . '/dist/main.build.css', array(), ZMEET_VERSION, 'all' );

			// Localize script to pass data to JavaScript.
			wp_localize_script(
				'zmeet-frontend-build',
				'zmeet_frontend_data',
				array(
					'ajax_url'        => admin_url( 'admin-ajax.php' ),
					'nonce'           => wp_create_nonce( 'zmeet_frontend_nonce' ),
					'strings'         => Zmeet_Translate::strings(),
					'is_admin'        => current_user_can( 'manage_options' ),
					'page_id'         => get_the_ID(),
					'visibility_mode' => get_post_meta( get_the_ID(), '_zmeet_visibility_mode', true ) ? get_post_meta( get_the_ID(), '_zmeet_visibility_mode', true ) : 'user',
					'selected_roles'  => get_post_meta( get_the_ID(), '_zmeet_visibility_roles', true ) ? get_post_meta( get_the_ID(), '_zmeet_visibility_roles', true ) : array(),
					'selected_users'  => get_post_meta( get_the_ID(), '_zmeet_visibility_users', true ) ? get_post_meta( get_the_ID(), '_zmeet_visibility_users', true ) : array(),
				)
			);

			wp_localize_script(
				'zmeet-frontend-js',
				'zmeet_frontend_params',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'nonce'    => wp_create_nonce( 'zmeet_frontend_nonce' ),
				)
			);

			wp_enqueue_script( 'zmeet-countdown-js', ZMEET_ASSETS_PATH . '/js/countdown.js', array( 'jquery', 'moment' ), ZMEET_VERSION, true );

			wp_localize_script(
				'zmeet-countdown-js',
				'zmeettran',
				array(
					'strings' => Zmeet_Translate::strings(),
				)
			);
			wp_enqueue_style( 'zmeet-meeting-css', ZMEET_ASSETS_PATH . '/css/meeting-style.css', array(), ZMEET_VERSION, 'all' );
		}
	}
}
