import React, { useState } from "react";
const { __ } = wp.i18n;
import '../../../scss/settings/shortcode-tab.scss';
import { langString } from "../../../Helpers";
import { Tooltip } from 'react-tooltip';
import 'react-tooltip/dist/react-tooltip.css';
import { useZmeetAdmin } from "../../../App/ZmeetAdminContext";

const shortcodes = [
  {
    title: langString('zoom_meeting_list'),
    description: langString('embed_zoom_meetings'),
    code: "[zmeet-zoom-meeting-list]",
    tooltip: "This shortcode displays all your upcoming Zoom meetings in a list format. Simply copy and paste it into any page or post."
  },
  {
    title: langString('zoom_webinar_list'),
    description: langString('embed_zoom_webinars'),
    code: "[zmeet-zoom-webinar-list]",
    tooltip: "This shortcode displays all your upcoming Zoom webinars in a list format. Simply copy and paste it into any page or post."
  },
  {
    title: langString('zoom_single_recording'),
    description: langString('embed_single_recording'),
    code: '[zmeet-zoom-single-recording type="meeting" post_id="123" limit="6" show_upload="true" show_passcode="true"]',
    tooltip: "Replace '123' with your actual meeting/webinar post ID. You can find the post ID from ZooMeet Meetings Dashboard.",
    parameterInfo: {
      title: "Available Parameters:",
      items: [
        { param: 'type', description: 'Type of recording: "meeting" or "webinar" (required)' },
        { param: 'post_id', description: 'The post ID of the meeting/webinar (required)' },
        { param: 'limit', description: 'Maximum number of recordings to display' },
        { param: 'show_upload', description: 'Enable thumbnail upload buttons for administrators (use "true" to enable)' },
        { param: 'show_passcode', description: 'Display recording passcode in the modal dialog (use "true" to enable)' }
      ]
    }
  },
  {
    title: langString('zoom_recording_archive'),
    description: langString('embed_zoom_recordings'),
    code: '[zmeet-zoom-recording-archive host_email="host@example.com" from_date="2024-01-01" to_date="2024-12-31" limit="10" show_upload="true" show_passcode="true"]',
    tooltip: "Replace the parameter values with your actual data. All parameters are optional - you can use any combination or none at all.",
    parameterInfo: {
      title: "Available Parameters:",
      items: [
        { param: 'host_email', description: 'Filter recordings by specific host email address' },
        { param: 'from_date', description: 'Show recordings from this date onwards (format: YYYY-MM-DD)' },
        { param: 'to_date', description: 'Show recordings up to this date (format: YYYY-MM-DD)' },
        { param: 'limit', description: 'Maximum number of recordings to display' },
        { param: 'show_upload', description: 'Enable thumbnail upload buttons for administrators (use "true" to enable)' },
        { param: 'show_passcode', description: 'Display recording passcode in the modal dialog (use "true" to enable)' }
      ]
    }
  },
];

const ShortcodesTab = () => {
  const [copiedIndex, setCopiedIndex] = useState(null);
  const isPro = window.zmeetMeetingsData?.active;

  const { openProModal } = useZmeetAdmin();

  const handleCopyShortcode = (shortcode, index) => {
    const tempTextArea = document.createElement('textarea');
    tempTextArea.value = shortcode;
    document.body.appendChild(tempTextArea);

    tempTextArea.select();
    document.execCommand('copy');

    document.body.removeChild(tempTextArea);

    setCopiedIndex(index);

    // Reset copy status after a short delay
    setTimeout(() => setCopiedIndex(null), 1500);
  };

  return (
    <div className="shortcodes-tab">
      <h2>{langString('shortcodes')}</h2>
      <p>{langString('shortcodes_instruction')}</p>
      <div className="shortcodes-list">
        {shortcodes.map((shortcode, index) => {
          const isSimple = !shortcode.tooltip && !shortcode.parameterInfo;
          const isRestricted = !isPro && (shortcode.code.includes('zmeet-zoom-single-recording') || shortcode.code.includes('zmeet-zoom-recording-archive'));
          const displayCode = shortcode.code;

          return (
            <div key={index} className={`shortcode-item ${isSimple ? 'shortcode-item-simple' : ''}`}>
              {isSimple ? (
                // Simple shortcode layout (vertical pattern like complex ones)
                <div className="shortcode-content">
                  <div className="shortcode-header">
                    <h3>{shortcode.title}</h3>
                  </div>
                  <div className="shortcode-description-simple">
                    {shortcode.description}
                    {shortcode.tooltip && (
                      <>
                        <a className="zmeet-info" data-tooltip-id={`shortcode-${index}`} data-tooltip-content={shortcode.tooltip}>
                          <i className="dashicons dashicons-info"></i>
                        </a>
                        <Tooltip id={`shortcode-${index}`} place="right" type="info" effect="float" style={{ fontSize: '14px', maxWidth: '400px' }} />
                      </>
                    )}
                  </div>

                  <div className="shortcode-code-wrapper">
                    <code className="main-shortcode">{shortcode.code}</code>
                    <button
                      onClick={() => handleCopyShortcode(shortcode.code, index)}
                      className="copy-shortcode-btn"
                    >
                      {copiedIndex === index ? (<span> {langString('copied_success')}</span>) : (<><i className='dashicons dashicons-admin-page'></i> <span> {langString('copy_shortcode')}</span> </>)}
                    </button>
                  </div>
                </div>
              ) : (
                // Complex shortcode layout (with parameters/tooltip)
                <div className="shortcode-content">
                  <div className="shortcode-header">
                    <h3>{shortcode.title}</h3>
                  </div>
                  <div className="shortcode-description">
                    {shortcode.description}
                    {shortcode.tooltip && (
                      <>
                        <a className="zmeet-info" data-tooltip-id={`shortcode-${index}`} data-tooltip-content={shortcode.tooltip}>
                          <i className="dashicons dashicons-info"></i>
                        </a>
                        <Tooltip id={`shortcode-${index}`} place="right" type="info" effect="float" style={{ fontSize: '14px', maxWidth: '400px' }} />
                      </>
                    )}
                  </div>

                  <div className={`shortcode-code-wrapper shortcode-code-wrapper-zoom ${isRestricted ? 'zmeet-pro-restricted' : ''}`}>
                    <code className="main-shortcode">{displayCode}</code>
                    {isRestricted && (
                      <div className="zmeet-pro-overlay">
                        <div className="zmeet-upgrade-btn" onClick={openProModal}>
                          <i className="dashicons dashicons-lock"></i> {langString('upgrade_to_pro')}
                        </div>
                      </div>
                    )}
                    <button
                      onClick={() => !isRestricted && handleCopyShortcode(shortcode.code, index)}
                      className="copy-shortcode-btn"
                      title={isRestricted ? langString('available_in_pro') : ''}
                    >
                      {copiedIndex === index ? (<span> {langString('copied_success')}</span>) : (<><i className='dashicons dashicons-admin-page'></i> <span> {langString('copy_shortcode')}</span> </>)}
                    </button>
                  </div>

                  {shortcode.parameterInfo && (
                    <div className="shortcode-parameters">
                      <div className="parameters-header">
                        <strong>{shortcode.parameterInfo.title}</strong>
                      </div>
                      <div className="parameters-list">
                        {shortcode.parameterInfo.items.map((item, paramIndex) => (
                          <div key={paramIndex} className="parameter-item">
                            <code className="parameter-name">{item.param}</code>
                            <span className="parameter-description">{item.description}</span>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </div>
              )}
            </div>
          );
        })}
      </div>
    </div>
  );
};

export default ShortcodesTab;
