import React, { useState } from 'react';
import { __ } from '@wordpress/i18n';

/**
 * ZmeetNotice Component
 * 
 * A premium administrative notice component with modern styling and persistence support.
 */
const ZmeetNotice = ({ notices }) => {
    const [activeNotices, setActiveNotices] = useState(notices || []);
    const [dismissingId, setDismissingId] = useState(null);
    const [showOptionsFor, setShowOptionsFor] = useState(null);

    if (activeNotices.length === 0) return null;

    const getNoticeContent = (notice) => {
        // If the notice from PHP already has content, use it!
        if (notice.title && notice.message) {
            return {
                title: notice.title,
                message: notice.message,
                linkText: notice.hasOwnProperty('linkText') ? notice.linkText : __('View Settings', 'zoomeet'),
                link: notice.link || zmeetMeetingsData.settingsUrl || '#/',
            };
        }

        const id = notice.id;
        const state = notice.state;

        const contents = {
            zoom_oauth: {
                unprepared: {
                    title: __('Zoom Configuration Required', 'zoomeet'),
                    message: __('Please configure Server to Server OAuth Credentials to enable creation of Zoom meetings.', 'zoomeet'),
                    linkText: __('Configure OAuth', 'zoomeet'),
                    link: zmeetMeetingsData.settingsUrl || '#/',
                },
                unconnected: {
                    title: __('Zoom Account Connection', 'zoomeet'),
                    message: __('Zoom OAuth credentials are configured. Please ensure your token is active.', 'zoomeet'),
                    linkText: __('Check Connection', 'zoomeet'),
                    link: zmeetMeetingsData.settingsUrl || '#/',
                }
            },
            zoom_sdk: {
                unprepared: {
                    title: __('Zoom SDK Configuration', 'zoomeet'),
                    message: __('Configure Zoom Meeting SDK to enable "Join In Browser" functionality.', 'zoomeet'),
                    linkText: __('Configure SDK', 'zoomeet'),
                    link: zmeetMeetingsData.settingsUrl || '#/',
                }
            }
        };

        return contents[id]?.[state] || {
            title: __('Notice', 'zoomeet'),
            message: __('Action required.', 'zoomeet'),
            linkText: '',
            link: '',
        };
    };

    const handleDismiss = (id, duration) => {
        setDismissingId(id);
        setShowOptionsFor(null);

        const data = new FormData();
        data.append('action', 'zmeet_dismiss_notice');
        data.append('nonce', zmeetMeetingsData.nonce);
        data.append('notice_id', id);
        data.append('duration', duration);

        fetch(zmeetMeetingsData.ajax_url, {
            method: 'POST',
            body: data,
        })
            .then(response => response.json())
            .then(res => {
                if (res.success) {
                    setActiveNotices(prev => prev.filter(n => n.id !== id));
                }
                setDismissingId(null);
            })
            .catch(() => {
                setDismissingId(null);
            });
    };

    const toggleOptions = (id) => {
        setShowOptionsFor(prev => prev === id ? null : id);
    };

    return (
        <div className="zmeet-notices-container" style={{ marginBottom: '20px' }}>
            {activeNotices.map((notice, index) => {
                const content = getNoticeContent(notice);
                const isWarning = notice.type === 'warning';
                const optionsVisible = showOptionsFor === notice.id;

                return (
                    <div
                        key={notice.id}
                        className={`zmeet-modern-notice ${notice.type}`}
                        style={{
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'space-between',
                            background: isWarning ? '#FFF9EB' : '#F0F7FF',
                            borderLeft: `4px solid ${isWarning ? '#FFA800' : '#007CBA'}`,
                            padding: '16px 24px',
                            borderRadius: '8px',
                            boxShadow: '0 2px 4px rgba(0,0,0,0.05)',
                            marginBottom: '12px',
                            marginTop: index === 0 ? '10px' : '0',
                            animation: 'slideDown 0.3s ease-out',
                            position: 'relative'
                        }}
                    >
                        <div style={{ display: 'flex', alignItems: 'center', gap: '16px' }}>
                            <div className="notice-icon" style={{
                                background: isWarning ? '#FFA800' : '#007CBA',
                                color: '#FFF',
                                width: '32px',
                                height: '32px',
                                borderRadius: '50%',
                                display: 'flex',
                                alignItems: 'center',
                                justifyContent: 'center'
                            }}>
                                <span className={`dashicons dashicons-${isWarning ? 'warning' : 'info'}`} style={{ fontSize: '18px', width: '18px', height: '18px' }}></span>
                            </div>
                            <div>
                                <h4 style={{ margin: '0 0 4px 0', fontSize: '16px', fontWeight: '600', color: '#1d2327' }}>{content.title}</h4>
                                <p style={{ margin: 0, color: '#50575e', fontSize: '14px' }}>
                                    {content.message}
                                </p>
                            </div>
                        </div>

                        <div className="dismiss-logic" style={{ position: 'relative' }}>
                            <button
                                className="zmeet-notice-close"
                                onClick={() => toggleOptions(notice.id)}
                                style={{
                                    background: 'transparent',
                                    border: '1px solid #c3c4c7',
                                    width: '32px',
                                    height: '32px',
                                    borderRadius: '50%',
                                    cursor: 'pointer',
                                    color: '#8c8f94',
                                    display: 'flex',
                                    alignItems: 'center',
                                    justifyContent: 'center',
                                    transition: 'all 0.2s ease',
                                    position: 'relative'
                                }}
                                onMouseOver={(e) => {
                                    e.currentTarget.style.borderColor = '#8c8f94';
                                    e.currentTarget.style.color = '#1d2327';
                                    e.currentTarget.style.background = '#fff';
                                }}
                                onMouseOut={(e) => {
                                    e.currentTarget.style.borderColor = '#c3c4c7';
                                    e.currentTarget.style.color = '#8c8f94';
                                    e.currentTarget.style.background = 'transparent';
                                }}
                            >
                                <span className={`dashicons dashicons-${optionsVisible ? 'arrow-up-alt2' : 'no-alt'}`} style={{ fontSize: optionsVisible ? '16px' : '20px', width: optionsVisible ? '16px' : '20px', height: optionsVisible ? '16px' : '20px' }}></span>

                                {optionsVisible && (
                                    <div className="dropdown-menu" style={{
                                        display: 'block',
                                        position: 'absolute',
                                        right: 0,
                                        top: 'calc(100% + 10px)',
                                        background: '#fff',
                                        border: '1px solid #c3c4c7',
                                        borderRadius: '8px',
                                        zIndex: 100,
                                        minWidth: '180px',
                                        boxShadow: '0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 8px 10px -6px rgba(0, 0, 0, 0.05)',
                                        overflow: 'hidden',
                                        textAlign: 'left'
                                    }}
                                        onClick={(e) => e.stopPropagation()}
                                    >
                                        <div style={{ padding: '12px 16px', fontSize: '11px', textTransform: 'uppercase', color: '#8c8f94', fontWeight: '700', borderBottom: '1px solid #f0f0f1', background: '#f6f7f7', letterSpacing: '0.5px' }}>
                                            {__('Dismiss for...', 'zoomeet')}
                                        </div>
                                        {[
                                            { label: __('7 Days', 'zoomeet'), value: '7' },
                                            { label: __('14 Days', 'zoomeet'), value: '14' },
                                            { label: __('1 Month', 'zoomeet'), value: '30' },
                                            { label: __('Don\'t show again', 'zoomeet'), value: 'forever' }
                                        ].map(opt => (
                                            <div
                                                key={opt.value}
                                                onClick={() => handleDismiss(notice.id, opt.value)}
                                                style={{
                                                    padding: '12px 16px',
                                                    fontSize: '14px',
                                                    cursor: 'pointer',
                                                    transition: 'all 0.2s',
                                                    color: '#1d2327',
                                                    display: 'flex',
                                                    justifyContent: 'space-between',
                                                    alignItems: 'center'
                                                }}
                                                onMouseOver={(e) => {
                                                    e.currentTarget.style.background = '#f0f7ff';
                                                    e.currentTarget.style.color = '#007CBA';
                                                }}
                                                onMouseOut={(e) => {
                                                    e.currentTarget.style.background = 'transparent';
                                                    e.currentTarget.style.color = '#1d2327';
                                                }}
                                            >
                                                {opt.label}
                                                <span className="dashicons dashicons-arrow-right-alt2" style={{ fontSize: '14px', opacity: 0.3 }}></span>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </button>
                        </div>
                    </div>
                );
            })}
        </div>
    );
};

export default ZmeetNotice;
