
const { InspectorControls } = wp.blockEditor;
const { registerBlockType } = wp.blocks;
const { __ } = wp.i18n;

const { Fragment, useState, useEffect } = wp.element;
const { PanelBody, TextControl, RangeControl, CheckboxControl, SelectControl } = wp.components;

import './blocks.scss';
import axios from 'axios';
const config = Object.assign({}, window.zmeetGutenbergData);

function getZoomUsers() {
    const zoomUsersData = config.zoom_users;

    if (!zoomUsersData || !Array.isArray(zoomUsersData)) {
        return [{ label: __('--Select a host--', 'zoomeet'), value: '' }];
    }

    const usersOptions = zoomUsersData.map(user => ({
        label: __(user.email),
        value: user.id
    }));

    // Add the default option "Select host" at the beginning of the options array
    return [{ label: __('--Select a host--', 'zoomeet'), value: '' }, ...usersOptions];
}


const BlockIcon = () => (
    <svg width="128" height="128" viewBox="0 0 128 128" fill="none" xmlns="http://www.w3.org/2000/svg">
        <g clip-path="url(#clip0_110_15)">
            <path d="M18.9763 -0.0126719C19.1363 -0.0126719 19.2963 -0.0126719 19.4563 -0.0126719C19.8953 -0.0136719 20.3343 -0.0136719 20.7733 -0.0126719C21.2513 -0.0126719 21.7293 -0.0126719 22.2073 -0.0136719C23.0443 -0.0136719 23.8823 -0.0136719 24.7193 -0.0136719C25.9653 -0.0126719 27.2103 -0.0126719 28.4563 -0.0136719C30.6803 -0.0136719 32.9043 -0.0136719 35.1283 -0.0136719C37.0743 -0.0136719 39.0213 -0.0136719 40.9673 -0.0136719C41.0963 -0.0136719 41.2253 -0.0136719 41.3583 -0.0136719C41.8813 -0.0136719 42.4053 -0.0136719 42.9293 -0.0136719C44.3933 -0.0136719 45.8573 -0.0136719 47.3213 -0.0136719C47.4213 -0.0136719 47.4213 -0.0136719 47.5233 -0.0136719C49.6473 -0.0136719 51.7713 -0.0136719 53.8963 -0.0136719C53.9993 -0.0136719 53.9993 -0.0136719 54.1043 -0.0136719C55.2233 -0.0136719 56.3423 -0.0136719 57.4613 -0.0126719C61.8993 -0.0126719 66.3373 -0.0126719 70.7753 -0.0126719C71.8933 -0.0136719 73.0123 -0.0136719 74.1303 -0.0136719C74.1993 -0.0136719 74.2683 -0.0136719 74.3393 -0.0136719C76.4613 -0.0136719 78.5843 -0.0136719 80.7063 -0.0136719C80.8063 -0.0136719 80.8063 -0.0136719 80.9083 -0.0136719C82.4353 -0.0136719 83.9633 -0.0136719 85.4913 -0.0136719C86.0143 -0.0136719 86.5373 -0.0136719 87.0593 -0.0136719C87.2523 -0.0136719 87.2523 -0.0136719 87.4493 -0.0136719C89.3953 -0.0136719 91.3403 -0.0136719 93.2853 -0.0136719C95.4973 -0.0136719 97.7083 -0.0136719 99.9193 -0.0136719C101.158 -0.0126719 102.396 -0.0126719 103.634 -0.0136719C104.461 -0.0136719 105.287 -0.0136719 106.113 -0.0126719C106.585 -0.0126719 107.056 -0.0126719 107.527 -0.0126719C107.956 -0.0136719 108.384 -0.0136719 108.813 -0.0126719C108.967 -0.0126719 109.121 -0.0126719 109.276 -0.0126719C111.053 -0.0156719 111.053 -0.0156719 111.7 0.200328C112.064 0.272328 112.429 0.340328 112.794 0.408328C114.775 0.822328 116.807 1.81233 118.508 2.88733C118.71 3.01533 118.71 3.01533 119 3.10033C119 3.16633 119 3.23233 119 3.30033C119.053 3.32433 119.107 3.34933 119.162 3.37533C119.661 3.63733 120.085 3.96533 120.506 4.33633C120.689 4.49033 120.876 4.63033 121.071 4.76933C121.527 5.10933 121.925 5.49533 122.325 5.90033C122.394 5.96833 122.463 6.03733 122.535 6.10733C122.909 6.48333 123.22 6.84933 123.5 7.30033C123.617 7.44033 123.735 7.58033 123.853 7.72033C124.447 8.44633 124.937 9.21233 125.385 10.0343C125.552 10.3393 125.72 10.6303 125.914 10.9203C126.075 11.1623 126.186 11.3733 126.287 11.6433C126.401 11.9443 126.518 12.2423 126.644 12.5383C127.013 13.4113 127.26 14.2733 127.456 15.2013C127.501 15.4073 127.554 15.6103 127.607 15.8143C127.707 16.2303 127.741 16.6233 127.765 17.0483C127.783 17.2963 127.783 17.2963 127.9 17.4753C128.033 17.7743 128.025 18.0163 128.025 18.3433C128.025 18.4403 128.025 18.4403 128.026 18.5393C128.026 18.7573 128.026 18.9753 128.025 19.1943C128.025 19.3523 128.025 19.5103 128.026 19.6693C128.026 20.1063 128.026 20.5443 128.025 20.9813C128.025 21.4563 128.025 21.9313 128.026 22.4063C128.027 23.2403 128.027 24.0733 128.026 24.9073C128.025 26.1463 128.026 27.3863 128.026 28.6253C128.027 30.8383 128.027 33.0513 128.027 35.2643C128.026 37.2013 128.026 39.1373 128.026 41.0743C128.026 41.2663 128.026 41.2663 128.026 41.4633C128.027 41.9833 128.027 42.5043 128.027 43.0253C128.027 44.4813 128.027 45.9363 128.027 47.3923C128.027 47.4583 128.027 47.5253 128.027 47.5933C128.027 49.7063 128.027 51.8193 128.026 53.9323C128.026 54.0353 128.026 54.0353 128.026 54.1403C128.026 55.2533 128.026 56.3673 128.026 57.4803C128.025 61.8963 128.025 66.3123 128.026 70.7283C128.026 71.8413 128.026 72.9543 128.026 74.0673C128.026 74.1703 128.026 74.1703 128.026 74.2743C128.027 76.3853 128.027 78.4963 128.027 80.6073C128.027 80.7063 128.027 80.7063 128.027 80.8083C128.027 82.3273 128.027 83.8463 128.027 85.3663C128.027 85.8863 128.026 86.4053 128.026 86.9253C128.026 87.0533 128.026 87.1813 128.026 87.3133C128.026 89.2493 128.026 91.1853 128.027 93.1203C128.027 95.3213 128.027 97.5213 128.026 99.7213C128.026 100.954 128.026 102.186 128.026 103.419C128.027 104.241 128.027 105.063 128.026 105.885C128.025 106.354 128.025 106.824 128.026 107.293C128.026 107.719 128.026 108.145 128.025 108.572C128.025 108.725 128.025 108.879 128.026 109.033C128.026 109.241 128.026 109.449 128.025 109.657C128.025 109.83 128.025 109.83 128.025 110.006C128 110.298 127.939 110.449 127.8 110.7C127.764 110.963 127.735 111.223 127.712 111.487C127.63 112.295 127.446 113.025 127.2 113.8C127.167 113.922 127.134 114.045 127.1 114.171C127 114.5 127 114.5 126.8 114.8C126.754 114.946 126.712 115.094 126.674 115.243C126.492 115.87 126.209 116.432 125.912 117.012C125.86 117.115 125.809 117.219 125.756 117.325C124.775 119.264 123.415 120.98 121.893 122.525C121.819 122.601 121.819 122.601 121.743 122.678C121.421 123.002 121.087 123.253 120.7 123.5C120.559 123.617 120.419 123.735 120.279 123.853C119.553 124.447 118.788 124.937 117.966 125.385C117.66 125.552 117.369 125.72 117.079 125.914C116.837 126.075 116.626 126.186 116.356 126.287C116.055 126.401 115.757 126.518 115.461 126.644C114.588 127.013 113.726 127.26 112.799 127.455C112.593 127.501 112.389 127.554 112.185 127.607C111.769 127.707 111.377 127.741 110.951 127.765C110.703 127.783 110.703 127.783 110.524 127.9C110.225 128.033 109.984 128.025 109.657 128.025C109.56 128.025 109.56 128.025 109.46 128.026C109.242 128.026 109.024 128.025 108.805 128.025C108.647 128.025 108.489 128.025 108.331 128.026C107.893 128.026 107.455 128.026 107.018 128.025C106.543 128.025 106.068 128.025 105.593 128.026C104.759 128.027 103.926 128.027 103.092 128.026C101.853 128.025 100.614 128.026 99.3743 128.026C97.1613 128.027 94.9483 128.027 92.7353 128.027C90.7993 128.026 88.8623 128.026 86.9253 128.026C86.7973 128.026 86.6693 128.026 86.5363 128.026C86.0163 128.026 85.4953 128.027 84.9743 128.027C83.5193 128.027 82.0633 128.027 80.6073 128.027C80.5413 128.027 80.4753 128.027 80.4063 128.027C78.2933 128.027 76.1803 128.027 74.0673 128.026C73.9643 128.026 73.9643 128.026 73.8593 128.026C72.7463 128.026 71.6323 128.026 70.5193 128.026C66.1033 128.025 61.6873 128.025 57.2713 128.026C56.1583 128.026 55.0453 128.026 53.9323 128.026C53.8303 128.026 53.8303 128.026 53.7253 128.026C51.6143 128.027 49.5033 128.027 47.3923 128.027C47.3263 128.027 47.2603 128.027 47.1913 128.027C45.6723 128.027 44.1533 128.027 42.6333 128.027C42.1143 128.026 41.5943 128.026 41.0743 128.026C40.9463 128.026 40.8183 128.026 40.6863 128.026C38.7503 128.026 36.8153 128.026 34.8793 128.027C32.6793 128.027 30.4783 128.027 28.2783 128.026C27.0453 128.026 25.8133 128.026 24.5803 128.026C23.7583 128.027 22.9363 128.026 22.1143 128.026C21.6453 128.025 21.1763 128.025 20.7063 128.026C20.2803 128.026 19.8543 128.026 19.4283 128.025C19.2743 128.025 19.1203 128.025 18.9663 128.026C18.7583 128.026 18.5503 128.026 18.3433 128.025C18.2273 128.025 18.1123 128.025 17.9933 128.025C17.7013 128 17.5513 127.939 17.3003 127.8C17.0363 127.764 16.7773 127.735 16.5123 127.712C15.7043 127.63 14.9753 127.446 14.2003 127.2C14.0773 127.167 13.9553 127.134 13.8293 127.1C13.5003 127 13.5003 127 13.2003 126.8C13.0533 126.754 12.9053 126.712 12.7573 126.674C12.1293 126.492 11.5673 126.209 10.9873 125.912C10.8843 125.86 10.7813 125.809 10.6743 125.755C9.30032 125.06 8.08732 124.179 6.90032 123.2C6.83532 123.147 6.77132 123.094 6.70532 123.039C5.78532 122.258 4.88232 121.399 4.20032 120.4C4.05932 120.21 3.91832 120.022 3.77132 119.838C2.45632 118.165 1.39632 116.032 0.764322 114.005C0.721322 113.867 0.673322 113.73 0.622322 113.594C0.403322 113.001 0.319322 112.411 0.243322 111.784C0.230322 111.51 0.230322 111.51 0.100322 111.4C-0.0416776 110.515 -0.0146781 109.61 -0.0126781 108.718C-0.0126781 108.558 -0.0136776 108.399 -0.0136776 108.24C-0.0136776 107.803 -0.0136776 107.366 -0.0136776 106.929C-0.0136776 106.453 -0.0136776 105.977 -0.0136776 105.501C-0.0146776 104.668 -0.0146785 103.834 -0.0146785 103C-0.0146785 101.761 -0.014678 100.521 -0.015678 99.2823C-0.016678 96.9393 -0.0166774 94.5963 -0.0166774 92.2533C-0.0166774 90.4443 -0.0166784 88.6353 -0.0176784 86.8263C-0.0176784 86.3053 -0.0176784 85.7853 -0.0176784 85.2643C-0.0176784 85.1343 -0.0176784 85.0053 -0.0176784 84.8723C-0.0186784 81.2313 -0.0186779 77.5903 -0.0186779 73.9493C-0.0186779 73.8103 -0.0186779 73.6723 -0.0186779 73.5333C-0.0186779 72.9073 -0.0186779 72.2813 -0.0186779 71.6553C-0.0186779 71.3763 -0.0186779 71.0973 -0.0186779 70.8173C-0.0186779 70.6783 -0.0186779 70.5383 -0.0186779 70.3983C-0.0186779 65.9803 -0.0196783 61.5623 -0.0206783 57.1433C-0.0206783 56.0303 -0.0216777 54.9173 -0.0216777 53.8043C-0.0216777 53.7353 -0.0216777 53.6673 -0.0216777 53.5963C-0.0226777 51.4833 -0.0226787 49.3703 -0.0226787 47.2573C-0.0226787 47.1573 -0.0226787 47.1573 -0.0226787 47.0553C-0.0226787 45.5333 -0.0226787 44.0113 -0.0226787 42.4893C-0.0226787 41.9683 -0.0226787 41.4483 -0.0226787 40.9273C-0.0226787 40.7993 -0.0226787 40.6703 -0.0226787 40.5383C-0.0226787 38.4743 -0.0236776 36.4103 -0.0246776 34.3463C-0.0256776 32.2723 -0.0256776 30.1973 -0.0246776 28.1233C-0.0246776 26.8913 -0.0246786 25.6593 -0.0256786 24.4263C-0.0256786 23.6033 -0.0256786 22.7803 -0.0256786 21.9563C-0.0246786 21.4873 -0.0246786 21.0183 -0.0256786 20.5493C-0.0266786 20.1223 -0.0256786 19.6953 -0.0256786 19.2673C-0.0246786 19.1143 -0.0246786 18.9613 -0.0256786 18.8083C-0.0296786 17.7003 0.00632237 16.5753 0.300322 15.5003C0.328322 15.3873 0.357322 15.2733 0.386322 15.1573C0.420322 15.0273 0.454322 14.8983 0.487322 14.7683C0.520322 14.6413 0.553322 14.5133 0.586322 14.3823C0.665322 14.1153 0.745322 13.8823 0.856322 13.6303C0.971322 13.3643 1.05132 13.1293 1.11832 12.8503C1.31232 12.1793 1.66532 11.5903 2.00032 10.9833C2.14532 10.7153 2.27932 10.4423 2.41232 10.1683C2.83332 9.32133 3.31432 8.54233 3.90032 7.80033C3.96332 7.71933 4.02632 7.63933 4.09132 7.55633C4.69532 6.79333 5.32332 6.06833 6.00632 5.37533C6.05532 5.32433 6.10532 5.27333 6.15632 5.22133C6.47932 4.89733 6.81232 4.64633 7.20032 4.40033C7.34732 4.27833 7.49332 4.15733 7.63932 4.03433C8.23832 3.53433 8.87132 3.12533 9.55032 2.74333C9.71132 2.65233 9.87332 2.56033 10.0343 2.46833C10.1513 2.40133 10.1513 2.40133 10.2713 2.33433C10.4923 2.20433 10.7073 2.06833 10.9223 1.92833C11.2383 1.72833 11.5393 1.60733 11.9003 1.50033C12.1363 1.40233 12.3713 1.30233 12.6063 1.20033C12.8373 1.09933 13.0683 0.999328 13.3003 0.900328C13.3793 0.865328 13.4593 0.829328 13.5413 0.793328C13.8963 0.665328 14.2573 0.583328 14.6253 0.500328C14.7393 0.473328 14.7393 0.473328 14.8553 0.447328C15.0043 0.413328 15.1533 0.379328 15.3033 0.345328C15.4613 0.309328 15.6183 0.271328 15.7763 0.233328C16.8323 -0.0186719 17.8963 -0.0146719 18.9763 -0.0126719Z" fill="url(#paint0_linear_110_15)" />
            <path d="M31.3327 44.688C31.4447 44.688 31.5567 44.688 31.6726 44.688C32.0476 44.688 32.4216 44.688 32.7966 44.689C33.0657 44.689 33.3347 44.689 33.6037 44.689C34.1827 44.689 34.7617 44.689 35.3407 44.689C36.1777 44.69 37.0137 44.691 37.8507 44.691C39.2087 44.691 40.5666 44.692 41.9246 44.692C43.2426 44.693 44.5616 44.694 45.8806 44.695C45.9617 44.695 46.0427 44.695 46.1267 44.695C46.5347 44.695 46.9426 44.695 47.3507 44.695C50.7337 44.696 54.1166 44.698 57.4996 44.7C57.6166 44.933 57.6146 45.074 57.6186 45.333C57.6206 45.423 57.6226 45.513 57.6236 45.606C57.6256 45.753 57.6256 45.753 57.6286 45.903C57.6296 46.005 57.6317 46.106 57.6337 46.211C57.6397 46.539 57.6446 46.866 57.6496 47.194C57.6776 48.982 57.6777 48.982 57.7467 50.77C57.7507 50.846 57.7537 50.922 57.7577 51C57.7836 51.478 57.8426 51.935 57.9596 52.399C58.0076 52.636 58.0027 52.859 57.9996 53.1C57.9137 53.1 57.9137 53.1 57.8256 53.101C54.4207 53.111 51.0146 53.122 47.6096 53.133C47.2016 53.135 46.7926 53.136 46.3847 53.137C46.3027 53.138 46.2216 53.138 46.1376 53.138C44.8197 53.142 43.5007 53.147 42.1827 53.15C40.8307 53.155 39.4786 53.159 38.1266 53.164C37.2927 53.167 36.4577 53.169 35.6227 53.172C35.0507 53.173 34.4797 53.175 33.9077 53.177C33.5777 53.179 33.2467 53.18 32.9167 53.18C32.5597 53.181 32.2017 53.183 31.8447 53.184C31.6866 53.184 31.6867 53.184 31.5247 53.185C31.3827 53.185 31.3827 53.185 31.2367 53.186C31.1546 53.186 31.0716 53.187 30.9867 53.187C30.7987 53.18 30.7987 53.18 30.6997 53.3C30.6907 53.519 30.6877 53.738 30.6877 53.957C30.6877 54.062 30.6877 54.062 30.6877 54.169C30.6867 54.406 30.6877 54.643 30.6877 54.88C30.6877 55.05 30.6877 55.221 30.6877 55.391C30.6877 55.862 30.6877 56.333 30.6887 56.803C30.6887 57.31 30.6887 57.817 30.6887 58.323C30.6887 59.202 30.6887 60.081 30.6897 60.959C30.6907 62.23 30.6907 63.5 30.6907 64.77C30.6907 66.831 30.6917 68.892 30.6927 70.953C30.6937 72.955 30.6937 74.958 30.6947 76.96C30.6947 77.084 30.6947 77.207 30.6947 77.334C30.6947 77.953 30.6947 78.571 30.6947 79.19C30.6957 84.327 30.6977 89.463 30.6997 94.6C36.8377 94.65 36.8377 94.65 43.0997 94.7C43.0997 94.007 43.0997 93.314 43.0997 92.6C43.4987 89.501 45.0177 86.589 47.5126 84.65C48.9277 83.632 50.4646 82.893 52.1567 82.481C52.2247 82.465 52.2937 82.448 52.3637 82.431C53.2867 82.218 54.2096 82.185 55.1526 82.185C55.3186 82.185 55.3186 82.185 55.4887 82.184C55.8517 82.184 56.2156 82.183 56.5786 82.183C56.8336 82.183 57.0886 82.183 57.3436 82.182C57.8766 82.182 58.4106 82.181 58.9436 82.181C59.5556 82.182 60.1686 82.181 60.7806 82.179C61.3746 82.177 61.9686 82.177 62.5616 82.177C62.8126 82.177 63.0626 82.177 63.3137 82.176C65.1446 82.171 66.8447 82.216 68.5997 82.8C68.7057 82.835 68.8116 82.87 68.9197 82.905C71.7516 83.916 74.0446 86.063 75.3326 88.775C76.1987 90.701 76.4026 92.459 76.4996 94.6C83.3307 94.633 90.1616 94.666 97.1996 94.7C97.1946 92.454 97.1886 90.207 97.1826 87.893C97.1796 86.473 97.1776 85.053 97.1756 83.634C97.1746 82.888 97.1736 82.143 97.1726 81.397C97.1726 81.304 97.1726 81.211 97.1726 81.114C97.1696 79.605 97.1656 78.096 97.1616 76.586C97.1566 75.038 97.1536 73.489 97.1526 71.941C97.1526 70.985 97.1497 70.029 97.1456 69.073C97.1436 68.418 97.1426 67.763 97.1436 67.108C97.1436 66.73 97.1426 66.352 97.1406 65.974C97.1376 65.564 97.1386 65.154 97.1396 64.745C97.1386 64.625 97.1367 64.505 97.1357 64.381C97.1437 63.487 97.1436 63.487 97.4397 63.179C97.6826 63.012 97.8696 62.909 98.1516 62.83C98.2796 62.792 98.2796 62.792 98.4097 62.753C98.4986 62.727 98.5886 62.702 98.6816 62.675C100.064 62.245 101.213 61.588 102.345 60.695C102.84 60.317 102.84 60.317 103.1 60.2C103.455 60.311 103.736 60.477 104.036 60.693C104.115 60.75 104.193 60.807 104.274 60.865C104.396 60.954 104.396 60.954 104.519 61.044C104.642 61.132 104.642 61.132 104.767 61.221C105.375 61.662 105.375 61.662 105.6 62C105.626 62.246 105.626 62.246 105.627 62.546C105.627 62.66 105.628 62.774 105.629 62.892C105.629 63.018 105.628 63.144 105.628 63.274C105.629 63.408 105.629 63.543 105.63 63.678C105.632 64.049 105.632 64.42 105.632 64.791C105.632 65.191 105.634 65.591 105.635 65.99C105.638 66.683 105.639 67.376 105.64 68.069C105.642 69.071 105.644 70.073 105.647 71.075C105.651 72.7 105.655 74.326 105.658 75.951C105.661 77.53 105.665 79.109 105.669 80.688C105.669 80.786 105.669 80.883 105.669 80.983C105.671 81.471 105.672 81.959 105.673 82.447C105.683 86.498 105.692 90.549 105.7 94.6C105.755 94.601 105.81 94.602 105.866 94.602C106.437 94.611 107.008 94.621 107.579 94.632C107.792 94.636 108.005 94.639 108.217 94.642C108.524 94.647 108.831 94.653 109.138 94.659C109.232 94.66 109.326 94.661 109.424 94.662C110.63 94.689 111.785 95.05 112.695 95.864C113.556 96.827 113.984 97.908 113.959 99.203C113.892 100.347 113.485 101.253 112.682 102.065C111.53 103.026 110.29 103.13 108.856 103.126C108.697 103.126 108.539 103.126 108.38 103.127C107.943 103.128 107.505 103.127 107.068 103.127C106.593 103.127 106.118 103.127 105.642 103.128C104.809 103.129 103.975 103.129 103.142 103.129C101.903 103.129 100.663 103.13 99.4236 103.131C97.0816 103.133 94.7396 103.133 92.3977 103.133C90.5896 103.134 88.7807 103.134 86.9726 103.135C86.4517 103.135 85.9316 103.135 85.4116 103.135C85.2816 103.135 85.1526 103.135 85.0196 103.135C81.3806 103.136 77.7417 103.137 74.1027 103.137C73.9647 103.137 73.8256 103.137 73.6866 103.137C73.0606 103.137 72.4356 103.137 71.8096 103.137C71.5306 103.137 71.2516 103.137 70.9726 103.137C70.8326 103.137 70.6926 103.137 70.5527 103.137C66.1367 103.137 61.7197 103.138 57.3027 103.141C56.1897 103.142 55.0766 103.143 53.9646 103.143C53.8616 103.143 53.8616 103.143 53.7566 103.143C51.6446 103.145 49.5327 103.145 47.4207 103.146C47.3207 103.146 47.3206 103.146 47.2197 103.146C45.6986 103.146 44.1776 103.146 42.6556 103.146C42.1356 103.146 41.6157 103.146 41.0947 103.146C40.9027 103.146 40.9027 103.146 40.7067 103.146C38.6427 103.146 36.5797 103.147 34.5157 103.149C32.4427 103.15 30.3687 103.151 28.2957 103.149C27.0637 103.149 25.8307 103.149 24.5987 103.151C23.7757 103.152 22.9527 103.152 22.1297 103.15C21.6607 103.15 21.1927 103.149 20.7237 103.151C20.2967 103.153 19.8697 103.152 19.4427 103.15C19.2887 103.15 19.1357 103.15 18.9827 103.151C17.5747 103.161 16.3167 102.909 15.1997 102C14.3717 101.171 13.8957 100.167 13.8687 98.994C13.8797 97.942 14.2687 97.125 14.8997 96.3C14.9727 96.201 15.0457 96.102 15.1207 96C15.4187 95.647 15.7677 95.469 16.1817 95.275C16.2487 95.241 16.3157 95.207 16.3857 95.173C17.2017 94.776 17.9427 94.67 18.8407 94.68C18.9857 94.681 18.9857 94.681 19.1337 94.681C19.4387 94.682 19.7447 94.685 20.0497 94.688C20.2587 94.689 20.4667 94.689 20.6747 94.69C21.1837 94.692 21.6917 94.696 22.1997 94.7C22.1997 94.567 22.1997 94.567 22.1997 94.432C22.1957 89.191 22.1917 83.95 22.1897 78.71C22.1897 78.08 22.1897 77.451 22.1887 76.822C22.1887 76.696 22.1887 76.571 22.1887 76.442C22.1877 74.414 22.1867 72.385 22.1847 70.357C22.1827 68.275 22.1817 66.194 22.1817 64.112C22.1807 62.828 22.1797 61.544 22.1787 60.26C22.1777 59.379 22.1767 58.498 22.1777 57.617C22.1777 57.109 22.1777 56.601 22.1767 56.093C22.1757 55.627 22.1757 55.162 22.1757 54.696C22.1757 54.528 22.1757 54.36 22.1747 54.192C22.1657 51.463 22.9197 49.08 24.8187 47.069C26.6057 45.39 28.9297 44.681 31.3327 44.688Z" fill="white" />
            <path d="M72.6143 23.8725C72.8583 23.8725 73.1013 23.8715 73.3453 23.8705C73.9353 23.8695 74.5253 23.8695 75.1153 23.8695C75.5963 23.8695 76.0763 23.8695 76.5563 23.8695C76.6243 23.8695 76.6933 23.8695 76.7633 23.8695C76.9033 23.8695 77.0423 23.8685 77.1823 23.8685C78.4853 23.8675 79.7893 23.8675 81.0933 23.8685C82.2833 23.8695 83.4743 23.8675 84.6643 23.8655C85.8903 23.8645 87.1153 23.8635 88.3413 23.8635C89.0283 23.8635 89.7143 23.8635 90.4013 23.8625C91.0473 23.8605 91.6933 23.8615 92.3383 23.8625C92.5753 23.8625 92.8113 23.8625 93.0483 23.8615C95.3343 23.8545 95.3343 23.8545 96.3003 24.1995C96.4703 24.2585 96.4703 24.2585 96.6443 24.3175C97.4793 24.6715 98.3303 25.0755 99.0003 25.6995C99.0003 25.7655 99.0003 25.8315 99.0003 25.8995C99.0663 25.8995 99.1323 25.8995 99.2003 25.8995C99.2003 25.9655 99.2003 26.0315 99.2003 26.0995C99.2563 26.1195 99.3123 26.1385 99.3693 26.1575C99.7003 26.3615 99.8793 26.6555 100.087 26.9745C100.131 27.0415 100.175 27.1095 100.22 27.1785C100.926 28.2855 101.294 29.4095 101.5 30.6995C101.6 30.6185 101.6 30.6185 101.702 30.5345C101.896 30.3755 102.091 30.2165 102.286 30.0565C102.485 29.8945 102.684 29.7315 102.883 29.5675C103.323 29.2065 103.765 28.8475 104.212 28.4935C104.61 28.1765 104.989 27.8455 105.367 27.5055C105.702 27.2095 106.05 26.9305 106.398 26.6485C106.676 26.4195 106.948 26.1855 107.22 25.9495C109.646 23.8425 109.646 23.8425 111.115 23.8665C112.007 23.9705 112.681 24.3725 113.238 25.0655C113.625 25.6265 113.912 26.1875 113.913 26.8795C113.914 26.9785 113.914 27.0785 113.914 27.1805C113.914 27.3445 113.914 27.3445 113.914 27.5115C113.915 27.6275 113.915 27.7435 113.916 27.8635C113.916 28.1865 113.917 28.5085 113.917 28.8315C113.918 29.1785 113.919 29.5265 113.919 29.8745C113.921 30.7135 113.922 31.5535 113.923 32.3935C113.924 32.7895 113.924 33.1845 113.925 33.5805C113.926 34.8955 113.928 36.2105 113.929 37.5255C113.929 37.8675 113.929 38.2085 113.929 38.5495C113.929 38.6345 113.929 38.7195 113.929 38.8065C113.93 40.1805 113.933 41.5545 113.936 42.9285C113.939 44.3385 113.941 45.7495 113.941 47.1605C113.942 47.9525 113.943 48.7435 113.945 49.5355C113.947 50.2105 113.948 50.8845 113.947 51.5595C113.947 51.9035 113.947 52.2475 113.949 52.5915C113.951 52.9645 113.95 53.3385 113.948 53.7115C113.95 53.8735 113.95 53.8735 113.952 54.0395C113.944 55.0105 113.653 55.7105 112.993 56.4205C112.238 57.1205 111.569 57.2385 110.559 57.2265C109.562 57.1525 108.87 56.5915 108.1 55.9995C107.921 55.8705 107.742 55.7415 107.562 55.6125C107.112 55.2855 106.672 54.9475 106.233 54.6065C105.678 54.1785 105.108 53.7705 104.538 53.3615C104.094 53.0425 103.658 52.7165 103.23 52.3765C102.698 51.9685 102.147 51.5865 101.6 51.1995C101.534 51.2325 101.468 51.2655 101.4 51.2995C101.322 51.5395 101.253 51.7765 101.187 52.0185C100.806 53.3115 100.165 54.3505 99.2003 55.2995C99.1173 55.3825 99.1173 55.3825 99.0333 55.4665C97.6513 56.7425 96.0163 57.2155 94.1773 57.2145C94.0763 57.2155 93.9743 57.2155 93.8693 57.2155C93.5303 57.2165 93.1903 57.2175 92.8513 57.2175C92.6073 57.2185 92.3633 57.2185 92.1193 57.2195C91.3883 57.2215 90.6573 57.2225 89.9263 57.2235C89.5813 57.2235 89.2363 57.2245 88.8913 57.2255C87.7433 57.2265 86.5953 57.2285 85.4463 57.2285C85.1493 57.2295 84.8513 57.2295 84.5543 57.2295C84.4803 57.2295 84.4063 57.2295 84.3303 57.2295C83.1333 57.2305 81.9373 57.2325 80.7403 57.2365C79.5093 57.2395 78.2783 57.2415 77.0483 57.2415C76.3583 57.2415 75.6673 57.2425 74.9773 57.2455C74.3283 57.2475 73.6783 57.2485 73.0283 57.2475C72.7903 57.2465 72.5533 57.2475 72.3153 57.2485C69.7603 57.2625 67.7323 56.7765 65.8183 54.9625C65.3913 54.5175 65.0883 54.0425 64.8003 53.4995C64.7563 53.4245 64.7123 53.3485 64.6663 53.2705C63.8793 51.8465 63.8413 50.3445 63.8503 48.7625C63.8493 48.5515 63.8493 48.3415 63.8483 48.1315C63.8473 47.5655 63.8483 46.9985 63.8493 46.4325C63.8513 45.8375 63.8503 45.2425 63.8503 44.6475C63.8493 43.6495 63.8513 42.6515 63.8533 41.6535C63.8563 40.5025 63.8563 39.3505 63.8553 38.1985C63.8543 37.0865 63.8543 35.9755 63.8563 34.8635C63.8563 34.3925 63.8563 33.9205 63.8563 33.4495C63.8553 32.8935 63.8563 32.3385 63.8583 31.7825C63.8593 31.5795 63.8593 31.3765 63.8593 31.1735C63.8543 29.2205 63.8543 29.2205 64.2003 28.2995C64.2303 28.2135 64.2603 28.1265 64.2913 28.0375C64.8203 26.5445 65.8043 25.5165 67.2003 24.7995C67.5953 24.6215 67.9953 24.4575 68.4003 24.2995C68.4773 24.2675 68.5543 24.2345 68.6343 24.2005C69.8713 23.7335 71.3173 23.8705 72.6143 23.8725Z" fill="white" />
            <path d="M64.9001 63.1C66.4781 64.41 67.6151 66.279 68.0001 68.3C68.0431 68.792 68.0481 69.281 68.0441 69.775C68.0431 69.841 68.0431 69.908 68.0431 69.976C68.0251 72.334 67.1451 74.287 65.4731 75.941C64.3801 76.974 63.1381 77.515 61.7001 77.9C61.6101 77.927 61.6101 77.927 61.5191 77.954C60.0141 78.271 58.0891 78.134 56.7001 77.5C56.6331 77.471 56.5671 77.442 56.4981 77.412C55.5541 76.999 54.7611 76.504 54.0001 75.8C53.9141 75.724 53.8281 75.648 53.7391 75.57C52.3591 74.29 51.4581 72.239 51.3791 70.362C51.3161 68.056 52.0841 65.84 53.6161 64.091C56.6311 60.917 61.3771 60.538 64.9001 63.1Z" fill="white" />
            <path d="M97.0996 62.2478L102.833 60.1506L105.611 60.4796V94.6999H97.0996V62.2478Z" fill="white" />
        </g>
        <defs>
            <linearGradient id="paint0_linear_110_15" x1="37.3196" y1="43.4704" x2="102.243" y2="108.89" gradientUnits="userSpaceOnUse">
                <stop stop-color="#7654FF" />
                <stop offset="1" stop-color="#193EF2" />
            </linearGradient>
            <clipPath id="clip0_110_15">
                <rect width="128" height="128" fill="white" />
            </clipPath>
        </defs>
    </svg>
);

const formatDateTime = (dateString) => {
    const options = { year: 'numeric', month: 'short', day: 'numeric', hour: 'numeric', minute: 'numeric', second: 'numeric' };
    return new Date(dateString).toLocaleString(undefined, options);
};



const ZoomBlockEdit = (props) => {
    const { attributes, setAttributes } = props;
    const { linkOnly, meetingType, hostId, selectedMeeting, selectedWebinar } = attributes;
    const zoomUsersOptions = getZoomUsers();
    const [meetings, setMeetings] = useState([]);
    const [webinars, setWebinars] = useState([]);
    const [loading, setLoading] = useState(true);

    const fetchMeetingsOrWebinarsByHostId = async (selectedHostId) => {
        setAttributes({ hostId: selectedHostId }); // Update hostId attribute

        // Check if a valid host ID is selected (not the default "Select host" option)
        if (selectedHostId) {
            setLoading(true);
            try {
                const path = `zmeet/v2/zoom/meetings?host_id=${selectedHostId}&meeting_type=${meetingType}`;
                const response = await wp.apiFetch({
                    path: path,
                    method: 'GET',
                });

                if (meetingType === 'meeting') {
                    if (response && response.meetings && response.meetings.length > 0) {
                        setMeetings(response.meetings);
                    } else {
                        setMeetings([]); // Ensure the state is cleared if no data is found
                    }
                } else if (meetingType === 'webinar') {
                    if (response && response.webinars && response.webinars.length > 0) {
                        setWebinars(response.webinars);
                    } else {
                        setWebinars([]); // Ensure the state is cleared if no data is found
                    }
                }
            } catch (error) {
                console.error('API Error:', error);
            } finally {
                setLoading(false);
            }
        } else {
            setMeetings([]);
            setWebinars([]);
        }
    };

    // Effect to handle changes in meetingType
    useEffect(() => {
        if (hostId) {
            fetchMeetingsOrWebinarsByHostId(hostId);
        }
    }, [meetingType]);

    const meetingSelectOptions = () => {
        const zoomMeetings = meetings.map(meeting => ({
            label: meeting.topic,
            value: meeting.id,
        }));

        // Add the default option "Select a meeting" at the beginning of the options array
        return [{ label: __('--Select a meeting--', 'zoomeet'), value: '' }, ...zoomMeetings];
    };

    const webinarSelectOptions = () => {
        const zoomWebinars = webinars.map(webinar => ({
            label: webinar.topic,
            value: webinar.id,
        }));

        // Add the default option "Select a webinar" at the beginning of the options array
        return [{ label: __('--Select a webinar--', 'zoomeet'), value: '' }, ...zoomWebinars];
    };

    const handleMeetingChange = (selectedValue) => {
        const selected = meetings.find(meeting => meeting.id == selectedValue);
        setAttributes({ selectedMeeting: selected });
    };

    const handleWebinarChange = (selectedValue) => {
        const selected = webinars.find(webinar => webinar.id == selectedValue);
        setAttributes({ selectedWebinar: selected });
    };


    return (
        <Fragment>
            <InspectorControls>
                <PanelBody title={__('Settings', 'zoomeet')} initialOpen={true}>
                    <CheckboxControl
                        label={__('Link Only ?', 'zoomeet')}
                        checked={linkOnly}
                        onChange={(val) => setAttributes({ linkOnly: val })}
                    />

                    <SelectControl
                        label={__('Meeting Type', 'zoomeet')}
                        value={meetingType}
                        options={[
                            { label: __('Meeting', 'zoomeet'), value: 'meeting' },
                            { label: __('Webinar', 'zoomeet'), value: 'webinar' },
                        ]}
                        onChange={(val) => setAttributes({ meetingType: val })}
                    />

                    <SelectControl
                        label={__('Select A Host', 'zoomeet')}
                        value={hostId}
                        options={zoomUsersOptions}
                        onChange={fetchMeetingsOrWebinarsByHostId}
                    />

                    {meetingType === 'meeting' && meetings.length > 0 && (
                        <SelectControl
                            label={__('Select A Meeting', 'zoomeet')}
                            value={selectedMeeting ? selectedMeeting.id : ''}
                            options={meetingSelectOptions()}
                            onChange={handleMeetingChange}
                        />
                    )}

                    {meetingType === 'webinar' && webinars.length > 0 && (
                        <SelectControl
                            label={__('Select A Webinar', 'zoomeet')}
                            value={selectedWebinar ? selectedWebinar.id : ''}
                            options={webinarSelectOptions()}
                            onChange={handleWebinarChange}
                        />
                    )}
                </PanelBody>
            </InspectorControls>
            <div>
                <h3>{__('Show Direct Zoom Meeting or Webinar', 'zoomeet')}</h3>

                {meetingType === 'meeting' && selectedMeeting && linkOnly ? (
                    <a className="zmeet_join_btn" href={selectedMeeting.join_url} target="_blank" rel="noopener noreferrer">
                       {__('Join meeting', 'zoomeet')}
                    </a>
                ) : (
                    meetingType === 'meeting' && selectedMeeting && (
                        <table className="zmeet-table">
                            <tbody>
                                <tr>
                                    <td>{__('Meeting ID', 'zoomeet')}</td>
                                    <td>{selectedMeeting.id}</td>
                                </tr>
                                <tr>
                                    <td>{__('Topic', 'zoomeet')}</td>
                                    <td>{selectedMeeting.topic}</td>
                                </tr>
                                <tr>
                                    <td>{__('Start Time', 'zoomeet')}</td>
                                    <td>{formatDateTime(selectedMeeting.start_time)}</td>
                                </tr>
                                <tr>
                                    <td>{__('Timezone', 'zoomeet')}</td>
                                    <td>{selectedMeeting.timezone}</td>
                                </tr>
                                <tr>
                                    <td>{__('Duration', 'zoomeet')}</td>
                                    <td>{selectedMeeting.duration} {__('minutes', 'zoomeet')}</td>
                                </tr>
                                <tr>
                                    <td>{__('Join via Zoom App', 'zoomeet')}</td>
                                    <td><a href={selectedMeeting.join_url} target="_blank" rel="noopener noreferrer">{__('Join', 'zoomeet')}</a></td>
                                </tr>
                            </tbody>

                        </table>
                    )
                )}

                {meetingType === 'webinar' && selectedWebinar && linkOnly ? (
                    <a href={selectedWebinar.join_url} className="zmeet_join_btn" target="_blank" rel="noopener noreferrer">
                       {__('Join webinar', 'zoomeet')}
                    </a>
                ) : (
                    meetingType === 'webinar' && selectedWebinar && (
                        <table className="zmeet-table">
                            <tbody>
                                <tr>
                                    <td>{__('Webinar ID', 'zoomeet')}</td>
                                    <td>{selectedWebinar.id}</td>
                                </tr>
                                <tr>
                                    <td>{__('Topic', 'zoomeet')}</td>
                                    <td>{selectedWebinar.topic}</td>
                                </tr>
                                <tr>
                                    <td>{__('Start Time', 'zoomeet')}</td>
                                    <td>{formatDateTime(selectedWebinar.start_time)}</td>
                                </tr>
                                <tr>
                                    <td>{__('Timezone', 'zoomeet')}</td>
                                    <td>{selectedWebinar.timezone}</td>
                                </tr>
                                <tr>
                                    <td>{__('Duration', 'zoomeet')}</td>
                                    <td>{selectedWebinar.duration} {__('minutes', 'zoomeet')}</td>
                                </tr>
                                <tr>
                                    <td>{__('Join via Zoom App', 'zoomeet')}</td>
                                    <td><a href={selectedWebinar.join_url} target="_blank" rel="noopener noreferrer">{__('Join', 'zoomeet')}</a></td>
                                </tr>
                            </tbody>
                        </table>
                    )
                )}

            </div>
        </Fragment>
    );
};


// Register Zoom block
registerBlockType('zmeet/zoom', {
    title: __('Zoom Direct Meeting or Webinar (ZooMeet)'),
    icon: BlockIcon,
    category: 'embed',
    keywords: [
        __('zoom', 'zoomeet'),
        __('video', 'zoomeet'),
        __('conference', 'zoomeet'),
        __('meeting', 'zoomeet'),
        __('zoomeet', 'zoomeet'),
    ],
    attributes: {
        linkOnly: {
            type: 'boolean',
            default: false,
        },
        meetingType: {
            type: 'string',
            default: 'meeting',
        },
        hostId: {
            type: 'string',
            default: 'meeting',
        },
        selectedMeeting: {
            type: 'object',
            default: null,
        },
        selectedWebinar: {
            type: 'object',
            default: null,
        },
    },
    edit: ZoomBlockEdit,
    save: (props) => {
        const { meetingType, linkOnly, selectedMeeting, selectedWebinar } = props.attributes;
        return (
            <div>
                { meetingType === 'meeting' && selectedMeeting && linkOnly ? (
                    <a href={selectedMeeting.join_url} className="zmeet_join_btn" target="_blank" rel="noopener noreferrer">
                       {__('Join meeting', 'zoomeet')}
                    </a>
                ) : (
                    meetingType === 'meeting' && selectedMeeting && (
                    <table className="zmeet-table">
                       <tbody>
                            <tr>
                                <td>{__('Meeting ID', 'zoomeet')}</td>
                                <td>{selectedMeeting.id}</td>
                            </tr>
                            <tr>
                                <td>{__('Topic', 'zoomeet')}</td>
                                <td>{selectedMeeting.topic}</td>
                            </tr>
                            <tr>
                                <td>{__('Start Time', 'zoomeet')}</td>
                                <td>{formatDateTime(selectedMeeting.start_time)}</td>
                            </tr>
                            <tr>
                                <td>{__('Timezone', 'zoomeet')}</td>
                                <td>{selectedMeeting.timezone}</td>
                            </tr>
                            <tr>
                                <td>{__('Duration', 'zoomeet')}</td>
                                <td>{selectedMeeting.duration} {__('minutes', 'zoomeet')}</td>
                            </tr>
                            <tr>
                                <td>{__('Join via Zoom App', 'zoomeet')}</td>
                                <td><a href={selectedMeeting.join_url} target="_blank" rel="noopener noreferrer">{__('Join', 'zoomeet')}</a></td>
                            </tr>
                        </tbody>

                    </table>
                    )
                )}

                { meetingType === 'meeting' && ! selectedMeeting && (
                   <h3>{__('No meeting found', 'zoomeet')}</h3>
                )}

                { meetingType === 'webinar' && selectedWebinar && linkOnly ? (
                    <a href={selectedWebinar.join_url} className="zmeet_join_btn" target="_blank" rel="noopener noreferrer">
                      {__('Join webinar', 'zoomeet')}
                    </a>
                ) : (
                    meetingType === 'webinar' && selectedWebinar && (
                    <table className="zmeet-table">
                       <tbody>
                            <tr>
                                <td>{__('Webinar ID', 'zoomeet')}</td>
                                <td>{selectedWebinar.id}</td>
                            </tr>
                            <tr>
                                <td>{__('Topic', 'zoomeet')}</td>
                                <td>{selectedWebinar.topic}</td>
                            </tr>
                            <tr>
                                <td>{__('Start Time', 'zoomeet')}</td>
                                <td>{new Date(selectedWebinar.start_time).toLocaleString()}</td>
                            </tr>
                            <tr>
                                <td>{__('Timezone', 'zoomeet')}</td>
                                <td>{selectedWebinar.timezone}</td>
                            </tr>
                            <tr>
                                <td>{__('Duration', 'zoomeet')}</td>
                                <td>{selectedWebinar.duration} {__('minutes', 'zoomeet')}</td>
                            </tr>
                            <tr>
                                <td>{__('Join via Zoom App', 'zoomeet')}</td>
                                <td><a href={selectedWebinar.join_url} target="_blank" rel="noopener noreferrer">{__('Join', 'zoomeet')}</a></td>
                            </tr>
                        </tbody>

                    </table>
                    )
                )}

                { meetingType === 'webinar' && ! selectedWebinar && (
                    <h3>{__('No Webinar found', 'zoomeet')}</h3>
                )}
            
            </div>
        );
    },
});

