import React, { useState, useEffect, useCallback } from 'react';
import axios from 'axios';
import moment from 'moment';
import Select from 'react-select';
import AsyncSelect from 'react-select/async';
import Spinner from '../components/common/Spinner';
import { langString, select2Styles } from '../Helpers';

const { __ } = wp.i18n;

const MeetingListApp = ({ platform, meeting_type }) => {
	const config = window.zmeet_frontend_data || {};
	const [meetings, setMeetings] = useState([]);
	const [loading, setLoading] = useState(true);
	const [sortBy, setSortBy] = useState('date');
	const [sortOrder, setSortOrder] = useState('desc');
	const [searchQuery, setSearchQuery] = useState('');
	const [currentPage, setCurrentPage] = useState(1);
	const [itemsPerPage] = useState(10);

	// Visibility states
	const [visibilityMode, setVisibilityMode] = useState(config.visibility_mode || 'all');
	const [selectedRoles, setSelectedRoles] = useState(config.selected_roles || []);
	const [selectedUsers, setSelectedUsers] = useState(config.selected_users || []);
	const [allRoles, setAllRoles] = useState([]);
	const [selectedUserOptions, setSelectedUserOptions] = useState([]);
	const [userSearchLoading, setUserSearchLoading] = useState(false);

	useEffect(() => {
		const fetchData = async () => {
			setLoading(true);
			try {
				const formData = new FormData();
				formData.append('action', 'zmeet_meeting_list');
				formData.append('nonce', config.nonce);
				formData.append('page_id', config.page_id);
				formData.append('visibility_mode', visibilityMode);
				formData.append('roles', JSON.stringify(selectedRoles));
				formData.append('users', JSON.stringify(selectedUsers));
				formData.append('use_passed_values', config.is_admin ? '1' : '0');

				const response = await axios.post(config.ajax_url, formData);
				if (response.data.success) {
					// Filter by meeting type (meeting or webinar)
					const filtered = response.data.data.filter(meeting => {
						const type = meeting.settings && meeting.settings.meeting_type ? meeting.settings.meeting_type : 0;
						return String(type) === String(meeting_type);
					});
					setMeetings(filtered);
				}
			} catch (error) {
				console.error(langString('error_fetching_data'), error);
			} finally {
				setLoading(false);
			}
		};

		fetchData();
	}, [platform, meeting_type, visibilityMode, selectedRoles, selectedUsers]);

	// Fetch roles when mode is 'role'
	useEffect(() => {
		if (!config.is_admin) return;

		const fetchRoles = async () => {
			try {
				const formData = new FormData();
				formData.append('action', 'zmeet_fetch_roles'); // Note: Need to implement this in Zmeet_Ajax if not exists.
				// Wait, I didn't implement zmeet_fetch_roles yet. I should add it to Zmeet_Ajax.
				formData.append('nonce', config.nonce);
				const response = await axios.post(config.ajax_url, formData);
				if (response.data.success) {
					setAllRoles(response.data.data);
				}
			} catch (error) {
				console.error("Failed to load roles", error);
			}
		};

		if (visibilityMode === 'role') {
			fetchRoles();
		}
	}, [visibilityMode, config.is_admin, config.nonce, config.ajax_url]);

	// Fetch user details for initial value
	useEffect(() => {
		if (!config.is_admin || selectedUsers.length === 0) return;

		const fetchUsersByIds = async () => {
			try {
				const formData = new FormData();
				formData.append('action', 'zmeet_fetch_users_by_ids');
				formData.append('nonce', config.nonce);
				selectedUsers.forEach(id => formData.append('user_ids[]', id));
				const response = await axios.post(config.ajax_url, formData);
				if (response.data.success) {
					setSelectedUserOptions(response.data.data);
				}
			} catch (error) {
				console.error("Failed to fetch users", error);
			}
		};

		fetchUsersByIds();
	}, [config.is_admin, config.nonce, config.ajax_url, selectedUsers]);

	const loadUsers = async (inputValue) => {
		if (inputValue.length < 2) return [];
		setUserSearchLoading(true);
		try {
			const formData = new FormData();
			formData.append('action', 'zmeet_fetch_users');
			formData.append('nonce', config.nonce);
			formData.append('search', inputValue);
			const response = await axios.post(config.ajax_url, formData);
			setUserSearchLoading(false);
			return response.data.success ? response.data.data : [];
		} catch (error) {
			console.error("User search failed", error);
			setUserSearchLoading(false);
			return [];
		}
	};

	const debouncedLoadUsers = useCallback((inputValue, callback) => {
		const timeoutId = setTimeout(() => {
			loadUsers(inputValue).then(callback);
		}, 600);
		return () => clearTimeout(timeoutId);
	}, [config.nonce, config.ajax_url]);

	const handleSort = (column) => {
		if (sortBy === column) setSortOrder(sortOrder === 'asc' ? 'desc' : 'asc');
		else {
			setSortBy(column);
			setSortOrder('asc');
		}
	};

	const getSortableValue = (meeting, column) => {
		switch (column) {
			case 'title': return meeting.title;
			case 'date': return moment(meeting.settings.startDateTime).format("YYYY-MM-DD HH:mm");
			default: return '';
		}
	};

	const getSortedMeetings = () => {
		return [...meetings].sort((a, b) => {
			const aVal = getSortableValue(a, sortBy);
			const bVal = getSortableValue(b, sortBy);
			return sortOrder === 'asc' ? aVal.localeCompare(bVal) : bVal.localeCompare(aVal);
		});
	};

	const filteredMeetings = getSortedMeetings().filter(meeting =>
		meeting.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
		moment(meeting.settings.startDateTime).format("MMMM Do YYYY").toLowerCase().includes(searchQuery.toLowerCase())
	);

	const indexOfLastItem = currentPage * itemsPerPage;
	const indexOfFirstItem = indexOfLastItem - itemsPerPage;
	const currentMeetings = filteredMeetings.slice(indexOfFirstItem, indexOfLastItem);
	const totalPages = Math.ceil(filteredMeetings.length / itemsPerPage);

	const handlePageChange = (page) => {
		if (page >= 1 && page <= totalPages) setCurrentPage(page);
	};

	const handleChangeVisibility = (mode) => {
		setVisibilityMode(mode);
		setCurrentPage(1);
		saveVisibilitySettings(mode, selectedRoles, selectedUsers);
	};

	const handleRoleChange = (selectedOptions) => {
		const roleValues = selectedOptions ? selectedOptions.map(option => option.value) : [];
		setSelectedRoles(roleValues);
		setCurrentPage(1);
		saveVisibilitySettings(visibilityMode, roleValues, selectedUsers);
	};

	const handleUserChange = (selectedOptions) => {
		const userValues = selectedOptions ? selectedOptions.map(option => String(option.value)) : [];
		setSelectedUsers(userValues);
		setSelectedUserOptions(selectedOptions || []);
		setCurrentPage(1);
		saveVisibilitySettings(visibilityMode, selectedRoles, userValues);
	};

	const saveVisibilitySettings = async (mode, roles, users) => {
		try {
			const formData = new FormData();
			formData.append('action', 'zmeet_save_visibility_mode');
			formData.append('nonce', config.nonce);
			formData.append('page_id', config.page_id);
			formData.append('mode', mode);
			formData.append('roles', JSON.stringify(roles));
			formData.append('users', JSON.stringify(users));
			await axios.post(config.ajax_url, formData);
		} catch (error) {
			console.error("Error saving visibility settings", error);
		}
	};

	const getSelectedRoleOptions = () => {
		return allRoles.filter(role => selectedRoles.includes(role.value));
	};

	const getUserSearchStrings = () => ({
		loadingMessage: () => langString('loading_users'),
		noOptionsMessage: ({ inputValue }) =>
			!inputValue || inputValue.length < 2
				? langString('type_to_search_users')
				: langString('no_users_found_search'),
		placeholder: langString('search_users_placeholder')
	});

	if (loading) return <Spinner />;

	const handleViewMeeting = (url) => {
		window.open(url, '_blank');
	};

	return (
		<div id="zmeet">
			<div className="zmeet-common-dashboard">
				<div className="zmeet-table-wrapper">
					{meetings.length > 0 && (
						<div className="search-container">
							<input
								type="text"
								placeholder={langString('search')}
								value={searchQuery}
								onChange={(e) => {
									setSearchQuery(e.target.value);
									setCurrentPage(1);
								}}
								className="search-input"
							/>
						</div>
					)}

					<div className={meetings.length ? 'zmeet-meeting-list' : 'zmeet-no-meeting'}>
						{meetings.length > 0 ? (
							<div className="table-container">
								<table className="zmeet-table">
									<thead>
										<tr>
											<th onClick={() => handleSort('title')} className={sortBy === 'title' ? `sortable ${sortOrder}` : 'sortable'}>{langString('topic')}</th>
											<th onClick={() => handleSort('date')} className={sortBy === 'date' ? `sortable ${sortOrder}` : 'sortable'}>{langString('start_time')}</th>
											<th>{langString('actions')}</th>
										</tr>
									</thead>
									<tbody>
										{currentMeetings.map((meeting) => (
											<tr key={meeting.id}>
												<td>{meeting.title}</td>
												<td>{moment(meeting.settings.startDateTime).format("dddd, MMMM D, YYYY h:mm A")}</td>
												<td>
													<button className="action-btn action-text-btn" onClick={() => handleViewMeeting(meeting.settings.url)}>
														{langString('meeting_link')}
													</button>
												</td>
											</tr>
										))}
									</tbody>
								</table>
							</div>
						) : (
							<div className="empty-meeting-wrapper">
								<h1>{platform === "zoom" && meeting_type == 1 ? langString('no_webinar') : langString('no_meetings')}</h1>
							</div>
						)}

						{!filteredMeetings.length && meetings.length > 0 && (
							<div className="empty-meeting-wrapper">
								<p>{langString('no_records')}</p>
							</div>
						)}

						{filteredMeetings.length > itemsPerPage && (
							<div className="pagination">
								<span className={`page-link ${currentPage === 1 ? 'disabled' : ''}`} onClick={() => handlePageChange(currentPage - 1)}>&lt; {langString('previous')}</span>
								{[...Array(totalPages).keys()].map((page) => (
									<span key={page + 1} className={`page-link ${currentPage === page + 1 ? 'active' : ''}`} onClick={() => handlePageChange(page + 1)}>{page + 1}</span>
								))}
								<span className={`page-link ${currentPage === totalPages ? 'disabled' : ''}`} onClick={() => handlePageChange(currentPage + 1)}>{langString('next')} &gt;</span>
							</div>
						)}
					</div>
				</div>

				{config.is_admin && (
					<div className={`zmeet-visibility-control ${loading ? 'loading' : ''}`}>
						<div className="visibility-main-control">
							<label>{langString('meeting_visibility')}</label>
							<select value={visibilityMode} onChange={(e) => handleChangeVisibility(e.target.value)} className="visibility-select">
								<option value="all">{platform === "zoom" && meeting_type == 1 ? langString('show_all_webinars') : langString('show_all_meetings')}</option>
								<option value="user">{langString('user_specific')}</option>
								<option value="role">{langString('role_based')}</option>
							</select>
						</div>

						{(visibilityMode === 'role' || visibilityMode === 'user') && (
							<div className="visibility-sub-controls">
								{visibilityMode === 'role' && (
									<div className="visibility-roles">
										<label>{langString('select_roles')}</label>
										<Select
											options={allRoles}
											value={getSelectedRoleOptions()}
											onChange={handleRoleChange}
											isMulti
											className="zmeet-select2"
											placeholder={langString('select_roles')}
											styles={select2Styles()}
											isClearable
										/>
									</div>
								)}

								{visibilityMode === 'user' && (
									<div className="visibility-users">
										<label>{langString('select_users')}</label>
										<AsyncSelect
											cacheOptions
											loadOptions={debouncedLoadUsers}
											defaultOptions={false}
											value={selectedUserOptions}
											onChange={handleUserChange}
											isMulti
											className="zmeet-select2"
											placeholder={getUserSearchStrings().placeholder}
											styles={select2Styles()}
											isClearable
											isLoading={userSearchLoading}
											loadingMessage={getUserSearchStrings().loadingMessage}
											noOptionsMessage={getUserSearchStrings().noOptionsMessage}
											filterOption={() => true}
											menuPortalTarget={document.body}
											menuPosition={'fixed'}
										/>
									</div>
								)}
							</div>
						)}
					</div>
				)}
			</div>
		</div>
	);
};

export default MeetingListApp;
