<?php
/**
 * Plugin Name:       ZML-Visit
 * Plugin URI:        https://www.zimoli.me/zml-plugin-visit/
 * Description:       Zimoli Lightweight Visit Statistics。
 * Version:           1.0.0
 * Author:            紫茉莉.ME
 * Author URI:        https://www.zimoli.me
 * Requires at least: 5.6
 * Requires PHP:      7.3
 * License:           GPLv3
 * License URI:       https://www.gnu.org/licenses/gpl-3.0.html
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class ZMLVIS_Stats {
    
    private static $instance = null;
    private $today;
    private $cookie_name = 'zml_visit_session';
    private $cookie_expire = 2592000; // 30天
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->today = gmdate('Y-m-d');
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // 记录访问
        add_action('wp', array($this, 'track_visit'), 1);
        
        // 管理页面
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // 仪表板小工具
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        
        // 清理过期数据
        add_action('zmlvis_daily_cleanup', array($this, 'cleanup_old_data'));
        $this->schedule_cleanup();
		
		// 加载文件
		\add_action( 'admin_enqueue_scripts', array($this, 'load_file'));
    }
    
    /**
     * 记录访问
     */
    public function track_visit() {
        // 排除条件
        if ($this->should_skip_tracking()) {
            return;
        }
        
        // 获取会话ID
        $session_id = $this->get_session_id();
        
        // 获取用户标识（会话ID + IP前两段）
        $user_key = $this->get_user_key($session_id);
        
        // 获取统计数据
        $stats = $this->get_stats();
        
        // 记录今日访问
        $today_visited = $this->has_visited_today($user_key);
        
        if (!$today_visited) {
            // 今日首次访问
            $stats['today']['visits']++;
            $stats['today']['unique_ips'][] = $this->get_ip_prefix();
            
            // 保存用户今日已访问标记
            $this->mark_user_visited_today($user_key);
        }
        
        // 增加今日PV
        $stats['today']['pageviews']++;
        
        // 增加总PV
        $stats['total']['pageviews']++;
        
        // 如果是新会话（今日第一次），增加总访问量
        if (!$today_visited) {
            $stats['total']['visits']++;
            
            // 更新最高访问日
            $this->update_peak_day($stats['today']['visits']);
        }
        
        // 更新最后访问时间
        $stats['today']['last_updated'] = current_time('timestamp');
        
        // 保存数据
        $this->save_stats($stats);
    }
	
    /**
     * 加载文件
     */
	public function load_file() {
		\wp_enqueue_style( 'zml-visit-style', plugin_dir_url( __FILE__ ) . 'sytle.css', [], '1.0.0', 'all' );
	}
	
    /**
     * 获取会话ID
     */
    private function get_session_id() {
        if (isset($_COOKIE[$this->cookie_name])) {
            return sanitize_text_field(wp_unslash($_COOKIE[$this->cookie_name]));
        }
        
        $session_id = $this->generate_session_id();
        setcookie(
            $this->cookie_name,
            $session_id,
            time() + $this->cookie_expire,
            COOKIEPATH,
            COOKIE_DOMAIN,
            is_ssl(),
            true
        );
        
        return $session_id;
    }
    
    /**
     * 生成会话ID
     */
    private function generate_session_id() {
		if (isset($_SERVER['REMOTE_ADDR'])) {
			return md5(uniqid(wp_rand(), true) . sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) . microtime());
		} else {
			return md5(uniqid(wp_rand(), true) . microtime());
		}
        
    }
    
    /**
     * 获取用户唯一标识
     */
    private function get_user_key($session_id) {
        $ip_prefix = $this->get_ip_prefix();
        return md5($session_id . $ip_prefix);
    }
    
    /**
     * 获取IP前缀（保护隐私）
     */
    private function get_ip_prefix() {
        $ip = $this->get_client_ip();
        // 只保留IP前两段，保护用户隐私
        $parts = explode('.', $ip);
        if (count($parts) >= 2) {
            return $parts[0] . '.' . $parts[1] . '.*.*';
        }
        return 'unknown';
    }
    
    /**
     * 获取客户端IP
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (isset($_SERVER[$key])) {
                $ip = trim(sanitize_text_field(wp_unslash($_SERVER[$key])));
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '0.0.0.0';
    }
    
    /**
     * 检查是否应该跳过统计
     */
    private function should_skip_tracking() {
        // 排除条件
        $skip_conditions = array(
            is_admin(),
            wp_doing_ajax(),
            wp_doing_cron(),
            is_preview(),
            $this->is_bot(),
            current_user_can('manage_options'), // 排除管理员
            defined('DOING_AUTOSAVE') && DOING_AUTOSAVE,
        );
        
        return in_array(true, $skip_conditions, true);
    }
    
    /**
     * 检测是否为爬虫
     */
    private function is_bot() {
        if (!isset($_SERVER['HTTP_USER_AGENT'])) {
            return false;
        }
        
        $user_agent = strtolower(sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])));
        $bots = array('bot', 'spider', 'crawl', 'slurp', 'googlebot', 'bingbot', 'yandexbot');
        
        foreach ($bots as $bot) {
            if (strpos($user_agent, $bot) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * 检查用户今日是否已访问
     */
    private function has_visited_today($user_key) {
        $today_visitors = get_option('zmlvis_today_visitors', array());
        return in_array($user_key, $today_visitors, true);
    }
    
    /**
     * 标记用户今日已访问
     */
    private function mark_user_visited_today($user_key) {
        $today_visitors = get_option('zmlvis_today_visitors', array());
        $today_visitors[] = $user_key;
        update_option('zmlvis_today_visitors', $today_visitors);
    }
    
    /**
     * 获取统计数据结构
     */
    private function get_stats() {
        $default_stats = array(
            'today' => array(
                'date' => $this->today,
                'visits' => 0,
                'pageviews' => 0,
                'unique_ips' => array(),
                'last_updated' => 0
            ),
            'total' => array(
                'visits' => 0,
                'pageviews' => 0,
                'peak_day' => array(
                    'date' => '',
                    'visits' => 0
                )
            )
        );
        
        $stats = get_option('zmlvis_statistics', $default_stats);
        
        // 如果是新的一天，重置今日数据
        if ($stats['today']['date'] !== $this->today) {
            $stats['today'] = $default_stats['today'];
            $stats['today']['date'] = $this->today;
            
            // 清空今日访客记录
            delete_option('zmlvis_today_visitors');
        }
        
        return $stats;
    }
    
    /**
     * 更新最高访问日
     */
    private function update_peak_day($today_visits) {
        $stats = $this->get_stats();
        
        if ($today_visits > $stats['total']['peak_day']['visits']) {
            $stats['total']['peak_day'] = array(
                'date' => $this->today,
                'visits' => $today_visits
            );
            
            $this->save_stats($stats);
        }
    }
    
    /**
     * 保存统计数据
     */
    private function save_stats($stats) {
        update_option('zmlvis_statistics', $stats, true);
    }
    
    /**
     * 安排每日清理任务
     */
    private function schedule_cleanup() {
        if (!wp_next_scheduled('zmlvis_daily_cleanup')) {
            wp_schedule_event(strtotime('tomorrow 02:00'), 'daily', 'zmlvis_daily_cleanup');
        }
    }
    
    /**
     * 清理过期数据
     */
    public function cleanup_old_data() {
        // 清理过期的今日访客记录（保持最近3天）
        $today_visitors = get_option('zmlvis_today_visitors', array());
        if (count($today_visitors) > 10000) {
            // 如果记录过多，清理一些旧的
            delete_option('zmlvis_today_visitors');
        }
    }
    
    /**
     * 添加管理菜单
     */
    public function add_admin_menu() {
        add_menu_page(
            'Visit Statistics',
            'Visit Statistics',
            'manage_options',
            'zmlvis-statistics',
            array($this, 'display_admin_page'),
            'dashicons-chart-line',
            30
        );
    }
    
    /**
     * 显示管理页面
     */
    public function display_admin_page() {
        $stats = $this->get_stats();
        $unique_ips = count($stats['today']['unique_ips']);
        $peak_day = $stats['total']['peak_day'];
        
        ?>
        <div class="wrap">
            <h1>Visit Statistics</h1>
            
            <div class="lvs-stats-grid">
                <div class="lvs-stat-card">
                    <div class="lvs-stat-icon">📊</div>
                    <div class="lvs-stat-content">
                        <h3>Today's visits</h3>
                        <p class="lvs-stat-number"><?php echo esc_html( number_format($stats['today']['visits']) ); ?></p>
                        <p class="lvs-stat-desc">Unique visitor: <?php echo esc_html( number_format($unique_ips) ); ?></p>
                    </div>
                </div>
                
                <div class="lvs-stat-card">
                    <div class="lvs-stat-icon">👁️</div>
                    <div class="lvs-stat-content">
                        <h3>Today's views</h3>
                        <p class="lvs-stat-number"><?php echo esc_html( number_format($stats['today']['pageviews']) ); ?></p>
                        <p class="lvs-stat-desc">Per capita: <?php echo esc_html( $stats['today']['visits'] > 0 ? round($stats['today']['pageviews'] / $stats['today']['visits'], 2) : '0' ); ?> Page</p>
                    </div>
                </div>
                
                <div class="lvs-stat-card">
                    <div class="lvs-stat-icon">🏆</div>
                    <div class="lvs-stat-content">
                        <h3>Total Visits</h3>
                        <p class="lvs-stat-number"><?php echo esc_html( number_format($stats['total']['visits']) ); ?></p>
                        <p class="lvs-stat-desc">All-time high: <?php echo esc_html( number_format($peak_day['visits']) ); ?> (<?php echo esc_html( $peak_day['date'] ); ?>)</p>
                    </div>
                </div>
                
                <div class="lvs-stat-card">
                    <div class="lvs-stat-icon">📈</div>
                    <div class="lvs-stat-content">
                        <h3>Total views</h3>
                        <p class="lvs-stat-number"><?php echo esc_html( number_format($stats['total']['pageviews']) ); ?></p>
                        <p class="lvs-stat-desc">Per capita: <?php echo esc_html( $stats['total']['visits'] > 0 ? round($stats['total']['pageviews'] / $stats['total']['visits'], 2) : '0' ); ?> Page</p>
                    </div>
                </div>
            </div>
            
            <div class="lvs-actions">
                <button type="button" class="button button-secondary" onclick="location.reload()">Refresh data</button>
                <button type="button" class="button button-primary" onclick="window.print()">Print statistics</button>
            </div>
        </div>
        <?php
        
    }
    
    /**
     * 添加仪表板小工具
     */
    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'zmlvis_dashboard_widget',
            'Visit Statistics',
            array($this, 'display_dashboard_widget')
        );
    }
    
    /**
     * 显示仪表板小工具
     */
    public function display_dashboard_widget() {
        $stats = $this->get_stats();
        $unique_ips = count($stats['today']['unique_ips']);
        ?>
        <div class="lvs-dashboard-widget">
            <div class="lvs-widget-row">
                <span class="lvs-widget-label">Today's visit:</span>
                <span class="lvs-widget-value"><?php echo esc_html( number_format($stats['today']['visits']) ); ?></span>
            </div>
            <div class="lvs-widget-row">
                <span class="lvs-widget-label">Unique visitor:</span>
                <span class="lvs-widget-value"><?php echo esc_html( number_format($unique_ips) ); ?></span>
            </div>
            <div class="lvs-widget-row">
                <span class="lvs-widget-label">Today's views:</span>
                <span class="lvs-widget-value"><?php echo esc_html( number_format($stats['today']['pageviews']) ); ?></span>
            </div>
            <div class="lvs-widget-row">
                <span class="lvs-widget-label">Total Visits:</span>
                <span class="lvs-widget-value"><?php echo esc_html( number_format($stats['total']['visits']) ); ?></span>
            </div>
            
            <div class="lvs-widget-footer">
                <small>Last updated: <?php echo esc_html( gmdate('H:i:s', $stats['today']['last_updated']) ); ?></small>
                <a href="<?php echo esc_url( admin_url('admin.php?page=zmlvis-statistics') ); ?>" class="button button-small">Detailed statistics</a>
            </div>
        </div>
        <?php
    }
    
    /**
     * 获取统计数据的短代码
     */
    public function get_stats_shortcode($atts) {
        $atts = shortcode_atts(array(
            'type' => 'today_visits', // today_visits, today_pageviews, total_visits, total_pageviews
            'format' => 'number' // number, formatted
        ), $atts);
        
        $stats = $this->get_stats();
        
        switch ($atts['type']) {
            case 'today_pageviews':
                $value = $stats['today']['pageviews'];
                break;
            case 'total_visits':
                $value = $stats['total']['visits'];
                break;
            case 'total_pageviews':
                $value = $stats['total']['pageviews'];
                break;
            default:
                $value = $stats['today']['visits'];
        }
        
        if ($atts['format'] === 'formatted') {
            return number_format($value);
        }
        
        return $value;
    }
    
    /**
     * 插件激活时的操作
     */
    public static function activate() {
        // 初始化数据
        $instance = self::get_instance();
        $instance->get_stats(); // 确保数据初始化
    }
    
    /**
     * 插件停用时的操作
     */
    public static function deactivate() {
        // 清理计划任务
        wp_clear_scheduled_hook('zmlvis_daily_cleanup');
    }
}

// 初始化插件
ZMLVIS_Stats::get_instance();

// 注册短代码
add_shortcode('zmlvis_stats', array(ZMLVIS_Stats::get_instance(), 'get_stats_shortcode'));

// 注册激活/停用钩子
register_activation_hook(__FILE__, array('ZMLVIS_Stats', 'activate'));
register_deactivation_hook(__FILE__, array('ZMLVIS_Stats', 'deactivate'));