<?php
/*
Plugin Name: zmadmin - Admin Access Key
Plugin URI: https://wordpress.org/plugins/zmadmin-admin-access-key/
Description:a simple yet powerful security plugin that adds an extra layer of protection to your WordPress admin dashboard. By requiring a custom URL parameter, it prevents unauthorized users from even reaching your login page.
Version: 1.1.2
Author: zmaxcoder
Author URI: https://www.zmax99.com/
License: GPLv2 or later
Text Domain: zmadmin-admin-access-key
Domain Path: /languages
Requires at least: 5.0
Tested up to: 6.9
Requires PHP: 7.0
*/

if (!defined('ABSPATH')) exit;

define('ZMADMIN_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ZMADMIN_PLUGIN_URL', plugin_dir_url(__FILE__));

// Language files are automatically loaded by WordPress (4.6+)

// Add parameter filter hook to ensure parameters are preserved in WordPress redirects
add_filter('login_url', 'zmadmin_preserve_login_url', 10, 2);

// Add plugin settings link
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'zmadmin_add_settings_link');
function zmadmin_add_settings_link($links) {
    $settings_link = '<a href="' . admin_url('options-general.php?page=zmadmin-access-key-settings') . '">' . __('Settings', 'zmadmin-admin-access-key') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}

// Add plugin metadata links
add_filter('plugin_row_meta', 'zmadmin_plugin_row_meta', 10, 2);
function zmadmin_plugin_row_meta($links, $file) {
    if (plugin_basename(__FILE__) === $file) {
        $links[] = '<a href="https://wordpress.org/support/plugin/zmadmin/" target="_blank">' . __('Support', 'zmadmin-admin-access-key') . '</a>';
        $links[] = '<a href="https://www.zmax99.com/wp/zmadmin" target="_blank">' . __('Documentation', 'zmadmin-admin-access-key') . '</a>';
    }
    return $links;
}

// Register activation/deactivation hooks
register_activation_hook(__FILE__, 'zmadmin_activate');
register_deactivation_hook(__FILE__, 'zmadmin_deactivate');

// Main functionality initialization
add_action('plugins_loaded', 'zmadmin_init');
function zmadmin_init() {
    // Execute verification on login page
    add_action('login_init', 'zmadmin_admin_check');
    // Execute verification on admin page
    add_action('admin_init', 'zmadmin_admin_check');
    if (is_admin()) {
        add_action('admin_menu', 'zmadmin_add_settings_page');
    }
}

// Create options on activation
function zmadmin_activate() {
    // Set default configuration
    $default_config = array(
        'name' => 'who',
        'value' => ''
    );
    add_option('zmadmin_config', $default_config);
}

// Clean up data on deactivation
function zmadmin_deactivate() {
    //delete_option('zmadmin_config');
}

// Admin backend verification logic
function zmadmin_admin_check() {
    // Exclude AJAX and media library requests
    if (defined('DOING_AJAX') && DOING_AJAX) return;
    
    $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
    
    // Exclude REST API requests
    if (strpos($request_uri, '/wp-json/') === 0) return;
    
    // Exclude Ajax requests using proper WordPress URL detection
    $ajax_url = admin_url('admin-ajax.php');
    $ajax_path = parse_url($ajax_url, PHP_URL_PATH);
    if ($ajax_path && strpos($request_uri, $ajax_path) === 0) return;

    // Allow access if user is already logged in
    if (is_user_logged_in()) {
        return;
    }

    // Exclude login form submission (POST requests)
    // Note: No nonce verification needed, we are just checking for WordPress standard login form fields
    $request_method = isset($_SERVER['REQUEST_METHOD']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD'])) : '';
    if ($request_method === 'POST' && isset($_POST['log']) && isset($_POST['pwd'])) {
        return;
    }

    // Only enable verification if a key is set
    $config = get_option('zmadmin_config');
    $key_name = $config['name'] ?? 'who';
    $key_value = $config['value'] ?? '';
    if($key_value === '') {
        return; // No key set, allow access
    }
    
    // Access key verification - no nonce needed as this is first-layer access control
    // This is a public-facing check to determine if user can reach login page
    // The actual login process still requires proper WordPress authentication
    $key = isset($_GET[$key_name]) ? sanitize_text_field(wp_unslash($_GET[$key_name])) : '';
    
    // Additional security: Validate key format and length
    if (!empty($key) && !preg_match('/^[a-zA-Z0-9_-]+$/', $key)) {
        zmadmin_denial_response();
        exit;
    }
    
    // Verify key
    if (!zmadmin_validate_key($key)) {
        zmadmin_denial_response();
        exit;
    }
}

// Key verification function
function zmadmin_validate_key($input_key) {
    
    $config = get_option('zmadmin_config');
    $stored_key = $config['value'] ?? '';
    
    // Allow access if no key is set (for initial setup)
    if (empty($stored_key)) {
        return true;
    }
    
    return hash_equals($stored_key, sanitize_text_field($input_key));
}

// Preserve parameters in login URL
function zmadmin_preserve_login_url($login_url, $redirect) {
    $config = get_option('zmadmin_config');
    $key_name = $config['name'] ?? 'who';
    $key_value = $config['value'] ?? '';
    
    // Security: Only preserve parameters if they match our stored config
    // This prevents parameter injection attacks
    if (!empty($key_value) && isset($_GET[$key_name])) {
        $provided_key = isset($_GET[$key_name]) ? sanitize_text_field(wp_unslash($_GET[$key_name])) : '';
        
        // Validate that the provided key matches our stored key
        // This prevents unauthorized parameter preservation
        if (hash_equals($key_value, $provided_key)) {
            $login_url = add_query_arg($key_name, $provided_key, $login_url);
        }
    }
    
    return $login_url;
}



// Access denied response
function zmadmin_denial_response() {
    // Redirect directly to frontend
    wp_safe_redirect(home_url());
    exit;
}

// Add settings page
function zmadmin_add_settings_page() {
    add_submenu_page(
        'options-general.php',
        __('zmadmin Access Key Settings', 'zmadmin-admin-access-key'),
        __('Admin Access Key', 'zmadmin-admin-access-key'),
        'manage_options',
        'zmadmin-access-key-settings',
        'zmadmin_settings_page'
    );
}

// Settings page content
function zmadmin_settings_page() {
    // Check user permissions
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'zmadmin-admin-access-key'));
    }
    
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Admin Access Key Settings', 'zmadmin-admin-access-key'); ?></h1>
        
        <?php
        // Display settings errors
        settings_errors();
        
        // Display current configuration status
        $config = get_option('zmadmin_config');
        $key_name = $config['name'] ?? 'who';
        $key_value = $config['value'] ?? '';
        
        if (!empty($key_value)) {
            $access_url = admin_url() . '?' . $key_name . '=' . $key_value;
            echo '<div class="notice notice-success">
                <p><strong>' . esc_html__('Secure Access Link:', 'zmadmin-admin-access-key') . '</strong></p>
                <p><code>' . esc_html($access_url) . '</code></p>
                <p><em>' . esc_html__('Use this link to safely access the admin dashboard.', 'zmadmin-admin-access-key') . '</em></p>
            </div>';
        } else {
            echo '<div class="notice notice-warning">
                <p>' . esc_html__('No access key set. Anyone can access the admin dashboard. Please set a key to enhance security.', 'zmadmin-admin-access-key') . '</p>
            </div>';
        }
        ?>
        
        <form method="post" action="options.php">
            <?php
                settings_fields('zmadmin_settings_group');
                do_settings_sections('zmadmin-access-key');
                submit_button(__('Save Settings', 'zmadmin-admin-access-key'));
            ?>
        </form>
    </div>
    <?php 
}

// Register settings
add_action('admin_init', 'zmadmin_register_settings');
function zmadmin_register_settings() {
    register_setting(
        'zmadmin_settings_group',
        'zmadmin_config',
        array(
            'sanitize_callback' => 'zmadmin_sanitize_config'
        )
    );

    add_settings_section(
        'zmadmin_settings_section',
        __('Security Settings', 'zmadmin-admin-access-key'),
        '__return_empty_string',
        'zmadmin-access-key'
    );

    add_settings_field(
        'name',
        __('Parameter Name', 'zmadmin-admin-access-key'),
        'zmadmin_key_name_callback',
        'zmadmin-access-key',
        'zmadmin_settings_section'
    );

    add_settings_field(
        'value',
        __('Parameter Value', 'zmadmin-admin-access-key'),
        'zmadmin_key_value_callback',
        'zmadmin-access-key',
        'zmadmin_settings_section'
    );
}

// Parameter name callback function
function zmadmin_key_name_callback() {
    $option = get_option('zmadmin_config');
    $key_name = $option['name'] ?? '';
    echo '<input type="text" name="zmadmin_config[name]" value="' . esc_attr($key_name) . '" class="regular-text" placeholder="who" />';
    $description = sprintf(
        /* translators: %s: URL parameter name */
        __('URL parameter name, e.g: who. Access link: wp-admin?%s=value', 'zmadmin-admin-access-key'),
        $key_name
    );
    echo '<p class="description">' . esc_html($description) . '</p>';
}

// Parameter value callback function
function zmadmin_key_value_callback() {
    $option = get_option('zmadmin_config');
    $key_value = $option['value'] ?? '';
    echo '<input type="text" name="zmadmin_config[value]" value="' . esc_attr($key_value) . '" class="regular-text" placeholder="admin" />';
    echo '<p class="description">' . esc_html__('Access key value. Use complex strings for better security.', 'zmadmin-admin-access-key') . '</p>';
}

// Configuration data validation and sanitization function
function zmadmin_sanitize_config($input) {
    if (!is_array($input)) {
        return array();
    }
    
    // Verify nonce for security
    if (!isset($_POST['option_page']) || $_POST['option_page'] !== 'zmadmin_settings_group') {
        add_settings_error('zmadmin_config', 'nonce_error', __('Security verification failed. Please try again.', 'zmadmin-admin-access-key'));
        return get_option('zmadmin_config', array());
    }
    
    // Verify user permissions
    if (!current_user_can('manage_options')) {
        add_settings_error('zmadmin_config', 'permission_error', __('You do not have sufficient permissions to modify these settings.', 'zmadmin-admin-access-key'));
        return get_option('zmadmin_config', array());
    }
    
    $name = isset($input['name']) ? sanitize_text_field($input['name']) : '';
    $value = isset($input['value']) ? sanitize_text_field($input['value']) : '';
    
    // Validate parameter name - allow only alphanumeric, hyphens and underscores
    if (!empty($name) && !preg_match('/^[a-zA-Z0-9_-]+$/', $name)) {
        add_settings_error('zmadmin_config', 'invalid_name', __('Parameter name can only contain letters, numbers, hyphens and underscores.', 'zmadmin-admin-access-key'));
        return get_option('zmadmin_config', array());
    }
    
    // Validate key format and length
    if (!empty($value)) {
        if (strlen($value) < 4) {
            add_settings_error('zmadmin_config', 'short_key', __('Access key must be at least 4 characters long.', 'zmadmin-admin-access-key'));
            return get_option('zmadmin_config', array());
        }
        
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $value)) {
            add_settings_error('zmadmin_config', 'invalid_key', __('Access key can only contain letters, numbers, hyphens and underscores.', 'zmadmin-admin-access-key'));
            return get_option('zmadmin_config', array());
        }
    }
    
    return array(
        'name' => $name ?: 'who', // Default to 'who' if empty
        'value' => $value
    );
}