<?php
/**
 * Plugin Name: ZillaMetrics Chat
 * Description: This plugin enables seamless integration of ZillaMetrics Chat with your WordPress site, allowing you to engage and support your visitors directly.
 * Version:     1.0.2
 * Author:      Zillametrics
 * Author URI:  https://zillametrics.com/
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: zillametrics-chat
 *
 * @package ZillametricsChat
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define constants.
define( 'ZILLAMETRICS_PLUGIN_FILE', __FILE__ );
define( 'ZILLAMETRICS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'ZILLAMETRICS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'ZILLAMETRICS_PLUGIN_VERSION', '1.0.1' );

// Include required classes.
require_once ZILLAMETRICS_PLUGIN_DIR . 'includes/class-zillametrics-settings.php';
require_once ZILLAMETRICS_PLUGIN_DIR . 'admin/class-zillametrics-admin.php';

/**
 * Initialize the plugin.
 */
function zillametrics_bot_integration_init() {
    // Only run in admin.
    if ( is_admin() ) {
        new Zillametrics_Admin();
    }
}
add_action( 'plugins_loaded', 'zillametrics_bot_integration_init' );

/**
 * Check if current request is a valid preview request with proper nonce verification.
 *
 * @return array|false Array with bot_id if valid preview, false otherwise.
 */
function zillametrics_is_valid_preview() {
    // First check if we have the required parameters
    if ( ! isset( $_GET['zillametrics_preview'] ) || $_GET['zillametrics_preview'] !== 'true' ) {
        return false;
    }
    
    if ( ! isset( $_GET['bot_id'] ) || ! isset( $_GET['zm_nonce'] ) ) {
        return false;
    }
    
    // Sanitize the bot_id first
    $bot_id = sanitize_text_field( wp_unslash( $_GET['bot_id'] ) );
    
    // Verify nonce for security
    $nonce = sanitize_text_field( wp_unslash( $_GET['zm_nonce'] ) );
    if ( ! wp_verify_nonce( $nonce, 'zillametrics_preview_' . $bot_id ) ) {
        return false;
    }
    
    return array( 'bot_id' => $bot_id );
}

/**
 * Load bot script on frontend if enabled.
 */
function zillametrics_load_bot_script() {
    // Only load on frontend
    if ( is_admin() ) {
        return;
    }
    
    // Check if it's a valid preview
    $preview_data = zillametrics_is_valid_preview();
    $is_preview = ( $preview_data !== false );

    // Check if bot should be displayed
    if ( ! $is_preview && ! Zillametrics_Settings::should_display_bot() ) {
        return;
    }
    
    // Enqueue scripts and styles properly
    add_action( 'wp_enqueue_scripts', 'zillametrics_enqueue_bot_assets' );
}
add_action( 'wp', 'zillametrics_load_bot_script' );

/**
 * Enqueue bot scripts and styles properly.
 */
function zillametrics_enqueue_bot_assets() {
    // Enqueue external scripts
    wp_enqueue_script( 
        'zillametrics-chat-widget', 
        'https://cdn.zillametrics.com/chat-widget.js', 
        array(), 
        ZILLAMETRICS_PLUGIN_VERSION, 
        true 
    );
    
    wp_enqueue_script( 
        'zillametrics-widget-bundle', 
        'https://cdn.zillametrics.com/widget-bundle.js', 
        array( 'zillametrics-chat-widget' ), 
        ZILLAMETRICS_PLUGIN_VERSION, 
        true 
    );
    
    // Enqueue external styles
    wp_enqueue_style( 
        'zillametrics-chat-widget-css', 
        'https://cdn.zillametrics.com/chat-widget.css', 
        array(), 
        ZILLAMETRICS_PLUGIN_VERSION 
    );
    
    // Add inline script for bot initialization
    add_action( 'wp_footer', 'zillametrics_output_bot_initialization' );
}

/**
 * Output the bot initialization script.
 */
function zillametrics_output_bot_initialization() {
    $init_config = array();

    // Check for valid preview mode (with proper nonce verification)
    $preview_data = zillametrics_is_valid_preview();

    if ( $preview_data !== false ) {
        $init_config['license'] = 'Preview';
        $init_config['botId']   = $preview_data['bot_id'];
    } else {
        $license_key = Zillametrics_Settings::get( 'license_key' );

        if ( empty( $license_key ) ) {
            return;
        }

        $init_config['license'] = $license_key;
    }

    if ( empty( $init_config ) ) {
        return;
    }

    // Use wp_add_inline_script to properly enqueue the initialization code
    $inline_script = "window.addEventListener('load', function() {
        if (window.ZillaChat) {
            window.ZillaChat.init(" . wp_json_encode( $init_config ) . ");
        }
    });";
    
    wp_add_inline_script( 'zillametrics-widget-bundle', $inline_script );
}

/**
 * Generate a secure preview URL with nonce.
 *
 * @param string $bot_id The bot ID for preview.
 * @param string $url    The base URL to add parameters to.
 * @return string The secure preview URL.
 */
function zillametrics_generate_preview_url( $bot_id, $url = '' ) {
    if ( empty( $url ) ) {
        // Safely get the current request URI with proper validation and sanitization
        if ( isset( $_SERVER['REQUEST_URI'] ) ) {
            $request_uri = sanitize_url( wp_unslash( $_SERVER['REQUEST_URI'] ) );
            $url = home_url( $request_uri );
        } else {
            // Fallback to home URL if REQUEST_URI is not available
            $url = home_url();
        }
    }
    
    $nonce = wp_create_nonce( 'zillametrics_preview_' . $bot_id );
    
    return add_query_arg( array(
        'zillametrics_preview' => 'true',
        'bot_id'               => $bot_id,
        'zm_nonce'             => $nonce,
    ), $url );
} 
