<?php
/**
 * Settings management for Zillametrics chat.
 *
 * @package ZillametricsChat
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'Zillametrics_Settings' ) ) :

class Zillametrics_Settings {

    /**
     * Option prefix for all settings.
     */
    const OPTION_PREFIX = 'zillametrics_';

    /**
     * Default settings values.
     */
    private static $defaults = array(
        // License settings
        'license_key' => '',
        'license_status' => '',
        'license_data' => array(),
        
        // Preview settings
        'preview_selected_bot' => '',
        'preview_selected_page' => array(
            'id' => '',
            'title' => '',
            'url' => '',
            'type' => ''
        ),
    );

    /**
     * Get a setting value.
     *
     * @param string $key Setting key.
     * @param mixed  $default Default value if setting doesn't exist.
     * @return mixed Setting value.
     */
    public static function get( $key, $default = null ) {
        $option_name = self::OPTION_PREFIX . $key;
        $value = get_option( $option_name, null );
        
        // Return default if value is null and default is provided
        if ( $value === null && $default !== null ) {
            return $default;
        }
        
        // Return default from defaults array if value is null
        if ( $value === null && isset( self::$defaults[ $key ] ) ) {
            return self::$defaults[ $key ];
        }
        
        return $value;
    }

    /**
     * Set a setting value.
     *
     * @param string $key Setting key.
     * @param mixed  $value Setting value.
     * @return bool True if option was updated, false otherwise.
     */
    public static function set( $key, $value ) {
        $option_name = self::OPTION_PREFIX . $key;
        return update_option( $option_name, $value );
    }

    /**
     * Get multiple settings at once.
     *
     * @param array $keys Array of setting keys.
     * @return array Array of setting values.
     */
    public static function get_multiple( $keys ) {
        $settings = array();
        foreach ( $keys as $key ) {
            $settings[ $key ] = self::get( $key );
        }
        return $settings;
    }

    /**
     * Set multiple settings at once.
     *
     * @param array $settings Array of key => value pairs.
     * @return bool True if all settings were updated successfully.
     */
    public static function set_multiple( $settings ) {
        $success = true;
        foreach ( $settings as $key => $value ) {
            if ( ! self::set( $key, $value ) ) {
                $success = false;
            }
        }
        return $success;
    }

    /**
     * Delete a setting.
     *
     * @param string $key Setting key.
     * @return bool True if option was deleted, false otherwise.
     */
    public static function delete( $key ) {
        $option_name = self::OPTION_PREFIX . $key;
        return delete_option( $option_name );
    }

    /**
     * Check if bot should be displayed.
     *
     * @return bool True if bot should be displayed.
     */
    public static function should_display_bot() {
        // Check license status
        if ( self::get( 'license_status' ) !== 'valid' ) {
            return false;
        }

        return true;
    }

    /**
     * Get bot script URL from license data.
     *
     * @return string Bot script URL.
     */
    public static function get_script_url() {
        $license_data = self::get( 'license_data' );
        if ( isset( $license_data['script_url'] ) ) {
            return $license_data['script_url'];
        }
        
        // Default script URL if not in license data
        return 'https://cdn.zillametrics.com/bot.js';
    }

    /**
     * Get available bots from license data.
     *
     * @return array Array of available bots.
     */
    public static function get_available_bots() {
        $license_key = self::get( 'license_key' );
        if ( empty( $license_key ) ) {
            return array();
        }

        $cached_bots = get_transient( 'zillametrics_available_bots' );
        if ( false !== $cached_bots ) {
            return $cached_bots;
        }

        $api_url = sprintf(
            'https://api.zillametrics.com/wp-json/zillametrics/v1/chatbot/find-bots?license_key=%s',
            esc_attr( $license_key )
        );

        $response = wp_remote_get( $api_url );

        if ( is_wp_error( $response ) ) {
            return array();
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! $data || ! isset( $data['success'] ) || ! $data['success'] || empty( $data['bots'] ) ) {
            return array();
        }

        set_transient( 'zillametrics_available_bots', $data['bots'], HOUR_IN_SECONDS );

        return $data['bots'];
    }

    /**
     * Get selected bot for preview.
     *
     * @return string Selected bot ID.
     */
    public static function get_selected_bot() {
        return self::get( 'preview_selected_bot', '' );
    }

    /**
     * Get selected page for preview.
     *
     * @return array Selected page data.
     */
    public static function get_selected_page() {
        return self::get( 'preview_selected_page', array() );
    }

    /**
     * Save preview settings.
     *
     * @param array $settings Preview settings.
     * @return bool True if settings were saved successfully.
     */
    public static function save_preview_settings( $settings ) {
        $validated = self::validate_preview_settings( $settings );
        return self::set_multiple( $validated );
    }

    /**
     * Validate preview settings.
     *
     * @param array $settings Settings to validate.
     * @return array Validated settings.
     */
    private static function validate_preview_settings( $settings ) {
        $validated = array();
        
        foreach ( $settings as $key => $value ) {
            switch ( $key ) {
                case 'preview_selected_bot':
                    $validated[ $key ] = sanitize_text_field( $value );
                    break;
                    
                case 'preview_selected_page':
                    if ( is_array( $value ) ) {
                        $validated[ $key ] = array(
                            'id' => sanitize_text_field( $value['id'] ?? '' ),
                            'title' => sanitize_text_field( $value['title'] ?? '' ),
                            'url' => esc_url_raw( $value['url'] ?? '' ),
                            'type' => sanitize_text_field( $value['type'] ?? '' )
                        );
                    }
                    break;
            }
        }
        
        return $validated;
    }
}

endif;
