// ZillaMetrics Admin JavaScript

function zillaMetricsOpenPreview(botId, page, triggerBtn) {
  if (!botId || !page || !page.url) return;
  if (triggerBtn) triggerBtn.disabled = true;
  var formData = new FormData();
  formData.append('action', 'zillametrics_generate_preview_url');
  formData.append('nonce', ZillaMetricsAdmin.nonce);
  formData.append('bot_id', botId);
  formData.append('page_url', page.url);
  fetch(ZillaMetricsAdmin.ajaxUrl, { method: 'POST', body: formData })
    .then(response => response.json())
    .then(data => {
      if (data.success && data.data && data.data.preview_url) {
        window.open(data.data.preview_url, '_blank');
      } else if (data.data && data.data.message) {
        alert(data.data.message);
      } else if (data.message) {
        alert(data.message);
      } else {
        alert('Unable to generate preview link. Please try again.');
      }
    })
    .catch(function() {
      alert('Unable to generate preview link. Please try again.');
    })
    .then(function() {
      if (triggerBtn) triggerBtn.disabled = false;
    });
}

(function($){
  $(function(){
    // Handle license form submission
    $('#zillametrics-license-form').on('submit', function(e) {
      e.preventDefault();
      
      const $form = $(this);
      const $button = $('#zillametrics-save-license');
      const $buttonText = $button.find('.button-text');
      const $spinner = $button.find('.spinner');
      const $messageContainer = $('#zillametrics-message-container');
      const $statusContainer = $('#zillametrics-license-status');
      const licenseKey = $('#zillametrics_license').val().trim();

      // Validate license key
      if (!licenseKey) {
        showMessage('error', 'Please enter a license key.');
        return;
      }

      // Show loading state
      $button.prop('disabled', true);
      $buttonText.text('Authenticating...');
      $spinner.show();
      $messageContainer.empty();

      // Make AJAX request
      $.ajax({
        url: ZillaMetricsAdmin.ajaxUrl,
        type: 'POST',
        data: {
          action: 'zillametrics_authenticate_license',
          license_key: licenseKey,
          nonce: ZillaMetricsAdmin.nonce
        },
        success: function(response) {
          if (response.success) {
            showMessage('success', response.data.message);
            updateLicenseStatus('valid', response.data.data);
            // Notify other scripts (e.g., combined settings view) to reveal preview section
            document.dispatchEvent(new CustomEvent('zillametrics:licenseStatusChanged', {
              detail: { status: 'valid', data: response.data.data || {} }
            }));
          } else {
            showMessage('error', response.data.message);
            updateLicenseStatus('invalid');
            document.dispatchEvent(new CustomEvent('zillametrics:licenseStatusChanged', {
              detail: { status: 'invalid' }
            }));
          }
        },
        error: function(xhr, status, error) {
          showMessage('error', 'An error occurred while authenticating the license. Please try again.');
          updateLicenseStatus('invalid');
          document.dispatchEvent(new CustomEvent('zillametrics:licenseStatusChanged', {
            detail: { status: 'invalid' }
          }));
        },
        complete: function() {
          // Reset button state
          $button.prop('disabled', false);
          $buttonText.text('Save License');
          $spinner.hide();
        }
      });
    });

    /**
     * Show success or error message
     */
    function showMessage(type, message) {
      const $messageContainer = $('#zillametrics-message-container');
      const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
      
      const messageHtml = `
        <div class="notice ${noticeClass} is-dismissible">
          <p><strong>${message}</strong></p>
          <button type="button" class="notice-dismiss">
            <span class="screen-reader-text">Dismiss this notice.</span>
          </button>
        </div>
      `;
      
      $messageContainer.html(messageHtml);
      
      // Auto-dismiss success messages after 5 seconds
      if (type === 'success') {
        setTimeout(function() {
          $messageContainer.find('.notice').fadeOut();
        }, 5000);
      }
    }

    /**
     * Update license status display
     */
    function updateLicenseStatus(status, data) {
      const $statusContainer = $('#zillametrics-license-status');
      
      if (status === 'valid' && data) {
        let statusHtml = `
          <div class="notice notice-success inline">
            <p><strong>License Status: Active</strong></p>
        `;
        
        if (data.plan) {
          statusHtml += `<p>Plan: ${data.plan}</p>`;
        }
        
        if (data.expires_at) {
          const expiryDate = new Date(data.expires_at);
          statusHtml += `<p>Expires: ${expiryDate.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric' 
          })}</p>`;
        }
        
        statusHtml += '</div>';
        $statusContainer.html(statusHtml);
      } else if (status === 'invalid') {
        $statusContainer.html(`
          <div class="notice notice-error inline">
            <p><strong>License Status: Invalid</strong></p>
          </div>
        `);
      } else {
        $statusContainer.empty();
      }
    }

    // Handle notice dismiss buttons
    $(document).on('click', '.notice-dismiss', function() {
      $(this).closest('.notice').fadeOut();
    });
  });
})(jQuery);

// License Page JavaScript
document.addEventListener('DOMContentLoaded', function() {
  // Reveal settings when clicking the CTA
  var enterBtn = document.getElementById('enter-license-btn');
  var wrapper = document.querySelector('.wrap.zillametrics-admin.zm-lic');
  if (enterBtn) {
    enterBtn.addEventListener('click', function(e){
      e.preventDefault();
      var settings = document.getElementById('zillametrics-settings-section');
      var about = document.getElementById('zillametrics-about-wrapper');
      if (settings) settings.style.display = '';
      if (about) about.style.display = 'none';
      if (wrapper) wrapper.classList.remove('zm-welcome-active');
      // Ensure deactivate button hidden when entering settings from welcome
      var deBtnOnEnter = document.getElementById('zillametrics-deactivate-license');
      if (deBtnOnEnter) deBtnOnEnter.style.display = 'none';
      var input = document.getElementById('zillametrics_license');
      if (input) {
        input.focus();
        // Smooth scroll to settings
        document.getElementById('zillametrics-settings-section').scrollIntoView({ behavior: 'smooth', block: 'start' });
      }
    });
  }

  // Tabs
  const tabButtons = document.querySelectorAll('.zm-tab-btn');
  const tabContents = document.querySelectorAll('.zm-tab-content-inner');
  tabButtons.forEach((btn, idx) => {
    btn.addEventListener('click', () => {
      tabButtons.forEach(b => b.classList.remove('active'));
      tabContents.forEach(c => c.classList.remove('active'));
      btn.classList.add('active');
      tabContents[idx].classList.add('active');
    });
  });

  // Toggle sections when license status changes (no reload)
  document.addEventListener('zillametrics:licenseStatusChanged', function(e){
    var preview = document.getElementById('zillametrics-preview-wrapper');
    var about = document.getElementById('zillametrics-about-wrapper');
    var settings = document.getElementById('zillametrics-settings-section');
    var wrapper = document.querySelector('.wrap.zillametrics-admin.zm-lic');
    if (e.detail && e.detail.status === 'valid') {
      if (settings) settings.style.display = '';
      if (preview) preview.style.display = '';
      if (about) about.style.display = 'none';
      if (wrapper) wrapper.classList.remove('zm-welcome-active');
      // Reveal deactivate button once valid
      var deBtn = document.getElementById('zillametrics-deactivate-license');
      if (deBtn) deBtn.style.display = '';
    } else {
      if (preview) preview.style.display = 'none';
      if (about) about.style.display = '';
      // Keep settings hidden until user clicks CTA when invalid
      if (settings) settings.style.display = 'none';
      if (wrapper) wrapper.classList.add('zm-welcome-active');
      // Hide deactivate button when license is invalid
      var deBtnInvalid = document.getElementById('zillametrics-deactivate-license');
      if (deBtnInvalid) deBtnInvalid.style.display = 'none';
    }
  });

  // Deactivate license handler (AJAX)
  function bindDeactivate() {
    var deBtn = document.getElementById('zillametrics-deactivate-license');
    if (!deBtn) return;
    deBtn.addEventListener('click', function(e){
      e.preventDefault();
      if (!confirm('Are you sure you want to deactivate the license?')) return;
      var params = new URLSearchParams({ action: 'zillametrics_deactivate_license', nonce: ZillaMetricsAdmin.nonce });
      fetch(ZillaMetricsAdmin.ajaxUrl, { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: params })
        .then(r => r.json())
        .then(data => {
          // Broadcast status change so sections toggle instantly
          document.dispatchEvent(new CustomEvent('zillametrics:licenseStatusChanged', { detail: { status: 'invalid' } }));
          // Clear input value
          var input = document.getElementById('zillametrics_license');
          if (input) input.value = '';
          // Update status banner (clear, don't show Invalid when no key)
          var statusBox = document.getElementById('zillametrics-license-status');
          if (statusBox) {
            statusBox.innerHTML = '';
          }
        })
        .catch(() => alert('Failed to deactivate license.'));
    });
  }
  bindDeactivate();

  // From preview: dropdowns and saving
  // Get initial data from wp_localize_script
  var selectedBotId = ZillaMetricsAdmin.selectedBotId || '';
  var selectedPage = ZillaMetricsAdmin.selectedPage || {};
  
  // Only initialize dropdowns if they exist on the page
  if (document.getElementById('bot-dropdown')) {
    var botDropdown = document.getElementById('bot-dropdown');
    var botDropdownBtn = document.getElementById('bot-dropdown-btn');
    var botDropdownMenu = document.getElementById('bot-dropdown-menu');
    var botBtnText = botDropdownBtn ? botDropdownBtn.querySelector('.btn-text') : null;
    if (botDropdown && botDropdownBtn && botDropdownMenu) {
      botDropdownBtn.addEventListener('click', function(e){ e.preventDefault(); botDropdown.classList.toggle('open'); });
      botDropdownMenu.querySelectorAll('.bot-item').forEach(item => {
        item.addEventListener('click', function(e){
          e.preventDefault();
          selectedBotId = this.dataset.botId;
          if (botBtnText) botBtnText.textContent = this.dataset.botName || 'Select Bot';
          botDropdown.classList.remove('open');
          savePreviewSettings({ selected_bot: selectedBotId });
          updatePreviewButtonState();
        });
      });
    }

    var pagesDropdown = document.getElementById('pages-dropdown');
    var pagesDropdownBtn = document.getElementById('pages-dropdown-btn');
    var pagesDropdownMenu = document.getElementById('pages-dropdown-menu');
    var pagesBtnText = pagesDropdownBtn ? pagesDropdownBtn.querySelector('.btn-text') : null;
    var loadingSpinner = pagesDropdownBtn ? pagesDropdownBtn.querySelector('.loading-spinner') : null;
    var loadingItem = pagesDropdownMenu ? pagesDropdownMenu.querySelector('.loading-item') : null;
    var noPagesItem = pagesDropdownMenu ? pagesDropdownMenu.querySelector('.no-pages-item') : null;
    var pagesLoaded = false;

    if (pagesDropdown && pagesDropdownBtn && pagesDropdownMenu) {
      pagesDropdownBtn.addEventListener('click', function(e){
        e.preventDefault(); e.stopPropagation();
        pagesDropdown.classList.toggle('open');
        if (!pagesLoaded) loadPages();
      });
      pagesDropdownMenu.addEventListener('click', function(e){
        const item = e.target.closest('.dropdown-item');
        if (!item || item.classList.contains('loading-item') || item.classList.contains('no-pages-item')) return;
        const page = { id: item.dataset.pageId || '', title: item.dataset.pageTitle || item.textContent, url: item.dataset.pageUrl || '', type: item.dataset.pageType || '' };
        selectedPage = page;
        if (pagesBtnText) pagesBtnText.textContent = page.title || 'Select Page';
        pagesDropdown.classList.remove('open');
        savePreviewSettings({ selected_page: page });
        updatePreviewButtonState();
      });
    }

    document.addEventListener('click', function(e){
      if (botDropdown && !botDropdown.contains(e.target)) botDropdown.classList.remove('open');
      if (pagesDropdown && !pagesDropdown.contains(e.target)) pagesDropdown.classList.remove('open');
    });

    var previewBtn = document.getElementById('view-preview-btn');

    function updatePreviewButtonState() {
      if (!previewBtn) return;
      var enabled = !!(selectedBotId && selectedPage && selectedPage.url);
      previewBtn.disabled = !enabled;
    }
    updatePreviewButtonState();

    if (previewBtn) {
      previewBtn.addEventListener('click', function(e){
        e.preventDefault();
        if (!(selectedBotId && selectedPage && selectedPage.url)) return;
        zillaMetricsOpenPreview(selectedBotId, selectedPage, previewBtn);
      });
    }

    function loadPages() {
      if (!loadingSpinner || !loadingItem || !pagesDropdownMenu || !pagesBtnText) return;
      loadingSpinner.style.display = 'inline';
      pagesBtnText.textContent = 'Loading...';
      loadingItem.style.display = 'block';
      pagesDropdownMenu.style.display = 'block';
      fetch(ZillaMetricsAdmin.ajaxUrl, { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: new URLSearchParams({ action: 'zillametrics_get_pages', nonce: ZillaMetricsAdmin.nonce }) })
      .then(r => r.json())
      .then(data => {
        loadingSpinner.style.display = 'none';
        loadingItem.style.display = 'none';
        if (data.success && data.data.pages.length > 0) {
          pagesDropdownMenu.innerHTML = '';
          data.data.pages.forEach(page => {
            var item = document.createElement('div');
            item.className = 'dropdown-item';
            item.textContent = page.title;
            item.dataset.pageId = page.id;
            item.dataset.pageUrl = page.url;
            item.dataset.pageTitle = page.title;
            item.dataset.pageType = page.type;
            pagesDropdownMenu.appendChild(item);
          });
          if (selectedPage && selectedPage.title) { pagesBtnText.textContent = selectedPage.title; } else { pagesBtnText.textContent = 'Select Page'; }
          pagesLoaded = true;
        } else {
          pagesDropdownMenu.innerHTML = '';
          if (noPagesItem) noPagesItem.style.display = 'block';
          pagesBtnText.textContent = 'No Pages Found';
        }
      })
      .catch(() => {
        loadingSpinner.style.display = 'none';
        loadingItem.style.display = 'none';
        pagesBtnText.textContent = 'Error Loading Pages';
        if (pagesDropdownMenu) pagesDropdownMenu.innerHTML = '<div class="dropdown-item" style="color: red;">Error loading pages</div>';
      });
    }

    function savePreviewSettings(settings) {
      var formData = new FormData();
      formData.append('action', 'zillametrics_save_preview_settings');
      formData.append('nonce', ZillaMetricsAdmin.nonce);
      if (settings.selected_bot) formData.append('selected_bot', settings.selected_bot);
      if (settings.selected_page) {
        formData.append('selected_page[id]', settings.selected_page.id || '');
        formData.append('selected_page[title]', settings.selected_page.title || '');
        formData.append('selected_page[url]', settings.selected_page.url || '');
        formData.append('selected_page[type]', settings.selected_page.type || '');
      }
      fetch(ZillaMetricsAdmin.ajaxUrl, { method: 'POST', body: formData })
        .then(r => r.json())
        .then(data => {
          var container = document.getElementById('zillametrics-preview-message-container');
          if (!container) return;
          var noticeClass = data.success ? 'notice-success' : 'notice-error';
          var message = (data.data && data.data.message) ? data.data.message : 'Saved.';
          container.innerHTML = '<div class="notice ' + noticeClass + ' is-dismissible"><p><strong>' + message + '</strong></p></div>';
          if (data.success) setTimeout(function() { container.innerHTML = ''; }, 3000);
        })
        .catch(() => {
          var container = document.getElementById('zillametrics-preview-message-container');
          if (container) container.innerHTML = '<div class="notice notice-error is-dismissible"><p><strong>Error saving settings.</strong></p></div>';
        });
    }
  }
});

// Preview Page JavaScript (for separate preview page if exists)
document.addEventListener("DOMContentLoaded", function() {
  // Only run if we're on the preview page (not the combined license page)
  if (!document.querySelector('.zm-preview') || document.querySelector('.zm-lic')) {
    return; // Exit if not on standalone preview page or on combined page
  }

  // ===============================
  // Tab Functionality
  // ===============================
  var previewPageTabButtons = document.querySelectorAll(".zm-preview .zm-tab-btn");
  var previewPageTabContents = document.querySelectorAll(".zm-preview .zm-tab-content-inner");

  previewPageTabButtons.forEach(function(btn, index) {
    btn.addEventListener('click', function() {
      // Remove active from all
      previewPageTabButtons.forEach(function(b) { b.classList.remove("active"); });
      previewPageTabContents.forEach(function(c) { c.classList.remove("active"); });

      // Close all dropdowns when switching tabs
      document.querySelectorAll(".zm-preview .dropdown.open").forEach(function(dd) {
        dd.classList.remove("open");
      });

      // Activate clicked tab + content
      btn.classList.add("active");
      previewPageTabContents[index].classList.add("active");
    });
  });

  // ===============================
  // Bot Selection Dropdown
  // ===============================
  var previewBotDropdown = document.querySelector('.zm-preview #bot-dropdown');
  var previewBotDropdownBtn = document.querySelector('.zm-preview #bot-dropdown-btn');
  var previewBotDropdownMenu = document.querySelector('.zm-preview #bot-dropdown-menu');
  var previewBotBtnText = null;
  if (previewBotDropdownBtn) {
    previewBotBtnText = previewBotDropdownBtn.querySelector('.btn-text');
  }
  // Use localized data
  var previewSelectedBotId = ZillaMetricsAdmin.selectedBotId || '';
  var previewSelectedPage = ZillaMetricsAdmin.selectedPage || {};

  if (previewBotDropdown) {
    // Handle bot dropdown button click
    previewBotDropdownBtn.addEventListener('click', function(e) {
      e.preventDefault();
      e.stopPropagation();
      
      if (previewBotDropdown.classList.contains('open')) {
        closeBotDropdown();
      } else {
        openBotDropdown();
      }
    });

    // Select bot item
    previewBotDropdownMenu.querySelectorAll('.bot-item').forEach(item => {
      item.addEventListener('click', function(e) {
        e.preventDefault();
        const botId = this.dataset.botId;
        const botName = this.dataset.botName;
        previewSelectedBotId = botId;
        previewBotBtnText.textContent = botName || 'Select Bot';
        closeBotDropdown();
        savePreviewPageSettings({ selected_bot: botId });
        updatePreviewPageButtonState();
      });
    });
  }

  // ===============================
  // Page Selection Dropdown
  // ===============================
  var previewPagesDropdown = document.querySelector('.zm-preview #pages-dropdown');
  var previewPagesDropdownBtn = document.querySelector('.zm-preview #pages-dropdown-btn');
  var previewPagesDropdownMenu = document.querySelector('.zm-preview #pages-dropdown-menu');
  var previewPagesBtnText = null;
  var previewLoadingSpinner = null;
  var previewLoadingItem = null;
  var previewNoPagesItem = null;
  if (previewPagesDropdownBtn) {
    previewPagesBtnText = previewPagesDropdownBtn.querySelector('.btn-text');
    previewLoadingSpinner = previewPagesDropdownBtn.querySelector('.loading-spinner');
  }
  if (previewPagesDropdownMenu) {
    previewLoadingItem = previewPagesDropdownMenu.querySelector('.loading-item');
    previewNoPagesItem = previewPagesDropdownMenu.querySelector('.no-pages-item');
  }

  var previewPagesLoaded = false;

  if (previewPagesDropdown) {
    // Handle pages dropdown button click
    previewPagesDropdownBtn.addEventListener('click', function(e) {
      e.preventDefault();
      e.stopPropagation();
      
      if (previewPagesDropdown.classList.contains('open')) {
        closePagesDropdown();
      } else {
        openPagesDropdown();
      }
    });

    // Delegate click for items (in case of dynamic load)
    previewPagesDropdownMenu.addEventListener('click', function(e) {
      const item = e.target.closest('.dropdown-item');
      if (!item || item.classList.contains('loading-item') || item.classList.contains('no-pages-item')) return;
      const page = {
        id: item.dataset.pageId || '',
        title: item.dataset.pageTitle || item.textContent,
        url: item.dataset.pageUrl || '',
        type: item.dataset.pageType || ''
      };
      previewSelectedPage = page;
      previewPagesBtnText.textContent = page.title || 'Select Page';
      closePagesDropdown();
      savePreviewPageSettings({ selected_page: page });
      updatePreviewPageButtonState();
    });
  }

  // ===============================
  // Global Click Handler
  // ===============================
  document.addEventListener('click', function(e) {
    if (previewBotDropdown && !previewBotDropdown.contains(e.target)) {
      closeBotDropdown();
    }
    if (previewPagesDropdown && !previewPagesDropdown.contains(e.target)) {
      closePagesDropdown();
    }
  });

  // ===============================
  // Helper Functions
  // ===============================
  function openBotDropdown() {
    previewBotDropdown.classList.add('open');
  }

  function closeBotDropdown() {
    previewBotDropdown.classList.remove('open');
  }

  function openPagesDropdown() {
    previewPagesDropdown.classList.add('open');
    
    // Load pages if not already loaded
    if (!previewPagesLoaded) {
      loadPreviewPages();
    }
  }

  function closePagesDropdown() {
    previewPagesDropdown.classList.remove('open');
  }

  // Build preview URL and open new tab
  var previewPageBtn = document.querySelector('.zm-preview #view-preview-btn');
  function updatePreviewPageButtonState() {
    if (!previewPageBtn) return;
    var enabled = !!(previewSelectedBotId && previewSelectedPage && previewSelectedPage.url);
    previewPageBtn.disabled = !enabled;
  }
  if (previewPageBtn) {
    updatePreviewPageButtonState();
  }

  if (previewPageBtn) {
    previewPageBtn.addEventListener('click', function(e) {
      e.preventDefault();
      if (!(previewSelectedBotId && previewSelectedPage && previewSelectedPage.url)) return;
      zillaMetricsOpenPreview(previewSelectedBotId, previewSelectedPage, previewPageBtn);
    });
  }

  function loadPreviewPages() {
    // Show loading state
    previewLoadingSpinner.style.display = 'inline';
    previewPagesBtnText.textContent = 'Loading...';
    previewLoadingItem.style.display = 'block';
    previewPagesDropdownMenu.style.display = 'block';

    // Make AJAX request to fetch pages
    fetch(ZillaMetricsAdmin.ajaxUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      body: new URLSearchParams({
        action: 'zillametrics_get_pages',
        nonce: ZillaMetricsAdmin.nonce
      })
    })
    .then(response => response.json())
    .then(data => {
      previewLoadingSpinner.style.display = 'none';
      previewLoadingItem.style.display = 'none';

      if (data.success && data.data.pages.length > 0) {
        // Clear existing items
        previewPagesDropdownMenu.innerHTML = '';
        
        // Add pages to dropdown
        data.data.pages.forEach(function(page) {
          var item = document.createElement('div');
          item.className = 'dropdown-item';
          item.textContent = page.title;
          item.dataset.pageId = page.id;
          item.dataset.pageUrl = page.url;
          item.dataset.pageTitle = page.title;
          item.dataset.pageType = page.type;
          
          previewPagesDropdownMenu.appendChild(item);
        });
        
        // Keep existing selection if any, else default label
        if (previewSelectedPage && previewSelectedPage.title) {
          previewPagesBtnText.textContent = previewSelectedPage.title;
        } else {
          previewPagesBtnText.textContent = 'Select Page';
        }
        previewPagesLoaded = true;
      } else {
        // No pages found
        previewPagesDropdownMenu.innerHTML = '';
        previewNoPagesItem.style.display = 'block';
        previewPagesBtnText.textContent = 'No Pages Found';
      }
    })
    .catch(error => {
      console.error('Error loading pages:', error);
      previewLoadingSpinner.style.display = 'none';
      previewLoadingItem.style.display = 'none';
      previewPagesBtnText.textContent = 'Error Loading Pages';
      
      // Show error message
      previewPagesDropdownMenu.innerHTML = '<div class="dropdown-item" style="color: red;">Error loading pages</div>';
    });
  }

  function savePreviewPageSettings(settings) {
    const formData = new FormData();
    formData.append('action', 'zillametrics_save_preview_settings');
    formData.append('nonce', ZillaMetricsAdmin.nonce);
    
    if (settings.selected_bot) {
      formData.append('selected_bot', settings.selected_bot);
    }
    
    if (settings.selected_page) {
      formData.append('selected_page[id]', settings.selected_page.id);
      formData.append('selected_page[title]', settings.selected_page.title);
      formData.append('selected_page[url]', settings.selected_page.url);
      formData.append('selected_page[type]', settings.selected_page.type);
    }
    
    fetch(ZillaMetricsAdmin.ajaxUrl, {
      method: 'POST',
      body: formData
    })
    .then(response => response.json())
    .then(data => {
      if (data.success) {
        showPreviewMessage('success', data.data.message);
      } else {
        showPreviewMessage('error', data.data.message);
      }
    })
    .catch(error => {
      console.error('Error saving preview settings:', error);
      showPreviewMessage('error', 'An error occurred while saving settings. Please try again.');
    });
  }

  function showPreviewMessage(type, message) {
    const messageContainer = document.getElementById('zillametrics-preview-message-container');
    if (!messageContainer) return;
    const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
    
    const messageHtml = '<div class="notice ' + noticeClass + ' is-dismissible"><p><strong>' + message + '</strong></p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button></div>';
    messageContainer.innerHTML = messageHtml;
    
    // Auto-dismiss success messages after 3 seconds
    if (type === 'success') {
      setTimeout(function() {
        const notice = messageContainer.querySelector('.notice');
        if (notice) {
          notice.style.display = 'none';
        }
      }, 3000);
    }
  }
  
  // Handle notice dismiss buttons
  document.addEventListener('click', function(e) {
    if (e.target.closest('.notice-dismiss')) {
      e.target.closest('.notice').style.display = 'none';
    }
  });
});
