<?php
/**
 * Admin functionality for Zillametrics chat.
 *
 * @package ZillametricsChat
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'Zillametrics_Admin' ) ) :

class Zillametrics_Admin {

    /**
     * Constructor.
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'register_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        // Add AJAX handlers for license authentication
        add_action( 'wp_ajax_zillametrics_authenticate_license', array( $this, 'ajax_authenticate_license' ) );
        // Add AJAX handler for fetching pages
        add_action( 'wp_ajax_zillametrics_get_pages', array( $this, 'ajax_get_pages' ) );
        // Add AJAX handlers for preview settings
        add_action( 'wp_ajax_zillametrics_save_preview_settings', array( $this, 'ajax_save_preview_settings' ) );
        add_action( 'wp_ajax_zillametrics_get_preview_settings', array( $this, 'ajax_get_preview_settings' ) );
        add_action( 'wp_ajax_zillametrics_generate_preview_url', array( $this, 'ajax_generate_preview_url' ) );
        // Add AJAX handler to deactivate license
        add_action( 'wp_ajax_zillametrics_deactivate_license', array( $this, 'ajax_deactivate_license' ) );
    }

    /**
     * Register top-level menu and submenus.
     */
    public function register_menu() {
        // Single top-level: Zillametrics Chat → combined settings
        add_menu_page(
            __( 'Zillametrics Chat', 'zillametrics-chat' ),
            __( 'Zillametrics Chat', 'zillametrics-chat' ),
            'manage_options',
            'zillametrics-chat',
            array( $this, 'load_license' ),
            ZILLAMETRICS_PLUGIN_URL . 'assets/images/dashicon.png',
            56
        );
    }

    /**
     * Enqueue admin assets only on our pages.
     *
     * @param string $hook Current admin hook.
     */
    public function enqueue_assets( $hook ) {
        if ( false === strpos( $hook, 'zillametrics' ) ) {
            return;
        }

        $style_version  = file_exists( ZILLAMETRICS_PLUGIN_DIR . 'assets/css/admin.css' ) ? filemtime( ZILLAMETRICS_PLUGIN_DIR . 'assets/css/admin.css' ) : ZILLAMETRICS_PLUGIN_VERSION;
        $script_version = file_exists( ZILLAMETRICS_PLUGIN_DIR . 'assets/js/admin.js' ) ? filemtime( ZILLAMETRICS_PLUGIN_DIR . 'assets/js/admin.js' ) : ZILLAMETRICS_PLUGIN_VERSION;

        wp_enqueue_style( 'zillametrics-admin', ZILLAMETRICS_PLUGIN_URL . 'assets/css/admin.css', array(), $style_version );
        wp_enqueue_script( 'zillametrics-admin', ZILLAMETRICS_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery' ), $script_version, true );

        // Get current settings
        $selected_bot = Zillametrics_Settings::get_selected_bot();
        $selected_page = Zillametrics_Settings::get_selected_page();

        // Pass data to JS if needed.
        wp_localize_script( 'zillametrics-admin', 'ZillaMetricsAdmin', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'zillametrics_admin_nonce' ),
            'selectedBotId' => isset( $selected_bot['id'] ) ? (string) $selected_bot['id'] : '',
            'selectedPage' => ! empty( $selected_page ) ? $selected_page : array(),
        ) );
    }

    /**
     * Handle AJAX bot activation toggle.
     */

    /**
     * Handle AJAX license authentication.
     */
    public function ajax_authenticate_license() {
        // Verify nonce
        if ( ! check_ajax_referer( 'zillametrics_admin_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'zillametrics-chat' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'zillametrics-chat' ) ) );
        }

        // Get license key from POST data
        $license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';

        if ( empty( $license_key ) ) {
            wp_send_json_error( array( 'message' => __( 'License key is required.', 'zillametrics-chat' ) ) );
        }

        // Authenticate license with ZillaMetrics API
        $auth_result = $this->authenticate_license_with_api( $license_key );

        if ( $auth_result['success'] ) {
            // Save license key if authentication successful
            $license_data = $auth_result['data'];
            // Normalize bots payload to expected keys: id, bot_name, chat_domain
            if ( isset( $license_data['bots'] ) && is_array( $license_data['bots'] ) ) {
                $normalized_bots = array();
                foreach ( $license_data['bots'] as $bot ) {
                    $normalized_bots[] = array(
                        'id'          => isset( $bot['id'] ) ? $bot['id'] : ( $bot['bot_id'] ?? '' ),
                        'bot_name'    => isset( $bot['bot_name'] ) ? $bot['bot_name'] : ( $bot['name'] ?? '' ),
                        'chat_domain' => isset( $bot['chat_domain'] ) ? $bot['chat_domain'] : ( $bot['domain'] ?? ( $bot['description'] ?? '' ) ),
                    );
                }
                $license_data['bots'] = $normalized_bots;
            }

            Zillametrics_Settings::set( 'license_key', $license_key );
            Zillametrics_Settings::set( 'license_status', 'valid' );
            Zillametrics_Settings::set( 'license_data', $license_data );
            
            wp_send_json_success( array( 
                'message' => __( 'License authenticated successfully!', 'zillametrics-chat' ),
                'data' => $license_data
            ) );
        } else {
            // Clear license data if authentication failed
            Zillametrics_Settings::set( 'license_status', 'invalid' );
            Zillametrics_Settings::delete( 'license_data' );
            
            wp_send_json_error( array( 
                'message' => $auth_result['message'] ?: __( 'License authentication failed. Please check your license key.', 'zillametrics-chat' )
            ) );
        }
    }

    /**
     * Handle AJAX request to fetch pages.
     */
    public function ajax_get_pages() {
        // Verify nonce
        if ( ! check_ajax_referer( 'zillametrics_admin_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'zillametrics-chat' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'zillametrics-chat' ) ) );
        }

        $formatted_pages = array();
        
        // Add homepage option
        $formatted_pages[] = array(
            'id' => 0,
            'title' => __( 'Homepage', 'zillametrics-chat' ),
            'url' => home_url( '/' ),
            'slug' => 'homepage',
            'type' => 'homepage'
        );
        
        // Get pages from WordPress
        $pages = get_pages( array(
            'post_status' => 'publish',
            'sort_column' => 'post_title',
            'sort_order' => 'ASC'
        ) );

        foreach ( $pages as $page ) {
            $formatted_pages[] = array(
                'id' => $page->ID,
                'title' => $page->post_title,
                'url' => get_permalink( $page->ID ),
                'slug' => $page->post_name,
                'type' => 'page'
            );
        }

        // Include recent posts
        $posts = get_posts( array(
            'post_status' => 'publish',
            'numberposts' => 20, // Limit to prevent too many results
            'orderby' => 'title',
            'order' => 'ASC'
        ) );

        foreach ( $posts as $post ) {
            $formatted_pages[] = array(
                'id' => $post->ID,
                'title' => $post->post_title . ' (Post)',
                'url' => get_permalink( $post->ID ),
                'slug' => $post->post_name,
                'type' => 'post'
            );
        }

        // Get custom post types
        $custom_post_types = get_post_types( array(
            'public' => true,
            '_builtin' => false
        ), 'names' );
        
        foreach ( $custom_post_types as $post_type ) {
            $custom_posts = get_posts( array(
                'post_type' => $post_type,
                'post_status' => 'publish',
                'numberposts' => 10, // Limit custom post types
                'orderby' => 'title',
                'order' => 'ASC'
            ) );
            
            foreach ( $custom_posts as $custom_post ) {
                $post_type_label = ucfirst( str_replace( array( '-', '_' ), ' ', $post_type ) );
                $formatted_pages[] = array(
                    'id' => $custom_post->ID,
                    'title' => $custom_post->post_title . ' (' . $post_type_label . ')',
                    'url' => get_permalink( $custom_post->ID ),
                    'slug' => $custom_post->post_name,
                    'type' => $post_type
                );
            }
        }

        // Sort all pages by title
        usort( $formatted_pages, function( $a, $b ) {
            return strcmp( $a['title'], $b['title'] );
        });

        wp_send_json_success( array(
            'pages' => $formatted_pages,
            'total' => count( $formatted_pages )
        ) );
    }

    /**
     * Handle AJAX request to save preview settings.
     */
    public function ajax_save_preview_settings() {
        // Verify nonce
        if ( ! check_ajax_referer( 'zillametrics_admin_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'zillametrics-chat' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'zillametrics-chat' ) ) );
        }

        // Get settings from POST data
        $settings = array();
        
        if ( isset( $_POST['selected_bot'] ) ) {
            $settings['selected_bot'] = sanitize_text_field( wp_unslash( $_POST['selected_bot'] ) );
        }
        
        if ( isset( $_POST['selected_page'] ) && is_array( $_POST['selected_page'] ) ) {
            $page_data = array_map( 'sanitize_text_field', wp_unslash( $_POST['selected_page'] ) );
            $settings['selected_page'] = array(
                'id' => sanitize_text_field( $page_data['id'] ?? '' ),
                'title' => sanitize_text_field( $page_data['title'] ?? '' ),
                'url' => esc_url_raw( $page_data['url'] ?? '' ),
                'type' => sanitize_text_field( $page_data['type'] ?? '' )
            );
        }

        // Save settings
        $success = Zillametrics_Settings::save_preview_settings( $settings );

        if ( $success ) {
            wp_send_json_success( array( 
                'message' => __( 'Preview settings saved successfully!', 'zillametrics-chat' ),
                'settings' => $settings
            ) );
        } else {
            wp_send_json_error( array( 
                'message' => __( 'Failed to save preview settings.', 'zillametrics-chat' )
            ) );
        }
    }

    /**
     * Handle AJAX request to get preview settings.
     */
    public function ajax_get_preview_settings() {
        // Verify nonce
        if ( ! check_ajax_referer( 'zillametrics_admin_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'zillametrics-chat' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'zillametrics-chat' ) ) );
        }

        // Get current settings
        $selected_bot = Zillametrics_Settings::get_selected_bot();
        $selected_page = Zillametrics_Settings::get_selected_page();
        $available_bots = Zillametrics_Settings::get_available_bots();

        wp_send_json_success( array(
            'selected_bot' => $selected_bot,
            'selected_page' => $selected_page,
            'available_bots' => $available_bots
        ) );
    }

    /**
     * Generate a nonce-protected preview URL.
     */
    public function ajax_generate_preview_url() {
        if ( ! check_ajax_referer( 'zillametrics_admin_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'zillametrics-chat' ) ) );
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'zillametrics-chat' ) ) );
        }

        $bot_id = isset( $_POST['bot_id'] ) ? sanitize_text_field( wp_unslash( $_POST['bot_id'] ) ) : '';
        $page_url = isset( $_POST['page_url'] ) ? esc_url_raw( wp_unslash( $_POST['page_url'] ) ) : '';

        if ( empty( $bot_id ) || empty( $page_url ) ) {
            wp_send_json_error( array( 'message' => __( 'Missing preview parameters.', 'zillametrics-chat' ) ) );
        }

        $preview_url = zillametrics_generate_preview_url( $bot_id, $page_url );

        wp_send_json_success( array(
            'preview_url' => $preview_url,
        ) );
    }

    /**
     * Deactivate the license and clear stored data.
     */
    public function ajax_deactivate_license() {
        // Verify nonce
        if ( ! check_ajax_referer( 'zillametrics_admin_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'zillametrics-chat' ) ) );
        }

        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'zillametrics-chat' ) ) );
        }

        // Clear license related data
        Zillametrics_Settings::set( 'license_key', '' );
        Zillametrics_Settings::set( 'license_status', 'invalid' );
        Zillametrics_Settings::delete( 'license_data' );
        Zillametrics_Settings::set( 'preview_selected_bot', '' );
        Zillametrics_Settings::set( 'preview_selected_page', array( 'id' => '', 'title' => '', 'url' => '', 'type' => '' ) );

        wp_send_json_success( array( 'message' => __( 'License deactivated.', 'zillametrics-chat' ) ) );
    }

    /**
     * Authenticate license key with ZillaMetrics API.
     *
     * @param string $license_key The license key to authenticate.
     * @return array Authentication result with success status and data/message.
     */
    private function authenticate_license_with_api( $license_key ) {
        // For testing purposes, allow specific test keys
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            // Test license keys for development
            if ( $license_key === 'test-valid-key' ) {
                return array(
                    'success' => true,
                    'data' => array(
                        'plan' => 'Professional',
                        'expires_at' => gmdate( 'Y-m-d H:i:s', strtotime( '+1 year' ) ),
                        'domain' => home_url(),
                        'status' => 'active',
                        'script_url' => 'https://cdn.zillametrics.com/bot.js',
                        'bots' => array(
                            array(
                                'id' => 'pro-bot-1',
                                'bot_name' => 'Professional Bot 1',
                                'chat_domain' => 'professional.example.com'
                            ),
                            array(
                                'id' => 'pro-bot-2', 
                                'bot_name' => 'Professional Bot 2',
                                'chat_domain' => 'support.example.com'
                            ),
                            array(
                                'id' => 'pro-bot-3',
                                'bot_name' => 'Professional Bot 3', 
                                'chat_domain' => 'sales.example.com'
                            )
                        )
                    )
                );
            } elseif ( $license_key === 'test-invalid-key' ) {
                return array(
                    'success' => false,
                    'message' => __( 'Invalid test license key.', 'zillametrics-chat' )
                );
            }
        }

        // API endpoint for license authentication
        $api_url = 'https://api.zillametrics.com/wp-json/zillametrics/v1/license/verify-basic';
        
        // Prepare request data
        $request_data = array(
            'license_key' => $license_key,
            'product_id' => 1
        );

        // Make API request
        $response = wp_remote_post( $api_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'ZillaMetrics-WordPress-Plugin/' . ZILLAMETRICS_PLUGIN_VERSION
            ),
            'body' => wp_json_encode( $request_data )
        ) );

        // Handle request errors
        if ( is_wp_error( $response ) ) {
            // In development mode, provide a fallback response for testing
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                return array(
                    'success' => false,
                    'message' => __( 'API connection failed (development mode). Use "test-valid-key" for testing.', 'zillametrics-chat' ),
                    'response' => array(
                        'error' => $response->get_error_message(),
                    ),
                );
            }
            
            return array(
                'success' => false,
                'message' => __( 'Unable to connect to ZillaMetrics servers. Please try again later.', 'zillametrics-chat' ),
                'response' => array(
                    'error' => $response->get_error_message(),
                ),
            );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $response_data = json_decode( $response_body, true );

        // Handle successful authentication
        if ( $response_code === 200 && ! empty( $response_data['valid'] ) && $response_data['valid'] ) {
            return array(
                'success' => true,
                'data' => $response_data['license_details'] ?? array(),
            );
        }

        // Handle authentication failure
        $error_message = '';
        if ( isset( $response_data['message'] ) ) {
            $error_message = $response_data['message'];
        } elseif ( $response_code === 401 ) {
            $error_message = __( 'Invalid license key.', 'zillametrics-chat' );
        } elseif ( $response_code === 403 ) {
            $error_message = __( 'License key is not authorized for this domain.', 'zillametrics-chat' );
        } elseif ( $response_code >= 500 ) {
            $error_message = __( 'Server error. Please try again later.', 'zillametrics-chat' );
        } else {
            $error_message = __( 'License authentication failed.', 'zillametrics-chat' );
        }

        return array(
            'success' => false,
            'message' => $error_message,
            'response' => array(
                'code' => $response_code,
                'body' => $response_body,
                'data' => $response_data,
            ),
       );
    }

    /** Loaders for each view file **/
    public function load_about()   { include ZILLAMETRICS_PLUGIN_DIR . 'admin/views/about.php'; }
    public function load_license() { include ZILLAMETRICS_PLUGIN_DIR . 'admin/views/license.php'; }
    public function load_preview() { include ZILLAMETRICS_PLUGIN_DIR . 'admin/views/license.php'; }
}

endif;
