<?php
namespace ZeroV99\ShipmentTracking\tasks;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;

/**
 * Handles automatic completion of orders in the "Shipped" status after a specified number of days.
 *
 * This class schedules and executes a recurring task to check for orders in the "Shipped" status
 * and automatically marks them as "Completed" after a configurable number of days.
 */
class ZeroV99_Auto_Complete_Shipped_Orders
{
    /**
     * Constructor.
     *
     * Hooks into WordPress initialization to schedule the auto-complete task and defines
     * the callback function to execute the task.
     */
    public function __construct()
    {

        add_action('init', [$this, 'schedule_auto_complete_orders']);
        add_action(ZeroV99_Shipment_Tracking_Config::TASK_AUTO_COMPLETE_SHIPPED_ORDERS, [$this, 'auto_complete_shipped_orders']);
    }

    /**
     * Schedules the recurring auto-complete task.
     *
     * This function checks if the task is already scheduled and, if not, schedules it to run
     * hourly starting at the next full hour.
     */
    public function schedule_auto_complete_orders()
    {

        if (!as_next_scheduled_action(ZeroV99_Shipment_Tracking_Config::TASK_AUTO_COMPLETE_SHIPPED_ORDERS)) {

            // Calculate the next full hour for the first run
            $first_run = strtotime(gmdate('Y-m-d H:00:00', current_time('timestamp')) . ' +1 hour');

            as_schedule_recurring_action(
                $first_run,
                HOUR_IN_SECONDS,
                ZeroV99_Shipment_Tracking_Config::TASK_AUTO_COMPLETE_SHIPPED_ORDERS
            );
        }
    }

    /**
     * Automatically completes orders in the "Shipped" status after the configured number of days.
     *
     * This function checks all orders in the "Shipped" status and marks them as "Completed" if
     * the specified number of days have passed since they were shipped. It logs the process for
     * debugging purposes.
     */
    public function auto_complete_shipped_orders()
    {

        if (get_option(ZeroV99_Shipment_Tracking_Config::OPTION_ENABLE_AUTO_COMPLETE) !== 'yes') {
            return;
        }

        $auto_complete_days = (int) get_option(ZeroV99_Shipment_Tracking_Config::OPTION_AUTO_COMPLETE_DAYS, 3);

        $now = current_time('timestamp');

        $orders = wc_get_orders([
            'status' => 'wc-shipped',
            'limit'  => -1,
        ]);

        foreach ($orders as $order) {

            $shipped_date = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_LAST_SHIPPED_DATE, true);

            if (!$shipped_date) {
                continue; // if no date, skip this order
            }

            $shipped_timestamp = strtotime($shipped_date);
            if ($shipped_timestamp === false) {
                continue; // if invalid date, skip this order
            }

            $completion_timestamp = strtotime("+$auto_complete_days days", $shipped_timestamp);

            if ($completion_timestamp <= $now) {
                $message = sprintf( /* translators: %d is the number of days before the order is completed */
                    __('Order ended automatically after %d days in "Shipped" status.', 'zerov99-shipment-tracking'),
                    $auto_complete_days
                );
                $order->update_status('completed', $message);
            }
        }
    }
}


